;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/misc/RCS/scroll-widget.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:32 $
;;;

;;;
;;;  Scroll-widget:  a scrollable collection.
;;;
;;;  A scroll-widget provides a "viewer window" into a collection of
;;;  arbitrary size and provides controls for scrolling horizontally and
;;;  vertically through the contents of the collection through this viewer.  
;;;  The controls are scroll-bars (horizontal and/or/nor vertical) equipped 
;;;  with single-step, page/screen, and dynamic drag scrolling.
;;;
;;;  The interface for creating a scroll-widget is exactly the same as that
;;;  used in creating a normal collection.  The :children argument 
;;;  is used to specify the contents of the scroll-widget.  If the a scroll-
;;;  widget is created with a :children list of length one, the scroll-widget
;;;  will manage the child specially as follows:  the size of the child will 
;;;  always be at least the size of the viewer window.
;;;
;;;  When a scroll-widget is created with multiple children, it creates a 
;;;  sub-collection called a "holder" with the indicated children and other
;;;  properties (eg. gm).  The holder can be accessed through the value slot.
;;;
;;;  For example:
;;;
;;;	(setq sw (make-scroll-widget 
;;;		  :gm 'packed-gm
;;;		  :children '((make-button :value "Press Me"
;;;			       :geom-spec :top)
;;;			      (make-entry-widget :value "Hello"
;;;			       :geom-spec :fill))))
;;;	<picasso> (gm sw)
;;;	
;;;	ANCHOR-GM
;;;	<picasso> (gm (value sw))
;;;
;;;	PACHED-GM
;;;
;;;
;;;  SLOTS OF INTEREST
;;;
;;;	horizontal-step:	amount in pixels to scroll right/left.
;;;
;;;	vertical-step:		amount in pixels to scroll up/down.
;;;
;;;	value:			handle on actual widget/collection being
;;;				displayed through viewer.  VALUE never changes 
;;;				once the scroll-widget is created.  If the
;;;				scroll-widget is created with a single child,
;;;				the VALUE slot will contain the child.  
;;;				Otherwise VALUE contains the holder (see above
;;;				notes on holder).
;;;
;;;  INITARGS OF INTEREST
;;;
;;;	:horiz-scroll-bar	if non-nil, a horizontal scroll-bar 
;;;				is created.  Default t.
;;;
;;;	:vert-scroll-bar	if non-nil, a vertical scroll-bar 
;;;				is created.  Default t.
;;;
;;;
;;;     :horiz-scroll-pos       :top or :bottom (default) controls
;;;                             placement of horiz scroll bar.
;;;
;;;     :vert-scroll-bar        :left or :right (default) controls
;;;                             placement of the vertical scroll bar.
;;;
;;;

(in-package "PT")

;;;
;;;  define the scroll-widget class
;;;
(defclass scroll-widget (collection-widget)
  ((horizontal-step
    :initarg :horizontal-step
    :initform 10
    :type integer
    :accessor horizontal-step) 
   (vertical-step
    :initarg :vertical-step
    :initform 10
    :type integer
    :accessor vertical-step)
   (horiz-scroll-bar
    :initform t
    :type t
    :reader horiz-scroll-bar)
   (vert-scroll-bar
    :initform t
    :type t
    :reader vert-scroll-bar)
   (horiz-scroll-pos
    :initform :bottom
    :type t)
   (vert-scroll-pos
    :initform :right
    :type t)
   (gm :initform 'anchor-gm)))

(defmethod (setf value) (val (self scroll-widget))
  (declare (ignore val))
  (warn "You can't set the value of a scroll-widget (yet)"))

(defun make-scroll-widget (&rest keys &aux kids) 
  (setq kids (getf keys :children))
  (remf keys :children)
  (setf (getf keys :kids) kids)
  (apply #'make-instance 'scroll-widget :allow-other-keys t keys))

(defmethod new-instance ((self scroll-widget)
			 &rest args
			 &key kids 
			 (horiz-scroll-bar t)
			 (vert-scroll-bar t)
                         (horiz-scroll-pos :bottom)
                         (vert-scroll-pos :right)
			 &allow-other-keys 
			 &aux holder (r-pad 0) (b-pad 0) (l-pad 0) (t-pad 0))
  
  (call-next-method)
  
  (if (getf args :children)
      (error "Scroll-widget:  when bypassing the make-scroll-widget function, specify :KIDS instead of :CHILDREN"))

  (if horiz-scroll-bar
      (if (eq horiz-scroll-pos :bottom)
          (setq b-pad 24)
        (setq t-pad 24)))

  (if vert-scroll-bar
      (if (eq vert-scroll-pos :right)
          (setq r-pad 24)
        (setq l-pad 24)))

  
  ;;  Put children (kids) in collection if needed
  (when (> (length kids) 1) 
	(remf args :parent)
	(setf (getf args :children) kids)
	(setq kids
	      (list (cons 'make-collection-widget args))))
  
  ;;  Create scrolling region
  (setq holder 
	(make-collection-widget 
	 :parent self
	 :border-width 0
	 :background :parent-relative
	 :gm 'sw-gm
	 :geom-spec `(:anchor (:left ,l-pad :top ,t-pad :right ,r-pad
                                     :bottom ,b-pad))
	 :children kids))
  
  ;;  Create scroll-bars
  (when vert-scroll-bar 
	(setq vert-scroll-bar
	      (make-scroll-bar
	       :parent self
	       :orientation :vertical
	       :geom-spec  `(:anchor (:top 0 ,vert-scroll-pos 0 :bottom 0))
	       :base-width 15
	       :lower-limit 0
	       :data self
	       :prev-line-func '(sw-up self (data self))
	       :next-line-func '(sw-down self (data self))
	       :next-page-func '(sw-down-screen self (data self))
	       :prev-page-func '(sw-up-screen self (data self))
	       :moved-func '(sw-move-vert self (data self) event)))
	(setf (base-width vert-scroll-bar) 20))
  
  (when horiz-scroll-bar 
	(setq horiz-scroll-bar
	      (make-scroll-bar
	       :parent self
	       :orientation :horizontal
	       :geom-spec `(:anchor (:left ,l-pad
                                     :right ,r-pad
                                     ,horiz-scroll-pos 0))
	       :base-height 15
	       :lower-limit 0
	       :data self
	       :prev-line-func '(sw-left self (data self))
	       :next-line-func '(sw-right self (data self))
	       :next-page-func '(sw-right-screen self (data self))
	       :prev-page-func '(sw-left-screen self (data self))
	       :moved-func '(sw-move-horiz self (data self) event)))
	(setf (base-height vert-scroll-bar) 20))
  
  (setf (slot-value self 'value) (car (children holder))
	(slot-value self 'gm) 'anchor-gm
	(slot-value self 'vert-scroll-bar) vert-scroll-bar
	(slot-value self 'horiz-scroll-bar) horiz-scroll-bar))

(defun sw-update-cache (self &aux vsb hsb form)
  (setq vsb (vert-scroll-bar self)
	hsb (horiz-scroll-bar self)
	form (value self))
  (if vsb
      (setf (upper-limit vsb) (virtual-height form)
	    (slider-size vsb) (height (parent form))
	    (slider-location vsb) (abs (y-offset form))))
  (if hsb
      (setf (upper-limit hsb) (virtual-width form)
	    (slider-size hsb) (width (parent form))
	    (slider-location hsb) (abs (x-offset form)))))

(defmethod resize-window-handler ((self scroll-widget))
  (call-next-method)
  (sw-update-cache self))

;;;
;;;	Vertical Scrolling functions
;;;

(defun sw-up (sb self &aux form step new)
  (setq form (value self)
	step (vertical-step self))
  (setq new (max (- (y-offset form) step)
		 (- (height (parent form)) (virtual-height form))))
  (setf (y-offset form) new
	(slider-location sb) (abs new))
  (repaint form))

(defun sw-down (sb self &aux form step new)
  (setq form (value self)
	step (vertical-step self))
  (setq new (min 0 (+ (y-offset form) step)))
  (setf (y-offset form) new
	(slider-location sb) (abs new))
  (repaint form))

(defun sw-down-screen (sb self &aux form step new)
  (setq form (value self))
  (setq step (height (parent form)))
  (setq new (max (- (y-offset form) step)
		 (- step (virtual-height form))))
  (setf (y-offset form) new
	(slider-location sb) (abs new))
  (repaint form))

(defun sw-up-screen (sb self &aux form step new)
  (setq form (value self))
  (setq step (height (parent form)))
  (setq new (min 0 (+ (y-offset form) step)))
  (setf (y-offset form) new
	(slider-location sb) (abs new))
  (repaint form))

(defun sw-move-vert (sb self ev &aux form step old new)
  (setq form (value self)
	step (vertical-step self))
  (setq old (abs (y-offset form)))
  (drag-scroll-bar sb
		   #'(lambda (sb self)
                             (declare (ignore self))
			     (setq new (round (slider-location sb)))
			     (if (< old new)
				 ;;  scroll up
				 (setq new (max (- (y-offset form) step)
						(- (height (parent form)) 
						   (virtual-height form))))
				 ;;  scroll down
				 (setq new (min 0 (+ (y-offset form) step))))
			     (setf (y-offset form) new)
			     (setq old (abs new)))
		   ev))
;;;
;;;	Horizontal Scrolling functions
;;;

(defun sw-left (sb self &aux form step new)
  (setq form (value self)
	step (horizontal-step self))
  (setq new (max (- (x-offset form) step)
		 (- (width (parent form)) (virtual-width form))))
  (setf (x-offset form) new
	(slider-location sb) (abs new))
  (repaint form))

(defun sw-right (sb self &aux form step new)
  (setq form (value self)
	step (horizontal-step self))
  (setq new (min 0 (+ (x-offset form) step)))
  (setf (x-offset form) new
	(slider-location sb) (abs new))
  (repaint form))

(defun sw-right-screen (sb self &aux form step new)
  (setq form (value self))
  (setq step (width (parent form)))
  (setq new (max (- (x-offset form) step)
		 (- step (virtual-width form))))
  (setf (x-offset form) new
	(slider-location sb) (abs new))
  (repaint form))

(defun sw-left-screen (sb self &aux form step new)
  (setq form (value self))
  (setq step (width (parent form)))
  (setq new (min 0 (+ (x-offset form) step)))
  (setf (x-offset form) new
	(slider-location sb) (abs new))
  (repaint form))

(defun sw-move-horiz (sb self ev &aux form step old new)
  (setq form (value self)
	step (horizontal-step self))
  (setq old (abs (x-offset form)))
  (drag-scroll-bar sb
		   #'(lambda (sb self)
			     (declare (ignore self))
			     (setq new (round (slider-location sb)))
			     (if (< old new)
				 ;;  scroll up
				 (setq new (max (- (x-offset form) step)
						(- (width (parent form)) 
						   (virtual-width form))))
				 ;;  scroll down
				 (setq new (min 0 (+ (x-offset form) step))))
			     (setf (x-offset form) new)
			     (setq old (abs new)))
		   ev))
;;;
