;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/misc/RCS/meter-widget.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:32 $
;;;
;;; METER VISUALIZATION AND FUNCTION
;;;
;;; A meter widget consists of a meter-slider and three numeric-fields.
;;; Meter widgets are used as one-dimensional indicators, similar to the 
;;; indicator (slider-bar) of a scroll-bar.  The indicator is the meter-slider
;;; which consists of a diamond-shaped locator and a horizontal grid.  The
;;; three numeric-fields specify the lower and upper bounds and the current
;;; position of the locator.  
;;;
;;; INTERFACE FOR EDITING BOUNDS
;;;
;;; The lower and upper bounds may be EDITED by clicking on them, typing in 
;;; the new value, and pressing return.  The current-value can only be set
;;; by the programmer.
;;;
;;; ACCESSORS AND INSTANTIATION
;;;
;;; The following accessors are provided for customization of the meter-widget:
;;; 	low	-	lower bound (number)
;;;	high	-	upper bound (number)
;;;	value	-	current position of locator relative to low and high
;;;				(number)
;;;	increment  -	grid increment (number)
;;;	update-flag -	used when more than one of above values is set at
;;;			once (for optimization)
;;;			for example:
;;;				(setf (update-flag mw) nil)
;;;				(setf (low mw) 0
;;;				      (high mw) 100
;;;				      (value mw) 25)
;;;				(setf (update-flag mw) t)
;;;
;;; NOTE: when value is nil, the locator is not drawn.
;;;

(in-package "PT")

(defclass meter-slider (widget)
  ((low
    :initarg :low 
    :initform 0
    :type number
    :reader low)
   (high
    :initarg :high 
    :initform 0
    :type number
    :reader high)
   (value
    :initarg :value 
    :initform 0
    :type number
    ;; :reader value
    )
   (increment
    :initarg :increment 
    :initform 5
    :type number
    :reader increment)
   (pad
    :initarg :pad 
    :initform 0
    :type integer
    :reader pad)
   (update-flag				;;  Like repaint-flag or repack-flag.
    :initarg :update-flag				 
    :initform t				;;  Use when changing more than one
    :type atom				;;  attribute at a time.
    :reader update-flag)
;;	-----------------------------------------------------------------
;;					;;  Internal use only
   (scale
    :initform 1
    :type number
    :reader scale)
   (grid-vertices
    :initform nil
    :type t
    :accessor grid-vertices)
   (gc-spec :initform '((gc-res (:foreground "white" :background "black"))))
   (foreground :initform "white")
   (background :initform "black")
   (border-width :initform 0)
   (base-beight :initform 25)))

(defun make-meter-slider (&rest args)
  (apply #'make-instance 'meter-slider :allow-other-keys t args))

;;;
;;;	Accessors signal repaint
;;;

(defmethod value ((self meter-slider)
		  &key 
		  &allow-other-keys)
  (slot-value self 'value))

(defmethod (setf value) (val (self meter-slider))
  (setf (slot-value self 'value) val)
  (when (update-flag self)
	(repaint self)))

(defmethod (setf low) (val (self meter-slider))
  (setf (slot-value self 'low) val)
  (when (update-flag self)
	(mw-update-cache self :invalidate t)
	(repaint self)))

(defmethod (setf high) (val (self meter-slider))
  (setf (slot-value self 'high) val)
  (when (update-flag self)
	(mw-update-cache self :invalidate t)
	(repaint self)))

(defmethod (setf increment) (val (self meter-slider))
  (setf (slot-value self 'increment) val)
  (when (update-flag self)
	(mw-update-cache self :invalidate t)
	(repaint self)))

(defmethod (setf update-flag) (val (self meter-slider))
  (setf (slot-value self 'update-flag) val)
  (when val
	(mw-update-cache self)))

;;;
;;;	Function to calculate and cache vertex and scaling info
;;;

(defun mw-update-cache (self &key (invalidate nil) 
			     &aux pad scale vlist len inc incr diff high low)
  ;;	First calculate and store scale
  (setq pad (+ (pad self) 8)
	high (high self)
	low (low self)
	inc (increment self)) 
  (unless (and low high inc) (return-from mw-update-cache))
  (setq scale (/ (max 0 (- (width self) pad pad))
		 (max 1 (- high low))))
  
  (cond (invalidate
	 ;;	Create vertex-list
	 (setq incr (round (* inc scale)))
	 (setq diff (max 0 (- high low)))
	 (setq len (max 8 (round (* (1+ (/ diff inc)) 4))))
	 (when (> (* incr len) (- (width self) pad pad))
	       (setq incr (1- incr)))
	 (setq diff (+ diff pad))
	 (do ((x pad (+ x inc)))
	     ((> x diff))
	     (setq vlist (nconc vlist (list incr 0 0 -3 0 6 0 -3))))
	 (setf (car vlist) pad
	       (cadr vlist) 20)
	 (setq inc incr)
	 (setq len (* 2 len)))
	(t
	 ;;	Reposition vertices
	 (setq vlist (grid-vertices self)
	       inc (round (* inc scale)))
	 (setq len (length vlist))
	 (when (> (* inc (/ len 4)) (- (width self) pad pad))
	       (setq inc (1- inc)))
	 (do ((x 8 (+ x 8)))
	     ((>= x len))
	     (setf (nth x vlist) inc))))
  
  (setf (grid-vertices self) vlist)
  (setf (slot-value self 'scale)
	(/ (* inc (1- (/ len 8)))
	   (max 1 (- high low)))))

;;;
;;;	Resizing should just update the cache
;;;

(defmethod resize-window-handler ((self meter-slider))
  (mw-update-cache self)
  (repaint self))

;;;
;;;	Instantiate a new meter-slider
;;;

(defmethod new-instance ((self meter-slider) &rest args)
  (declare (ignore args))
  (call-next-method)

  ;;	Set up bitmaps if necessary 
  (make-image :name "meter-slider" :file "gray-triang.bitmap")

  ;;	Set up vertex-list
  (mw-update-cache self :invalidate t))

;;;
;;;	Draw meter-slider
;;;

(defmethod do-repaint ((self meter-slider)
		       &key 
		       &allow-other-keys
		       &aux res gc loc w low high)
  (call-next-method)
  
  ;;	Draw indicator
  (setq res (res self)
	gc (gc-res self)
	loc (value self)
	low (low self)
	high (high self))
  (when (and loc low high (<= low loc high))
	(setq loc (max 0 (round (* (scale self) (- loc low)))))
	(xlib:put-image res gc 
			(res (get-image "meter-slider"))
			:x (1+ loc) :y 1
			:bitmap-p t))
  
  ;;	Draw grid
  (xlib:draw-lines res gc (grid-vertices self) :relative-p t)
  (setq w (width self))
  (xlib:draw-rectangle res gc (- w 2) 2 1 29)
  (xlib:draw-point res gc (- w 1) 1))

;;;
;;;	Meter-widget class
;;;

(defclass meter-widget (collection-widget)
  ((low-win
    :initarg :low-win 
    :initform nil
    :type num-entry
    :accessor low-win)
   (high-win
    :initarg :high-win 
    :initform nil
    :type num-entry
    :accessor high-win)
   (current-win
    :initarg :current-win 
    :initform nil
    :type simple-text-gadget
    :accessor current-win)
   (meter-slider
    :initarg :meter-slider 
    :initform nil
    :type meter-slider
    :accessor meter-slider)
   (current-field
    :initarg :current-field 
    :initform nil
    :type window)
   (background :initform "gray75")
   (base-width :initform 20)
   (base-height :initform 50)
   (border-width :initform 0)
   (event-mask :initform '(:exposure :button-press :key-press))
   (gm :initform 'anchor-gm)))

(defun make-meter-widget (&rest args)
  (apply #'make-instance 'meter-widget :allow-other-keys t args))

;;;
;;;	Accessor Methods
;;;

(defmethod value ((self meter-widget) 
		  &key 
		  &allow-other-keys
		  &aux cw)
  (when (setq cw (current-win self))
	(read-from-string (car (value cw)))))

(defmethod (setf value) (val (self meter-widget) &aux mw)
  (when (setq mw (meter-slider self))
	(setf (value mw) val)))

(defmethod low ((self meter-widget))
  (value (low-win self)))

(defmethod (setf low) (val (self meter-widget))
  (setf (low (meter-slider self)) val)
  (setf (value (low-win self)) val))

(defmethod high ((self meter-widget))
  (value (high-win self)))

(defmethod (setf high) (val (self meter-widget))
  (setf (high (meter-slider self)) val)
  (setf (value (high-win self)) val))

(defmethod increment ((self meter-widget))
  (increment (meter-slider self)))

(defmethod (setf increment) (val (self meter-widget))
  (setf (increment (meter-slider self)) val))

(defmethod (setf update-flag) (val (self meter-widget))
  (setf (update-flag (meter-slider self)) val))

;;;
;;;	Event stuff
;;;

(defun mw-toggle-event-mask (win &aux em)
  (setq em (event-mask win))
  (setf (event-mask win)
	(append (car (last em)) (list (butlast em)))))

(defhandler char-dispatch ((self meter-widget) &rest args &aux cur mapping
			   &default :key-press)
  (when (setq cur (slot-value self 'current-field))
	(setq mapping (lookup-event-mapping cur (descriptor args)))
	(when (function-p mapping)
	      (apply mapping (cons cur args)))))

(defhandler select ((self meter-widget) &key child &allow-other-keys &aux cur
		    &default :button-press)
  (setq cur (slot-value self 'current-field))
  (when cur 
	(deactivate cur)
	(mw-toggle-event-mask cur)
	(setf (slot-value self 'current-field) nil))
  (unless (or (null child) 
	      (not (typep (setq child (find-window child (display self))) 
			  'num-entry))) 
	  (setf (slot-value self 'current-field) child)
	  (mw-toggle-event-mask child)
	  (activate child)))

;;;
;;;	Instantiate a meter-widget
;;;

(defmethod new-instance ((self meter-widget)
			 &key
			 (value 0)
			 (font	(get-font))
			 (low	0)
			 (high	0)
			 (increment	5)
			 &allow-other-keys
			 &aux ms cw)
  (call-next-method)
  
  (setf (meter-slider self)
	(setq ms (make-meter-slider :low low :high high :value value 
				    :increment increment :parent self 
				    :base-height 26
				    :geom-spec '(:anchor 
						 (:left 0 :top 0 :right 0)
						 :arrow (:horiz)))))
  (setf (low-win self)
	(make-num-entry :value low 
			   :font font
			   :parent self 
			   :geom-spec '(0 0 1/3 1 
					:anchor (:left 0 :top 26 :bottom 0)
					:arrow (:horiz))
			   :gray t
			   :current nil
			   :event-mask '(:exposure (:key-press
						    :button-press
						    :expose-region
						    :double-click
						    :button-1-motion))
			   :return-function `(setf (low ',ms) (value self))))
  (setf (high-win self)
	(make-num-entry :value high 
			   :font font
			   :parent self
			   :geom-spec '(2/3 0 1/3 1 
					:anchor (:right 0 :top 26 :bottom 0)
					:arrow (:horiz))
			   :gray t
			   :current nil
			   :event-mask '(:exposure (:key-press
						    :button-press
						    :expose-region
						    :double-click
						    :button-1-motion))
			   :return-function `(setf (high ',ms) (value self))))
  (setf (current-win self)
	(setq cw
	      (make-text-gadget :font font :value (princ-to-string value) 
				:background "black" :foreground "white"
				:gray t
				:parent self 
			   :geom-spec '(1/3 0 1/3 1 
					:anchor (:top 26 :bottom 0)
					:arrow (:horiz)))))
  (bind-slot 'value cw `(princ-to-string (var value ,ms))))
