
;;;
;;;  PANEL-SUPPORT.CL
;;;  Support functions for manipulating the panel cache and choosing
;;;  and setting up a panel when a new node is to be displayed.
;;;

(in-package "PT")


;;;  Functions for adjusting the panel cache size:
(defun new-panel-cache (&optional (n *num-panels*))
  "sets up list of panels to be used and reused in displaying nodes"
  (when *hip-panels*
        (remove-from-panel-cache (length *hip-panels*)))
  (add-to-panel-cache n))

(defun update-panel-cache-size (new-length)
  "adds or removes panels from cache according to desired new-length"
  (when (not (= new-length *num-panels*))
        (if (> new-length *num-panels*)
            (add-to-panel-cache (- new-length *num-panels*))
          (remove-from-panel-cache (- *num-panels* new-length)))
        (setq *num-panels* new-length)))
      
(defun add-to-panel-cache (&optional (n 3))
  "adds n panels to panel cache"
  (dotimes (i n)
     (let ((p (generate-hyper-panel)))
       (push p *hip-panels*)
       ;; oughta save ~10 sec. later:
       (attach p))))

(defun remove-from-panel-cache (&optional (n 1))
  "removes n panels from panel cache"
  (dotimes (i n)
       (detach (pop *hip-panels*))))

;;;  Convenient accessors:
(defun current-hip-panel ()
  *current-hip-panel*)

(defun chp () 
  *current-hip-panel*)
(defun cn ()
  #!node@(chp))
(defun chw ()
  #!hw@(chp))
(defun ch ()
  #!*current-hyperdoc*)

(defun make-current (panel)
  "makes panel the current-hip-panel, adjusting the panel cache
   appropriately; returns panel"
  (setq *hip-panels* 
        (append (remove panel *hip-panels*) (list panel)))
  (if (exposed-p panel) (raise panel))  ;; make sure it's fully visible
  (format t "~%Changing current panel to ~a~%" panel)
  (setq *current-hip-panel* panel))

;;;
;;;  Choosing and setting up panels:
;;;

(defun choose-panel ()
  "returns a panel to be used to display the next node"
  (let ((next (find-if-not #'panel-locked? *hip-panels*))) 
    (cond ((not next)  ;; make sure there's one to choose
           (error "Can't choose a panel: all locked"))
          ;; make it current:
          (t (make-current next)))))

;;  Function to call if node isn't already assigned to a panel:
(defun use-new-panel (&key node) 
  (let ((p (choose-panel)))
    (when p
          ;; need a way to find out if panel is iconified - slot
          ;; doesn't seem to be getting set!!!
          (if (or (exposed-p p)) ;;; (not (invisible-p p))
            ;;  (setup-panel-for-new-node p node)
	      (reassign-panel p node)
            (call p :node node)))
    p))

(defun reassign-panel (p node)
  (close-node #!node@p)
  (call p :node node)
  (make-current p))

(defun setup-panel-for-new-node (p node) 
  "sets up panel when its node is reassigned"
  (let* ((env (if (null #!hw@p)  ;; first time being called, so children not yet
                                 ;; migrated up from form
                  (get-form p)
                p))              ;; else can look in panel itself
         (widget (get-display-widget node env)))
    (setf #!node@p node)
    (setf #!node-name@p (string (name node)))
    (setf #!hw@p widget)
    (setf (node #!hw@p) node)
    (unbind-fast #!rcpt@p)
    (if (eq (type node) 'video)
        (setf #!rcpt@p
              (bind-slot 'current-entry #!video-wid@p
                         `(var curr-key
                               :ref
                               ,#!video-panel@(current-tool))
                         :receipt t)))
   ;  (scroll-to #!hw@p offset)
    (if (concealed-p p)
        (expose p))
    ))

;;; REOPEN-PANEL is called when the selected node is still being
;;; displayed in a panel (i.e., its panel hasn't been reclaimed yet)
(defun reopen-panel (p)  
  (make-current p)
  (setf #!node@p #!node@p)  ;; trigger a few state-oriented propagations
  (cond ((concealed-p p)
         ;; let it show its face again:
         (attach p)
         (expose p))
        ((iconified-p p)
         (expose p)))
  )

(defun close-panel (panel)
  (if (typep panel 'form)  ;; because of what #!po evals to sometimes
      (setf panel (parent panel)))
  (unlock-panel panel)
  (ret panel nil)
  ;; reset current panel to some other exposed one:
  ;;  (should probably keep a visit-order list to do this better)
  (setq *current-hip-panel* 
        (find-if #'(lambda (p) (exposed-p p)) *hip-panels*)))

;;; Utilities for locking/unlocking panels.
;;; A locked panel won't be chosen for reuse when a new panel is
;;; needed. 
(defun lock-panel (panel)
  (setf #!locked@panel t))

(defun unlock-panel (panel)
  (setf #!locked@panel nil))

(defun panel-locked? (panel)
  #!locked@panel)

(defun nodes-in-use ()
  "returns set of nodes currently being viewed in panels"
  (mapcar #'node
          (remove-if #'null (mapcar #'(lambda (p) #!hw@p) *hip-panels*))))

(defun node-panel (node)
  "returns panel displaying node"
  (if (stringp node) (setq node (get-node-named node)))
  (find-if #'(lambda (x) (eql #!node@x node)) *hip-panels*))
#|  This is breaking on un-returned forms:
  (let ((v (viewer node)))
    (when v
          (my-panel v))))
|#


