
(in-package "PT")

;;;;  Miscellaneous support functions and macros....

;;;
;;;  Stuff for manipulating color:
;;;

;;; MAKE-HIP-COLORS is called from hip.tool to initialize all the
;;; colors used in the interface.
(defun make-hip-colors ()
  (dolist (color *all-hip-colors*)
	  (when color (make-color :name color))))

;;
;;  SET-BACKGROUND-RECURSIVELY lets you set the background of a PO and
;;  all of its children to a given color
(defun set-background-recursively (po color)
  (setf (background po) color)
  (if (slot-exists-p po 'children)
      (dolist (c (children po))
	      (set-background-recursively c color)))
  (repaint po))


;;;
;;; Functions for interacting with the user via dialogs, etc:
;;;

;;; The SET-xxx functions call the corresponding dialogs to allow
;;; the user to modify various interface parameters.
(defun set-panels ()
  (let ((new-num (call (find-po-named '("new-hip" "set-panels" . "dialog")))))
    (when new-num
	  (update-panel-cache-size new-num))))

(defun set-mode ()
  (let ((new-mode (call (find-po-named '("new-hip" "set-mode" . "dialog")))))
    (when new-mode
	  (setf #!*user-mode*
		(case new-mode
		      (0 :browse) 
		      (1 :author)
		      (2 :modify))))))

;;;  ANNOUNCE-ERROR checks to see whether we're actually running a
;;;  tool, and if so uses a dialog widget to display the error; else goes to
;;;  standard output.  ASK-USER-TO-CONFIRM does the same for getting
;;;  the user to confirm a potentially destructive operation.

(defun announce-error (message)	   
  (if *widgets-loaded*
      (call-dialog (find-po-named '("new-hip" "error" . "dialog")) :msg message)
    (warn message)))

(defun ask-user-to-confirm (message)
  (if *widgets-loaded*
      (call (find-po-named '("picasso" "confirmer" . "dialog")) :msg  message)
    (yes-or-no-p message)))

(defun get-string (&key (prompt "") (default ""))
  (call (find-po-named '("picasso" "str-prompter" . "dialog"))
	:prompt prompt :default default))

;;;  FEEDBACK prints a message in the message window of the top-level
;;;  frame, or to standard output if widgets not loaded yet.

(defun msg-box ()
  #!top-frame@(current-tool)/hyperdoc-form/message-box)

(defun feedback (msg &optional where)
  "prints msg in the messages window in the top-level frame"
  (if *widgets-loaded*
      ;; show the message in the appropriate window in the top frame:
      (put (text-widget (or where (msg-box)))
	   :string (concatenate 'string msg (string '#\Newline))
	   :overwrite nil)
    (princ msg)))

;;;
;;;  User interface facilitation functions:
;;;

;;;  STRINGIFY-LIST is used, for example, in turning a list of
;;;  keywords into a string suitable for displaying in a form.
;;;  LISTIFY-STRING does the inverse, for reading keywords from an
;;;  entry field.

(defun stringify-list (list-o-symbols)
  "turns a list of symbols into a string consisting of the symbols separated by commas"
  (when list-o-symbols
	(let ((str (format nil "~{~a, ~}" list-o-symbols)))
	  (subseq str 0 (- (length str) 2)))))  ;; drop last comma
	  
(defun stringify-region (list-o-tuples)
  (apply #'(lambda (x y) (concatenate 'string x " " y))
	 (mapcar #'(lambda (x) (format nil "( ~{~a ~})" x))
		 list-o-tuples)))

(defun listify-string (str &aux (lis nil) (pos 0))
  "accepts a string of words separated by commas and/or spaces, and
   returns a list of those words as symbols"
  (do ((s str (subseq s pos))
       (wd nil))
      ((string= s "") lis)
      (multiple-value-setq (wd pos) (read-from-string s))
      (setq lis (append lis (list wd)))
      (if (and (< pos (length s)) (char= (elt s pos) #\,)) (setq pos (1+ pos)))))


;;;  CURRENT-ROW returns the row number of the currently-selected item
;;;  in a table:
(defun current-row (table)
  (let ((ci (current-indices (matrix-field table))))
    (when ci (caar ci))))

;;;  MAKE-MENU-ITEM-LIST turns a list of symbols into a set of items
;;;  appropriate for a pop-button menu.

(defun make-menu-item-list (list-of-symbols &optional (field '#!type-field))
  "returns a list of pairs to be used in a pop-button menu to choose
   the value for some field of a dialog box (e.g., link types)"
  (mapcar #'(lambda (lt) (list (string lt) `(setf (value ,field) ,(string lt))))
	  list-of-symbols))

;;;  Detabification is needed to make text files look right in widgets:
(defun detabify (str &aux (newstr ""))
  (dotimes (i (length str) newstr)
      (let ((c (elt str i)))
       (setq newstr (concatenate 'string
				 newstr 
				 (if (char= c #\tab) "     " (string c)))))))
		
(defun detabify-file-contents (text-file-contents)
  (map 'array #'detabify text-file-contents))


;;;
;;; Some stuff for manipulating tuples:
;;;

(defun tuple-distance (t1 t2)
  "returns euclidean distance between t1 and t2"
  (sqrt 
   (apply #'+ 
	  (mapcar #'(lambda (a b) (expt (- b a) 2)) t1  t2))))

(defun tuple-< (t1 t2)
  (or (and (null t1) (null t2))
      (< (car t1) (car t2))
      (and (= (car t1) (car t2))
	   (if (listp (cdr t1))
	       (tuple-< (cdr t1) (cdr t2))
	     (< (cdr t1) (cdr t2))))))

(defun tuple-min (tuple &rest tuples)
  ;; a recursive solution would be more elegant here, but I think this
  ;; is easier...
  (let ((min tuple))
    (dolist (p tuples min)
	    (if (tuple-< p min) (setq min p)))))

(defun tuple-> (p1 p2)
  (or (and (null p1) (null p2))
      (> (car p1) (car p2))
      (and (= (car p1) (car p2))
	   (if (listp (cdr p1))
	       (tuple-> (cdr p1) (cdr p2))
	     (> (cdr p1) (cdr p2))))))

(defun tuple-max (tuple &rest tuples)
  (let ((max tuple))
    (dolist (p tuples max)
	    (if (tuple-> p max) (setq max p)))))


(defun increment-point (pt)
  "returns list consisting of each element of pt incremented by 1"
  (mapcar #'1+ pt))

;;;
;;  CONTAINS-POINT is used in collection-widgets to find the child in
;;  which a given point falls
;;;

(defun contains-point (region point)
  "returns t if point falls within region"
  ;; region looks like (x y width height), as in the region specs for widgets
  (let ((max-x (+ (car region) (third region)))
	(max-y (+ (second region) (fourth region)))
	(x (car point))
	(y (second point)))
    (and (> x (car region)) (< x max-x)
	 (> y (second region)) (< y max-y))))

;;;
;;;  The next two functions are for use in converting WIP log data
;;;  into lists suitable for display in a table.  They should probably
;;;  go in the wip-node file, but I'm putting them here in case they
;;;  prove to be more generally useful.  8/10 BSB
;;;
(defun string-to-pair-list (str)
  "turns a string of the form '<:key> <val> <:key> <val>...' into a list
   of lists of those key/val pairs"
  (multiple-value-bind  (pair pos) 
		(read-pair-from-string str)
		(if (zerop pos) 
		    (list pair)
		  (append (list pair) (string-to-pair-list (subseq str pos))))))

(defun read-pair-from-string (str)
  (let* ((colon-pos (position #\: str :start 1))
	 (pair-string (subseq str 0 (or colon-pos (length str))))
	 (space-pos (position #\space pair-string)))
    (values
     (list (subseq pair-string 1 space-pos)
	   (subseq str (1+ space-pos) (if colon-pos (1- colon-pos) (length str))))
     (or colon-pos 0))))

;;;
;;; Functions for fetching objects:
;;;

(defun get-hyperdoc (hyperdoc-name)
  (find hyperdoc-name #!*hyperdocs* :key #'name))

(defun get-node-in-doc (node-name doc)
  (get-node node-name doc))

(defun get-node (node-name &optional (hyperdoc (ch)))
  (if hyperdoc
      (find node-name (nodes hyperdoc) :key #'name)
    (gethash node-name *nodes*)))

(defun get-all-nodes (&aux nlist)
  (maphash #'(lambda (key val) 
		     (declare (ignore key))
		     (pushnew val nlist)) 
	   *nodes*)
  (reverse nlist))

(defun get-link (link-name)
  (gethash link-name *links*))

(defun get-all-links (&aux nlist)
  (maphash #'(lambda (key val) 
		     (declare (ignore key))
		     (pushnew val nlist)) 
	   *links*)
  (reverse nlist))

;;; Potentially useful, but not currently used:
(defun get-node-named (node-name)
  (if (stringp node-name) 
      (get-node (read-from-string node-name))
    (get-node node-name)))

(defun get-link-named (link-name)
  (if (stringp link-name) 
      (get-link (read-from-string link-name))
    (get-link link-name)))

(defun get-hyperdoc-named (hyperdoc-name)
  (if (stringp hyperdoc-name)
      (get-hyperdoc (read-from-string hyperdoc-name))
    (get-hyperdoc hyperdoc-name)))

(defmethod visible ((self t)) nil)

(defmethod (setf visible) (new-v (self t)) (declare (ignore new-v)) nil)
