(in-package "PT")

;;;**************************************************************
;;;
;;;  MARKER class, subclasses, methods, and support functions:
;;;

(defdbclass marker (hyper-object)
  ((label :initform nil :type string :accessor label)
   (parent :initarg :parent :type node :accessor parent)
   (region :initarg :region :type list :initform nil :accessor region
	   :documentation "List of points describing marker region")
  )
  (:documentation "Defines a selectable region within a node"))

(defun make-marker (&rest args)
  (apply #'make-instance 'marker args))

(defmethod new-instance ((m marker) 
                         &key name parent region
                         &allow-other-keys)
  (if (not (or name (name m)))
      (setf (name m) (intern (gensym "marker-") (find-package 'pt))))
  ;; if no endpt specified, make it one more than offset:
  (if (= (length region) 1)  ;; i.e., a single point
      (setf (region m) (list (car region) (increment-point (car region)))))
  ;; add this marker to list of parent's markers
  (when parent (add-obj m parent)))

(defmethod parent ((foo list))
  ;; for cases where a link endpt hasn't been resolved into a marker yet
  (if (symbolp (second foo))
      (get-node (second foo))))
      
      
(defmethod offset ((m marker))
  (car (region m)))
(defmethod (setf offset) (pt (m marker))
  (rplaca (region m) pt))

(defmethod endpt ((m marker))
  (car (last (region m))))
(defmethod (setf endpt) (pt (m marker))
  (setf (car (last (region m))) pt))

(defun legal-region (region)
  (tuple-> (second region) (first region)))

(defmethod marker-area ((m marker))
  (apply #'* 
	 (mapcar #'(lambda (x y) (1+ (- y x)))
		 (offset m) (endpt m))))

(defmethod store ((m marker))
  )
(defmethod unstore ((m marker))
  )

(defmethod open-marker ((m marker) &key (in-link nil))
  "asks parent node to open at location of marker"
  (stamp m)  ;; note current time
  (open-node (parent m) :open-at (offset m) :opener in-link))

(defmethod edit ((m marker))
  (let ((result (call (find-po-named '("new-hip" "edit-marker" . "dialog"))
				     :marker m)))
    (when result
	  (if (legal-region (cdr (assoc 'region result)))
	      (update m result)
	    (announce-error "Illegal region specified - no upate performed")))))

(defmethod update :after ((m marker) info)
  (declare (ignore info))
  (modify (parent m))
  (if (links-into m)
      (mapcar #'make-label (append (links-into m)))))

;;;
;;;  BOOKMARK subclass:
;;;
(defdbclass bookmark (marker)
  (
  )
  (:documentation "Class of marker w/o links, for use as bookmarks"))

(defun bookmarkp (m)
  (typep m 'bookmark))

(defun make-bookmark (&rest args)
  (apply #'make-instance 'bookmark args))

(defmethod new-instance ((b bookmark) 
                         &key name 
                         &allow-other-keys)
  (if (not name)
      (setf (name b) (intern (gensym "bookmark-") (find-package 'pt))))
  (call-next-method))

(defun bookmark-summary (b)
  (list (label b) (string (name (parent b)))))

;;;
;;;  LINK-MARKER subclass:
;;;

(defdbclass link-marker (marker)
  ((links-from :initform nil  :type list :accessor links-from)
   (links-into :initform nil :type list :accessor links-into)
   )
  (:documentation "Anchor for link within a node"))

(defun make-link-marker (&rest args)
  (apply #'make-instance 'link-marker args))

(defmethod new-instance ((lm link-marker)
                         &key name 
                         &allow-other-keys)
  (if (not name)
      (setf (name lm) (intern (gensym "link-marker-") (find-package 'pt))))
  (call-next-method))



(defun get-link-marker-in-node (name-or-label node)
  (if (not (typep node 'node)) (setf node (get-node-named node)))
  (if (stringp name-or-label)
      (find name-or-label (link-markers node) :key #'label :test #'string=)
    (find (string-downcase (string name-or-label))
	  (link-markers node)
	  :key #'(lambda (x) (string-downcase (string (name x))))
	  :test #'string=))) 

;;; Need these to prevent setup-code in dialogs from blowing up:
(defmethod links-from ((foo t))
  nil)

(defmethod links-into ((foo t))
  nil)

;;; And these are useful for getting all the links to and from a set
;;; of markers (e.g., when choosing from all the links to/from a node):
(defmethod links-from ((marker-list list))
  (apply #'append
	 (mapcar #'links-from marker-list)))

(defmethod links-into ((marker-list list))
  (apply #'append
	 (mapcar #'links-into marker-list)))

(defun access-links-from (lm)
  ;; this version brings up a dialog with several functions available
  ;; (follow, edit, delete) and actually performs the functions before
  ;; returning.
  (call (find-po-named '("new-hip" "link-utility" . "dialog"))
	:link-markers lm))

(defun choose-link-from (node-or-lm &key (op 'follow))
  ;; this one just brings up lists of links (in and out) to select
  ;; from, and returns the selected link
  (call (find-po-named '("new-hip" "choose-link" . "dialog"))
	:node-or-link-marker node-or-lm
	:op op))
