
(in-package "PT")

;;;*********************************************************
;;;
;;;  LINK-TYPE class
;;;

(defdbclass link-type (pmc)
  ((name :type symbol :initform nil :initarg :name :accessor name)
   (parent :type link-type :initform nil :initarg :parent :accessor parent)
   (children :type list :initform nil :accessor children)
   (filtered :initform nil :initarg :filtered :accessor filtered?)
   (graph-box :initform nil :type shape 
	      :documentation "Pointer to graphic object representing type")
   (action :type list :initform nil :accessor action
	   :documentation "May someday be used to attach arbitrary
                           actions to link-following")
   )
  (:documentation "Implements link taxonomy"))

(defmethod (setf graph-box) (value (self link-type))
  (setf (slot-value self 'graph-box) value))

(defmethod graph-box ((self link-type) &optional shape)
  (declare (ignore shape))
  (slot-value self 'graph-box))

(defun make-link-type (&rest args)
  (apply #'make-instance 'link-type args))

(defmethod new-instance ((lt link-type) &key (name nil) (parent nil))
  (when (get-link-type name)
	(error "A link type name ~a already exists" name))
  (when parent (set-parent lt parent))
  (register-new-type lt))

(defun add-link-subtype (parent)
  (let ((new-type-name (get-string 
			:prompt (format nil "Name for subtype of ~a"
					(name parent)))))
    (when new-type-name
	  (make-link-type :name (read-from-string new-type-name)
			  :parent parent))))

(defmethod set-parent ((lt link-type) par)
  (if (symbolp par) (setf par (get-link-type par)))
  (if (parent lt) (remove-child-type lt (parent lt)))
  (setf (parent lt) par)
  (add-child-type lt par))

(defun get-link-type (ltname)
  (if (stringp ltname) (setf ltname (read-from-string ltname)))
  (find-if #'(lambda (x) (eq (name x) ltname)) #!*link-types*))

(defun type-in-use (lt &aux found)
  (maphash #'(lambda (key val) 
		     (declare (ignore key))
		     (if (has-type val lt) (setf found t)))
	   *links*)
  found)

(defun remove-type (lt)
  (if (symbolp lt) (setf lt (get-link-type lt)))
  (if (type-in-use lt)
      (announce-error 
       (format nil "Links of type ~a exist - can't delete" (name lt)))
    (setf #!*link-types* (remove lt #!*link-types*))))

(defmethod add-child-type ((child link-type) (parent link-type))
  (pushnew child (children parent)))
(defmethod add-child-type ((child link-type) (parent symbol))
  (add-child-type child (get-link-type parent)))

(defmethod remove-child-type ((child link-type) (parent link-type))
  (setf (children parent)
	(remove child (children parent))))
(defmethod remove-child-type ((child link-type) (parent symbol))
  (when parent (remove-child-type child (Get-link-type parent))))

(defun all-supers (lt)
  (if (parent lt)
    (append (all-supers (parent lt)) (list (parent lt)))))

(defun subtype-tree (lt)
  "returns link type subtree rooted at lt"
  (when lt
	(if (children lt) 
	    (cons lt (apply #'append (mapcar #'subtype-tree (children lt))))
	  (list lt))))

(defmethod filter-type ((lt link-type))
  (setf (filtered? lt) t)
  (when (graph-box lt) (update-style (graph-box lt))))
(defmethod unfilter-type ((lt link-type))
  (setf (filtered? lt) nil)
  (when (graph-box lt) (update-style (graph-box lt))))

;;;*********************************************************
;;;
;;;  LINK class, methods, and support functions:
;;;
;;;  At present, I have not defined subclasses of links (to represent
;;;  link types).  we need to decide what those types should be, and
;;;  what behavior they should have.  BSB 4/19

(defdbclass link (hyper-object)
     ((label :initform nil :type string :accessor label)
      (type :initform nil :initarg :type :type link-type :accessor link-type)
      (source :initarg :source :initform nil :type link-marker :accessor source)
      (dest :initarg :dest :initform nil :type link-marker :accessor dest)
      (hyperdocs :initform nil :accessor hyperdocs 
            :documentation "List of hyperdocs to which this link belongs")
      (followed :initform nil :type symbol :accessor followed?)
      )
     (:documentation "Represents relationship between nodes"))

;;; Link creation/deletion:

(defun make-link (&rest keys)
  (apply #'make-instance 'link keys))

(defmethod new-instance ((l link) &key name 
                         (source nil)
                         (dest nil)
			 (type (get-link-type 'link))
                         (hyperdocs (list #!*current-hyperdoc*))
                         &allow-other-keys)
  ;; should we be checking for name uniqueness?
    ;; we'll want to do something better than this eventually...
    (if (not name)
        (setf (name l) (intern (gensym "link-") (find-package 'pt))))
    (if (symbolp type) (setf (link-type l) (get-link-type type)))
    ;; (if (null (link-type l)) (announce-error (format nil "No link type ~a" type)))
    (if (not (label l))
        (make-label l))
    (if source (add-obj l (source l)))
    (if (and dest (typep dest 'link-marker))
	(add-obj l (dest l) :to))
    (dolist (h hyperdocs) (if h (add-obj l h)))
    (store l))  ;; add to global set of links

(defmethod make-label ((l link))
  (setf (label l)
        (concatenate 'string  
                     (if (link-type l) (string (name (link-type l))) "" )
		     (if (and (dest l) (typep (dest l) 'link-marker))
			(concatenate 'string ": " (label (dest l)))
			(string (name l)))
                     )))

(defmethod store ((l link))
  (setf (gethash (name l) *links*) l))

(defmethod unstore ((l link))
  (db-unstore-object l)
  (remhash (name l) *links*))

(defmethod lmark ((l link))
  (stamp l)  ;; note current time
  (setf (followed? l) t))

(defmethod unmark ((l link))
  (setf (followed? l) nil))

;;; Previewing and following links:  these will be called from the
;;; widget displaying the source node.  
;;;   Again, this could probably go away...
(defmethod preview ((l link))
  "displays descriptor for other endpoint of link"
  (if (dest l)
      (preview (dest l))
    (announce-error (format nil "Preview error: Link ~a has no destination node" (name l)))))

(defmethod follow ((l link) &key (dir :forw))
  "invokes appropriate browser for node at other endpoint of link"
  (let ((slot (if (eq dir :forw) 'dest 'source)))
    (if (slot-value l slot)
	(with-feedback (format nil "Following link ~a to ~a" (label l) slot)
		       (progn
			 (lmark l)
			 (open-marker (slot-value l slot) :in-link l)))
      (announce-error (format nil "Follow error: Link ~a has no ~a node"
			      (name l) slot)))))

;;; make sure that whenever a link gets a new source, the old and new
;;;   source blocks are updated appropriately:
(defmethod set-source ((l link) (src link-marker))
  (setf (source l) src)
  ;; if l already has a source, remove l from its links:
  (if (src l)
      (remove-obj l (src l)))
  ;; add this link to those emanating from src:
  (add-obj l src :from)
  (make-label l))

;; need to update the label if dest or type changes:
(defmethod set-dest ((l link) (dest link-marker))
  (if (typep (dest l) 'link-marker)
      (remove-obj l (dest l) :to))
  (setf (dest l) dest)
  (add-obj l dest :to)
  (make-label l))

(defmethod set-link-type ((l link) type)
  (if (symbolp type) (setf type (get-link-type type)))
  (setf (link-type l) type)
  (make-label l))

(defmethod has-type ((l link) type)
  (if (symbolp type) (setf type (get-link-type type)))
  (or (eq type (link-type l))
      (member type (all-supers (link-type l)))))

(defun get-link-labeled (string link-list)
  "a stupid function to find a link with a given label out of a list
   of links"
  (find-if #'(lambda (l) (string= string (label l))) link-list))

(defmethod edit ((l link))
  (let ((result (call (find-po-named '("new-hip" "edit-link" . "dialog"))
				     :link l)))
    (when result
	  (update l result))))

(defmethod update ((l link) values)
  (call-next-method)   ;; an :after method wasn't doing the trick here..
  (set-link-type l (cdr (assoc 'type values)))
  (modify (source-node l)))


(defun source-node (link)
  (parent (source link)))

(defun dest-node (link)
  (parent (dest link)))
