
;;;
;;;  FILTERS.CL
;;;  Variables and functions for filtering the links and nodes of the
;;;  current hyperdocument.

(in-package "PT")

(defun initialize-types ()
  (initialize-link-types)
  (initialize-node-types))

;;;  Link types are stored in a file, which must be read in at the
;;;  beginning of each session to create the taxonomy:
(defun initialize-link-types ()
  (when (null #!*link-types*)
       (make-link-type :name 'link)
       (make-link-type :name 'path :filtered t)
       (with-open-file (str $link-type-file :direction :input)
             (setf $raw-link-types (read str)))
       (dolist (type $raw-link-types)
               (dolist (sub (cdr type))
                       (make-link-type :name sub :parent (car type))))
       ))

(defun initialize-node-types ()
  (when (null #!*node-types*)
        (setf #!*node-types* $raw-node-types)
        (setf *node-type-selections* #!*node-types*)))

;;  when the user adds a new type, make sure it gets installed everywhere:
(defun register-new-type (lt)
  (push lt #!*link-types*))

;;;  Link types are written out at the end of a session, in case the
;;;  user has added new types to the taxonomy:
(defun write-link-types (&optional (link-types #!*link-types*))
  (with-open-file (str $link-type-file :direction :output
                       :if-exists :new-version)
       (princ "(" str)
       (dolist (lt (reverse link-types))  ;;; make sure type 'link
					  ;;; goes first
               (when (children lt)
                     (format str "~%(~a . ~s)" (name lt) (mapcar #'name
                                                                 (children lt)))))
       (terpri str)
       (princ ")" str)))

;;;
;;;   Functions for setting and applying filters:
;;;

(defun set-filters ()
  (let ((ft (call (find-po-named '("new-hip" "choose-filter-type" .  "dialog")))))
    (case ft
	  (link-type (call (find-po-named '("new-hip" "filter-links" . "dialog"))))
	  (node-type (call (find-po-named '("new-hip" "filter" .  "dialog"))
			   :obj-type 'node))
	  (keyword (call (find-po-named '("new-hip" "filter" . "dialog"))
			 :obj-type nil)))
    (update-filter-state)))
	  
(defun set-keyword-restrictions (new-keys &key (links? t) (nodes? t))
  (feedback (format nil "Setting keywords to ~s" new-keys))
  (if links? (set-link-keywords new-keys))
  (if nodes? (set-node-keywords new-keys)))

(defun filter (list-of-obj)
  (let ((slot (if (typep (car list-of-obj) 'link) 'links 'nodes)))
    ;;;  Get rid of anybody who (1) doesn't belong to the current
    ;;;  hyperdoc; (2) doesn't have one of the selected types; or (3)
    ;;;  doesn't have a selected keyword
    (remove-if
     #'(lambda (l) (or (and #!*current-hyperdoc*
                             (not (member l (slot-value #!*current-hyperdoc* slot))))
		       (filtered? l)))
     list-of-obj)))

(defmethod displayed? ((obj t))
  (not (filtered? obj)))

;;;  Link filtering variables and functions.
;;;  Link types are filtered by setting a slot in the objects
;;;  representing the types to filter out.  Link keyword filters are
;;;  represented by a global list.

(defun filter-link-type (lt)
  (dolist (type (subtype-tree lt))
	  (filter-type type)))
(defun display-link-type (lt)
  (dolist (type (subtype-tree lt))
	  (unfilter-type type)))
(defun toggle-link-type (lt)
  (if (filtered? lt) (display-link-type lt) (filter-link-type lt)))

(defun set-link-type-selections (types)
  "filters all link types except those specified"
  (dolist (lt #!*link-types*) (filter-link-type lt))
  (dolist (lt types) (display-link-type lt)))

(defun set-link-keywords (keys)
  (setf *link-keyword-restrictions* keys))

(defun clear-link-type-filters ()
  (dolist (lt #!*link-types*) (display-link-type lt))
  (update-filter-state))

(defmethod filtered? ((l link))
  (or (not (link-type l))
      (filtered? (source-node l))
      (when (dest-node l) (filtered? (dest-node l)))
      (not (satisfies-keyword-filter l))
      (not (satisfies-type-filter l))))

(defmethod satisfies-type-filter ((l link))
  (displayed? (link-type l)))

(defmethod satisfies-keyword-filter ((l link))
  (if *link-keyword-restrictions*
      (some #'(lambda (k) (member k *link-keyword-restrictions*))
	    (keywords l))
    t))  ;; if no restrictions, everybody passes

;;;  Node filtering variables and functions.
;;;  Node type and keyword selections are stored in global lists.
(defvar *node-type-selections* nil
  "list of node types being shown; used in filtering function")
(defvar *node-keyword-restrictions* nil
  "list of symbols used to filter set of nodes shown")

(defun set-node-type-selections (types)
  (setf *node-type-selections* types))
(defun set-node-keywords (keys)
  (setf *node-keyword-restrictions* keys))

(defmethod filtered? ((n node))
  (or (not (satisfies-type-filter n))
      (not (satisfies-keyword-filter n))))

(defmethod satisfies-type-filter ((n node))
  (member (type n) *node-type-selections*))
(defmethod satisfies-keyword-filter ((n node))
  (if *node-keyword-restrictions*
      (some #'(lambda (k) (member k *node-keyword-restrictions*))
            (keywords n))
    t))

;;
;;  Functions for accessing current filter state of the system:
;;

(defun filter-state ()
  (let ((fl (filtered-link-types))
	(fn (filtered-node-types))
	(kl (filtered-link-keywords))
	(kn (filtered-node-keywords)))
    (concatenate 'string
	       (if fl (format nil "Link types:~{ ~S~}; " (mapcar #'name fl)))
	       (if fn (format nil "Node types:~{ ~S~}; " fn))
	       (if kl (format nil "Link keywords:~{ ~S~}; " kl))
	       (if kn (format nil "Node keywords:~{ ~S~}" kn))
	       )))
(defun update-filter-state ()
  (setf #!*filter-state* (filter-state)))

(defun filtered-link-types ()
  "returns set of currently-filtered link types"
  (remove (get-link-type 'path)
	  (remove-if-not #'filtered? #!*link-types*)))
(defun filtered-node-types ()
  (set-difference $raw-node-types *node-type-selections*))
(defun filtered-link-keywords ()
  *link-keyword-restrictions*)
(defun filtered-node-keywords ()
  *node-keyword-restrictions*)

