
;;
;;  BASE-CLASSES.CL
;;  This file contains the base class definitions for the hypermedia framework
;;  

(in-package "PT")

;; SOH DOES NOT WORK
;; Make sure db open for class defs
;; (db-open)

;;;*****************************************************
;;;
;;;  HYPER-OBJECT class
;;;  Abstract parent for blocks, links, and nodes; provides basic
;;;  attributes like name, description, and keywords.

(defdbclass hyper-object (pmc dbobject)
  ((name :initarg :name :initform 'an-object :type symbol :accessor name)
   (description :initarg :description :initform "" :type string
                :accessor description
                :documentation "Description of this object")
   (keywords :initarg :keywords :initform nil :type list :accessor keywords
             :documentation "Keywords used to search hyperdocs")
   (timestamp :initform 0 :type number :accessor timestamp
	      :documentation "Last access time of object")
   )
  (:documentation "Abstract parent class for hypermedia objects"))

(defmethod has-keyword ((self hyper-object) keywd)
  (find keywd (keywords self)))

(defmethod stamp ((self hyper-object))
  (setf (timestamp self) (get-universal-time)))

;;  Methods to be specialized by subclasses:
(defmethod edit ((self hyper-object))
  self)

(defmethod update ((obj hyper-object) values)
  ;; values is a list of slot . new-value pairs
  (dolist (v values)
	  ;; changing the name could have other implication:
	  (if (eq (car v) 'name)
		(unless (eq (cdr v) (name obj))
			(rename obj (cdr v)))
	    (setf (slot-value obj (car v)) (cdr v)))))

(defmethod rename ((obj hyper-object) new-name)
  (setf (name obj) new-name))


;;;*********************************************************
;;;
;;; OWNED-OBJ mixin class
;;; Provides access control for objects, using standard Unix
;;; permissions.
;;;

(defclass owned-obj (pmc)
  ((owner :initarg :owner :type string :initform *user* :accessor owner)
   ;; perms is an alist of (uid . <perms>)
      ;;  (actually, I think perms should be structures, with uid and
      ;;  perm-list, so we can manipulate them more abstractly, but
      ;;  then we need to make the perms list here a hash table to
      ;;  minimize searching on uid)
   (perms :initform (list '(all . (read write edit)))
          :type list :accessor perms)))

(defmethod new-instance ((self owned-obj) &key (owner *user*) perms
                         &allow-other-keys) 
  ;; give owner all privileges plus grant:
  (call-next-method)
  (unless perms (add-perms self owner '(all))))

;;; Need to add grant-perm methods, too

(defmethod add-perms ((self owned-obj) uid perm-list)
  "adds permissions in perm-list for uid on self, if not already present"
  ;; perm is a list of permissions, e.g., (read write modify)
  ;;;; Instead of dealing with special checks for 'all, let's just
  ;;;; turn an 'all specifier into the appropriate list of specific perms:
  (if (member 'all perm-list) 
      (setq perm-list *perms*))
  ;; add each permission to the list for this object:
  (dolist (p perm-list)
          (add-perm self uid p)))

(defmethod add-perm ((self owned-obj) uid perm)
  "adds perm to perm-list for uid on self, if legal and not present"
  (if (not (member perm *perms*))
      (announce-error (format nil "Unknown permission type ~a" perm))
    (let ((user-perms (cdr (assoc uid (perms self)))))
      (cond (user-perms                ;; if user already has some perms...
             ;; add the new one to the entry for this user:
             (setf (cdr (assoc uid (perms self))) (pushnew perm user-perms)))
            ;; else create a new entry for user and perm:
            (t (pushnew (cons uid (list perm)) (perms self)))))))

(defmethod remove-perms ((self owned-obj) uid perm-list)
  "revokes permissions in perm-list for uid on self, if present"
  ;; perm is a list of permissions, e.g., (read write modify)
  (if (member 'all perm-list) 
      (setq perm-list *perms*))
  ;; add each permission to the list for this object:
  (dolist (p perm-list)
          (remove-perm self uid p)))

(defmethod remove-perm ((self owned-obj) uid perm)
  "revokes perm for uid on self"
  (if (not (member perm *perms*))
      (announce-error (format nil "Unknown permission type ~a" perm))
    (let ((user-perms (cdr (assoc uid (perms self)))))
      (or (not user-perms)    ;; user didn't have any perms to begin with...
          (not (member perm user-perms))  ;;  or not this one...
          (setf (cdr (assoc uid (perms self))) (remove perm user-perms))))))

(defmethod has-access (user (obj owned-obj) perm)
  "Returns t if user has specific perm for accessing obj"
  (or (string= user (owner obj))  ;; owner can do whatever
      ;; see if everybody has access...
      (member perm (cdr (assoc 'all (perms obj))))
      ;; and if not, whether this user does..
      (let ((user-perms (cdr (assoc user (perms obj) :test #'string=))))
        (or (member 'all user-perms)
            (member perm user-perms)))))

;;; unspecialized version: for accessing objects without ownership
;;; restrictions; always returns t
(defmethod has-access (user obj perm)
  "Returns t if user has specific perm for accessing obj"
  (declare (ignore user obj perm))
  t)
