;;;
;;;  HYPER-VIDEO-WIDGET
;;;  Widget for displaying video nodes in HIP:
;;;     

(in-package "PT")

(defvar *video-connected* nil)

(defun play-video-node (node)
  (let* ((disk (videodisk node))
	 (index (video::disk-index disk))
	 (range nil))
	(maphash #'(lambda (k v) (setq range v)) index)
	(apply #'video:play-range range)))

(defclass hyper-video-widget (hypermedia-mixin collection-widget)
  (
  (event-mask :initform '(:button-press :keypress :exposure))
  (current-entry :initform nil :accessor current-entry
		 :documentation "Selected entry in video panel")
  (frame :initform 0 :type integer :accessor frame :documentation
	 "Current frame being displayed on video-disk")
  ;;; Children of collection:
  (marker-scroll-controls :initform nil :type scroll-bar :accessor marker-scroll-controls)
  (display-area :initform nil :type widget :accessor display-area)
  (control-button :initform nil :type button :accessor control-button)
  (index-panel-button :initform nil :type button :accessor index-panel-button)
   )
  (:documentation "Video widgets with hypermedia behavior"))

(defun make-hyper-video-widget (&rest args)
  (apply #'make-instance 'hyper-video-widget args))

(defmethod new-instance ((self hyper-video-widget) 
			 &key node 
			 &allow-other-keys)
  (call-next-method)
  ;; Set up the display area and buttons:
  (unless *video-connected*
	  (setq *video-connected* t)
	  (video:connect-player)
	  (video:start-disc))
  (setf (gm self) 'anchor-gm)
  (setf (display-area self) 
	(make-video-widget :geom-spec '(0 0 1 .68)
			   :border-width 2
			   :base-size '(320 200)
			   :horiz-just :center
			   :vert-just :center)
	(parent (display-area self)) self)
  (setf (control-button self)
	(make-gray-button :geom-spec '(.1 .85 .35 .15 :arrow (:horiz :vert))
			  :value "Play"
			  :press-func `(progn
					(reset-video (display-area ',self))
					(play-video-node (node ',self))
					(start-video)))
	(parent (control-button self)) self)
  (setf (index-panel-button self)
	(make-gray-button :geom-spec '(.55 .85 .35 .15 :arrow (:horiz :vert))
			  :value "Pause"
			  :press-func '(progn (stop-video)
					      (video:pause)))
	(parent (index-panel-button self)) self)
  ;; Usual stuff...
  (when node
	(setf (viewer node) self)
	(setup-node self node))
  ;; temporary kludge to get these guys to listen to the right events:
  #|
  (register-callback self #'show-link-menu 
		     :button-press :detail :left-button :state :control)
  (register-callback self #'follow-default-link
		     :button-press :detail :middle-button :state :control)
  |#
)

(defmethod widget ((self hyper-video-widget))
  self)

;;;
;;;  Methods for setting up and saving node correctly:
;;;

(defmethod setup-node ((self hyper-video-widget) (n video-node))
  (when (videodisk n) 
	(video::load-disk (videodisk n))
	(video::load-index (videodisk n) (dataset n))
	(with-feedback "Loading video index..."
		       (maphash #'(lambda (key val)
				    (if (not (get-link-marker-in-node key n))
					(make-marker-from-index-entry key val n)
				      (confirm-marker-data key val n)))
				(video::disk-index (videodisk n)))))
  (unmodify n)
  )

(defun make-marker-from-index-entry (label data node)
  (make-link-marker :parent node :label (format nil "~a" label)
	      :region (list (list (car data)) (list (second data)))))
  
(defun confirm-marker-data (label data node)
  "make sure marker corresponding to index entry has same start and end
   frame" 
  (let ((b (get-link-marker-in-node label node)))
    (setf (region b) (list (list (car data)) (list (second data))))))

(defmethod save ((vn video-node) &optional hw)
  (declare (ignore hw))
  (let ((path (call (find-po-named '("save-file" . "dialog")) :dir
		    (picasso-path "lib/video/cim-disk")))
	overwrite msg)
    (when (not path) (setf path (dataset vn)))
    (setq msg (list "Overwrite file" (namestring path))
	  overwrite
	  (if (probe-file path)
	      (call (find-po-named '("overwrite" . "dialog")) :msg msg)
	    t))
    (when overwrite
	  (video:save-index (videodisk vn) path))
    (unmodify vn)))

;;;
;;; Methods for setting and querying the current state of the widget:
;;;

(defmethod get-current-position ((self hyper-video-widget))
  (frame self))

(defmethod get-mark-region ((self hyper-video-widget))
  ;; returns marked region as (start end)
  (let* ((ip #!index@(current-tool))
	 (start (value #!start-entry@ip))
	 (end (value #!end-entry@ip)))
    (and start
	 (list start end))))

(defun get-index-key ()
  (value #!index@(current-tool)/key-entry))

(defmethod get-current-link-marker ((self hyper-video-widget))
  ;; I think this should change - too tied to selection of entries vs.
  ;; selection of new ranges of video
  (get-link-marker-in-node (current-entry self) (node self)))


(defmethod unmark-marker ((self hyper-video-widget) (b marker))
  nil)

(defmethod exposed-markers ((self hyper-video-widget))
  (when (node self)
	(remove-if-not #'(lambda (b) (and (<= (car (offset b)) (frame self))
					  (>= (car (endpt b)) (frame self))))
		       (visible-markers (node self)))))

(defmethod scroll-to ((self hyper-video-widget) offset)
  ;; make sure video controls get activated:
  (when (and offset
	     (not (exposed-p #!video@(current-tool))))
	(eval (press-func (control-button self))))
  ;; offset is frame number in the video
  (when (consp offset) (setf offset (car offset)))
  (when (and offset (not (zerop offset)))
	(video:search-to-frame offset)))

;;;
;;;  Event handlers:
;;;

(defhandler position-mark ((self hyper-video-widget)
			   &key &allow-other-keys
			   &default
			   (:button-press :detail :left-button))
  (make-current (my-panel self)))

  
;;;  Methods for outlining marker regions:

(defmethod marker-outline-points (marker (widget hyper-video-widget))
  "just returns entire region around display area of widget"
  (declare (ignore marker))
  (let ((reg (region (display-area widget))))
    (list (cons (first reg) (second reg))
	  (cons (+ (first reg) (third reg)) (second reg))
	  (cons (+ (first reg) (third reg)) (+ (second reg) (fourth  reg)))
	  (cons (first reg) (+ (second reg) (fourth  reg))))))

(defmethod translate-point (pt (self hyper-video-widget) &optional (adjust :tl) (outer? nil))
  "translates pt to coordinates of widget, adjusting so actual point
   will be outside of text region"
  ;; adjust values are :tl, :tr, :bl, :br (for top/bottom,
  ;; right/left)
  (declare (ignore pt))
  (let* ((widget (display-area self))
	 (top (y-offset widget))
	 (left (x-offset widget))
	 (wi (width widget))
	 (hi (height widget))
	 (wi-adjust (if outer? -4 -3))
	 (hi-adjust (if outer? -4 -2))
	 )
    ;; adjust x and y so lines go in proper spaces between rows/cols
    (case adjust
	  (:tl (list (- left wi-adjust) (- top hi-adjust)))
	  (:tr (list (+ left wi wi-adjust) (- top hi-adjust)))
	  (:bl (list (- left wi-adjust) (+ top hi hi-adjust)))
	  (:br (list (+ left wi wi-adjust) (+ top hi hi-adjust)))
	  (otherwise (list left top))
	  )))

