
;;;
;;;  HYPER-TEXT-WIDGET
;;;  Widget for displaying text nodes in HIP:
;;;     

(in-package "PT")

(defclass hyper-text-widget (hypermedia-mixin scrolling-text-widget)
  (
;;  (event-mask :initform '(:button-press :button-1-motion :keypress
;;                                       :exposure :double-click))  
   )
  (:documentation "Text widgets with hypermedia behavior"))

(defun make-hyper-text-widget (&rest args)
  (apply #'make-instance 'hyper-text-widget args))

(defmethod new-instance ((self hyper-text-widget) 
                         &key node 
                         (prev-page-func `(htw-prev-page ',self))
                         (next-page-func `(htw-next-page ',self))
                         (prev-line-func `(htw-prev-line ',self))
                         (next-line-func `(htw-next-line ',self))
                         (move-func `(htw-sb-move self ',self event))
                         &allow-other-keys)
  (call-next-method)
  ;; for some reason, these args aren't being passed on correctly to
  ;; the make-scroll-bar call in new-instance for
  ;; scrolling-text-widget, so for now I'm explicitly assigning them
  ;; here  (8/9 BSB) :
  (setf (prev-page-func (scroll-bar self)) prev-page-func)
  (setf (next-page-func (scroll-bar self)) next-page-func)
  (setf (prev-line-func (scroll-bar self)) prev-line-func)
  (setf (next-line-func (scroll-bar self)) next-line-func)
  (setf (moved-func (scroll-bar self)) move-func)
  (when node
        (setf (viewer node) self)
        (setup-node self node))
  ;; temporary kludge to get these guys to listen to the right events:
  (register-callback (widget self) #'show-link-menu 
                     :button-press :detail :left-button :state :control)
  (register-callback (widget self) #'follow-default-link
                     :button-press :detail :middle-button :state :control)
  (register-callback (widget self) #'hyper-position-mark
                     :button-press :detail :right-button))
  

(defmethod widget ((self hyper-text-widget))
  (text-widget self))

;;;
;;;  Method for setting up node correctly:
;;;

(defmethod setup-node ((self hyper-text-widget) (n text-node))
  ;; get the contents of the text file and put them into the buffer:
  (load-file (text-widget self) (dataset n))
  (unmodify n)
  )

;;;
;;; Methods for setting and querying the current state of the widget:
;;;

(defmethod get-current-position ((self hyper-text-widget))
  (list (row self) (column self)))

(defmethod set-position ((self hyper-text-widget)
                         &key x y &allow-other-keys)
  (text-widget-position-cursor (widget self) :x x :y y))

;;  Seems that we need this one as well, until we can move the event
;;  handlers from the widget to the hyper-widget:
(defmethod set-position ((self text-widget) &key pos
                         &allow-other-keys)
  (text-widget-position-cursor (widget self) :x (car pos) :y (cadr pos)))
  
(defmethod get-mark-region ((self hyper-text-widget))
  ;; returns list of (row col) points defining marked region
  (let ((cur-pos (get-current-position self))
        (mark-pos (list (mark-row self) (mark-column self))))
    ;; if there's really a region selected, return it:
    (if (car mark-pos) 
        (sort (list mark-pos cur-pos) #'tuple-<)
      ;; else just return the current position
      (list cur-pos))))

(defmethod unmark-marker ((self hyper-text-widget) (b marker))
  "sets point and mark to beginning of marker, un-highlighting it"
  (let* ((tw (text-widget self)))
    (setf (row tw) (car (offset b)))
    (setf (column tw) (cadr (offset b)))
    (setf (mark-row tw) (car (offset b)))
    (setf (mark-column tw) (cadr (offset b)))
    (tbg-repaint tw :start-row (row tw) :start-column (column tw)
                 :end-row (mark-row tw) :end-column (mark-column tw))))

(defmethod exposed-markers ((self hyper-text-widget))
  (let* ((tr (top-of-screen (text-widget self)))
         (br (+ tr (rows self))))
    (remove-if-not #'(lambda (b) (or (and (>= (car (offset b)) tr)
                                          (<  (car (offset b)) br))
                                     (and (>= (car (endpt b)) tr)
                                          (< (car (endpt b)) br))))
                   (visible-markers (node self)))))


(defmethod scroll-to ((self hyper-text-widget) offset)
  "offset is (row . col) in the text widget"
  (let  ((tw (text-widget self))
         (row (car offset))
         (col (cadr offset))  ;; ignored for now
         (tos (top-of-screen (widget self)))
         (r (row self)))
	(declare (ignore col))
    (when (not (and (>= row tos)
                    (<= row (+ tos (rows (widget self))))))
          (if (< r row)
              (tbg-scroll-down tw (- row r))
            (tbg-scroll-up tw (- r row))))))

;;;
;;;  Event handlers:
;;;

(defhandler position-mark ((self hyper-text-widget)
                           &key x y &allow-other-keys
                           &default
                           (:button-press :detail :right-button))
  (hyper-position-mark self :x x :y y))

(defun hyper-position-mark (self &key x y &allow-other-keys)
  (cond  ((eq (class-of self) (find-class 'text-widget))
          (text-widget-position-mark self :x x :y y)))
  ;; make this panel the current panel:
  (make-current (my-panel self)))
  
;;;
;;;  The following methods provide access to attributes of the
;;;  underlying text widget:
;;;

(defmethod row ((self hyper-text-widget))
  (row (text-widget self)))
(defmethod (setf row) (row (self hyper-text-widget))
  (setf (row (text-widget self)) row))

(defmethod mark-row ((self hyper-text-widget))
  (mark-row (text-widget self)))
(defmethod (setf mark-row) (row (self hyper-text-widget))
  (setf (mark-row (text-widget self)) row))

(defmethod column ((self hyper-text-widget))
  (column (text-widget self)))
(defmethod (setf column) (column (self hyper-text-widget))
  (setf (column (text-widget self)) column))

(defmethod mark-column ((self hyper-text-widget))
  (mark-column (text-widget self)))
(defmethod (setf mark-column) (column (self hyper-text-widget))
  (setf (mark-column (text-widget self)) column))

;;;  Methods on text-widget:

;;;  KLUDGE AHOY!!!
;;;  I'm adding the following :after method to text-widget because I
;;;  can't seem to get hyper-text-widgets to do anything for
;;;  themselves.  Once I get them to start accepting events, this
;;;  problem should also be solved.
(defmethod do-repaint :after ((self text-widget) &key &allow-other-keys)
  (if (and (eq (class-of (parent self)) (find-class 'hyper-text-widget))
           (markers-shown? (parent self)))
      (hm-repaint (parent self))))
(defmethod do-repaint-region :after ((self text-widget) x y w h 
				     &key &allow-other-keys)
  (declare (ignore x y w h))
  (if (and (eq (class-of (parent self)) (find-class 'hyper-text-widget))
           (markers-shown? (parent self)))
      (hm-repaint (parent self))))

(defmethod (setf changed) :after (val (self text-widget))
  (when (eq (class-of (parent self)) (find-class 'hyper-text-widget))
        (setf (modified? (node (parent self))) val)))

(defmethod max-col ((self text-widget))
  (columns self))

;;; this is so I can treat text and tables similarly - but will
;;; probably get too complicated with additional media...
(defmethod current-indices ((self text-widget))
  (list (list (row self) (column self))))

;;;  Methods for outlining marker regions:

(defmethod marker-outline-points (marker (widget text-widget))
  "returns a set of (row, col) points defining the region around block
   of text"
  (text-block-outline-pts (offset marker) (endpt marker) widget))

(defun text-block-outline-pts (beg end &optional widget)
  "returns the set of points defining the boundaries of a text region
   from beg to end in widget"
  (let* ((max-col (if widget (max-col widget) 80))
         (x1 (car beg)) (y1 (second beg))
         (x2 (car end)) (y2 (second end)))
    ;;  if second point is the beginning of a line, make it the end of
    ;;  the previous line:
    (if (zerop y2) (setq x2 (1- x2) 
                         y2 max-col
                         end (list x2 y2)))
    ;;;  Given (x1 y1) (x2 y2), cases are:
    ;;;    rectangular block all on one line (x1 = x2, y1 < y2) - 4 pts
    ;;;    rectangular block >1 line high    (x1 < x2, y1 < y2) - 4 pts
    ;;;    irreg. with indented first point  (y1 > y0, y2 = ym) - 6 pts
    ;;;    irreg. with indented last point   (y1 = y0, y2 < ym) - 6 pts
    ;;;    totally irregular                 (y1 > y0, y2 < ym) - 8 pts
    ;;; (where y0 = left margin, ym = max y value = # columns)

    (if (eq x1 x2)               ;; same-line block
        (list beg end end beg)   ;; translate-point will
                                 ;; differentiate these...
      (if (zerop y1)             ;; flush at left
          (if (= y2 max-col)     ;; and right
              (list beg (list x1 y2) end (list x2 y1))
            ;; else it's indented at lower right
            (list beg (list x1 max-col) (list (1- x2) max-col)
                  (list (1- x2) y2) end (list x2 y1)))
        ;; else it's indented at left
        (if (= y2 max-col)     ;; flush right
            (list beg (list x1 y2) end (list x2 0) (list (1- x1) 0)
                  (list (1- x1) y1))
            (list beg (list x1 max-col) (list (1- x2) max-col)
                  (list (1- x2) y2) end (list x2 0) (list (1+ x1) 0)
                  (list (1+ x1) y1)))))))

(defmethod translate-point (pt (widget text-widget) &optional (adjust :tl) (outer? nil))
  "translates pt to coordinates of widget, adjusting so actual point
   will be outside of text region"
  ;; adjust values are :tl, :tr, :bl, :br (for top/bottom,
  ;; right/left)
  (let* ((top (top-of-screen widget))
         (left (left-of-screen widget))
         (wi (width-increment widget))
         (half-wi (floor wi (if outer? 2 3)))
         (hi (height-increment widget))
         (half-hi (floor hi (if outer? 4 6)))
         ;; remember that the y-val of the point indicates l-r
         ;; position (column), and the x-val is up-down position
         ;; (row), so we have to switch and then translate them:
         (new-x  (- (* (cadr pt) wi) (* left wi)))
         (new-y  (- (* (car pt) hi) (* top hi))))
    ;; adjust x and y so lines go in proper spaces between rows/cols
    (case adjust
       (:tl (list (max (- new-x half-wi) left) (- new-y half-hi)))
       (:tr (list (+ new-x wi half-wi) (- new-y half-hi)))
       (:bl (list (max (- new-x half-wi) left) (+ new-y hi half-hi)))
       (:br (list (+ new-x wi half-wi) (+ new-y hi half-hi)))
       (otherwise (list new-x new-y))
       )
    ))

;;;
;;;  These functions are used to replace the usual scroll-bar
;;;  functions, to make sure the right repaint gets called:
;;;

(defun htw-prev-page (htw)
  (sb-prev-page (scroll-bar htw))
  (hm-repaint htw))

(defun htw-next-page (htw)
  (sb-next-page (scroll-bar htw))
  (hm-repaint htw))

(defun htw-prev-line (htw)
  (sb-prev-line (scroll-bar htw))
  (hm-repaint htw))

(defun htw-next-line (htw)
  (sb-next-line (scroll-bar htw))
  (hm-repaint htw))

(defun htw-sb-move (sb htw ev)
  (declare (ignore sb))
  (stw-sb-move (scroll-bar htw) htw ev)
  (hm-repaint htw))
