(in-package "PT")

;;;
;;;  HIP-specific node accessors:
;;;

(defun node-info-fn (node)
  (list (format nil "~a: ~a" (type node) (name node))))

(defun get-node-successors (node)
  "returns all nodes you can get to via a single link from given node,
   after applying link and node filters and removing the node itself
   (to avoid circularity if the node is linked to itself)"
  (filter
   (remove node
	   (remove-duplicates
	    (mapcar #'parent
		    (remove-if #'null
			       (mapcar #'dest
				       (filter (links-from node)))))))))
  

;;;
;;;  Specialized class of box for displaying nodes:
;;;

(defclass node-box (box)
  ((my-obj :initform nil :initarg :my-obj :type node :accessor my-obj)))

(defun make-node-box (&rest args)
  (apply #'make-instance 'node-box args))

(defmethod graph-box ((n node) 
		      &optional 
		      (shape #!top-frame@(current-tool)/browser/top-shape))
  (when shape
	(find-if #'(lambda (x) (eql (my-obj x) n)) 
		 (remove-if-not #'(lambda (x) (box-p x)) 
				(sub-objs shape)))))

(defmethod graph-box ((lt link-type) &optional ignore )
  (declare (ignore ignore))
  (let ((shape #!graph-shape@(find-po-named '("new-hip" "type-browser" . "form"))))
    (when shape
	  (find-if #'(lambda (x) (eql (my-obj x) lt))
		   (remove-if-not #'(lambda (x) (box-p x))
				  (sub-objs shape))))))

(defmethod update-style ((self t)) nil)

(defmethod update-style ((nb node-box))
  (let ((new-color (box-color (my-obj nb))))
    (setf (line-style nb) (box-line (my-obj nb)))
    (setf (color nb) new-color)
    (dolist (so (sub-objs nb))
	    (setf (color so) new-color))))

(defmethod box-color ((n node))
  (if (visited? n)
      *visited-node-color*
    *unvisited-node-color*))
(defmethod box-color ((lt link-type))
  (get-color "black"))

(defmethod box-line ((n node))
  (if (and (node-in-use n)
	   (exposed-p (node-panel n)))
      :dash :solid))
(defmethod box-line ((lt link-type))
  (if (displayed? lt)
      :solid
    :dash))

;;;
;;;  Accessor and functions for graphing link types:
;;;

(defun link-type-info-fn (lt)
  (list (format nil "~a" (name lt))))

(defun make-link-type-graph (&key (lt (get-link-type 'link))
				  (passes 1)
				  (orientation :horizontal) (sf 1.0))
  (create-shapes-from-graph
   :shape-name (string (name lt))
   :graph-list (make-digraph-node-list (list lt)
		      :backend-text-function #'link-type-info-fn
		      :backend-children-function #'children
		      :min-level-gap 50
		      :x-scale sf
		      :y-scale sf
		      :depth-limit 10
		      :bary-center-passes passes
		      :orientation orientation)
	 :orientation orientation
	 :scale sf))

;;;
;;;  Functions to convert graph elements to Picasso shapes
;;;

(defun add-node-to-graph (graph-node &optional (orientation :horizontal))
  (if (graph-node-dummy-p graph-node)
      (add-object *hyperdoc-graph* (dummy-node-to-segment graph-node orientation))
    (add-object *hyperdoc-graph* (graph-node-to-box graph-node))))

(defun add-line-to-graph (x1 y1 x2 y2 &optional (edge-weight 1))
  (add-object *hyperdoc-graph* (graph-line-to-segment x1 y1 x2 y2 edge-weight)))

(defun graph-node-to-box (graph-node)
  "returns a box object representing the graph-node, for use in a graphical
   browser"
  (let* ((ll (make-2d-point :x (graph-node-left-x graph-node)
			    :y (- (graph-node-top-y graph-node)
				  (graph-node-y-size graph-node))))
	 (obj (graph-node-backend-object graph-node))
	 (color (cond ((typep obj 'node)
		       (if (visited? obj)  *visited-node-color*
			 *unvisited-node-color*))
		      (t (get-color "black"))))
	 (box
	  (make-node-box 
	   :name (name obj)
	   :my-obj obj
	   :line-width 2 
	   :hook-pt ll
	   :color color
	   :width (graph-node-x-size graph-node)
	   :height (graph-node-y-size graph-node)
	   :sub-objs
	   (list 
	    (make-annotation  
	     :allow-other-keys t
	     :name (format nil "~a-label" (name obj))
	     :text (car (graph-node-text-list graph-node)) 
	     :color color
	     :lower-left ll
	     :height (graph-node-y-size graph-node)
	     :width (graph-node-x-size graph-node)
	     :just :CC )))))
    (setf (lower-left (car (sub-objs box))) ll)
    box))

(defun dummy-node-to-segment (graph-node orientation)
  (let* ((start (make-2d-point :x (graph-node-left-x graph-node)
			      :y (graph-node-top-y graph-node)))
	(end (if (eq orientation :horizontal)
		 (make-2d-point :x (+ (graph-node-left-x graph-node)
				      (graph-node-x-size graph-node))
				:y (graph-node-top-y graph-node))
	       (make-2d-point :x (graph-node-left-x graph-node)
			      :y (+ (graph-node-top-y graph-node)
				    (graph-node-y-size  graph-node)))))
	(seg (make-line-segment :line-width 1
				:color *link-color*
				:hook-pt start)))
    (setf (end-pt seg) end)
    seg))
	       

(defun graph-line-to-segment (x1 y1 x2 y2 &optional (edge-weight 1))
  ;; should modify this to allow different line styles according to
  ;; edge weight
  (declare (ignore edge-weight))
  (let ((seg (make-line-segment :line-width 1 
				:color *link-color*
				:hook-pt (make-2d-point :x x1 :y y1))))
    ;; the new-instance method for line-segments seems to throw away
    ;; the ctrl-pts argument, so for now I'm setting the endpt explicitly:
    (setf (end-pt seg) (make-2d-point :x x2 :y y2))
    seg))
  
;;;
;;;  Top-level function to create a graph structure out of a hyperdocument:
;;;

(defun make-hyperdoc-graph (hyperdoc &key (passes $hip-passes) (orientation :horizontal) (sf 1.0))

  (when (nodes hyperdoc)
	(create-shapes-from-graph
	 :shape-name (string (name hyperdoc))
	 :graph-list (make-digraph-node-list 
		     ;; (list (start-node hyperdoc))
		      ;; roots are all nodes w/no incoming links:
		      (remove-if #'(lambda (n) (filter (links-into n)))
				 (filter (nodes hyperdoc)))
		      :backend-text-function #'node-info-fn
		      :backend-children-function #'get-node-successors
		      :min-level-gap 100
		      :x-scale sf
		      :y-scale sf
		      :depth-limit 3
		      :bary-center-passes passes
		      :orientation orientation)
	 :orientation orientation
	 :scale sf)))

(defun make-hyperdoc-ps-graph (hyperdoc &optional 
					(orientation :horizontal) 
					(sf 0.8)
					(file (picasso-path "lib/hip/hyperdoc-ps-graph.ps")))
  (make-postscript-digraph
   (list (start-node hyperdoc))
   :split-near-children-p nil
   :backend-text-function #'node-info-fn
   :backend-children-function #'get-node-successors
   :min-level-gap 100
   :x-scale sf
   :y-scale sf
   :depth-limit 2
   :bary-center-passes 2
   :orientation orientation
   :output-filename file))

;;;  This function returns a list of lists, each element of which is
;;;  an ordered list of the nodes at that level of the digraph:
(defun make-digraph-node-list (graph-list &key
				(backend-text-function '$no-function$ text-fn-p)
				(backend-children-function '$no-function$ children-fn-p)
				(backend-edge-weight-function #'default-edge-weight-function)
				(make-levels-from-top-p nil)
				(generate-all-children-p nil)
				(depth-limit nil)
				(bary-center-passes 2)
				(verbose-execution-p t)
				(verbose-nodes-p nil)
				(split-near-children-p t)
				(orientation :horizontal)
				(min-level-gap 150)
				(min-node-gap 25)
				(x-scale 1.0)
				(y-scale 1.0)
				(output-filename (picasso-path "lib/hip/browser-graph.ps"))
				)
  (declare (ignore output-filename x-scale y-scale))
  "same as make-postscript-digraph, except it stops before calling the
   function to generate postscript"

  ;; Make sure that the caller of this function provided all of the
  ;; interesting information before proceeding 
   (assert text-fn-p ()
	  "You did not provide a function to describe your backend objects")
   (assert children-fn-p ()
	  "You did not provide a function to generate the children of your backend objects")
   (setf *backend-text-function* backend-text-function
	 *backend-children-function* backend-children-function
	 *backend-edge-weight-function* backend-edge-weight-function)

  
   ;; Control the printing of status messages as we proceed
   (setf *graph-loudly* verbose-execution-p)

   ;; Control the amount of text displayed at each node in the final graph
   (setf *verbose-node-text* verbose-nodes-p)

   ;; Indicate whether or not to include individuals when selecting nodes for the graph
   (setf *generate-all-children-p* generate-all-children-p)

   ;; Tell the PostScript output routine whether or not to try
   ;; splitting lines to adjacent sets of children 
   (setf *split-near-children-p* split-near-children-p)
  
   (setf *graph-list*
	(place-levels-in-graph
	  (place-nodes-within-level
	    (determine-size-of-nodes
	      (sort-levels-by-bary-center
		(make-nested-list-from-sorted-list
		  (remove-long-edges-from-graph-list
		    (make-graph-list-from-roots graph-list
						:depth-limit depth-limit
						:make-levels-from-top-p make-levels-from-top-p))
		  #'graph-node-level)
		bary-center-passes))
	    min-node-gap orientation)
	  min-level-gap orientation))
   *graph-list*)


;;;
;;;   Functions to convert the graph-node-list into a set of Picasso
;;;   shapes.
;;;

(defun hip-draw-edge-to-single-child (x1 y1 parent child orientation)
  "Given the starting position for the edge (either the end of the
   line from the parent node or the split point for clusters of children) 
   this produces a line segment to the appropriate point on the child"

  ;; A New! Improved!! Whiter!!! feature that allows you to specify the type of edges to draw between the
  ;; given parent and child.  The function stored in *BACKEND-EDGE-WEIGHT-FUNCTION* should return a small
  ;; integer in the approximate range 1-5 for specifying the thickness of the connecting edge.
  (let ((edge-weight (funcall *backend-edge-weight-function*
			      (graph-node-backend-object (find-real-parent parent))
			      (graph-node-backend-object (find-real-child child))))
	x2 y2)
    (ccase orientation
      (:vertical (setf
		   ;; This line connects to the center of the top edge of the child box
		   x2 (+ (/ (graph-node-x-size child) 2.0)
			     (graph-node-left-x child))
		   y2 (graph-node-top-y child)))
      (:horizontal (setf
		     ;; This line connects to the center of the left edge of the child box
		     x2 (graph-node-left-x child)
		     y2 (- (graph-node-top-y child)
			   (/ (graph-node-y-size child) 2.0)))))
    (hip-emit-code-for-typed-edge x1 y1 x2 y2 edge-weight)))

(defun hip-emit-code-for-typed-edge (x1 y1 x2 y2 edge-weight)
  "this is supposed to use the edge-weight to make the line style
   different, but since we're not using postscript, that stuff belongs
   with the link creation"
  (assert (typep edge-weight '(integer 1 5)) (edge-weight) "The edge type must be an integer 1-5")
  (add-line-to-graph x1 y1 x2 y2 edge-weight))

(defun hip-draw-edges-to-child-group (x1 y1 parent child-group inter-level-gap orientation)
  "Given a point near the parent an a list of adjacent children this function produces
   a line segment from the parent to a split-point  near the children,
   and then individual line segments from the split point to each child." 

  (if (= (length child-group) 1)
      (hip-draw-edge-to-single-child x1 y1 parent (first child-group) orientation)

      ;; There are at least two children in this group so we need a split point
      (let* (x2 y2
	     (last-child (first (last child-group)))

	     ;; Calculate the distance into the gap for the split point
	     (offset-1 (* 0.8 inter-level-gap))
	     ;; And the center of the children contained by this
	     ;; group, which is the average of the BOTTOM 
	     ;; of the first child and the TOP of the last child
	     (offset-2 (ccase orientation
			 (:vertical (/ (+ (graph-node-left-x (first child-group))
					  (+ (graph-node-left-x
					      last-child) 
					     (graph-node-x-size last-child)))
				       2.0))
			 
             ;;;  THE FOLLOWING MAY WELL NEED TO BE REVISED FOR
	     ;;;  OUTPUTTING SHAPES, SINCE WE THINK Y=0 IS THE TOP...
	     ;; REMEMBER THIS WELL, MY SON... We start at postion 0
	     ;; for the first node in the level, adding as we go along
	     ;; the level. PostScript thinks Y=0 is the BOTTOM of the
	     ;; page, so the first node in the list gets drawn at the
	     ;; lowest position in the graph.  HENCE, the bottom of
	     ;; the group is the top of the first child MINUS its
	     ;; height, and the top of the group is the top of the
	     ;; last child
			 (:horizontal (/ (+ (- (graph-node-top-y
						(first child-group))
					       (graph-node-y-size (first child-group)))
					    (graph-node-top-y last-child))
				       2.0)))))
	(ccase orientation
	  (:horizontal (setf x2 (+ x1 offset-1)
			   y2 offset-2))
	  (:vertical (setf x2 offset-2
			     y2 (- y1 offset-1))))

	;;Now draw a line from the parent to the split, and then
	;;connect the split to each child 
	(add-line-to-graph x1 y1 x2 y2) 
	(dolist (next-child child-group)
	  (hip-draw-edge-to-single-child x2 y2 parent next-child orientation)))))

(defun hip-extend-parent-to-end-of-level (parent orientation size-of-level)
  "This draws a line from the end of the parent box to the end of the
   level, and returns a list containing the x and y coordinates of the end of that line."
  (let ((parent-x (+ (graph-node-left-x parent)
		     (ccase orientation
		       (:horizontal (graph-node-x-size parent))
		       (:vertical (/ (graph-node-x-size parent)
				     2.0)))))
	(parent-y (- (graph-node-top-y parent)
		     (ccase orientation
		       (:vertical (graph-node-y-size parent))
			(:horizontal (/ (graph-node-y-size parent)
					2.0)))))
	(level-x (+ (graph-node-left-x parent)
		    (ccase orientation
		      (:horizontal size-of-level)
		      (:vertical (/ (graph-node-x-size parent)
				    2.0)))))
	(level-y (- (graph-node-top-y parent)
		    (ccase orientation
		      (:vertical size-of-level)
		      (:horizontal (/ (graph-node-y-size parent)
				      2.0))))))

    ;; Write a line from the edge of the box to the end of the level,
    ;; but not for the longest box(es) on the level 
    (unless (and (= parent-x level-x) (= parent-y level-y))
	    (add-line-to-graph parent-x parent-y level-x level-y))

    ;; Return the end of the level so that the other functions can use it
    (list level-x level-y)))

;;;  Corresponds to write-postscript-edges-to-children:
(defun create-edges-to-children (parent-node orientation size-of-level)
  "This function takes a parent node and creates shapes representing
   the edges connecting the parent to each of its children.  The
   edges are drawn as a straight line which extends to the 
   end of the current level and then branches out to each of the
   children.  For groups of 2 or more adjacent children, a single line
   is drawn to a split-point near the children and then lines fan 
   out to each child in the group.  Each parent can have multiple such groups."

  ;;  Original code grouped all edges as a single path, so we might
  ;;  want to make them sub-objs of a single composite shape - but for
  ;;  now I'm creating them as sub-objs of the top-level shape
  
  (let* ((inter-level-gap (ccase orientation
			    (:horizontal (- (graph-node-left-x 
					     (first (graph-node-children parent-node)))
					    (graph-node-left-x parent-node)
					    size-of-level))
			    (:vertical (- (graph-node-top-y parent-node)
					  (graph-node-top-y 
					   (first (graph-node-children parent-node)))
					  size-of-level))))
	 (parent-end-list 
	     (hip-extend-parent-to-end-of-level parent-node orientation size-of-level))
	 (parent-x (first parent-end-list))
	 (parent-y (second parent-end-list))
	 (grouped-children (find-groups-of-children parent-node)))

    ;; Tell the user HOW we have mung'ed the list of adjacent children
    (when (and *graph-loudly* (/= (length (graph-node-children parent-node))
				  (length grouped-children)))
      (format t "Decided to draw ~d children of ~a as ~d group~:P~%"
	      (length (graph-node-children parent-node))
	      (first (graph-node-text-list parent-node))
	      (length grouped-children)))
    
    (dolist (next-group grouped-children)
      (hip-draw-edges-to-child-group parent-x parent-y parent-node next-group 
				 inter-level-gap orientation))
    ))

;;;  Corresponds to write-postscript-file-from-graph:
(defun create-shapes-from-graph (&key shape-name graph-list orientation scale)
  "This takes a fully specified graph and produces a Picasso shape
   which is the graphical representation. In'shalla..." 

  ;; Start a new shape:
  (setf *hyperdoc-graph* (make-2d-shape :name shape-name))

  ;;  May not need these counters, but I'm hanging on to them in case
  ;;  I decide to gensym-up some names or something...
  (let ((node-counter 0)
	(last-node-counter 0)
	(level-counter 0))
    
    ;; Now put the boxes and edges for each node
    (dolist (next-level graph-list)
	    (let* ((size-accessor (ccase orientation
					 (:vertical #'graph-node-y-size)
					 (:horizontal #'graph-node-x-size)))
		   (size-of-level (apply #'max (mapcar size-accessor next-level))))

	  (dolist (next-node next-level)
	
	    ;; Create a box representing this node:
	    (add-node-to-graph next-node orientation)

	    ;; Connect this node to all of its children
	    (when (graph-node-children next-node)
	      (create-edges-to-children next-node orientation size-of-level))

	    (incf node-counter)))

	(if *graph-loudly* (format t "Creating Level ~d~%" level-counter))
	
	;; Adjust the counters to go on to the next level
	(incf level-counter)
	(setf last-node-counter node-counter))
    (2d-scale *hyperdoc-graph* scale 0 0)
    *hyperdoc-graph* ))
    
;;  Not using this for now - I don't think we really need to write
;;  stuff to a file or worry about panning:

#|
(defun write-graph-file-header (graph-list stream hyperdoc)
  "This writes out a header for the graph of the given hyperdoc"
  (format stream "~%;;Automatically generated digraph for hyperdocument ~a~%" (name hyperdoc)))
    
(defun write-postscript-panning-code (x-scale y-scale stream)
  "This repeatedly translates and redraws sections of the graph, one page at a time."

  ;; Figure out how many 8"x10.5" pages are needed to cover the graph
  (let* ((vertical-points-per-page (/ 756.0 y-scale))
	 (horizontal-points-per-page (/ 576.0 x-scale))
	 (vertical-pages (ceiling (/ *graph-y-size* vertical-points-per-page)))
	 (horizontal-pages (ceiling (/ *graph-x-size* horizontal-points-per-page))))

    ;; Tell the user how big this thing is, in case they are surprised
    (when *graph-loudly*
      (format t "~%The graph is ~d page~:P wide, ~d page~:P high, and was saved as ~A~%"
	      horizontal-pages vertical-pages (pathname stream)))
    
    ;; We have to redo the drawing for each page, after translating
    ;; the coordinate system first so that the current pane to be
    ;; displayed starts at the 0,0 corner of the printer 
    (do* ((horiz-page-count 0 (1+ horiz-page-count))
	  (x-translation 0 (- x-translation horizontal-points-per-page)))
	 ((= horiz-page-count horizontal-pages))
      (do* ((vert-page-count 0 (1+ vert-page-count))
	    (y-translation 0 (- y-translation vertical-points-per-page)))
	   ((= vert-page-count vertical-pages))

	;; The rest of the file written by
	;; WRITE-POSTSCRIPT-FILE-FROM-GRAPH defines a function called 
	;; drawGraph which contains all of the commands needed to
	;; produce the graph.  Just keep calling it to create the
	;; graph, which will be clipped to fit the pane currently visible under these 
	;; translations, until all portions of it have been drawn.

	(format stream "~d ~d scale~%" x-scale y-scale)
	(format stream "~d ~d translate~%drawGraph~%showpage~2%"
		x-translation y-translation)))))
	
|#
