;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; simple-graphic-browser: A very fast graphic browser.
;;;
;;; A simple-graphic-browser can display lines, annotations and symbols.
;;; Lines are line segments, annotations are scalable text and symbols
;;; are scalable bitmaps.
;;; 
;;; The speed is derived from the static nature of the data structures.
;;; The typical use is to load it up with a bunch of objects, then
;;; "initialize" the browser.  This function precomputes everything possible.
;;;
;;; $Author: johnb $
;;; $Source: /pic2/picasso/new/widgets/graphic/RCS/sgg-browser.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 1991/07/19 18:36:31 $
;;;

(in-package "PT")

;;;
;;; Class definition for simple-graphic-browser (sgg)
;;;
(defclass simple-graphic-browser (widget simple-graphic-gadget)
  ((foreground :initform "white")
   (background :initform "black")
   (selection :initform nil :type list :accessor selection)
   (grf-cursor :initform nil :accessor grf-cursor :allocation :class)
   (zoom-cursor :initform nil :accessor zoom-cursor :allocation :class)
   (pan-cursor :initform nil :accessor pan-cursor :allocation :class)
   (wait-cursor :initform nil :accessor wait-cursor :allocation :class)
   (search-rad :initform 50 :type integer :accessor search-rad)
   (event-mask :initform '(:button-press :key-press :exposure))
   (object-extents :initform nil :type array :accessor object-extents)
   (copy-gc :initform nil :type vector :reader copy-gc)
   (gc-spec :initform '((copy-gc (:foreground "black" :background "black"
					      :exposures :off))))
   ))

;;;
;;; Constructor Function...
;;;
(defun make-simple-graphic-browser (&rest keys)
  (apply #'make-instance 'simple-graphic-browser :allow-other-keys t keys))

(defmethod new-instance ((self simple-graphic-browser)
			 &key
			 (ignore nil)
			 &allow-other-keys)
  (declare (ignore ignore))
  (call-next-method)
  (unless (zoom-cursor self)
	  (setf (pan-cursor self)
		(make-cursor :file "arrow_cross.cursor" 
			     :mask-file "arrow_cross_mask.cursor"))
	  (setf (wait-cursor self) 
		(make-cursor :file "clock.cursor" 
			     :mask-file "clock_mask.cursor"))
	  (setf (zoom-cursor self)
		(make-cursor :file "zoom.cursor" 
			     :mask-file "zoom_mask.cursor"))
	  (setf (grf-cursor self)
		(make-cursor :file "tcross.cursor" 
			     :mask-file "tcross_mask.cursor")))
  (setf (cursor self) (grf-cursor self))
  self)

(defmethod (setf selection) (new-items (self simple-graphic-browser))
  (setf (cursor self) (wait-cursor self))
  (flush-display)
  (let* ((sel (selection self))
	 (to-del (set-difference sel new-items))
	 (to-add (set-difference new-items sel))
	 (ht (fgo-id->idx self))
	 (objects (objects self))
	 o)
	(setf (slot-value self 'selection) new-items)
	(dolist (obj to-del)
		(setq o (aref objects (gethash obj ht)))
		(setf (fgo-selected o) nil))
	(dolist (obj to-add)
		(setq o (aref objects (gethash obj ht)))
		(if (fgo-selectable o) (setf (fgo-selected o) t))))
  (repaint self)
  (setf (cursor self) (grf-cursor self)))

(defun sgb-find-sel-near (self x y)
  (let* ((rad (search-rad self))
	 (mx (mx self))
	 (my (my self))
	 (wc-rad (max (abs (/ rad mx)) (abs (/ rad my))))
	 (dist 0)
	 (min-dist most-positive-fixnum)
	 (objects (objects self))
	 wx wy obj
	 (items nil))
	(map-dc-to-wc self x y wx wy)
	(setq rad (* rad rad)
	      wc-rad (* wc-rad wc-rad))
	(dotimes (i (length objects))
		 (setq obj (aref objects i))
		 (when (and (fgo-selectable obj)
			    (fgo-visible obj)
			    (> wc-rad (dist-to-bb (fgo-extent obj) wx wy)))
		       (setq dist (dist-to-object self obj x y))
		       (when (< dist rad)
			     (if (= dist min-dist)
				 (push (fgo-id obj) items)
				 (when (< dist min-dist)
				       (setq min-dist dist)
				       (setq items (list (fgo-id obj))))))))
	items))

(defun sgb-find-inside (self bb)
  (let* ((objects (objects self))
	 (x1 (first bb))
	 (y1 (second bb))
	 (x2 (third bb))
	 (y2 (fourth bb))
	 wx1 wy1 wx2 wy2
	 ox1 oy1 ox2 oy2
	 obj obb
	 (items nil))
	(map-dc-to-wc self x1 y2 wx1 wy1)
	(map-dc-to-wc self x2 y1 wx2 wy2)
	(dotimes (i (length objects))
		 (setq obj (aref objects i))
		 (setq obb (fgo-extent obj)
		       ox1 (first obb)
		       oy1 (second obb)
		       ox2 (third obb)
		       oy2 (fourth obb))
		 (if (and (fgo-selectable obj)
			  (fgo-visible obj)
			  (pt-in-bb ox1 oy1 wx1 wy1 wx2 wy2)
			  (pt-in-bb ox2 oy2 wx1 wy1 wx2 wy2))
		     (push (fgo-id obj) items)))
	items))

(defhandler set-selection ((self simple-graphic-browser)
			   &key x y &allow-other-keys
			   &default (:button-press :detail :left-button))
  (setf (selection self) (sgb-find-sel-near self x y)))

(defhandler extend-del-selection ((self simple-graphic-browser) &key x y 
				  &allow-other-keys
				  &default (:button-press :detail
							  :right-button))
  (let ((sel (selection self))
	(new-items nil))
       (setq new-items (sgb-find-sel-near self x y))
       (setf (selection self) 
	     (set-difference (union sel new-items)
			     (intersection sel new-items)))))

(defhandler select-region ((self simple-graphic-browser)
			   &key x y &allow-other-keys
			   &default (:button-press :detail :middle-button))
  (let* ((bb (get-user-bb self x y)))
	(if bb
	    (setf (selection self) (sgb-find-inside self bb)))))

(defhandler extend-region ((self simple-graphic-browser)
			   &key x y &allow-other-keys
			   &default (:button-press :shift :middle-button))
  (let* ((bb (get-user-bb self x y))
	 (sel (selection self))
	 (new-items nil))
	(when bb
	      (setq new-items (sgb-find-inside self bb))
	      (setf (selection self) 
		    (set-difference (union sel new-items)
				    (intersection sel new-items))))))

(defhandler zoom-dynamic ((self simple-graphic-browser)
			  &key x y &allow-other-keys
			  &default (:button-press :control :left-button))
  (let* ((x1 x)
	 (y1 y)
	 (cursor (zoom-cursor self))
	 (x2 0)
	 (y2 0)
	 (fg (foreground self))
	 (wx1 0)
	 (wy1 0)
	 (wx2 0)
	 (wy2 0))
	(multiple-value-setq (x2 y2)
			     (get-region self x1 y1 :cursor cursor :fg fg))
	(if (or (< (abs (- x1 x2)) 5) (< (abs (- y1 y2)) 5))
	    (xlib:bell (res (current-display))) ;; Too small!
	    (progn
	     (map-dc-to-wc self (min x1 x2) (max y1 y2) wx1 wy1)
	     (map-dc-to-wc self (max x1 x2) (min y1 y2) wx2 wy2)
	     (set-world self wx1 wy1 wx2 wy2)))))

(defhandler pan-dynamic ((self simple-graphic-browser)
			 &key x y &allow-other-keys
			 &default (:button-press :control :middle-button))
  (let* ((res (res self))
	 (x1 x)
	 (y1 y)
	 (width (width self))
	 (height (height self))
	 (cursor (pan-cursor self))
	 (x2 0)
	 (y2 0)
	 (dx 0)
	 (dy 0)
	 (wx1 0)
	 (wy1 0)
	 (wx2 0)
	 (wy2 0)
	 (gc (copy-gc self))
	 (pm (xlib:create-pixmap :width (* 3 width) :height (* 3 height)
				 :depth (xlib:drawable-depth res)
				 :drawable res)))
	(when (null pm) 
	      (xlib:bell (res (current-display)))
	      (return-from pan-dynamic))
	(xlib:draw-rectangle pm gc 0 0 (* 3 width) (* 3 height) t)
	(sgg-repaint self pm width height)
	(xlib:copy-area pm gc width height width height res 0 0)
	(grab-mouse self :cursor cursor
		    :event-mask '(:pointer-motion :button-release))
	;; start processing -- drag and wait for button-released
	(block event-loop
	       (event-loop :handler
#'(lambda (&rest args &key x y event-key &allow-other-keys)
	  (cond ((eq event-key :motion-notify)
		 (setq x2 x y2 y)
		 (xlib:draw-rectangle res gc 0 0 width height t)
		 (xlib:copy-area pm gc
				 (- width (- x2 x1)) (- height (- y2 y1))
				 width height 
				 res 0 0))
		((eq event-key :button-release)
		 (setq x2 x y2 y)
		 (return-from event-loop))
		(t
		 (apply #'dispatch-event args)))
	  nil)))
	;; Free up pixmap
	(xlib:free-pixmap pm)
	;; ungrab mouse
	(ungrab-mouse)
	(map-dc-to-wc self x1 y1 wx1 wy1)
	(map-dc-to-wc self x2 y2 wx2 wy2)
	(setq dx (- wx2 wx1))
	(setq dy (- wy2 wy1))
	(set-world self (- (xmin self) dx)
		   (- (ymin self) dy)
		   (- (xmax self) dx)
		   (- (ymax self) dy))))

(defhandler drop-on-floor ((self simple-graphic-browser) &rest args
			   &default ((:button-press)
				     (:button-release)
				     (:pointer-motion)
				     (:key-press)))
  (declare (ignore self args))
  nil)

