;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; 2d-mapper-mixin: A mixin class to map and clip 2d coordinates into
;;; device coordinates.
;;;
;;; A 2d-mapper-mixin has a world coordinate system, defining a lower
;;; left corner in the floating point coordinates 
;;; (xmin, ymin) and an upper right corner in floating point
;;; coordinates (xmax, ymax).  The setf methods can be used to change
;;; these values, or the set-lower-left, set-upper-right, and set-world
;;; methods can alternatively be used.  Conceptually, these get mapped into
;;; a space whose upper left corner is (0,0) and whose lower right corner is
;;; (width,height).  Clipping routines are provided to clip line segments and
;;; linestrings against the window.
;;;
;;; Giving these coordinates, one could easily define a coordinate system
;;; such that a unit square in world coordinates is anything but square,
;;; depending on the shape of the window.  To make a square come out square
;;; no matter what the shape of the window, the mapping slot can be set to
;;; :isotropic (preserves shape) or :anisotropic.  The setf method should
;;; be used to change this value.
;;;
;;; In summary:
;;;	(xmin, ymin)	lower left corner, in world coordinates.
;;;	(xmax, ymax)	upper right corner, in world coordinates.
;;;	mapping		type of mapping, :isotropic or :anisotropic. 
;;;

(in-package "PT")

;;;
;;; Class definition for 2d-mapper-mixin
;;;

(defclass 2d-mapper-mixin (pmc) 
  (
   (width :initform 1 :type integer 
	  ; :accessor width
	  )
   (height :initform 1 :type integer :accessor height)
   ;; World coordinate information
   (xmin :initarg :xmin :initform 0.0 :type float :reader xmin)
   (ymin :initarg :ymin :initform 0.0 :type float :reader ymin)
   (xmax :initarg :xmax :initform 1.0 :type float :reader xmax)
   (ymax :initarg :ymax :initform 1.0 :type float :reader ymax)
   (mapping 
    :initarg :mapping 
    :initform :isotropic 
    :type keyword 
    :reader mapping)
   ;; Stores mapping from world to device coordinates.
   (mx :initform 1.0 :type float :accessor mx)
   (bx :initform 0.0 :type float :accessor bx)
   (my :initform -1.0 :type float :accessor my)
   (by :initform 0.0 :type float :accessor by)
   ;; Stores the clip coordiantes of the left, right, top and bottom
   ;; of the window.
   (xleft :initform 0.0 :type float :reader xleft)
   (xright :initform 1.0 :type float :reader xright)
   (ybottom :initform 0.0 :type float :reader ybottom)
   (ytop :initform 1.0 :type float :reader ytop)))

(defmethod width ((self 2d-mapper-mixin)
		  &key &allow-other-keys)
  (slot-value self 'width))

(defmethod (setf width) (w (self 2d-mapper-mixin))
  (setf (slot-value self 'width) w))

;;;
;;; Constructor Function...
;;;

(defmethod new-instance ((self 2d-mapper-mixin)
			 &key
			 (ignore nil)
			 &allow-other-keys)
  (declare (ignore ignore))
  ;; Let any super-classes initialize themselves
  (call-next-method)

  ;; Make sure the map-params are OK
  (recache-map self)
  self)

;;;
;;; Define the setf methods that affect the world coordinate system
;;;

(defmethod (setf mapping) (value (self 2d-mapper-mixin))
  (unless (eq value (mapping self))
	  (setf (slot-value self 'mapping) value)
	  (recache-map self)))

(defmethod (setf xmin) (value (self 2d-mapper-mixin))
  (unless (= value (xmin self))
	  (setf (slot-value self 'xmin) value)
	  (recache-map self)))

(defmethod (setf ymin) (value (self 2d-mapper-mixin))
  (unless (= value (ymin self))
	  (setf (slot-value self 'ymin) value)
	  (recache-map self)))

(defmethod (setf xmax) (value (self 2d-mapper-mixin))
  (unless (= value (xmax self))
	  (setf (slot-value self 'xmax) value)
	  (recache-map self)))

(defmethod (setf ymax) (value (self 2d-mapper-mixin))
  (unless (= value (ymax self))
	  (setf (slot-value self 'ymax) value)
	  (recache-map self)))

(defmethod set-lower-left ((self 2d-mapper-mixin) x y)
  (unless (and (= (slot-value self 'xmin) x)
	       (= (slot-value self 'ymin) y))
	  (setf (slot-value self 'xmin) x)
	  (setf (slot-value self 'ymin) y)
	  (recache-map self)))

(defmethod set-upper-right ((self 2d-mapper-mixin) x y)
  (unless (and (= (slot-value self 'xmax) x)
	       (= (slot-value self 'ymax) y))
	  (setf (slot-value self 'xmax) x)
	  (setf (slot-value self 'ymax) y)
	  (recache-map self)))

(defmethod set-world ((self 2d-mapper-mixin) x1 y1 x2 y2)
  (unless (and (= (slot-value self 'xmin) x1)
	       (= (slot-value self 'ymin) y1)
	       (= (slot-value self 'xmax) x2)
	       (= (slot-value self 'ymax) y2))
	  (setf (slot-value self 'xmin) x1)
	  (setf (slot-value self 'ymin) y1)
	  (setf (slot-value self 'xmax) x2)
	  (setf (slot-value self 'ymax) y2)
	  (recache-map self)))


;;;
;;; Other methods that affect the world coordinate system
;;;
(defmethod pan ((self 2d-mapper-mixin) x-factor y-factor)
  (let* ((xmin (slot-value self 'xmin))
	 (ymin (slot-value self 'ymin))
	 (xmax (slot-value self 'xmax))
	 (ymax (slot-value self 'ymax))
	 (dx (* (- xmax xmin) (/ x-factor 2)))
	 (dy (* (- ymax ymin) (/ y-factor 2))))
	(set-world self
		   (+ (xmin self) dx) (+ (ymin self) dy)
		   (+ (xmax self) dx) (+ (ymax self) dy))))

(defmethod zoom-factor ((self 2d-mapper-mixin) factor)
  (if (<= factor 0.0)
      (error "2d-mapper-mixin.zoom-factor, illecgal factor ~s specified."
	     factor))
  (if (/= 1.0 factor)
      (let* ((xmin (slot-value self 'xmin))
	     (ymin (slot-value self 'ymin))
	     (xmax (slot-value self 'xmax))
	     (ymax (slot-value self 'ymax))
	     (cx (/ (+ xmin xmax) 2))
	     (cy (/ (+ ymin ymax) 2))
	     (dx (/ (- xmax xmin) (* 2 factor)))
	     (dy (/ (- ymax ymin) (* 2 factor))))
	    (set-world self (- cx dx) (- cy dy) (+ cx dx) (+ cy dy)))))

;;;
;;; Recalculate the world to device mapping function of a 2d-mapper-mixin.
;;;
(defmethod recache-map ((self 2d-mapper-mixin))
  (let* ((x (repaint-x self))
	 (y (repaint-y self))
	 (w (width self))
	 (h (height self))
	 (xmin (xmin self))
	 (ymin (ymin self))
	 (xmax (xmax self))
	 (ymax (ymax self))
	 (dx (- xmax xmin))
	 (dy (- ymin ymax))
	 (mapping (mapping self))
	 (mx 0.0)
	 (bx 0.0)
	 (my 0.0)
	 (by 0.0))
	(when (eq mapping :isotropic)
	      (if (> (* w (- dy)) (* h dx))
		  ;; Window is wider than high.
		  (let ((cx (/ (+ xmin xmax) 2)))
		       (setq dx (/ (* w (- dy)) h))
		       (setq xmin (- cx (/ dx 2)))
		       (setq xmax (+ cx (/ dx 2))))
		  ;; Window is higher than wide.
		  (let ((cy (/ (+ ymin ymax) 2)))
		       (setq dy (- (/ (* h dx) w)))
		       (setq ymax (- cy (/ dy 2)))
		       (setq ymin (+ cy (/ dy 2))))))
	(if (/= 0.0 dx) (setq mx (/ w dx)))
	(if (/= 0.0 dy) (setq my (/ h dy)))
	(setq bx (- x (* mx xmin)))
	(setq by (- y (* my ymax)))
	(setf (mx self) (float mx)
	      (bx self) (float bx)
	      (my self) (float my)
	      (by self) (float by)
	      (slot-value self 'xleft) (/ (- bx) mx)
	      (slot-value self 'ytop) (/ (- by) my)
	      (slot-value self 'xright) (/ (- w bx) mx)
	      (slot-value self 'ybottom) (/ (- h by) my))))

(defun ppu (self)
  (max (abs (mx self)) (abs (my self))))

