;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: johnb $
;;; $Source: /pic2/picasso/new/widgets/graphic/RCS/graphic-browser.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 1991/07/19 18:28:39 $
;;;
;;;
;;; The graphic-browser class is browser for graphic data structures,
;;; allowing the user to graphically select subparts of a graphics data
;;; structure, embodied as subclasses of the shape class.
;;;
;;; A graphic-browser introduces a few other conceptual structures
;;; beyond those of graphic gadgets.  Graphic-browsers have a selection
;;; set, stored in the "selection slot" which can be edited by the
;;; user, a list of visible scene-objects, which are just a flattened-out
;;; version of the graphical-object hierarchy stored in the "value" slot.
;;; a gravity radius (in pixels), and a list of selectables -- items that
;;; can be selected.  If the select-deep parameter is true, then all the
;;; sub-objects of a selected object are selected as well, but only the
;;; selected object is added to the selection.
;;;

(in-package "PT")

(defclass graphic-browser (widget graphic-gadget)
  ((selection :initform nil :type list :accessor selection)
   (selectables :initform nil :type list :accessor selectables)
   (select-deep :initform nil :type t :accessor select-deep)
   (grf-cursor :initform nil :accessor grf-cursor :allocation :class)
   (zoom-cursor :initform nil :accessor zoom-cursor :allocation :class)
   (pan-cursor :initform nil :accessor pan-cursor :allocation :class)
   (wait-cursor :initform nil :accessor wait-cursor :allocation :class)
   (search-rad :initform 50 :type integer :accessor search-rad)
   (background :initform "black")
   (foreground :initform "white")
   (copy-gc :initform nil :type vector :reader copy-gc)
   (gc-spec :initform '((copy-gc (:foreground "black" :background "black"
				  :exposures :off))))
   (highlight-font-list :initform nil :type list :accessor highlight-font-list)
   (event-mask :initform '(:button-press :key-press :exposure))))

(defun make-graphic-browser (&rest keys)
  (apply #'make-instance (cons 'graphic-browser keys)))

(defun make-wait-cursor ()
  (make-cursor :file "clock.cursor" :mask-file "clock_mask.cursor"))

(defun make-pan-cursor ()
  (make-cursor :file "arrow_cross.cursor" :mask-file "arrow_cross_mask.cursor"))

(defun make-grf-cursor ()
  (make-cursor :file "tcross.cursor" :mask-file "tcross_mask.cursor"))

(defun make-zoom-cursor ()
  (make-cursor :file "zoom.cursor" :mask-file "zoom_mask.cursor"))

(defmethod new-instance ((self graphic-browser)
			 &key
			 (highlight-font-list nil)
			 &allow-other-keys)
  (declare (ignore ignore))
  (call-next-method)
  (unless (zoom-cursor self)
	  (setf (wait-cursor self) (make-wait-cursor))
	  (setf (grf-cursor self) (make-grf-cursor))
	  (setf (zoom-cursor self) (make-zoom-cursor))
	  (setf (pan-cursor self) (make-pan-cursor)))
  (setf (cursor self) (grf-cursor self))
  (setf (highlight-font-list self)
	(if highlight-font-list 
	    highlight-font-list
	    (list (make-font :name "*helvetica-bold-o-*--34*")
		  (make-font :name "*helvetica-bold-o-*--20*")
		  (make-font :name "*helvetica-bold-o-*--14*")
		  (make-font :name "*helvetica-bold-o-*--10*")
		  (make-font :name "nil2"))))
  self)

(defmethod (setf selection) (new-items (self graphic-browser))
  (setf (cursor self) (wait-cursor self))
  (flush-display)
  (let* ((sel (selection self))
	 (to-del (set-difference sel new-items))
	 (ht (dl-tab self))
         (dl nil)
	 (sd (select-deep self))
	 (to-add (set-difference new-items sel))
	 (del-to-change (if sd (mapcan #'flatten to-del) to-del))
	 (add-to-change (if sd (mapcan #'flatten to-add) to-add)))
	(setf (slot-value self 'selection) new-items)
	(dolist (obj del-to-change)
	        (setq dl (gethash obj ht))
		(if (visible dl) (erase dl self))
		(unset-highlight dl)
		(if (visible dl) (draw dl self)))
	(dolist (obj add-to-change)
	        (setq dl (gethash obj ht))
		(if (visible dl) (erase dl self))
		(set-highlight self dl)
		(if (visible dl) (draw dl self))))
  (setf (cursor self) (grf-cursor self)))

(defun highlight-selection (self)
  (let* ((sel (selection self))
	 (ht (dl-tab self))
         (dl nil)
         (visible nil)
	 (sd (select-deep self))
	 (to-change (if sd (mapcan #'flatten sel) sel)))
	(dolist (obj to-change)
	        (setq dl (gethash obj ht))
	        (setq visible (visible dl))
		(if visible (erase dl self))
		(set-highlight self dl)
		(if visible (draw dl self)))))

(defun unhighlight-selection (self)
  (let* ((sel (selection self))
	 (ht (dl-tab self))
         (dl nil)
         (visible nil)
	 (sd (select-deep self))
	 (to-change (if sd (mapcan #'flatten sel) sel)))
	(dolist (obj to-change)
	        (setq dl (gethash obj ht))
	        (setq visible (visible dl))
		(if visible (erase dl self))
		(unset-highlight dl)
		(if visible (draw dl self)))))

(defun unset-highlight (dl)
  (cond ((annot-dl-p dl)
	 (setf (color dl) nil
	       (fonts dl) nil)
	 (invalidate dl))
	((segment-dl-p dl)
	 (setf (color dl) nil
	       (line-width dl) nil
	       (line-style dl) nil)
	 (invalidate dl))))

(defun set-highlight (self dl)
  (cond ((annot-dl-p dl)
	 (setf (color dl) "white"
	       (fonts dl) (highlight-font-list self))
	 (invalidate dl))
	((segment-dl-p dl)
	 (setf (color dl) "white"
	       (line-width dl) 2)
	 (invalidate dl))))

;;; ==================================================================
;;;
;;; class event handlers for graphic-browser
;;;

;; Repaint
(defhandler repaint ((self graphic-browser) &rest args
		     &default (:key-press :control #\l))
  (declare (ignore args))
  (setf (cursor self) (wait-cursor self))
  (flush-display)
  (repaint self)
  (setf (cursor self) (grf-cursor self)))

(defmethod dist-to-dl ((dl t) rad x y)
  (declare (ignore rad x y))
  most-positive-fixnum)

(defun find-sel-near (self x y)
  (let* ((selectables (selectables self))
	 (rad (search-rad self))
	 (dist 0)
	 (ht (dl-tab self))
	 (min-dist most-positive-fixnum)
	 (items nil))
	(dolist (obj selectables)
		(setq dist (dist-to-dl (gethash obj ht) rad x y))
		(if (= dist min-dist)
		    (push obj items)
		    (when (< dist min-dist)
			  (setq min-dist dist)
			  (setq items (list obj)))))
	items))

(defhandler set-selection ((self graphic-browser) &key x y &allow-other-keys
			   &default (:button-press :detail :left-button))
  (setf (selection self) (find-sel-near self x y)))

(defhandler extend-del-selection ((self graphic-browser) &key x y 
				  &allow-other-keys
				  &default (:button-press :detail
							  :right-button))
  (let ((sel (selection self))
	(new-items nil))
       (setq new-items (find-sel-near self x y))
       (setf (selection self) 
	     (set-difference (union sel new-items)
			     (intersection sel new-items)))))

(defmethod dl-bbox ((dl t))
  (list most-positive-fixnum most-positive-fixnum 
	most-negative-fixnum most-negative-fixnum))

(defun find-inside (self selectables x1 y1 x2 y2)
  (let ((rv nil)
	(ht (dl-tab self))
	(dl nil)
	(bb nil))
       (dolist (obj selectables)
	       (setq dl (gethash obj ht))
	       (when (and dl (visible dl))
		     (setq bb (dl-bbox dl))
		     (if (and bb
			      (<= x1 (first bb))
			      (<= y1 (second bb))
			      (>= x2 (third bb))
			      (>= y2 (fourth bb)))
			 (push obj rv))))
       rv))

(defun get-user-bb (self x1 y1)
  (let* ((cursor (zoom-cursor self))
	 (x2 0)
	 (y2 0)
	 (fg (foreground self))
	 )
	(multiple-value-setq (x2 y2)
			     (get-region self x1 y1 :cursor cursor :fg fg))
	(if (or (< (abs (- x1 x2)) 5) (< (abs (- y1 y2)) 5))
	    (progn
	     (xlib:bell (current-display))	;; Too small!
	     nil)
	    (list (min x1 x2) (min y1 y2) (max x1 x2) (max y1 y2)))))

(defhandler select-region ((self graphic-browser) &key x y &allow-other-keys
			   &default (:button-press :detail :middle-button))
  (let* ((bb (get-user-bb self x y))
	 (selectables (selectables self)))
	(if bb
	    (setf (selection self) 
		  (apply #'find-inside self selectables bb)))))

(defhandler extend-region ((self graphic-browser) &key x y &allow-other-keys
			   &default (:button-press :shift :middle-button))
  (let* ((bb (get-user-bb self x y))
	 (sel (selection self))
	 (selectables (selectables self))
	 (new-items nil))
	(when bb
	      (setq new-items (apply #'find-inside self selectables bb))
	      (setf (selection self) 
		    (set-difference (union sel new-items)
				    (intersection sel new-items))))))

;;;
;;; Do a two-pt drag to get the region of the zoom box.
;;;
(defhandler zoom-dynamic ((self graphic-browser) &key x y &allow-other-keys
			  &default (:button-press :control :left-button))
  (let* ((x1 x)
	 (y1 y)
	 (cursor (zoom-cursor self))
	 (x2 0)
	 (y2 0)
	 (fg (foreground self))
	 (wx1 0)
	 (wy1 0)
	 (wx2 0)
	 (wy2 0))
	(multiple-value-setq (x2 y2)
			     (get-region self x1 y1 :cursor cursor :fg fg))
	(if (or (< (abs (- x1 x2)) 5) (< (abs (- y1 y2)) 5))
	    (x-feep)	;; Too small!
	    (progn
	     (map-dc-to-wc self (min x1 x2) (max y1 y2) wx1 wy1)
	     (map-dc-to-wc self (max x1 x2) (min y1 y2) wx2 wy2)
	     (set-world self wx1 wy1 wx2 wy2)))))

;;;
;;; Do a dynamic pan on the graphic window.
;;;
(defhandler pan-dynamic ((self graphic-browser) &key x y &allow-other-keys
			 &default (:button-press :control :middle-button))
  (let* ((res (res self))
	 (x1 x)
	 (y1 y)
	 (width (width self))
	 (height (height self))
	 (cursor (pan-cursor self))
	 (x2 0)
	 (y2 0)
	 (dx 0)
	 (dy 0)
	 (wx1 0)
	 (wy1 0)
	 (wx2 0)
	 (wy2 0)
	 (gc (copy-gc self))
	 (pm (xlib:create-pixmap :width width :height height
				 :depth (xlib:drawable-depth res)
				 :drawable res)))
	(xlib:draw-rectangle pm gc 0 0 width height t)
	(xlib:copy-area res gc 0 0 width height pm 0 0)
	(grab-mouse self :cursor cursor
		    :event-mask '(:pointer-motion :button-release))
	;; start processing -- drag and wait for button-released
	(block event-loop
	       (event-loop :handler
#'(lambda (&rest args &key x y event-key &allow-other-keys)
	  (cond ((eq event-key :motion-notify)
		 (setq x2 x y2 y)
		 (clear self)
		 (xlib:copy-area pm gc 0 0 width height 
				 res (- x2 x1) (- y2 y1)))
		((eq event-key :button-release)
		 (setq x2 x y2 y)
		 (return-from event-loop))
		(t
		 (apply #'dispatch-event args)))
	  nil)))
	;; Free up pixmap
	(xlib:free-pixmap pm)
	;; ungrab mouse
	(ungrab-mouse)
	(map-dc-to-wc self x1 y1 wx1 wy1)
	(map-dc-to-wc self x2 y2 wx2 wy2)
	(setq dx (- wx2 wx1))
	(setq dy (- wy2 wy1))
	(set-world self (- (xmin self) dx)
		   (- (ymin self) dy)
		   (- (xmax self) dx)
		   (- (ymax self) dy))))

;; Catch garbage events...
(defhandler drop-on-floor ((self graphic-browser) &rest args
			   &default ((:button-press)
				     (:button-release)
				     (:pointer-motion)
				     (:key-press)))
  (declare (ignore self args))
  nil)
