;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Definition for annotation, a subclass of 2d-shapes.
;;;
;;; Annotation is a type of 2d-shape whose control points are the 9
;;; justification points of the text.
;;;
;;; The parametric description of a annotation is an integer between [1-9]
;;; that gives the control point.
;;;

(in-package "PT")

;;;
;;; Class definition for annotation
;;;
(defclass annotation (2d-shape)
  ((lower-left 
    :initform (make-2d-point :x 0.0 :y 0.0) 
    :type 2d-point 
    :reader lower-left)
   (width :initarg :width :initform 1.0 :type float 
	  ; :reader width
	  )
   (height :initarg :height :initform 1.0 :type float :reader height)
   (color :initarg :color :initform "white" :type t :reader color)
   (just :initarg :just :initform :LC :type keyword :reader just)
   (text :initarg :text :initform "" :type string :reader text)
   (fonts :initarg :fonts :initform nil :type font :reader fonts)))

(defmethod width ((self annotation) &key &allow-other-keys)
  (slot-value self 'width))

;;;
;;; Are we a annotation? Isn't that special!
;;;
(defun annotation-p (self) (typep self 'annotation))

(defun update-annotation-ctrl-pts (self)
  (let* ((ctrl-pts (ctrl-pts self))
	 (lower-left (lower-left self))
	 (x (2d-point-x lower-left))
	 (y (2d-point-y lower-left))
	 (w (width self))
	 (h (height self)))
	(when (null ctrl-pts)
	      (setf ctrl-pts (list (make-2d-point :x 0 :y 0)
				   (make-2d-point :x 0 :y 0)
				   (make-2d-point :x 0 :y 0)
				   (make-2d-point :x 0 :y 0)
				   (make-2d-point :x 0 :y 0)
				   (make-2d-point :x 0 :y 0)
				   (make-2d-point :x 0 :y 0)
				   (make-2d-point :x 0 :y 0)
				   (make-2d-point :x 0 :y 0)))
	      (setf (ctrl-pts self) ctrl-pts))
	(copy-2d (first ctrl-pts) x y)
	(copy-2d (second ctrl-pts) (+ x (/ w 2)) y)
	(copy-2d (third ctrl-pts) (+ x w) y)
	(copy-2d (fourth ctrl-pts) x (+ y (/ h 2)))
	(copy-2d (fifth ctrl-pts) (+ x (/ w 2)) (+ y (/ h 2)))
	(copy-2d (sixth ctrl-pts) (+ x w) (+ y (/ h 2)))
	(copy-2d (seventh ctrl-pts) x (+ y h))
	(copy-2d (eighth ctrl-pts) (+ x (/ w 2)) (+ y h))
	(copy-2d (ninth ctrl-pts) (+ x w) (+ y h))))

(defmethod (setf lower-left) (lower-left (self annotation))
  (copy-2d (slot-value self 'lower-left) 
	   (2d-point-x lower-left) (2d-point-y lower-left))
  (update-annotation-ctrl-pts self)
  (notify-geom-change self))

(defmethod (setf color) (value (self annotation))
  (unless (eq value (color self))
	  (setf (slot-value self 'color) value)
	  (notify-gc-change self)))

(defmethod (setf fonts) (fonts (self annotation))
  (unless (equalp fonts (fonts self))
	  (setf (slot-value self 'fonts) fonts)
	  (notify-geom-change self)))

(defmethod (setf text) (text (self annotation))
  (setf (slot-value self 'text) text)
  (notify-geom-change self))

(defmethod (setf width) (width (self annotation))
  (setf (slot-value self 'width) width)
  (update-annotation-ctrl-pts self)
  (notify-geom-change self))

(defmethod (setf height) (height (self annotation))
  (setf (slot-value self 'height) height)
  (update-annotation-ctrl-pts self)
  (notify-geom-change self))

(defmethod (setf just) (just (self annotation))
  (setf (slot-value self 'just) just)
  (notify-geom-change self))

(defmethod new-instance ((self annotation)
			 &key
			 (ignore nil)
			 &allow-other-keys)
  (declare (ignore ignore))
  (call-next-method)
  (setf (slot-value self 'fonts) 
	(list
	 (make-font :name "*helvetica-medium-r-*--34*")
	 (make-font :name "*helvetica-medium-r-*--20*")
	 (make-font :name "*helvetica-medium-r-*--14*")
	 (make-font :name "*helvetica-medium-r-*--10*")
	 (make-font :name "*helvetica-medium-r-*--8*")
	 (make-font :name "nil2")))
  (setf (slot-value self 'coord-list) self)
  (update-annotation-ctrl-pts self)
  self)

(defun make-annotation (&rest keys)
  (apply #'make-instance (cons 'annotation keys)))

;;;
;;; Make a copy of a annotation.
;;;
(defmethod copy ((self annotation))
  (let ((rv (call-next-method)))
       (setf (slot-value rv 'lower-left) (duplicate-2d (lower-left self))
	     (slot-value rv 'fonts) (copy-list (fonts self))
	     (slot-value rv 'text) (text self)
	     (slot-value rv 'width) (width self)
	     (slot-value rv 'height) (height self)
	     (slot-value rv 'just) (just self)
	     (slot-value rv 'coord-list) rv)
       (new-instance rv)))

;;;
;;; Return the gravity points for a annotation. This is a list of param-vals.
;;;
(defmethod gravity-pts ((self annotation) do-midpt)
  (declare (ignore do-midpt))
  '(1 2 3 4 5 6 7 8 9))

;;;
;;; Return the point on the annotation at the given parametric value.
;;; The value can range from [1-9].  The value returned
;;; can be used in any way seen fit -- it is a copy of the actual data.
;;;
(defmethod point ((self annotation) value)
  (if (<= 1 value 9)
      (duplicate-2d (nth (- value 1) (ctrl-pts self)))
      nil))

;;;
;;; Bounding box of a annotation.
;;;
(defmethod find-bbox ((self annotation))
  (let* ((lower-left (lower-left self))
	 (x (2d-point-x lower-left))
	 (y (2d-point-y lower-left))
	 (w (width self))
	 (h (height self)))
	(make-2d-bbox x y (+ x w) (+ y h))))

;;;
;;; Translate an annotation.  Just moves the lower-left and adjust ctrl-pts.
;;;
(defmethod 2d-translate ((self annotation) tx ty)
  (call-next-method)
  (let* ((lower-left (lower-left self))
	 (x (2d-point-x lower-left))
	 (y (2d-point-y lower-left)))
	(setf (2d-point-x lower-left) (+ x tx)
	      (2d-point-y lower-left) (+ y ty))
	(update-annotation-ctrl-pts self)
	(notify-geom-change self)))

;;;
;;; Rotate an annotation about (ox, oy).  Just rotates the lower-left.
;;;
(defmethod 2d-rotate ((self annotation) theta ox oy)
  (call-next-method)
  (let* ((lower-left (lower-left self))
	 (x (2d-point-x lower-left))
	 (y (2d-point-y lower-left))
	 (c (cos theta))
	 (s (sin theta)))
	(declare (float c s x y)
		 (number ox oy))
	(decf x ox)
	(decf y oy)
	(setf (2d-point-x lower-left) (+ (- (* x c) (* y s)) ox)
	      (2d-point-y lower-left) (+ (* x s) (* y c) oy))
	(update-annotation-ctrl-pts self)
	(notify-geom-change self)))

;;;
;;; Scale an annotation about (ox, oy).  Changes hook-p, width and height
;;;
(defmethod 2d-scale ((self annotation) sf ox oy)
  (call-next-method)
  (let* ((lower-left (lower-left self))
	 (x (2d-point-x lower-left))
	 (y (2d-point-y lower-left))
	 (w (width self))
	 (h (height self)))
	(declare (number sf x y w h ox oy))
	(decf x ox)
	(decf y oy)
	(setf (2d-point-x lower-left) (+ ox (* x sf))
	      (2d-point-y lower-left) (+ (* y sf) oy))
	(setf (slot-value self 'width) (* sf w))
	(setf (slot-value self 'height) (* sf h))
	(update-annotation-ctrl-pts self)
	(notify-geom-change self)))

;;;
;;; Return the coord-list of a annotation.  This either nil, if the
;;; annotation doesn't fit in its box in world coordinates, or the
;;; annotation itself.
;;;
(defmethod 2d-coord-list ((self annotation) xleft xright ybottom ytop w h)
  (declare (ignore w h))
  (let* ((lower-left (lower-left self))
	 (x (2d-point-x lower-left))
	 (y (2d-point-y lower-left))
	 (width (width self))
	 (height (height self)))
	;; Check for in-boundaries:
	(if (or (< (+ x width) xleft) (> x xright)
		(< (+ y height) ybottom) (> y ytop))
	    nil
	    self)))

;;;
;;; Find the nearest point on the annotation to a point.  Return a list
;;; of (parametric-value distance).
;;;
(defmethod nearest-pt ((self annotation) pt)
  (let* ((nptls (nearest-pt-to-linestr (ctrl-pts self) t pt))
	 (pv (1+ (mod (round (car nptls)) 9)))
	 (close (point self pv))
	 (dist (2d-pt-pt-dist pt close)))
	(free-2d close)
	(list pv dist)))
