;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Definition for annot-dl, a class for mapping
;;; from linestrings to lists that can be easily drawn in an X window.
;;;

(in-package "PT")

;;;
;;; Class definition for annot-dl
;;;
(defclass annot-dl (display-list)
  ((x :initform nil :type integer :accessor x)
   (y :initform nil :type integer :accessor y)
   (txt-width :initform nil :type integer :accessor txt-width)
   (txt-ascent :initform nil :type integer :accessor txt-ascent)
   (txt-descent :initform nil :type integer :accessor txt-descent)
   (str :initform nil :type string :accessor str)
   (color :initform nil :accessor color)
   (fonts :initform nil :accessor fonts)))

(defmethod new-instance ((self annot-dl)
			 &key
			 (ignore nil)
			 &allow-other-keys)
  (declare (ignore ignore))
  (setf (slot-value self 'prop-list)
	(list :background "black"
	      :foreground "white"
	      :font nil)))
  

(defun make-annot-dl (&rest args)
  (apply #'make-instance (cons 'annot-dl args)))

(defmethod draw ((self annot-dl) viewer)
  (validate self viewer)
  (let ((res (res viewer))
	(gc (draw-gc self)))
       (if gc
	   (xlib:draw-glyphs res gc (x self) (y self) (str self)))))

(defmethod erase ((self annot-dl) viewer)
  (validate self viewer)
  (let ((res (res viewer))
	(gc (erase-gc self)))
       (if gc
	   (xlib:draw-glyphs res gc (x self) (y self) (str self)))))

(defmethod dl-bbox ((self annot-dl))
  (let ((x (x self))
	(y (y self))
	(tw (txt-width self))
	(ta (txt-ascent self))
	(td (txt-descent self)))
       (list x (- y ta) (+ x tw) (+ y td))))

(defmethod dist-to-dl ((dl annot-dl) rad dx dy)
  (declare (ignore rad))
  (if (not (visible dl))
      (return-from dist-to-dl most-positive-fixnum))
  (let ((x (x dl))
	(y (y dl))
	(tw (txt-width dl))
	(ta (txt-ascent dl))
	(td (txt-descent dl)))
       (setq dx (cond ((< dx x) (- x dx))
		      ((> dx (+ x tw)) (- dx (+ x tw)))
		      (t 0)))
       (setq dy (cond ((< dy (- y ta)) (- (- y ta) dy))
		      ((> dy (+ y td)) (- dy (+ y td)))
		      (t 0)))
       (+ (* dx dx) (* dy dy))))

(defmethod do-validation ((self annot-dl) viewer
			  &aux 
			  (mx (mx viewer))
			  (my (my viewer))
			  (bx (bx viewer))
			  (by (by viewer)))
  (let* ((font nil)
	 (prop-list (prop-list self))
	 (shape (shape self))
	 (annot (shape self))
	 (fonts (or (fonts self) (fonts annot)))
	 (str (text annot))
	 (text-width 0)
	 (text-ascent 0)
	 (text-descent 0)
	 (ll (lower-left annot))
	 (annot-x (2d-point-x ll))
	 (annot-y (2d-point-y ll))
	 (annot-w (width annot))
	 (annot-h (height annot))
	 (dx 0)
	 (dy 0)
	 (dw (truncate (* mx annot-w)))
	 (dh (abs (truncate (* my annot-h))))
	 (just (just annot)))
	
	;; Find a font that fits...
	(do* ((f-list fonts (cdr f-list))
	      (f (car f-list) (car f-list)))
	     ((or font (null f)))
	     (if (not (attached-p f)) 
		 (font-attach f)) ; Make sure font is attached...
	     (if (and (<= (xlib:text-width (res f) str) dw)
		      (<= (height f) dh))
		 (setq font f)))
	
	;; If none fit, return
	(when (null font)
	      (setf (slot-value self 'draw-gc) nil)
	      (setf (slot-value self 'erase-gc) nil)
	      (setf (txt-width self) 0
		    (txt-ascent self) 0
		    (txt-descent self) 0)
	      (return-from do-validation nil))
	
	;; Adjust horiz justification...
	(cond ((or (eq just :CC) (eq just :CB) (eq just :CT))
	       (incf annot-x (/ annot-w 2)))
	      ((or (eq just :RC) (eq just :RB) (eq just :RT))
	       (incf annot-x annot-w)))
	
	;; Adjust vert justification...
	(cond ((or (eq just :LC) (eq just :CC) (eq just :RC))
	       (incf annot-y (/ annot-h 2)))
	      ((or (eq just :LB) (eq just :CB) (eq just :RB))
	       (incf annot-y annot-h)))
	
	(setq dx (+ (* mx annot-x) bx)
	      dy (+ (* my annot-y) by))
	
	;; Get size of actual string
	(multiple-value-setq
	 (text-width text-ascent text-descent)
	 (xlib:text-extents (res font) str))
	
	(setf (txt-width self) text-width
	      (txt-ascent self) text-ascent
	      (txt-descent self) text-descent)
	
	;; Adjust horiz justification...
	(cond ((or (eq just :CC) (eq just :CB) (eq just :CT))
	       (decf dx (/ text-width 2)))
	      ((or (eq just :RC) (eq just :RB) (eq just :RT))
	       (decf dx text-width)))
	
	;; Adjust vert justification...
	(cond ((or (eq just :LC) (eq just :CC) (eq just :RC))
	       (incf dy (/ (- text-ascent text-descent) 2)))
	      ((or (eq just :LB) (eq just :CB) (eq just :RB))
	       (incf dy (- text-ascent text-descent))))
	
	;; Update x-y-str...
	(setf (x self) (round dx)	;; x position
	      (y self) (round dy)	;; y position
	      (str self) str)		;; text to display
	
	;; Update rest of graphic props and make draw/erase gc's
	;; 33% of time
	(let ((bg (or (background viewer) "black"))
	      (fg (or (color self) (color shape) (foreground viewer))))
	     (setf (getf prop-list :font) font)
	     (setf (getf prop-list :background) bg)
	     (setf (getf prop-list :foreground) bg)
	     (setf (erase-gc self) (make-shared-gc viewer prop-list))
	     (setf (getf prop-list :foreground) fg)
	     (setf (draw-gc self) (make-shared-gc viewer prop-list)))
	))
