;;; 
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; $Author: bsmith $
;;; $Source: RCS/colormap.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 91/08/04 19:03:13 $
;;;

;;;
;;; A colormap can be attached or detached.
;;; An attached colormap can be installed or uninstalled.
;;; To simplify things, we install all attached colormaps, and
;;; uninstall all detached colormaps.  This can be changed later if
;;; it turns out to be a problem
;;;

(in-package "PT")

(defclass colormap (pmc)
  ((res
    :initform nil
    :type vector
    :reader res)
   (name
    :initarg :name
    :initform ""
    :type string
    :reader name)
   (visual
    :initarg :visual
    :initform nil
    :type visual
    :reader visual)
   (window
    :initarg :window
    :initform nil
    :type window
    :reader window)
   (screen
    :initarg :screen
    :initform nil
    :type screen
    :reader screen)
   (hashtab
    :initarg :hashtab
    :initform nil
    :type hash-table
    :reader hashtab)))

(defun make-colormap (&rest args)
  (apply #'make-instance 'colormap :allow-other-keys t args))

(defun get-color (&optional name spec)
  (cond ((null name) (white-color))
	((colormap-p spec) 
	 (gethash name (hashtab spec)))
	((window-p spec)
	 (gethash name (hashtab (colormap spec))))
	((display-p spec)
	 (gethash name (hashtab (default-colormap spec))))
	((screen-p spec)
	 (gethash name (hashtab (default-colormap spec))))
	(spec
	 (error "get-color: illegal second argument \'~s\'" spec))
	(t (gethash name (hashtab (current-colormap))))))

(defmethod new-instance ((self colormap)
			 &key
			 (visual	nil)
			 (window	nil)
			 (screen	nil)
			 (res		nil)
			 (name		nil)
			 &allow-other-keys)
  ;;
  ;; a colormap is created for a screen, but you may specify 
  ;; either a window on the screen or ther screen itself.
  ;;
  ;; if visual is not specified, we use the visual of the specified
  ;; window, or the visual of the root-window.
  ;;	

  (if screen
      (setq window (root screen))
      (setq screen (screen window)))
  (setf (slot-value self 'screen) screen)
  (setf (slot-value self 'window) window)
  (setf (slot-value self 'hashtab)
	(make-hash-table :size 5 :rehash-size 5 :test #'equal))
  
  (when name
	(when (null (colormap-table screen))
	      (let ((ctab (make-hashtable :size 10 :test #'equal)))
		   (setf (gethash nil ctab) (current-colormap screen))
		   (setf (slot-value screen 'colormap-table) ctab)))
	(setf (gethash name (colormap-table screen)) self)))

(defmethod do-attach ((self colormap))
  ;; attach colormap if not attached, and then install it.
  ;; the window of the colormap must be attached, or it will be attached.
  (let ((visual (slot-value self 'visual))
	(window (slot-value self 'window)))
       (unless window 
	       (setf (slot-value self 'window) 
		     (setq window (root-window))))
       (unless (attached-p window) (attach window))
       ;; if visual wasn't specified, use visual of window
       (unless visual
	       (setf (slot-value self 'visual)
		     (setq visual (xlib:window-visual (res window)))))
       (unless (attached-p self)
	       (setf (slot-value self 'res)
		     (xlib:create-colormap visual (res window))))
       
       ;; always install when attached.  This can be changed if there
       ;; is the need.
       (xlib:install-colormap (res self))))

(defmethod do-detach ((self colormap))
  ;; detached colormaps cannot remain installed.
  (xlib:uninstall-colormap (res self))
  (xlib:free-colormap (res self))
  (setf (slot-value self 'res) nil))

