;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: johnb $
;;; $Source: RCS/color.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 91/10/15 13:19:08 $
;;;

(in-package "PT")

;; There is an inconsistency in the way colors work in Picasso.
;; Normally the res slot contains the clx object.  If the res slot is
;; nil, then the object is detached.  
;; However, for colors, it is the "pixel" value that matters.  The
;; clx object doesn't not have to be detached because colors aren't
;; represented in the x server.  
;; So we use the "res" slot to store the pixel value, and the 
;; "clx-color" slot to store what usually goes into res.
;;

(defclass color (pmc)
  ((res
    :initform nil
    :type integer
    :reader res)
   (clx-color
    :initform nil
    :type vector
    :reader clx-color)
   (name
    :initarg :name
    :initform ""
    :type string
    :reader name)
   (colormap
    :initarg :colormap
    :initform nil
    :type colormap
    :reader colormap)
   (red 
    :initarg :red
    :initform 0 
    :type integer
    :reader red)
   (green 
    :initarg :green
    :initform 0 
    :type integer
    :reader green)
   (blue 
    :initarg :blue
    :initform 0 
    :type integer
    :reader blue)
   (ref-count
    :initarg :ref-count
    :initform 0
    :type integer)))

(defmethod pixel ((self color))
  (if (res self) (res self) -1))

(defun make-color (&rest args
			 &key
			 (name nil)
			 (colormap nil mapp)
			 (attach-p nil)
			 red green blue
			 &allow-other-keys 
			 &aux color)
  (unless (and mapp (colormap-p colormap))
	  (setq colormap (default-colormap)))
  (setf (getf args :colormap) colormap)

  ;; if color exists and rgb values are not specified, then just
  ;; lookup the color.
  (if (and 
       (setq color (get-color name colormap))
       (not (or red green blue))) 
      (progn 
       (when attach-p (do-attach color))
       color)
      (apply #'make-instance 'color :allow-other-keys t args)))

(defmethod new-instance ((self color)
			 &key
			 (name nil namep)
			 (colormap nil mapp)
			 (red 0 rp)
			 (green 0 gp)
			 (blue 0 bp)
			 (attach-p nil) ;; attach right now
			 (lookup-p t) ;; indicates that it should be looked-up
			 (pixel 0 pixelp) ;; only for advanced x-users
			 (hardware-color-p nil) ;; use hardware-color
			 &allow-other-keys 
			 &aux sc hc)
  (unless (and mapp (colormap-p colormap))
	  (setf (slot-value self 'colormap)
		(setq colormap (default-colormap))))
  (when (or rp gp bp) 
	(setq lookup-p nil))
  (if pixelp 
      (setf (slot-value self 'res) pixel)
      (progn
       (cond (lookup-p
	      (when (and namep (stringp name)) 
		    (multiple-value-setq 
		     (sc hc) 
		     (xlib:lookup-color (res colormap) name)) 
		    (if hardware-color-p 
			(setf (slot-value self 'clx-color) sc
			      (slot-value self 'red) (xlib:color-red sc)
			      (slot-value self 'green) (xlib:color-green sc)
			      (slot-value self 'blue) (xlib:color-blue sc) 
			      (gethash name (hashtab colormap)) self)
			(setf (slot-value self 'clx-color) hc
			      (slot-value self 'red) (xlib:color-red hc)
			      (slot-value self 'green) (xlib:color-green hc)
			      (slot-value self 'blue) (xlib:color-blue hc)
			      (gethash name (hashtab colormap)) self))))
	     ((and (<= 0 red 1) (<= 0 green 1) (<= 0 blue 1))
	      (setq hc (xlib:make-color :blue blue :green green :red red)) 
	      (setf (slot-value self 'clx-color) hc
		    (slot-value self 'red) (xlib:color-red hc)
		    (slot-value self 'green) (xlib:color-green hc)
		    (slot-value self 'blue) (xlib:color-blue hc))
	      (when name
		    (if (get-color name colormap)
			(warn "color.new-instance: name \'~s\' already exists in colormap. Aborted." name)
			(setf (gethash name (hashtab colormap)) self))))
	     (t
	      (warn "color.new-instance: bad rgb values (~s ~s ~s)" 
		    red green blue)))))
  
  ;; attach when desired
  (when attach-p (attach self))
  ;; return self
  self)

;;;
;;;	Do not normally detach colors
;;;

;; colors do not need to be detached because they are not represented
;; in the X server.  
;; Instead, attach installs the color in its colormap (and attaches the
;; colormap if necessary), and detach uninstalls it from the colormap.
;;

(defmethod do-detach ((self color))
  ;; don't detach if ref-count is 0
  (when (> (slot-value self 'ref-count) 1) 
	(decf (slot-value self 'ref-count))
	(return-from do-detach))
  (let ((p (res self)))
       (cond ((< p 0)
	      (warn "Color ~S already detached" self))
	     (t
	      (xlib:free-colors (res (colormap self)) (list (pixel self)))
	      (setf (slot-value self 'res) nil
		    (slot-value self 'ref-count) 0)))))

(defmethod do-attach ((self color))
  ;; don't reattach if ref-count is positive
  (when (> (slot-value self 'ref-count) 0)
	(incf (slot-value self 'ref-count))
	(return-from do-attach))
  ;; check colormap
  (unless (colormap self)
	  (error "Can't attach color ~S with no colormap" self))

  ;; attach. . .
  (unless (attached-p (colormap self)) (attach (colormap self)))
  (setf (slot-value self 'res)
	(xlib:alloc-color (res (colormap self)) (clx-color self))
	(slot-value self 'ref-count) 1))

