#include <stdio.h>

#define ID "GIF87a"
#define COLORMAP_MASK 0x80
#define PIXELBITS_MASK 0x07
#define IMAGESEPARATOR ','    
#define INTERLACEMASK 0x40
#define STACKSIZE 1024
#define TABLESIZE 4096    
/*
 * Initialize all static variables;
 */
void gif_init()
{ 
   unsigned char block_next_byte(); 
   unsigned char raster_ref();
   void add_pixel_data(); 

   (void) block_next_byte(0);
   (void) next_code(0,0);
   add_pixel_data(0,0,0,0,0);
   raster_ref(0,0);
}

unsigned char next_byte(fp)
FILE *fp;
{
    static unsigned char buf[2];
    fread(buf, 1, 1, fp);
    return buf[0];
}

/*
 * Assuming that data in fp is organized in the form:
 *  block byte count, byte1, byte2, ..., byten, block byte count, ... , 0
 * return the next byte.
 */
unsigned char block_next_byte(fp)
FILE *fp;
{
    static int left_in_block;

    if (!fp) {
	left_in_block = 0;
	return;
    }
    if (left_in_block == 0) {
	if ((left_in_block = next_byte(fp)) == 0)
	    return 0;
    }
    left_in_block--;
    return (next_byte(fp));
}


/*
 * This function basically fakes bytewise semi-random access to a sequential file.
 * A CACHESIZE number of bytes before the actual read position are stored for
 * reference.  
 * Assumption made that no one else is playing around with fp.
 * Also, access requests are assumed to be localized.
 */
#define NOT_INITIALIZED -9999
#define CACHESIZE 4
unsigned char raster_ref(fp, n)
FILE *fp;
int n;  /* offset into raster */
{
    static unsigned char cache[CACHESIZE];
    static int current_pos;  /* current read position in fp */
    int i;

    /* init if necessary */
    if (!fp) {
	current_pos = NOT_INITIALIZED;
	return;
    }

    if (current_pos == NOT_INITIALIZED) {
	for (i=0; i<CACHESIZE; i++) 
	    cache[i] = block_next_byte(fp);
	current_pos = CACHESIZE;
    }

    /* n is too big -- no problem */
    if (n >= current_pos) {
	/* scroll cache and try again.  We scroll only one position at a time
	 * because access is going to be localized anyway. */
	for (i=1; i<CACHESIZE; i++) 
	    cache[i-1] = cache[i];
	cache[CACHESIZE-1] = block_next_byte(fp);
	current_pos++;
	return (raster_ref(fp, n));  /* how elegant :) */
    }

    /* n is too small -- bad! */
    else if (n < (current_pos - CACHESIZE)) {
	fprintf (stderr, "raster_ref(): cache miss!\n");
	fclose(fp);
	exit(1);
    }

    /* cache hit */
    else {
	return (cache[CACHESIZE + n - current_pos]);
    }
    /* never reached */
    fprintf(stderr, "raster_ref(): fell off the end!\n"); 
    fclose(fp);
    exit(1);
}

/* returns the next codeside bits from fp, properly shifted and masked. */
int next_code(fp, codesize)
FILE *fp;
int codesize;
{
    static int bit_offset;
    int raw_code, byte_offset, readmask;

    if (!fp) {
	bit_offset = 0;
	return;
    }
    byte_offset = bit_offset / 8;

    /* get two or three bytes as necessary */
    raw_code =
	raster_ref(fp, byte_offset) + 0x100 * raster_ref(fp, byte_offset + 1);
    if (codesize > 8)
	raw_code += 0x10000 * raster_ref(fp, byte_offset + 2);
    
    /* shift to align least-significant-bit */
    raw_code >>= bit_offset % 8;

    /* keep track for next time. */
    bit_offset += codesize;

    /* make readmask to get only a codesize number of bits from the left */
    readmask = (1 << codesize) - 1;

    /* printf("(%d) %d\n", codesize, raw_code & readmask); */
    return (raw_code & readmask);
}

void add_pixel_data(data, pixel, width, height, interlaced, true_color_p)
unsigned char *data;
unsigned char pixel;
int width, height, interlaced;
unsigned char true_color_p;
{
    static int x;
    static int y;
    static int pass;

    if (!data) {
	x = y = pass = 0;
	return;
    }
    if (y < height)
	if (true_color_p) *(data+4*(y*width+x)) = pixel;
	else *(data+y*width+x) = pixel;
    /*
      data[y * width + x] = pixel;
     */

    if (++x == width) {
	x=0;
	if (!interlaced)
	    y++;
	else {
	    switch (pass) {
	    case 0:
		y+=8;
		if (y >= height) {
		    pass++;
		    y=4;
		}
		break;
	    case 1:
		y+=8;
		if (y >= height) {
		    pass++;
		    y=2;
		}
		break;
	    case 2:
		y+=4;
		if (y >= height) {
		    pass++;
		    y=1;
		}
		break;
	    case 3:
		y+=2;
		break;
	    default:
		break;
	    }
	}
    }
}

int load_gif (filename, data, red, green, blue, used, true_color_p)
char *filename;
unsigned char *data;
unsigned char *red;
unsigned char *green;
unsigned char *blue;
unsigned char *used;
unsigned char true_color_p;
{
    FILE *fp;
    unsigned char ch, inbuf[256];
    int i;

    int out_stack[1+STACKSIZE];   /* temporary buffer for decompressor output */
    int out_sp = 0;         /* decompressor sp */

    int prefix[1+TABLESIZE];
    int suffix[1+TABLESIZE];
    
    int r_width, r_height;  /* of the raster */
    int i_width, i_height;  /* of image */
    int i_top, i_left;      /* top and left of image */
    int has_colormap;       /* does this file have a global colormap? */
    int bits_per_pixel; 
    int num_colors;         /* number of colors used by image */
    int background;
    int interlaced;         /* 1: raster in interlaced order
			     * 0: raster in sequential order */
    int codesize, init_codesize;
    int clear_code, eof_code;
    int max_code;
    int code,               /* to read in codes */
        current_code,        
        old_code,
        in_code;
    int final_code;         /* properly selected and masked code */
    int next_free_code, first_free_code;
    int bitmask;            /* mask for data size */

    fp = fopen(filename, "r");
    if (!fp) {
	perror("fopen:");
	fclose(fp);
	return(1);
    }

    /* validate id header. */
    fread(inbuf, 6, 1, fp);
    if (strncmp(inbuf, ID, 6)) {
	fprintf(stderr, "Not GIF file.\n");
	fclose(fp);
	return(1);
    } 
    
    /* initialize everything. */
    gif_init();

    /* get the image dimensions */
    r_width = next_byte(fp);
    r_width += 0x100 * next_byte(fp);

    r_height = next_byte(fp);
    r_height += 0x100 * next_byte(fp);

    /* details... */
    ch = next_byte(fp);
    has_colormap = ((ch & COLORMAP_MASK) ? 1 : 0);
    bits_per_pixel = (ch & PIXELBITS_MASK) + 1;
    num_colors = 1 << bits_per_pixel;
    bitmask = num_colors - 1;
    
    background = next_byte(fp);

    /* The next byte must be a 0 */
    if (next_byte(fp)) {
	fprintf (stderr, "format error in GIF file.\n");
	fclose(fp);
	return(1);
    }

    if (has_colormap) {
	/* read in global colormap */
	int colormap_size = 1 << bits_per_pixel;
	int i;

	for (i=0; i<colormap_size; i++) {
	    red[i] = next_byte(fp);
	    green[i] = next_byte(fp);
	    blue[i] = next_byte(fp);
	}
    }

    /* next byte must be a image separator */
    if (next_byte(fp) != IMAGESEPARATOR) {
	fprintf (stderr, "format error in GIF file. \n");
	fclose(fp);
	return(1);
    }

    /* read in image region */
    i_left = next_byte(fp);
    i_left += 0x100 * next_byte(fp);

    i_top = next_byte(fp);
    i_top += 0x100 * next_byte(fp);
    
    i_width = next_byte(fp);
    i_width += 0x100 * next_byte(fp);

    i_height = next_byte(fp);
    i_height += 0x100 * next_byte(fp);

    /* more details...
     * Here we ignore everything but the interlace.
     * Could cause problems in the future...
     */
    ch = next_byte(fp);
    interlaced = ((ch & INTERLACEMASK) ? 1 : 0);

    if (ch & COLORMAP_MASK) {
	fprintf (stderr, "Sorry, local colormap not supported.\n");
	fclose(fp);
	return(1);
    }

    /*
     * Now, setup for the raster data.
     */
    codesize = next_byte(fp);
    clear_code = (1 << codesize);
    eof_code = 1 + clear_code;
    next_free_code = first_free_code = clear_code + 2;
    
    /*
     * undocumented detail of gif spec:
     * the code size used in compression/decompression is the one given
     * in the file plus one.
     */
    init_codesize = ++codesize;
    max_code = (1 << codesize);

    /*
     * finally the raster data.
     */

    code = next_code(fp, codesize);
    while (code != eof_code) {
	if (code == clear_code) {
	    codesize = init_codesize;
	    max_code = (1 << codesize);
	    next_free_code = first_free_code;
	    current_code = old_code = code = next_code(fp, codesize);
	    final_code = current_code & bitmask;
	    used[final_code] = 1;
	    add_pixel_data(data, final_code, i_width, i_height, interlaced,
			   true_color_p);
	} else {
	    current_code = in_code = code;
	    /*
	     * if not in table, add to table and repeat last character.
	     */
	    if (current_code >= next_free_code) {
		current_code = old_code;
		out_stack[out_sp++] = final_code;
	    }
	    while (current_code > bitmask) {
		if (out_sp > STACKSIZE) {
		    fprintf (stderr, "output stack overflow.\n"); 
		    fclose(fp); 
		    return(2);
		}
		out_stack[out_sp++] = suffix[current_code];
		current_code = prefix[current_code];
	    }
	    final_code = current_code & bitmask;
	    out_stack[out_sp++] = final_code;

	    /* now flush stack */
	    for (i=out_sp-1; i>=0; i--) {
		used[out_stack[i]] = 1;
		add_pixel_data(data, out_stack[i],
			       i_width, i_height, interlaced, true_color_p);
	    }
	    out_sp =0;

	    /* build table */
	    prefix[next_free_code] = old_code;
	    suffix[next_free_code] = final_code;
	    old_code = in_code;

	    /* next slot in table */
	    next_free_code++;
	    if (next_free_code >= max_code) {
		if (codesize < 12) {
		    codesize++;
		    max_code*=2;
		}
	    }
	}
	code = next_code(fp, codesize);
    }
    fclose(fp);
    return (num_colors);
}

/*
main()
{
    unsigned char red[256], green[255], blue[255], used[256];
    
    int i;
    unsigned char *foo;

    foo = (unsigned char *) malloc(70000);

    i = load_gif("moonflag.gif", foo, red, green, blue, used );

    for (i=0; i<64000; i++)
	printf("(%d) %d\n", i, foo[i]);
    
}
*/
