;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/toolkit/resource/cursor.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:03:13 $
;;;

(in-package "PT")

;;;
;;; cursor class
;;;

(defclass cursor (pmc)
  ((name 
    :initarg :name
    :initform *default-cursor-name*
    :type string
    :reader name)
   (res
    :initform nil
    :type vector
    :reader res)
   (foreground
    :initform "black"
    :initarg :foreground
    :type t
    :reader foreground)
   (background
    :initform "white"
    :initarg :background
    :type t
    :reader background)
   (display
    :initform nil
    :initarg :display
    :type t
    :reader display)))

(defmethod (setf foreground) (val (self cursor) &aux color)
  (cond ((stringp val) 
	 (setq color (get-color val))
	 (if (not (color-p color))
	     (warn "cursor.setf.foreground: illegal color ~S~%" val)
	     (progn 
	      (when (attached-p self)
		    (do-detach (slot-value self 'foreground))
		    (do-attach color)
		    (change-cursor-color self))
	      (setf (slot-value self 'foreground) color))))
	((color-p val)
	 (when (attached-p self)
	       (do-detach (slot-value self 'foreground))
	       (do-attach color)
	       (change-cursor-color self))
	 (setf (slot-value self 'foreground) color))))

(defmethod (setf background) (val (self cursor) &aux color)
  (cond ((stringp val) 
	 (setq color (get-color val))
	 (if (not (color-p color))
	     (warn "cursor.setf.background: illegal color ~S~%" val)
	     (progn 
	      (when (attached-p self)
		    (do-detach (slot-value self 'background))
		    (do-attach color)
		    (change-cursor-color self))
	      (setf (slot-value self 'background) color))))
	((color-p val)
	 (when (attached-p self)
	       (do-detach (slot-value self 'background))
	       (do-attach color)
	       (change-cursor-color self))
	 (setf (slot-value self 'background) color))))

(defun change-cursor-color (cursor)
  (when (attached-p cursor)
	(xlib:recolor-cursor (res cursor) (res (foreground cursor))
			     (res (background cursor)))))

;;;
;;;	Lookup cursor in table
;;;

(defun get-cursor (&optional name spec &aux display)
  (if (null name)
      nil
      (progn
       (setq display
	     (cond ((window-p spec)
		    (display spec))
		   ((display-p spec)
		    spec)
		   (t (current-display))))
       (gethash name (cursor-table display)))))

;;;
;;; cursor initalization method
;;;

(defun make-cursor (&rest args 
		    &key
		    (name nil)
		    (image nil)
		    (file nil)
		    (font nil)
		    (index nil)
		    (source-font nil)
		    (source-index nil)
		    (bitmap-file nil)
		    (display (current-display))
		    &allow-other-keys)
  (if font
      (setq source-font font))
  (if index
      (setq source-index index))
  (when bitmap-file 
	(setq file bitmap-file)
	(setf (getf args :file) file))
  (cond ((and name (not (or image file source-index)))
	 (let ((c (get-cursor name display)))
	      (unless c 
		      (warn "make-cursor: no cursor named \'~s\' exists." name))
	      c))
	(name (apply #'make-instance 'cursor :allow-other-keys t args))
	(file
	 (setf (getf args :name) file)
	 (apply #'make-instance 'cursor :allow-other-keys t args))
	(image 
	 (unless (image-p image)
		 (error "make-cursor: invalid image \`~s\`." image))
	 (setf (getf args :name) (name image))
	 (apply #'make-instance 'cursor :allow-other-keys t args))
	(t
	 (get-cursor *default-cursor-name* display))))

(defmethod new-instance ((self cursor)
			 &key
			 (image nil imagep)
			 (name nil)
			 (mask nil maskp)
			 (file "arrow.cursor")
			 (mask-file "arrow_mask.cursor") 

			 (font nil)
			 (index nil)
			 (source-font nil) 
			 (mask-font nil) 
			 (source-index nil) 
			 (mask-index nil)

			 (foreground "black")
			 (background "white")
			 (x-hot nil)
			 (y-hot nil)
			 (display nil)
			 &allow-other-keys
			 &aux pm (rw (res (root-window))) temp cursor)

  (unless (display-p display)
	  (setf (slot-value self 'display)
		(setq display (current-display))))
  (if font
      (setq source-font font))
  (if index
      (setq source-index index))
  ;; set foreground and background
  (when (stringp foreground)
	(let ((fg (get-color foreground display))) 
	     (unless fg
		   (warn "cursor.new-instance: can't find color ~s. 
			 Resetting to black."
			   foreground)
		     (setq fg (get-color "black" display)))
	     (setq foreground fg)))
  (when (stringp background)
	(let ((bg (get-color background display))) 
	     (unless bg 
		   (warn "cursor.new-instance: can't find color ~s. Resetting to white."
			   background)
		     (setq bg (get-color "white" display)))
	     (setq background bg)))
  (unless foreground (setq foreground (get-color "black" display)))
  (unless background (setq background (get-color "white" display)))
  (unless (color-p foreground) 
	  (warn "cursor.new-instance: invalid foreground color \`~S\`.  Resetting to black." foreground)
	  (setq foreground (get-color "white")))
  (setf (slot-value self 'foreground) foreground)
  (do-attach foreground)
  (setq foreground (res foreground))
  (unless (color-p background) 
	  (warn "cursor.new-instance: invalid background color \`~S\`.  Resetting to white." background)
	  (setq background (get-color "black")))
  (setf (slot-value self 'background) background)
  (do-attach background)
  (setq background (res background))

  ;; get image and mask from file or object
  (cond (source-index
	 (setq temp source-font)
	 (cond ((null source-font)
		(setq source-font (make-font :name "cursor"))
		(if (not (attached-p source-font))
		    (font-attach source-font)))
	       ((stringp source-font) 
		(setq source-font (get-font source-font))))
	 (if (null source-font)
	     (error "make-cursor: bad source-font ~S" temp))
	 (if (not (attached-p source-font))
	     (font-attach source-font))
	 (setq temp mask-font)
	 (cond ((null mask-font)
		(setq mask-font source-font))
	       ((stringp mask-font) 
		(setq mask-font (get-font mask-font))))
	 (if (null mask-font)
	     (error "make-cursor: bad mask-font ~S" temp))
	 (if (not (attached-p mask-font))
	     (font-attach mask-font))
	 (if (not (integerp source-index))
	     (error "make-cursor: bad source-index ~S" source-index))
	 (cond ((null (mask-index))
		(setq mask-index (1+ source-index)))
	       ((not (integerp mask-index)) 
		(error "make-cursor: bad mask-index ~S" mask-index)))
	 (setq cursor 
	       (xlib:create-glyph-cursor
		:source-font (res source-font)
		:mask-font (res mask-font)
		:source-char source-index
		:mask-char mask-index
		:foreground (res foreground)
		:background (res background))))
	(t
	 (setq image
	       (if file
		   (xlib:read-bitmap-file (find-library-file file))
		   (res image)))
	 (setq pm (xlib:create-pixmap :width (xlib:image-width image)
				      :height (xlib:image-height image)
				      :depth 1 :drawable rw))
	 (setq mask
	       (cond (mask-file (xlib:image-pixmap 
				 pm (xlib:read-bitmap-file 
				     (find-library-file mask-file))))
		     ((image-p mask) (xlib:image-pixmap
				      pm (res mask)))
		     (t nil)))
	 
	 ;; register hotspots
	 (unless (integerp x-hot)
		 (setq x-hot (xlib:image-x-hot image)))
	 (unless (integerp y-hot)
		 (setq y-hot (xlib:image-y-hot image)))
	 
	 ;; coerce image to pixmap
	 (setq image (xlib:image-pixmap pm image))
	 
	 ;; set id slot
	 (setq cursor
	       (if mask
		   (xlib:create-cursor :source image :mask mask
				       :x x-hot :y y-hot 
				       :foreground foreground 
				       :background background)
		   (xlib:create-cursor :source image
				       :x x-hot :y y-hot 
				       :foreground foreground 
				       :background background)))
	 
	 ;; Free the pixmaps when necessary
	 (xlib:free-pixmap pm)
	 (unless imagep 
		 (xlib:free-pixmap image))
	 (when (and mask (not maskp))
	       (xlib:free-pixmap mask))))

  ;; register in hashtable
  (setf (gethash name (cursor-table display)) self)

  ;; set resource slot
  (setf (slot-value self 'res) cursor)

  ;; return self
  self)

(defmethod do-attach ((self cursor))
  nil)

(defmethod do-detach ((self cursor))
  nil)

;;;
;;; 	Retrieves best size closest to desired width & height
;;;

(defun cursor-best-size (width height &optional (display (current-display)))
  (xlib:query-best-cursor width height (res display)))
