;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/toolkit/picasso/variable.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:03:07 $
;;;

(in-package "PT")

(defclass variable-holder (pmc)
  ((vtab 
    :initarg :vtab  
    :initform (make-hash-table) 
    :type hash-table 
    :reader vtab)))

(defmethod new-instance ((self variable-holder)
			 &key 
			 (variables nil)
			 (constants nil)
			 &allow-other-keys)
  (call-next-method)
  (dolist (v variables)
	  (cond ((and v (symbolp v)) (add-var v self))
		((and (consp v) (symbolp (car v)))
		 (add-var (car v) self (cadr v)))))
  (dolist (c constants)
	  (cond ((and c (symbolp c)) (add-const c self))
		((and (consp c) (symbolp (car c)))
		 (add-const (car c) 
			    self 
			    (if (or (gadget-p (cadr c)) 
				    (picasso-name-p (cadr c)))
				(cadr c)
				(eval (cadr c)))))))
  self)

(defclass variable (pmc)
  ((value :initarg :value  :initform nil :type t 
	  ;; :reader value
	  )))

(defmethod value ((self variable)
		  &key &allow-other-keys)
  (slot-value self 'value))

(defclass constant (variable) ())

(defclass alerter (variable) 
  ((code :initarg :code  :initform nil :type t :accessor code)))

(defun make-variable (&rest keys)
  (apply #'make-instance 'variable :allow-other-keys t keys))

(defun make-constant (&rest keys)
  (apply #'make-instance 'constant :allow-other-keys t keys))

(defun make-alerter (&rest keys)
  (apply #'make-instance 'alerter :allow-other-keys t keys))

(defun add-var (name place &optional (init nil))
  (unless (symbolp name) 
	  (error "add-var:  ~S not a valid variable name" name))
  (cond ((variable-holder-p place)
	 (setf (gethash name (vtab place)) (make-variable :value init)))
	((hash-table-p place)
	 (setf (gethash name place) (make-variable :value init)))
	(t (error "add-var: ~S not a valid variable place" place))))

(defun add-const (name place &optional (init nil))
  (unless (symbolp name) 
	  (error "add-const:  ~S not a valid variable name" name))
  (cond ((variable-holder-p place)
	 (setf (gethash name (vtab place)) (make-constant :value init)))
	((hash-table-p place)
	 (setf (gethash name place) (make-constant :value init)))
	(t (error "add-const: ~S not a valid variable place" place))))

(defmethod (setf value) (new-v (self variable))
  (unless (equal (value self) new-v)
	  (setf (slot-value self 'value) new-v)
	  (let ((key `(value ',self)))
	       (propagate (gethash key *prop-table*) key))))

(defmethod (setf value) (new-v (self alerter))
  (unless (equal (value self) new-v)
	  (setf (slot-value self 'value) new-v)
	  (execute 'code self)
	  (let ((key `(value ',self)))
	       (propagate (gethash key *prop-table*) key))))

(defmethod (setf value) (new-v (self constant))
  (if *constants-enforced*
      (warn "Constant cannot be set!")
      (progn (setf (slot-value self 'value) new-v)
	     (let ((key `(value ',self)))
		  (propagate (gethash key *prop-table*) key)))))

(defun find-var (name ref)
  (cond ((or (null name) (null ref))
	 (warn "Bad attempt to lookup var ~S from location ~S" name ref)
	 nil)
	((and (variable-holder-p ref)
	      (gethash name (vtab ref))))
	((lexical-parent ref) (find-var name (lexical-parent ref)))
	((parent ref) (find-var name (parent ref)))
	(t 
	 (warn "find-var:  Variable ~S not defined from reference ~S" name ref)
	 nil)))

(defun var-val (name ref)
  (value (find-var name ref)))

(defun dump-vars (holder)
  (maphash #'(lambda (k d) (format t "~S:   ~S~%" k (value d) ))
	   (vtab holder))
  t)


;;; bombproofing code

(defmethod value ((Self null)
		  &key 
		  &allow-other-keys)
  (warn "Attempt to take value of nil")
  nil)

(defmethod (setf value) (new-v (self null))
  (declare (ignore new-v))
  (warn "Attempt to set value of nil")
  nil)

(defun valid-var-expression (expr &optional (second nil))
  (and (consp expr)
       (or (and (eql (car expr) 'lookup)
		(or (symbolp (cadr expr))
		    (and (consp (cadr expr))
			 (eql (caadr expr) 'quote)
			 (symbolp (cadadr expr))))
		(<= (length expr) 3))
	   (and (eql (car expr) 'value)
                (not second)
		(= (length expr) 2)
		(valid-var-expression (cadr expr) t)))))
