;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/src/toolkit/gm/RCS/packed-gm.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 1991/08/06 18:50:19 $
;;;

(in-package "PT")

;;;  packed-gm is a geometry manager that allows for perpendicular packing of
;;;  subwindows in a style much like that of the SX toolkit.

;;;  packed-gm takes great advantage of the geom-spec field of children.  This
;;;  field is used in the following ways:

;;;  geom-spec		action
;;;
;;;  nil		treated as :top (base-height self) for tall windows
;;;			and as :left (base-height self) for wide windows
;;;			(wide windows have width > 3*height)
;;;
;;;  :fill 		indicates to pack this window in the remaining space
;;;			This is equivalent to (:top 0)
;;;
;;;  :top size		indicates to pack this window in the remaining space
;;;			along the top (full-width) with a height of size.
;;;			if size is 0, then use remaining height.  If size is
;;;			omitted, then use base-height.
;;;
;;;  :bottom size	similar to :top but pack along bottom
;;;
;;;  :left size, or
;;;  :right size	similar to :top but switch height and width
;;;
;;;  Widgets can also have "padding" on the left, right, top and bottom.
;;;  These act as blank sections that offset a widget from it's packed
;;;  neighbors.  Padding can be specified in the following ways:
;;;
;;;  :left-pad size	Make a pad on the left side of the widget size
;;;			pixels wide.
;;;  :right-pad size	Make a pad on the right side of the widget size
;;;			pixels wide.
;;;  :top-pad size	Make a pad on the top side of the widget size
;;;			pixels wide.
;;;  :bottom-pad size	Make a pad on the bottom side of the widget size
;;;			pixels wide.
;;;  :horiz-pad size	Make a pad on the left and right sides of the
;;;			widget size pixels wide.
;;;  :vert-pad size	Make a pad on the top and bottom sides of the
;;;			widget size pixels wide.
;;;  :pad size		Make a pad on all sides of the widget size pixels wide.
;;;
;;;  In addition the geom-spec may include one of these, which is deleted
;;;  after the appropriate action is taken
;;;
;;;  :after window	pack this window immediately after the designated 
;;;			window.  The other window must be a child of the 
;;;			collection.  If window is ommitted, then pack after
;;;			all other existing windows (this is also the default).
;;;
;;;  :before window	pack this window immediately before the designated 
;;;			window.  the other window must be a child of the 
;;;			collection.  If window is ommitted, then pack before
;;;			all other existing windows.

;;;  the children list for packed-gm is in packing order

;;; gm-initialize inherited

(defmethod gm-add-child ((gm (eql 'packed-gm)) self child)
  (multiple-value-bind (geom-spec bef-after) 
		       (parse-packed-geom-spec child)
            (declare (ignore geom-spec))
	    (case (car bef-after)
		  (:after 
		   (if (= (length bef-after) 2)
		       (let ((tail (member (cadr bef-after) (children self))))
			    (if tail
				(rplacd tail (cons child (cdr tail)))
				(progn
				 (warn "packed-gm.add-child :after not found~%")
				 (if (children self)
				     (nconc (children self) (list child))
				     (setf (children self) (list child))))))
		       (if (children self)
			   (nconc (children self) (list child))
			   (setf (children self) (list child)))))
		  (:before
		   (if (= (length bef-after) 2)
		       (let* ((children (children self))
			      (before (second bef-after))
			      (head (reverse 
				     (cdr (member before (reverse children)))))
			      (tail (member before children)))
			     (if tail
				 (setf (children self)
				       (nconc head (list child) tail))
				 (progn
				  (warn "packed-gm.add-child :before missing~%")
				  (push child (children self)))))
		       (push child (children self))))
		  (otherwise 
		   (error "packed-gm.add-child:  Illegal bef-after spec~%"))))
  (do-repack self))

;;; parse-packed-geom-spec  takes a child of a packed-collection and 
;;; parses the geom-spec field.  If the geom-spec was already parsed, it
;;; returns it (as a list of zero to two elements), otherwise, it does a 
;;; parse, sets the geom-spec, and RETURNS TWO VALUES, the geom-spec and the
;;; before-after specification (for adding children).
(defun parse-packed-geom-spec (child &aux (gs (geom-spec child)))
  (if (null gs)
      ;; Delay parsing until attach time -- then dynamically determine
      ;; default geom-spec
      (if (attached-p child)
	  (let* ((p (parent child))
		 (w (width p))
		 (h (height p)))
		(if (> w (* 3 h))
		    (setq gs :left)
		    (setq gs :top)))
	  ;; Null geom-spec and not yet attached -- let's get outta here!
	  (return-from parse-packed-geom-spec (values nil '(:after)))))
  (if (not (listp gs)) (setq gs (list gs)))
  (if (eq (car gs) *packed-gm-magic-id*)
      (values (cdr gs) '(:after))
      (let ((left-pad 0)
	    (right-pad 0)
	    (top-pad 0)
	    (bottom-pad 0)
	    (size nil)
	    (aux nil)
	    (win nil)
	    (pos :left)
	    (bef-after '(:after))
	    (geom-spec nil))
	   (do* ((key-l gs (cdr key-l))
		 (key (car key-l) (car key-l)))
		((null key-l))
		(setq aux nil)
		(setq win nil)
		(cond ((integerp (cadr key-l))
		       (setq key-l (cdr key-l)
			     aux (car key-l)))
		      ((window-p (cadr key-l))
		       (setq key-l (cdr key-l)
			     win (car key-l))))
		(case key
		      (:left 
		       (if (eq 0 aux)
			   (setq pos :fill)
			   (setq pos :left size aux)))
		      (:right 
		       (if (eq 0 aux)
			   (setq pos :fill)
			   (setq pos :right size aux)))
		      (:top 
		       (if (eq 0 aux)
			   (setq pos :fill)
			   (setq pos :top size aux)))
		      (:bottom 
		       (if (eq 0 aux)
			   (setq pos :fill)
			   (setq pos :bottom size aux)))
		      (:fill
		       (setq pos :fill))
		      (:left-pad
		       (if aux (setq left-pad aux)))
		      (:right-pad
		       (if aux (setq right-pad aux)))
		      (:top-pad
		       (if aux (setq top-pad aux)))
		      (:bottom-pad
		       (if aux (setq bottom-pad aux)))
		      (:horiz-pad
		       (if aux (setq left-pad aux
				     right-pad aux)))
		      (:vert-pad
		       (if aux (setq bottom-pad aux
				     top-pad aux)))
		      (:pad
		       (if aux (setq left-pad aux
				     right-pad aux
				     bottom-pad aux
				     top-pad aux)))
		      (:before
		       (if win
			   (setq bef-after (list key win))
			   (setq bef-after (list key))))
		      (:after
		       (if win
			   (setq bef-after (list key win))
			   (setq bef-after (list key))))))
	   (setq geom-spec 
		 (cons *packed-gm-magic-id* 
		       (cons (cons pos size)
			     (list
			      (list left-pad top-pad right-pad bottom-pad)))))
	   (setf (slot-value child 'geom-spec) geom-spec)
	   (values (cdr geom-spec) bef-after))))

(defmethod gm-delete-child ((gm (eql 'packed-gm)) self child)
  (setf (slot-value self 'children) (delete child (children self)))
  (do-repack self))

(defmethod gm-resize-hint-changed ((gm (eql 'packed-gm)) self child)
  ;;  Only repack if child is managed, and if not fixed size child
  (when (managed-p child)
	(let* ((ps (parse-packed-geom-spec child))
	       (pack-spec (car ps)))
	      (if (or (null (cdr pack-spec)) (eq (car pack-spec) :fill))
		  (do-repack self)))))

(defmethod gm-status-changed ((gm (eql 'packed-gm)) self child)
  (declare (ignore child old new))
  (do-repack self))

(defmethod gm-spec-changed ((gm (eql 'packed-gm)) self child)
  (declare (ignore child))
  (do-repack self))

(defmethod gm-calculate-min-size ((gm (eql 'packed-gm)) self
				  &aux ps gs displ bdw bdh
				  pad pl pt pr pb bl bt br bb)
  (let ((h 0)
	(w 0)
	(dw 0)
	(dh 0))
       (dolist (ch (reverse (managed-of (children self))))
	       (multiple-value-setq (bl bt br bb) (region-offset ch))
	       (decf bb bt)
	       (decf br bl)
	       (setq ps (parse-packed-geom-spec ch))
	       (setq gs (car ps))
	       (setq pad (cadr ps)
		     pl (first pad)
		     pt (second pad)
		     pr (third pad)
		     pb (fourth pad))
	       (setq displ (cdr gs))
	       (setq bdw (+ bl br pl pr)
		     bdh (+ bt bb pt pb))
	       (setq dw (if (member (car gs) '(:left :right))
			    (if displ
				(+ displ bdw)
				(+ bdw (base-width ch)))
			    (max (- (+ bdw (base-width ch)) w) 0)))
	       (setq dh (if (member (car gs) '(:top :bottom))
			    (if displ
				(+ displ bdh)
				(+ bdh (base-height ch)))
			    (max (- (+ bdh (base-height ch)) h) 0)))
	       (incf h dh)
	       (incf w dw))
       (setf (min-size self) (list w h))))

(defmethod gm-repack ((gm (eql 'packed-gm)) self
		      &aux gs ps pad pdw pdh pl pr pt pb geom side dist
		      bdw bdh nh nw bl bt br bb)
  (declare (integer nh nw bl bt br bb pl pr pt pb pdw pdh bdw bdh))
  (let ((h (height self))
	(w (width self))
	(x 0)
	(y 0)
	(children (managed-of (children self))))
       (declare (integer x y w h))
       (dolist (ch children)
	       ;; Extract a bunch of information
	       (multiple-value-setq (bl bt br bb) (region-offset ch))
	       (decf bb bt)
	       (decf br bl)
	       (setq ps (parse-packed-geom-spec ch))
	       (setq gs (car ps))
	       (setq ps (parse-packed-geom-spec ch))
	       (setq geom (car ps))
	       (setq pad (cadr ps)
		     pl (first pad)
		     pt (second pad)
		     pr (third pad)
		     pb (fourth pad))
	       (setq side (car geom)
		     dist (cdr geom))
	       (setq bdw (+ bl br)
		     bdh (+ bt bb)
		     pdw (+ pl pr)
		     pdh (+ pt pb))
	       ;; Coming into this case, we have:
	       ;;	x, y as upper-left of available space,
	       ;;	w, h as size of available space.
	       ;;	bl, br, bt and bb as the size of the left, right
	       ;;	    top and bottom borders.
	       ;;	pl, pr, pt and pb as the size of the left, right
	       ;;	    top and bottom padding.
	       ;;	bdw and bdh as the total width and height of the
	       ;;	    border
	       ;;	pdw and pdh as the total width and height of the pad
	       ;;
	       (case side
		     (:fill
		      (reshape ch (+ x bl pl) (+ y bt pt)
			       (- w bdw pdw) (- h bdh pdh))
		      (mapcar #'pend (cdr (member ch children)))
		      (return-from gm-repack))
		     (:top
		      (setq nh (or dist (base-height ch)))
		      (reshape ch (+ x bl pl) (+ y bt pt)
			       (- w bdw pdw) nh)
		      (incf nh (+ bdh pdh))
		      (decf h nh)
		      (incf y nh))
		     (:bottom
		      (setq nh (or dist (base-height ch)))
		      (reshape ch (+ x bl pl) (+ y (- h bb pb nh))
			       (- w bdw pdw) nh)
		      (incf nh (+ bdh pdh))
		      (decf h nh))
		     (:left
		      (setq nw (or dist (base-width ch)))
		      (reshape ch (+ x bl pl) (+ y bt pt)
			       nw (- h bdh pdh))
		      (incf nw (+ bdw pdw))
		      (decf w nw)
		      (incf x nw))
		     (:right
		      (setq nw (or dist (base-width ch)))
		      (reshape ch (+ x (- w br pr nw)) (+ y bt pt)
			       nw (- h bdh pdh))
		      (incf nw (+ bdw pdw))
		      (decf w nw)))))
  (repaint self))
