;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/toolkit/gm/anchor-gm.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:02:51 $
;;;

(in-package "PT")

;;;
;;; anchor-gm, handles reshaping of children according to the placement of 
;;; figurative "anchors" and "arrows".  
;;;
;;; ARROWS
;;; The two types of arrows, vertical & horizontal, specify in which 
;;; directions a given child should be resized.  Arrows always imply 
;;; proportional reshaping, as in rubber-gm.  If arrows are omitted, the 
;;; child is not resized, but moved to the center of the area which the child 
;;; would occupy if it were resized.
;;;
;;; ANCHORS
;;; The four types of anchors, left, right, top, & bottom, specify the 
;;; side of the collection to which the child should be anchored.  Anchors
;;; imply absolute reshaping.  Associated with each anchor is an integer
;;; which specifies the gap (in pixels) between the given side of the child
;;; and the given side of the collection.  For instance, a child can be 
;;; anchored at coordinates (20 35) by putting a left anchor at 20 and a
;;; top anchor 35.  Anchor gm takes into account border-width in repacking
;;; so that a window with border-width 5 will be positioned 5 pixels in 
;;; of its "desired" position (x and y) and will be 10 pixels shorter and 
;;; thinner than "desired".  Border-width is ignored if '(:border nil)
;;; occurs in the geom-spec.
;;;
;;; example: A scrolling text-wigdet of initial dimensions 100 x 100
;;;		;; horizontal scroll-bar
;;;	:geom-spec '(:anchor (:left 0 :top 0 :bottom 0)) :width 20
;;;		;; vertical scroll-bar
;;;	:geom-spec '(:anchor (:left 20 :bottom 0 :right 0)) :height 20
;;;		;; text-widget
;;;	:geom-spec '(:arrow (:horiz :vert)) :region '(0 0 80 80)
;;;
;;; example: A window which sticks to the bottom, grows upward, but remains 
;;;	       centered horizontally within its parent and always takes up
;;;	       1/2 of the width and 3/4 of the height of the parent 
;;;	:geom-spec '(1/4 1/4 1/2 3/4 :anchor (:bottom 0) :arrow (:vert)) 
;;;

(defmethod gm-spec-changed ((gm (eql 'anchor-gm)) self child)
  (if (managed-p child)
      (a-gm-reshape-one-child self child)))

(defmethod gm-calculate-min-size ((gm (eql 'anchor-gm)) self)
  (setf (min-size self) 
	(list (slot-value self 'base-width)
	      (slot-value self 'base-height))))

(defmethod gm-repack ((gm (eql 'anchor-gm)) self)
  (let ((nh (height self))
	(nw (width self))
	(children (managed-of (children self)))
	(temp nil))
       (setf (slot-value self 'repaint-flag) nil)
       (dolist (ch children)
	       (let* ((gs (a-gm-parse-spec ch self))
		      (pct-x (first gs))
		      (pct-y (second gs))
		      (pct-w (third gs))
		      (pct-h (fourth gs))
		      (chw (base-width ch))
		      (chh (base-height ch))
		      (xb (round (* pct-x nw)))
		      (yb (round (* pct-y nh)))
		      (wb (round (* pct-w nw)))
		      (hb (round (* pct-h nh)))
		      (arrows (getf gs :arrow))
		      (anchors (getf gs :anchor))
		      (x nil)
		      (y nil)
		      (w nil)
		      (h nil))
		     (if (not (or anchors arrows)) 
			 (apply #'reshape ch 
				(actual-region 
				 ch 
				 :x (round (+ xb (/ (max 0 (- wb chw)) 2)))
				 :y (round (+ yb (/ (max 0 (- hb chh)) 2)))
				 :width chw
				 :height chh))
			 (progn
			  (when arrows
				(when (member :horiz arrows)
				      (setq x xb 
					    w wb))
				(when (member :vert arrows)
				      (setq y yb
					    h hb)))
			  (when anchors 
				(if (setq temp (getf anchors :left)) 
				      (setq x temp)) 
				(if (setq temp (getf anchors :top))
				      (setq y temp)) 
				(if (null w) 
				    (setq w chw))
				(if (null h)
				    (setq h chh))
				(if (setq temp (getf anchors :right)) 
				      (if x
					  (setq w (max 1 (- nw x temp)))
					  (setq x (- nw chw temp))))
				(if (setq temp (getf anchors :bottom))
				      (if y
					  (setq h (max 1 (- nh y temp)))
					  (setq y (- nh chh temp))))
				(if (null x) 
					(setq x 
					      (round (+ xb (/ (max 0 (- wb chw)) 
							      2)))))
				(if (null y)
					(setq y
					      (round (+ yb (/ (max 0 (- hb chh)) 
							      2))))))
			  (if (null w) (setq w chw))
			  (if (null h) (setq h chh))
			  (if (null x) (setq x (round (+ xb (/ (max 0 (- wb chw)) 
							    2))))) 
			  (if (null y) (setq y (round (+ yb (/ (max 0 (- hb chh)) 
							    2)))))
			  (apply #'reshape ch 
				 (actual-region ch
						:x x 
						:y y 
						:width w 
						:height h))))))
       (setf (slot-value self 'repaint-flag) t)
       (repaint self)))

(defun a-gm-reshape-one-child (self ch)
  (let* ((nh (height self))
	 (nw (width self))
	 (temp nil)
	 (gs (a-gm-parse-spec ch self))
	 (pct-x (first gs))
	 (pct-y (second gs))
	 (pct-w (third gs))
	 (pct-h (fourth gs))
	 (chw (max (base-width ch) (width ch)))
	 (chh (max (base-height ch) (height ch)))
	 (xb (round (* pct-x nw)))
	 (yb (round (* pct-y nh)))
	 (wb (round (* pct-w nw)))
	 (hb (round (* pct-h nh)))
	 (arrows (getf gs :arrow))
	 (anchors (getf gs :anchor))
	 (x nil)
	 (y nil)
	 (w nil)
	 (h nil))
	(if (not (or anchors arrows)) 
	    (apply #'reshape
		   ch
		   (actual-region 
		    ch 
		    :x (round (+ xb (/ (max 0 (- wb chw)) 2)))
		    :y (round (+ yb (/ (max 0 (- hb chh)) 2)))
		    :width chw 
		    :height chh))
	    (progn
	     (when arrows
		   (when (member :horiz arrows)
			 (setq x xb 
			       w wb))
		   (when (member :vert arrows)
			 (setq y yb
			       h hb)))
	     (when anchors 
		   (when (setq temp (getf anchors :left)) 
			 (setq x temp)) 
		   (when (setq temp (getf anchors :top))
			 (setq y temp)) 
		   (if (null w) (setq w chw))
		   (if (null h) (setq w chh))
		   (when (setq temp (getf anchors :right)) 
			 (if x
			     (setq w (max 1 (- nw x temp)))
			     (setq x (- nw chw temp))))
		   (when (setq temp (getf anchors :bottom))
			 (if y
			     (setq h (max 1 (- nh y temp)))
			     (setq y (- nh chh temp))))
		   (unless x 
			   (setq x 
				 (round (+ xb (/ (max 0 (- wb chw)) 
						 2)))))
		   (unless y
			   (setq y
				 (round (+ yb (/ (max 0 (- hb chh)) 
						 2))))))
	     (if (null w) (setq w chw))
	     (if (null h) (setq w chh))
	     (unless x (setq x (round (+ xb (/ (max 0 (- wb chw)) 
					       2))))) 
	     (unless y (setq y (round (+ yb (/ (max 0 (- hb chh)) 
					       2)))))
	     (apply #'reshape
		    ch
		    (actual-region ch :x x :y y :width w :height h)))))
  (if (and (exposed-p ch) (not (x-window-p ch)))
      (repaint self)))

(defun a-gm-parse-spec (self parent &aux gs)
  (setq gs (geom-spec self))
  (cond ((and (consp gs) (eql (car gs) *anchor-gm-magic-id*))
	 (return-from a-gm-parse-spec (cdr gs)))
	((list gs)
	 (case (length gs)
	       (0 (setq gs (a-gm-calc-specs self parent)))
	       (2 (if (a-gm-valid-keys gs)
		      (setq gs (append (a-gm-calc-specs self parent) gs))
		      (progn
		       (warn "anchor-parse-spec: invalid geom-spec ~S" gs)
		       (setq gs (a-gm-calc-specs self parent)))))
	       (4 (cond ((every #'a-gm-valid-dims gs) 
			 (setq gs (cons *anchor-gm-magic-id* gs)))
			((a-gm-valid-keys gs)
			 (setq gs (append (a-gm-calc-specs self parent) gs)))
			(t
			 (warn "anchor-parse-spec: invalid geom-spec ~S" gs)
			 (setq gs (a-gm-calc-specs self parent)))))
	       #|(5 (if (and (every #'a-gm-valid-dims (butlast gs))
			   (a-gm-valid-keys (last gs)))
		      (setq gs (cons *anchor-gm-magic-id* gs))
		      (progn
		       (warn "anchor-parse-spec: invalid geom-spec ~S" gs)
		       (setq gs (a-gm-calc-specs self parent)))))|#
	       (6 (cond ((and (every #'a-gm-valid-dims (butlast gs 2))
			      (a-gm-valid-keys (cddddr gs)))
			 (setq gs (cons *anchor-gm-magic-id* gs)))
			((a-gm-valid-keys gs)
			 (setq gs (append (a-gm-calc-specs self parent) gs)))
			(t
			 (warn "anchor-parse-spec: invalid geom-spec ~S" gs)
			 (setq gs (a-gm-calc-specs self parent)))))
	       (8 (if (and (every #'a-gm-valid-dims (butlast gs 4))
			   (a-gm-valid-keys (cddddr gs)))
		      (setq gs (cons *anchor-gm-magic-id* gs))
		      (progn
		       (warn "anchor-parse-spec: invalid geom-spec ~S" gs)
		       (setq gs (a-gm-calc-specs self parent)))))
	       (10 (if (and (every #'a-gm-valid-dims (butlast gs 6))
			    (a-gm-valid-keys (cddddr gs)))
		       (setq gs (cons *anchor-gm-magic-id* gs))
		       (progn
			(warn "anchor-parse-spec: invalid geom-spec ~S" gs)
			(setq gs (a-gm-calc-specs self parent)))))
	       (12 (if (and (every #'a-gm-valid-dims (butlast gs 8))
			    (a-gm-valid-keys (cddr (cddddr gs))))
		       (setq gs (cons *anchor-gm-magic-id* gs))
		       (progn
			(warn "anchor-parse-spec: invalid geom-spec ~S" gs)
			(setq gs (a-gm-calc-specs self parent)))))
	       (t (warn "anchor-parse-spec: invalid geom-spec ~S" gs) 
		  (setq gs (a-gm-calc-specs self parent)))))
	(t
	 (warn "anchor-parse-spec: invalid geom-spec ~S" gs) 
	 (setq gs (a-gm-calc-specs self parent))))
  (setf (slot-value self 'geom-spec) gs)
  (cdr gs))

(defun a-gm-valid-dims (gs)
  (and (numberp gs) (>= gs 0) (<= gs 1)))

(defun a-gm-valid-keys (gs)
  (case (length gs)
	(2 (a-gm-valid-key gs))
	(4 (and (a-gm-valid-key gs) (a-gm-valid-key (cddr gs))))
	(t nil)))

(defun a-gm-valid-key (pair &aux l)
  (case (car pair)
	(:arrow 
	 (setq l (cadr pair))
	 (when (consp l)
	       (dolist (el l)
		       (unless (or (eq el :horiz) (eq el :vert))
			       (return-from a-gm-valid-key nil)))
	       t))
	(:anchor 
	 (setq l (cadr pair))
	 (when (consp l)
	       (do* ((subl l (cddr subl))
		     (el (car subl) (car subl))
		     (val (cadr subl) (cadr subl)))
		    ((null subl))
		    (unless (and (or (eq el :left) (eq el :right) 
				     (eq el :top) (eq el :bottom))
				 (integerp val) (>= val 0))
			    (return-from a-gm-valid-key nil)))
	       t))
	(t nil)))

(defun a-gm-calc-specs (self parent) 
  (let* ((pw (width parent))
	 (ph (height parent))
	 (cw (max (width self) (base-width self)))
	 (ch (max (height self) (base-height self)))
	 (cx (x-offset self))
	 (cy (y-offset self))
	 (pct-x (/ cx pw))
	 (pct-y (/ cy ph))
	 (pct-w (min (/ cw pw) (- 1 pct-x)))
	 (pct-h (min (/ ch ph) (- 1 pct-y))))
	(list *anchor-gm-magic-id* pct-x pct-y pct-w pct-h)))
