;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/toolkit/base/opaque-window.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:02:19 $

;;;

(in-package "PT")

;;;
;;; opaque-window class
;;;

(defclass opaque-window (x-window)
  ((name :initform "An Opaque Window")
   (width :initarg :width :initform 1)
   (height :initarg :height :initform 1)
   (parent :initarg :parent :initform nil )
   (event-mask
    :initarg :event-mask
    :initform '( :exposure )
    :type list)
   (border-type :initform :box)
   (border-width :initform 2)
   (icon
    :initarg :icon
    :initform nil
    :type icon
    :reader icon)
   (activated :initform t :accessor activated :type atom)
   (background :initform "white")
   (foreground :initform "black")))

(defun make-opaque-window (&rest keys)
  (apply #'make-instance 'opaque-window :allow-other-keys t keys))

;;;
;;; opaque-window setf methods
;;;

(defmethod (setf background) (val (self opaque-window) &aux old attp gc) 
  (setq old (slot-value self 'background)
	attp (attached-p self))
  (cond ((stringp val)
	 (if (not attp)
	     (setf (slot-value self 'background) val)
	     (if (setq old (get-paint val self)) 
		 (setf (background self) old) 
		 (error 
		  "opaque-window.setf.background: couldn't find paint \`~s\`."
		  val))))
	((color-p val) 
	 (when attp
	       (color-attach val)
	       (setf (xlib:window-background (res self)) (pixel val))
	       (when (xlib:gcontext-p (setq gc (slot-value self 'gc-res)))
		     (setf (xlib:gcontext-background gc) (pixel val))
		     (xlib:force-gcontext-changes gc))
	       (when (paint-p old)
		     (do-detach old)))
	 (setf (slot-value self 'background) val)
	 (clear self))
	((integerp val)
	 (when attp
	       (setf (xlib:window-background (res self)) val)
	       (when (xlib:gcontext-p (setq gc (slot-value self 'gc-res)))
		     (setf (xlib:gcontext-background gc) val)
		     (xlib:force-gcontext-changes gc))
	       (when (paint-p old)
		     (do-detach old)))
	 (setf (slot-value self 'background) val)
	 (clear self))
	((image-p val)
	 (when (and attp (paint-p old))
	       (do-detach old))
	 (setq val (make-tile :image val :window self))
	 (when attp
	       (setf (xlib:window-background (res self)) (res val)))
	 (setf (slot-value self 'background) val)
	 (clear self))
	((tile-p val) 
	 (when (and attp old (paint-p old))
	       (do-detach old))
	 (setf (window val) self)
	 (when attp
	       (setf (xlib:window-background (res self)) (res val)))
	 (setf (slot-value self 'background) val) 
	 (clear self))
	((eq :parent-relative val)
	 (when (and attp (paint-p old))
	       (do-detach old))
	 (when attp
	       (when (and (setq val (slot-value self 'cached-background))
			  (xlib:gcontext-p (setq gc (slot-value self 'gc-res))))
		     (setf (xlib:gcontext-background gc) val))
	       (setf (xlib:window-background (res self)) :parent-relative))
	 (setf (slot-value self 'background) :parent-relative)
	 (clear self))
	((null val)
	 (when attp
	       (when (and (setq val (slot-value self 'cached-background))
			  (xlib:gcontext-p (setq gc (slot-value self 'gc-res))))
		     (setf (xlib:gcontext-background gc) val))
	       (setf (xlib:window-background (res self)) :none)
	       (when (paint-p old)
		     (do-detach old)))
	 (setf (slot-value self 'background) nil)
	 (clear self))
	(t (error "opaque-window.setf.background: invalid paint \`~s\`." 
		  val))))

(defmethod (setf icon-name) (value (self opaque-window))
  (setf (name self) value)
  (when (attached-p self)
	(setf (xlib:wm-icon-name (res self)) value)))

(defmethod icon-name ((self opaque-window))
  (name self))

(defmethod (setf title) (value (self opaque-window))
  (when (attached-p self)
	(setf (xlib:wm-name (res self)) value)))

(defmethod (setf icon) (value (self opaque-window))
  "Set method for changing the icon of an opaque-window"
  (cond ((null value)
	 (when (attached-p self) 
	       (xlib:set-standard-properties 
		(res self)
		:icon-pixmap nil)))
	((eq (icon self) value)
	 (when (attached-p self)
	       (unless (attached-p value) (attach value))
	       (xlib:set-standard-properties 
		(res self)
		:icon-pixmap (res value))))
	((icon-p value)
	 (when (attached-p self)
	       (unless (attached-p value) (attach value))
	       (xlib:set-standard-properties 
		(res self)
		:icon-pixmap (res value))))
	(t
	 (error "opaque-window.icon: invalid icon ~s" value)))
  (setf (slot-value self 'icon) value)
  value)

;;;
;;;	Clear methods
;;;

(defmethod clear-region ((self opaque-window) x y w h)
  (when (and (>= x 0) (>= y 0) (>= w 0) (>= h 0) (exposed-p self)) 
	(xlib:clear-area (res self) 
			 :x x :y y :width w :height h :exposures-p nil)))

(defmethod clear ((self opaque-window) &key (ignore nil) &allow-other-keys)
  (declare (ignore ignore))
  (when (exposed-p self)
	(xlib:clear-area (res self) :exposures-p nil)))

;;;
;;;	Dummy activate, deactivate methods
;;;

(defmethod activate ((self opaque-window))
  (setf (activated self) t))

(defmethod deactivate ((self opaque-window))
  (setf (activated self) nil))

;;;
;;; opaque-window initialization method
;;;

(defmethod new-instance ((self opaque-window)
			 &rest args)
  "Initialize a new instance of the opaque-window class"
  (declare (ignore args))
  ;; call next method to interpret the shape of the window, colors, resize-hint
  (call-next-method)

  (setf (slot-value self 'partial-repaint-p)
	(member :expose-region (slot-value self 'event-mask)))
  self)

;;;
;;; opaque-window methods
;;;

(defmethod do-attach ((self opaque-window) &aux res background icon)
  (unless (attached-p self)
	  ;; create the window
	  (setf (slot-value self 'res)
		(setq res
		      (xlib:create-window
		       :parent (res (parent self))
		       :x (x-offset self)
		       :y (y-offset self)
		       :width (max 1 (width self))
		       :height (max 1 (height self))
		       :border-width (if (integerp (border-width self))
					 (border-width self)
					 0))))
	  
	  (unless res
		  (error "opaque-window.attach: can't create window"))
	  ;; set instance slots which require side-effects
	  (let ((name (slot-value self 'name))
		(status (slot-value self 'status))
		(event-mask (slot-value self 'event-mask))
		(cursor (slot-value self 'cursor))
		(base-width (slot-value self 'base-width))
		(base-height (slot-value self 'base-height))
		(width-increment (slot-value self 'width-increment))
		(height-increment (slot-value self 'height-increment)))
	       (setf (status self) status)
	       (if (not (zerop (length name))) (setf (name self) name))
	       (if (not (eq event-mask :no-event))
		   (setf (event-mask self) event-mask))
	       (if (and cursor (not (eq cursor (make-cursor))))
		   (setf (cursor self) cursor))
	       (if (or (not (eq base-width 0))
		       (not (eq base-height 0))
		       (not (eq width-increment 1))
		       (not (eq height-increment 1)))
		   (setf (resize-hint self) (resize-hint self))))
	  
	  ;; add instance to hash table
	  (append-window self)
	  
	  ;; Set the icon for side effects
	  (when (setq icon (icon self))
		(attach icon)
		(setf (icon self) icon)
		(setf (xlib:wm-icon-name res) (name self))
		(setf (xlib:wm-name res) (name self)))
	  
	  ;; Call next method after, so next methods have a real window to work with
	  (call-next-method)
	  
	  ;; Set background and border
	  (setq background (background self)) 
	  (setf (xlib:window-background res) 
		(cond ((color-p background)
		       (pixel background))
		      ((tile-p background)
		       (res background))
		      ((numberp background) 
		       background)
		      ((eq :parent-relative background)
		       :parent-relative)
		      (t :none)))
	  
	  (when (and (root-window-p (parent self))
		     (slot-value self 'position-specified))
		(xlib:set-standard-properties res
					      :user-specified-position-p t
					      :x (x-offset self)
					      :y (y-offset self)))
	  
	  ;; Map the window if it's not concealed and not pended
	  (when (exposed-p self)
;		(if (and (typep self 'collection-gadget)
;			 (typep (parent self) 'root-window))
;		    (repack-on self))
		(xlib:map-window (res self)))
	  
	  ;; return self
	  self))


;;;
;;; opaque-window event handler(s)
;;;

(defun conceal-window-handler (self &key child &allow-other-keys)
  "Handle conceal-window events for opaque-windows -- expose its icon"
  (when child (return-from conceal-window-handler))
  (let ((icon (icon self)))
       (if (and (window-p icon) (concealed-p icon))
	   (expose icon)))
  (when (exposed-p self)
	(conceal self :x-unmap nil)))

(defun expose-region-handler (self &key child x y width height 
				   &allow-other-keys)
  "Handle expose-region events for opaque-windows"
  (declare (type integer x y width height))
  (when child (return-from expose-region-handler))
  (repaint-region self x y width height))

(defun find-root-sup (w)
  (do ((sup (parent w) (parent w)))
      ((or (root-window-p sup) (null sup)) (if sup w nil))
      (setq w sup)))

(defun expose-window-handler (self &key child &allow-other-keys)
  "Handle expose-window events for opaque-windows"
  (declare (type integer x y width height))
  (when child (return-from expose-window-handler))

  
  ;; Sometimes X10 gives us the events in a strange order, ie, an expose
  ;; event for a child before the parent.  In this case, we need to expose
  ;; the parent recursively.
  ;; DOES THIS APPLY TO X11?
  #|(do ((win self (parent win)))
	((or (exposed-p self) (eq win (root-window))))
	(expose win :x-map nil))
  
  (let ((event-list (x-next-event-nohang
		     :window (id self)
		     :mask (logior *expose-window* *expose-region*)
		     :count t)))
       (dolist (ev event-list)
	       (if (zerop (x-event-subwindow ev))
		   (setq event ev))))|#
  
  (do-expose self)
  (repaint self))
