;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/toolkit/base/gc.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:02:19 $
;;;

(in-package "PT")

(defun parse-gc-spec (spec)
  (unless (zerop (mod (length spec) 2))
	  (error "parse-gc-spec: unpaired gc list \`~s\`" spec))
  (do* ((l spec (cddr l)))
       ((endp l))
       (unless (keywordp (car l))
	       (error "parse-gc-spec: illegal gc entry \`~s\`" (car l))))
  spec)

(defun final-parse-gc-spec 
  (&key arc-mode background cap-style clip-mask
	clip-ordering clip-x clip-y dash-offset dashes exposures 
	fill-rule fill-style font foreground function join-style 
	line-style line-width plane-mask stipple subwindow-mode
	tile ts-x ts-y paint &allow-other-keys)
  (list :arc-mode arc-mode 
	:background background
	:cap-style cap-style 
	:clip-mask clip-mask
	:clip-ordering clip-ordering
	:clip-x clip-x 
	:clip-y clip-y
	:dash-offset dash-offset
	:dashes dashes 
	:exposures exposures
	:fill-rule fill-rule
	:fill-style fill-style 
	:font font 
	:foreground foreground
	:function function 
	:join-style join-style
	:line-style line-style
	:line-width line-width 
	:plane-mask plane-mask
	:stipple stipple
	:subwindow-mode subwindow-mode
	:tile tile
	:paint paint
	:ts-x ts-x
	:ts-y ts-y))

;;;
;;;	Inserts named spec into hashtable.  Uses *default-gc-spec* for any
;;;	unspecified entries (spec should be a paired keyword-value list).
;;;
(defun register-gc (name spec &optional (template-name "default") &aux new-spec)
  (setq spec (parse-gc-spec spec)) 
  (setq new-spec 
	(if (equal name "default")
	    (copy-list *default-gc-spec*) 
	    (get-gc template-name)))
  (do* ((l spec (cddr l))
	(key (car l) (car l))
	(val (cadr l) (cadr l)))
       ((endp l))
       (setf (getf new-spec key) val))
  (setf (gethash name *global-gc-hashtab*) new-spec))

;;;
;;;	Retrieves named gc from hashtable.  Uses spec argument to override
;;;	desired entries (spec is a paired keyword-value list). 
;;;
(defun get-gc (&optional name spec &aux new-spec)
  (setq spec (parse-gc-spec spec))
  (unless name (setq name "default"))
  (setq new-spec (copy-list (gethash name *global-gc-hashtab*)))
  (do* ((l spec (cddr l))
	(key (car l) (car l))
	(val (cadr l) (cadr l)))
       ((endp l))
       (setf (getf new-spec key) val))
  (when new-spec (apply #'final-parse-gc-spec new-spec)))

;;;
;;;	Creates a clx gc structure from spec for window.
;;;	If shared:
;;;	     registers gc in hashtable with key: (window spec).
;;;	IMPORTANT NOTE:  a shared gc created from make-gc should never be
;;;	altered.
;;;

(defun cleanup-shared-gcs (window &aux keys)
  (maphash #'(lambda (k v) (when (eq (car k) window)
				 (xlib:free-gcontext v)
				 (push k keys)))
	   *shared-gc-cache*)
  (dolist (k keys)
	  (remhash k *shared-gc-cache*)))

(defun make-gc (window spec &optional (shared nil)
		       &aux copied-spec cache-val name col gc)
  (when shared 
	(setq col (find-window (res window))) 
	(if (setq cache-val (gethash (cons col spec) *shared-gc-cache*))
	    (return-from make-gc cache-val)
	    (setq copied-spec (copy-list spec))))
  (unless (attached-p window) 
	  (error "make-gc:  window ~s is not attached" window))
  (cond ((listp spec)
	 (setq name (if (stringp (car spec))
			(prog1
			 (car spec)
			 (setq spec (cdr spec)))
			"default")))
	((stringp spec)
	 (psetq name spec 
		spec nil))
	(t (error "make-gc:  bad spec \'~s\'" spec)))
  (setq spec (get-gc name spec))
  (setq spec (update-window-gc window spec))
  (remf spec :paint)
  (setq gc (apply #'xlib:create-gcontext
		  (nconc (list :drawable (res window)) spec)))
  (when shared
	(setf (gethash (cons col copied-spec) *shared-gc-cache*) gc))
  gc)


(defun alter-gc (gc atts &aux win f)
  (setq win (find-window (xlib::gcontext-drawable gc)))
  (unless win (warn "alter-gc: Couldn't find window"))
  (setq atts (update-window-gc win atts))
  (remf atts :paint)
  (do ((l atts (cddr l)))
      ((endp l))
      (setq f (symbol-function 
	       (read-from-string 
		(clos::string-append "xlib::set-gcontext-" 
				    (symbol-name (car l))))))
      (funcall f gc (cadr l))))
