(in-package "PT")

(defmacro lexical-environment ()
  `*current-environment*)

(defmacro refresh-environment ()
  `(setq *current-environment*
         (or
          (starting-po)
          (current-dialog)
          (current-panel)
          (current-frame)
          (current-tool)
          (root-window))))

(defmacro push-env (place)
  `(prog1 (push ,place *starting-po-list*)
          (refresh-environment)))

(defmacro pop-env ()
  `(prog1 (pop *starting-po-list*)
          (refresh-environment)))

(defmacro starting-po ()
  `(car *starting-po-list*))

(defmacro push-dialog (dialog)
  `(prog1 (push (get-form ,dialog) *current-dialogs*)
          (refresh-environment)))

(defmacro pop-dialog ()
  `(prog1 (pop *current-dialogs*)
          (refresh-environment)))

(defmacro real-current-dialog ()
  `(parent (car *current-dialogs*)))

(defmacro current-dialog ()
  `(car *current-dialogs*))

(defmacro push-panel (panel)
  `(prog1 (setq *current-panel* (get-form ,panel))
          (refresh-environment)))

(defmacro pop-panel ()
  `(prog1 (setq *current-panel* nil)
          (refresh-environment)))

(defmacro current-panel ()
  `*current-panel*)

(defmacro real-current-panel ()
  `(parent *current-panel*))

(defmacro push-frame (frame)
  `(prog1 (push (get-form ,frame) *current-frames*)
          (refresh-environment)))

(defmacro pop-frame ()
  `(prog1 (pop *current-frames*)
          (refresh-environment)))

(defmacro real-current-frame ()
  `(parent (car *current-frames*)))

(defmacro current-frame ()
  `(car *current-frames*))

(defmacro push-tool (tool)
  `(prog1 (setq *current-tool* ,tool)
          (refresh-environment)))

(defmacro pop-tool ()
  `(prog1 (setq *current-tool* nil)
          (refresh-environment)))

(defmacro current-tool ()
  `*current-tool*)

(defmacro ret-tool (&optional (retval nil))
  `(leave (current-tool) ,retval))

(defmacro exit-tool (&optional (retval nil))
  `(leave (current-tool) ,retval))

(defmacro ret-frame (&optional (retval nil))
  `(leave (real-current-frame) ,retval))

(defmacro ret-form (&optional (retval nil))
  `(leave (real-current-form) ,retval))

(defmacro ret-dialog (&optional (retval nil))
  `(leave (real-current-dialog) ,retval))

(defmacro ret (self &optional (retval nil))
  `(do ((window ,self (parent window)))
       ((picasso-object-p window) (if (form-p window)
                                         (leave (parent window) ,retval)
                                         (leave window ,retval)))
       (when (root-window-p window)
             (warn "ret:  called from outside any picasso object")
             (return))))

(defmacro package-search-list ()
  `(progn
    (if (eq *package-search-list* :uninitialized)
        (setup-package-search-list))
    *package-search-list*))

(defmacro current-package ()
  `(progn
    (if (eq *package-search-list* :uninitialized)
        (setup-package-search-list))
    (car *package-search-list*)))

(defmacro find-po-named (name-form &rest args &key reload destroy-old)
  `(if ,reload
       (if ,destroy-old 
           (apply #'reload-picasso-object-named ,name-form
                  :allow-other-keys t ',args)
           (reload-picasso-object-named ,name-form))
       (find-picasso-object-named ,name-form)))

;; propagation macros

;;; Update, expressions are expected to be without any remaining local
;;; variable references.  To accomplish this, they should generally
;;; be created in backquote form.  for instance,
;;;
;;; (bind-slot 'sname win1 `(concatenate 'string (var name ,win2)))
;;;

(defmacro bind-var (first second &rest others)
  (let* ((third (car others))
         (thirdp (and third (not (eql third :receipt))))
         (receipt (cadr (member :receipt others))))
        (if thirdp
            `(bind-var-internal ,first ,second ,third :receipt ,receipt)
	    (if (valid-var-expression first)
		(progn
		 (if (eql (car first) 'value) (setq first (cadr first)))
		 `(bind-var-internal ,(cadr first)
				     ,(or (caddr first)
					  '(lexical-environment))
				     ,second :receipt ,receipt))
		`(warn "Bind-var:  Invalid arguments")))))

;;; defpropagator sets up a slot for a class to always try to propagate
;;; as variable values do.  This is here for optimizing code.

(defmacro defpropagator (slot class-name)
  `(let  ((key (cons ',slot (find-class ',class-name))))
         (unless (gethash key *class-meth-table*)
                 (defmethod (setf ,slot) :around (val (self ,class-name))
                    (if (eql (find-class ',class-name) (class-of self))
                     (unless (equal (,slot self) val)
                             (call-next-method)
                             (let* ((key (list ',slot (list 'quote self))))
                                   (propagate
                                    (gethash key *prop-table*) key))
                     (call-next-method))))
                 (setf (gethash (cons ',slot (find-class ',class-name))
                                *class-meth-table*)
                       t))))

;; set-trigger
;; (set-trigger slot object function) or (set-trigger variable function)
;; the function argument is evaluated whenever the slot or variable changes

(defmacro set-trigger (s-or-v o-or-f &optional (func nil func-p))
   (if func-p
       `(bind-slot 'value *dummy-prop-var*
          '(progn (var ,(eval s-or-v) ,(eval o-or-f)) ,(eval func)))
       `(bind-slot 'value *dummy-prop-var*
          '(progn (var ,s-or-v :ref ,(lexical-environment)) ,(eval o-or-f)))))

;; blet - easy syntax for complicated bindings

(defmacro blet (what &rest stuff &aux v w bool form)
  (setq v (mapcar #'bind-helper (cadr (member :var stuff))))
  (setq w (mapcar #'bind-helper-2 (cadr (member :with stuff))))
  (setq bool nil)
  (dolist (s stuff)
          (if bool
              (setq bool nil)
              (if (member s '(:var :with))
                  (setq bool t)
                  (setq form (cons (list 'quote s) form)))))
  (setq form `(list 'quote
                    (list 'let (list ,@(append v w)) ,@(nreverse form))))

  (if (valid-var-expression what)
      (setq form  `(list 'bind-var ',what ,form))
      (setq form  `(list 'bind-slot '',(car what) ',(cadr what) ,form)))
  (repair-damage (eval form)))

(defun bind-helper (let-clause)
    (if (valid-var-expression (second let-clause))
        `(list ',(car let-clause)
               (list 'var ',(second let-clause)))
        `(list ',(car let-clause)
               (list 'var ',(car (second let-clause))
                          ,(cadr (second let-clause))))))

(defun bind-helper-2 (let-clause)
  `(list ',(car let-clause) ,(second let-clause)))

(defun repair-damage (form)
  (cond ((atom form) form)
        ((eql (car form) 'let)
                  (cons (car form)
                        (cons (repair-let-clauses (cadr form))
                              (cddr form))))
        ((cons (repair-damage (car form)) (repair-damage (cdr form))))))

(defun repair-let-clauses (list)
  (mapcar #'repair-one-let-clause list))

(defun repair-one-let-clause (cl)
  (if (and (listp (cadr cl)) (eql (caadr cl) 'var))
      cl
      `(,(car cl) ',(cadr cl))))

;; bind -- the simplest syntax of all

(defmacro bind (this that)
  `(blet ,this
         :var ((x ,that))
         x))

(defmacro lookup (name &optional (place '(lexical-environment)))
  `(find-var ,name ,place))

(defmacro symbolify (list)
  `(read-from-string (coerce ,list 'string)))

(defun value-var-reader (stream subchar arg)
  (list 'value (reference-var-reader stream subchar arg)))

(eval-when (compile load eval)
(set-dispatch-macro-character #\# #\! #'value-var-reader))

(defun reference-var-reader (stream subchar arg)
  (declare (ignore subchar arg))
  (let ((variable (read stream t nil t))
        (lvar nil)
        (vtail nil))
       (cond
        ((consp variable) (car variable))
        (t
         (setq lvar (coerce (string variable) 'list))
         (when (eql (car (last lvar)) #\@)
               (let ((next (read stream t nil t))
                     (after (peek-char nil stream t nil)))
                    (setq lvar (nconc lvar
                                      (coerce (prin1-to-string next) 'list)))
                    (when (eql after #\/)
                          (setq lvar
                                (nconc lvar
                                       (coerce (string (read stream t nil t))
                                               'list))))))
         (setq vtail (member #\/ lvar))
         (let* ((start (butlast lvar (length vtail)))
                (rest (cdr vtail))
                (retval (if  (setq vtail (member #\@ start))
                             `(lookup ',(symbolify
                                         (butlast start (length vtail)))
                                      ,(symbolify (cdr vtail)))
                             `(lookup ',(symbolify start)))))
               (dolist (v (get-vars-from rest))
                       (setq retval `(lookup ',v (value ,retval))))
               retval)))))

(defun get-vars-from (list &aux tail)
  (if (null list) nil
                  (if (setq tail (member #\/ list))
                      (cons (symbolify (butlast list (length tail)))
                            (get-vars-from (cdr tail)))
                      (list (symbolify list)))))

(eval-when (compile load eval)
  (set-dispatch-macro-character #\# #\? #'reference-var-reader))

(defmacro enforce-constants ()
  `(setq *constants-enforced* t))

(defmacro relax-constants ()
  `(setq *constants-enforced* nil))

;; define top-level variable

(defmacro defpvar (name &optional (init nil))
  `(add-var ',(unquote name) (root-window) ,init))

(defun unquote (n)
  (if (and (consp n) (eq (car n) 'quote))
      (cadr n)
      n))

(defmacro deftool (name arglist &rest params)
  `(def-internal 'tool ',name ',arglist ',params *deftool-clauses*))

(defmacro defframe (name arglist &rest params)
  `(def-internal 'frame ',name ',arglist ',params *defframe-clauses*))

(defmacro defform (name arglist &rest params)
  `(def-internal 'form ',name ',arglist ',params *defform-clauses*))

(defmacro defpanel (name arglist &rest params)
  `(def-internal 'panel ',name ',arglist ',params *defpanel-clauses*))

(defmacro defdialog (name arglist &rest params)
  `(def-internal 'dialog ',name ',arglist ',params *defdialog-clauses*))

(defmacro call (self &rest args)
  `(invoke ,self ,@args))

(defmacro run-dialog (self &rest args)
  `(call ,self ,@args))

(defmacro call-dialog (self &rest args)
  `(call ,self ,@args))

(defmacro open-dialog (self &rest args)
  `(call ,self ,@args))

(defmacro call-frame (self &rest args)
  `(call ,self ,@args))

(defmacro run-frame (self &rest args)
  `(call ,self ,@args))

(defmacro run-panel (self &rest args)
  `(call ,self ,@args))

(defmacro open-panel (self &rest args)
  `(call ,self ,@args))

(defmacro run-tool (self &rest args)
  `(call ,self ,@args))

(defmacro run-tool-named (self &rest args)
  `(call (find-po-named ,self) ,@args))

