
(in-package "PT")

(defconstant *card-path* (picasso-path "lib/po/vp/cards/"))
(defconstant *max-bet* 5)

(defclass hand-type ()
 ((name :initform "Unnamed hand" :type string :initarg :name :reader name)
  (poker-value :initform 0 :type integer :initarg :poker-value :accessor poker-value)
  (test-fxn :initform nil :type t :initarg :test-fxn :reader test-fxn)))

;; test-fxn will be a function of two arguments:  the sorted ranks of the 
;; hand and the suits of the hand.  The test-fxn returns true if the 
;; hand-type is satisfied.

(defun make-hand-type (name value test)
  (make-instance 'hand-type :name name :poker-value value :test-fxn test))

(defclass card ()
  ((suit :type symbol :initform :nothing :initarg :suit :reader suit)
   (rank :type integer :initform 0 :initarg :rank :reader rank)
   (picture :type image :initarg :picture :reader picture)))

(defun make-card (rank suit image)
  (make-instance 'card :rank rank :suit suit :picture image))

(defvar *all-ranks* '(1 2 3 4 5 6 7 8 9 10 11 12 13))
(defvar *all-suits* '(:clubs :hearts :diamonds :spades))

(defvar *deck-size* (* (length *all-ranks*) (length *all-suits*)))
(defvar *max-cards-per-hand* 10)

(defun image-of (rank suit)
  (make-image :gif-file (concatenate 'string 
				     *card-path* 
				     (write-to-string rank)
				     (write-to-string suit))))

(defvar *full-deck*
  (mapcan #'(lambda (suit) 
		    (mapcar #'(lambda (rank)
				   (make-card rank suit (image-of rank suit)))
			    *all-ranks*))
	  *all-suits*))

(defvar *card-back*
  (make-card 0 nil 
            (make-image :gif-file 
                        (concatenate 'string *card-path* "CARDBACK"))))

(defun shuffle (deck)
  (let ((source (copy-list deck))
	(dest nil)
	(rnd nil))
       (dotimes (n *max-cards-per-hand*)
		(setq rnd (random (- *deck-size* n)))
		(push (nth rnd source) dest)
		(setq source (delete (car dest) source)))
       dest))

(defmacro new-deck ()
  `(shuffle *full-deck*))

(defun royal-flush-p (ranks suits)
  (and (straight-flush-p ranks suits)
       (member 1 ranks)
       (member 10 ranks)))

(defun straight-flush-p (ranks suits)
  (and (straight-p ranks suits)
       (flush-p ranks suits)))

(defun four-of-a-kind-p (ranks suits)
  (declare (ignore suits))
  (or (eql (first ranks) (fourth ranks))
      (eql (second ranks) (fifth ranks))))

(defun full-house-p (ranks suits)
  (declare (ignore suits))
  (or (and (eql (first ranks) (third ranks))
	   (eql (fourth ranks) (fifth ranks)))
      (and (eql (first ranks) (second ranks))
	   (eql (third ranks) (fifth ranks)))))

(defun flush-p (ranks suits)
  (declare (ignore ranks))
  (let ((s (car suits)))
       (every #'(lambda (x) (eql s x)) (cdr suits))))

(defun straight-p (ranks suits)
  (declare (ignore suits))
  (and (eql (third ranks) (1+ (second ranks)))
       (eql (fourth ranks) (1+ (third ranks)))
       (eql (fifth ranks) (1+ (fourth ranks)))
       (or (eql (second ranks) (1+ (first ranks)))
	   (and (eql (first ranks) 1) (eql (fifth ranks) 13)))))

(defun three-of-a-kind-p (ranks suits)
  (declare (ignore suits))
  (or (eql (first ranks) (third ranks))
      (eql (second ranks) (fourth ranks))
      (eql (third ranks) (fifth ranks))))

(defun two-pair-p (ranks suits)
  (declare (ignore suits))
  (eql 2 (apply #'+ (maplist #'(lambda (list) 
				       (if (eql (first list) (second list))
					   1 
					   0))
			     ranks))))

(defun jacks-or-better-p (ranks suits)
  (declare (ignore suits))
  (mapcon #'(lambda (list)
		    (if (and (or (> (first list) 10)
				 (= (first list) 1))
			     (eql (first list) (second list)))
			'(t)
			nil))
	  ranks))

(defun losing-hand-p (ranks suits)
  (declare (ignore ranks suits))
  t)

(defvar *all-hand-types*
  (list
   (make-hand-type "Royal Flush" 500 #'royal-flush-p)
   (make-hand-type "Straight Flush" 100 #'straight-flush-p)
   (make-hand-type "Four-of-a-Kind" 25 #'four-of-a-kind-p)
   (make-hand-type "Full House" 8 #'full-house-p)
   (make-hand-type "Flush" 5 #'flush-p)
   (make-hand-type "Straight" 4 #'straight-p)
   (make-hand-type "Three-of-a-Kind" 3 #'three-of-a-kind-p)
   (make-hand-type "Two Pair" 2 #'two-pair-p)
   (make-hand-type "Jacks or Better" 1 #'jacks-or-better-p)
   (make-hand-type "Losing Hand" 0 #'losing-hand-p)))

(defun evaluate (hand)
  ;; takes a hand of five cards and returns the payout and title
  (let ((ranks (sort (mapcar #'rank hand) #'<))
	(suits (mapcar #'suit hand)))
       (mapc #'(lambda (ht)
		       (if (funcall (test-fxn ht) ranks suits)
			   (return-from evaluate
					(list (name ht) (poker-value ht)))))
	     *all-hand-types*)))

;; shortcut for painting less

(defmethod (setf value) :around (new-v (self synth-gadget))
  (unless (and (slot-value self 'value) (eql new-v (value self)))
       (call-next-method)))
