;;; -*-Mode: LISP; Package: PICASSO; Base: 10; Syntax: Common-lisp -*-
;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;;
;;;
;;;       $Source: /n/hermes/pic2/picasso/lib/po/mosaic/RCS/mosaic.cl,v $
;;;       $Author: johnb $
;;;       $Header: /n/hermes/pic2/picasso/lib/po/mosaic/RCS/mosaic.cl,v 1.7 1991/11/19 02:27:56 johnb Exp smoot $
;;;       $Locker: smoot $
;;;

; steve smoot
; Friday, Oct 26
; mosaic -- the common lisp version
; this is version 0.1
; i.e. a direct port of the C code.
; Its really bad common lisp, as I'm relearning the language

; from mosaic.h
;  Constants
(defconstant BoardSize   24 "Board width")
(defconstant BoardSize-1 (- BoardSize 1) "Last place for a piece")
(defconstant BoardSize-3 (- BoardSize 3) "Last piece which has neighbors")
(defconstant Nkinds      3 "Number of kinds of pieces")
(defconstant NPieces     (* Nkinds Nkinds Nkinds Nkinds) "Number of Pieces")
(defconstant NTiles      (* BoardSize BoardSize) "Number of Tiles on Board")
(defconstant MaxMix      1000 "Number of elements to shuffle in initital deck")
(defconstant *hs-file*   (picasso-path "lib/po/mosaic/hs")  "High Score File Name")
(defconstant MaxHSLength 10 "trim length of HS list to this")

;  Globals
(defvar tile)
(defvar piece)
(defvar nextpiece)
(defvar tscore)
(defvar pscore)
(defvar nscore)
(defvar remain)
(defvar size)
(defvar parent)
(defvar *hs-assoc*)

; Macros
(defmacro pc-ul (pc) `(aref ,pc 0))  ; Access pieces 
(defmacro pc-ur (pc) `(aref ,pc 1))
(defmacro pc-ll (pc) `(aref ,pc 2))
(defmacro pc-lr (pc) `(aref ,pc 3))
(defmacro decaref (a e) `(decf (aref ,a ,e)))
(defmacro square (x) `(* ,x ,x))
(defmacro neq (x y) `(not (eq ,x ,y)))
(defmacro nextpiece () `(aref piece nextpiece))

; new game command
(defun New-Game ()
  (Setup-game))

; new setup command
(defun Setup-Game ()
  (Init-Game))

(defun Closeup-Game ()
  (values))

(defun Read-High-Scores ()
  (load *hs-file* :verbose nil)
  (format-for-widget *hs-assoc*))

(defun format-for-widget (hs)
  (let ((str '()))
    (do ((each (car hs) (car rest))
	 (rest (cdr hs) (cdr rest)))
	((null each) (reverse str))
	(setf str (cons (format '() "~20a ~a" (car each) (cadr each))
			str)))))

; Check High Score
(defun Check-High-Scores ()
  (Read-High-Scores)
  (let ((total (+ (aref tscore 0) (aref tscore 1) (aref tscore 2)))
	(last-entry (car (last *hs-assoc*))))
    (cond ((>= total (cadr last-entry))         ; high enough score or
	   (setf *hs-assoc* (insert-score *user* total *hs-assoc*))
	   (write-high-scores)
	   t)
	  ((and (not (assoc *user* *hs-assoc* :test #'string-equal))
		(< (length *hs-assoc*) MaxHSLength)) ; not full yet & not here
	   (setf *hs-assoc* (append *hs-assoc* 
				    `((,*user* ,total))))
	   (write-high-scores)
	   t)
	  (t nil))))

; add a HS to the list
; just put it in and adjust if too many
(defun insert-score (u s lst)
  (let ((new (internal-insert-score u s lst)))
    (if (> (length new) MaxHSLength)
	(setf (cdr (nth (1- MaxHSLength) new)) '()))
    new))

; find the place in the list for the score, and insert
; remove the user from rest of list if present.
(defun internal-insert-score (u s lst)
  (if (null lst) 
      `((,u ,s))
    (let* ((current (car lst))
	   (current-n (car current))
	   (current-s (cadr current)))
      (cond ((< current-s s)
	     (cons `(,u ,s)
		   (remove-user u lst)))
	    ((string-equal u current-n) lst)
	    (t (cons current (internal-insert-score u s (cdr lst))))))))

; if a user is present in a list, remove first occurrence.
(defun remove-user (u l)
  (if (null l)
      '()
    (let* ((current (car l))
	   (current-n (car current)))
      (if (string-equal u current-n)
	  (cdr l)
	(cons current
	      (remove-user u (cdr l)))))))

; Write New High Score File
(defun Write-High-Scores ()
  (let ((file (open *hs-file* :direction :output
                   :if-exists :supersede)))
    (format file "(setf *hs-assoc* '~W)"
	    *hs-assoc*)
    (close file)))


; Init-Game
(defun Init-Game ()
  ; Create Board
  (setq tile (make-array `(,Ntiles) :element-type 'fixnum
			 :initial-element 0))

  ; Create Deck
  (setq piece (make-array `(,Npieces) :element-type '(simple-array fixnum (4))
	:initial-contents (create-initial-board)))
  
  ; Reorder Deck
  (setf *random-state* (make-random-state t))
  (let ((temp 0))
    (do ((i 0 (1+ i))
	 (swap (random Npieces) (random Npieces)))
	((= i MaxMix))
	(setq temp (aref piece swap))
	(setf (aref piece swap) (aref piece 0))
	(setf (aref piece 0) temp)))

  (setq nextpiece 0)

  ; Setup score data structures
  (setq size (make-array `(,NTiles) :element-type 'fixnum 
			 :initial-element 1))
  (setq parent (make-array `(,NTiles) 
			   :element-type 'fixnum 
			   :initial-contents 
			    (reverse (let ((temp '()))
				       (do ((i 0 (1+ i))) ((= i NTiles) temp)
					   (setq temp (cons i temp)))))))
  (setq tscore (make-array `(,Nkinds) :element-type 'fixnum 
			   :initial-element 0))
  (setq nscore (make-array `(,Nkinds) :element-type 'fixnum 
			   :initial-element 0))
  (setq pscore (make-array `(,Nkinds) :element-type 'fixnum 
			   :initial-element 0))
  (setq remain (make-array `(,(1+ Nkinds)) :element-type 'fixnum 
			   :initial-element (/ (* Npieces 4) Nkinds))))


; DropPiece
(defun DropPiece (row col pc)
  (let* ((idx (+ (* row Boardsize) col))
	 (UL idx)      (tUL (pc-ul pc))
	 (UR (1+ idx)) (tUR (pc-ur pc))
	 (LL (+ idx BoardSize)) (tLL (pc-ll pc))
	 (LR (1+ LL))           (tLR (pc-lr pc)))
;    (format t "= ~d ~d ~d ~d ~d ~d ~d~%" row col idx UL UR LL LR) 
    (if (or (= row BoardSize-1)    ; off edge
	    (= col BoardSize-1)
	    (neq (aref tile LR) 0); not empty
	    (neq (aref tile UL) 0)  
	    (neq (aref tile UR) 0)
	    (neq (aref tile LL) 0))
	'()                                ; if not valid, escape function
      (progn 
	(setf (aref tile LR) tLR)
	(setf (aref tile LL) tLL)
	(setf (aref tile UR) tUR)
	(setf (aref tile UL) tUL)
	(decaref remain tLR)
	(decaref remain tLL)
	(decaref remain tUL)
	(decaref remain tUR)
	(Update-and-Score row col UL UR LL LR)  ; sets nscore
	(do ((i 0 (1+ i)))
	    ((= i NKinds))
	    (setf (aref pscore i) (- (aref nscore i) (aref tscore i)))
	    (setf (aref tscore i) (aref nscore i)))
;	(format t "New Score ~a~%" nscore)
	t))))

; Update the DS and change the score.
; This goes through the heap operations, and returns the new score array.
(defun Update-and-Score (row col UL UR LL LR)
  ; merge piece with itself as much as possible
  (Possibly-Merge UR UL)
  (Possibly-Merge LR LL)
  (Possibly-Merge UR LR)
  (Possibly-Merge UL LL)
		   
  ; merge with left
  (if (>= col 1)
      (progn 
       (Possibly-Merge UL (1- UL))
       (Possibly-Merge LL (1- LL))))
  ; merge with top
  (if (>= row 1)
      (progn 
       (Possibly-Merge UR (- UR BoardSize))
       (Possibly-Merge UL (- UL BoardSize))))
  ; merge with right
  (if (<= col BoardSize-3)
      (progn 
       (Possibly-Merge UR (+ UR 1))
       (Possibly-Merge LR (+ LR 1))))
  ; merge with bot.
  (if (<= row BoardSize-3)
      (progn 
       (Possibly-Merge LR (+ LR BoardSize))
       (Possibly-Merge LL (+ LL BoardSize))))
  ; get new score
  (find-score))


; Possibly-Merge
; Consider two heaps, merge them properly if they can be merged
(defun Possibly-Merge (i j)
  (if (neq (aref tile i) (aref tile j))
      (values)
    (Really-Merge i j)))

; Auto-play has fun, selecting random locations, and trying to put tiles there.
(defun Auto-Play ()
  (do ()
      ((= nextpiece NPieces) (Check-and-Display-High-Scores))
      (let* ((location (do ((r (random BoardSize) (random BoardSize))
			    (c (random BoardSize) (random BoardSize)))
			   ((DropPiece r c (nextpiece)) `(,r ,c))))
	     (row (first location))
	     (col (second location)))
;	(format t "Auto Playing on row=~d col=~d np=~d~%" row col nextpiece)
	(setf nextpiece (1+ nextpiece))
	(Update-Board row col)
	(Update-Next-Piece)
	(Update-Totals)
	(flush-display)
	)))
  
  
; Merge two tiles
(defun Really-Merge (i j)
  (let ((i-rep (compress i))
	(j-rep (compress j)))
    (if (= i-rep j-rep)
	'()  ; if same -- done
      (if (> (aref size i-rep) (aref size j-rep))
	  (progn (setf (aref parent j-rep) i-rep)
		 (setf (aref size i-rep) (+ (aref size i-rep) (aref size j-rep))))
	  (progn (setf (aref parent i-rep) j-rep)
		 (setf (aref size j-rep) (+ (aref size i-rep) (aref size j-rep))))))))

    
; compress goes up a path to the root, setting parent up to the top,
; it returns the root's id.
(defun compress (x)
  (let ((rep (find-rep x)))
    (do ((scan x (aref parent scan)))
	((= scan (aref parent scan)) rep)
	(setf (aref parent scan) rep))))
  
  
(defun find-rep (x)
  (do ((p (aref parent x) (aref parent p))
	(i x p))
       ((= i p) p)))

; find-score
; go through all the tiles, 
; look at ones which are tops of heaps, 
; and square in their scores
(defun find-score ()
  (do ((i 0 (1+ i)))
      ((= i NKinds))
      (setf (aref nscore i) 0))
  (do ((i 0 (1+ i)))
      ((= i NTiles) nscore)
      (if (and (not (= 0 (aref tile i)))
	       (= (aref parent i) i))
	  (let ((kind (1- (aref tile i))))
	    (setf (aref nscore kind) (+ (aref nscore kind) (square (aref size i)))))))
  nscore)


(defun n-elts (n e)
  (let ((s '()))
    (do ((i 1 (1+ i)))
	((< n i) s)
	(setq s (cons e s)))))


(defun create-initial-board ()
  (let ((s '()))
    (do ((i 1 (1+ i))) ((> i NKinds))
	(do ((j 1 (1+ j))) ((> j NKinds))
	    (do ((k 1 (1+ k))) ((> k NKinds))
		(do ((m 1 (1+ m))) ((> m NKinds))
		    (setq s (cons (make-array '(4) :element-type 'fixnum
					      :initial-contents `(,i ,j ,k ,m)) 
				  s))))))
    s))



;; Debugging Code

(defun print-board ()
  (do ((i 0 (1+ i)))
      ((= i BoardSize))
      (do ((j 0 (1+ j)))
	  ((= j BoardSize) (format t "~%"))
	  (format t "~A " (format-char (aref tile (+ (* BoardSize i) j)))))))
(defun format-char (c)
  (cond ((= c 0) " ")
	((= c 1) "A")
	((= c 2) "B")
	((= c 3) "C")))

