;;; -*- Mode:LISP;Syntax: Common-Lisp;Package: GINA;Base:10-*-
;;;
;;; Copyright 1990 GMD (German National Research Center for Computer Science)
;;;
;;; Permission to use, copy, modify, distribute, and sell this software and its
;;; documentation for any purpose is hereby granted without fee, provided that
;;; the above copyright notice appear in all copies and that both that
;;; copyright notice and this permission notice appear in supporting
;;; documentation, and that the name of GMD not be used in advertising or
;;; publicity pertaining to distribution of the software without specific,
;;; written prior permission.  GMD makes no representations about the
;;; suitability of this software for any purpose.  It is provided "as is"
;;; without express or implied warranty.
;;;
;;; GMD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
;;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL GMD
;;; BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
;;; WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
;;; OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
;;; CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;; Authors: Project GINA (spenke@gmd.de)
;;;          P.O. Box 1316
;;;          D-5205 Sankt Augustin 1
;;;

(in-package :GINA)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; experimental extension for postscript printing
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defparameter *postscript-print-string*   "| lpr -h -P$PRINTER")
;;(defparameter *postscript-print-string* "| lpr -h -Pf3pxi")
;;(defparameter *postscript-print-string* "| lpr -h -PLWIIg+C5-040 ")
;;(defparameter *postscript-print-string* "| lpr -h -Pf3pxi ") ;;  Farbdrucker

(defparameter *postscript-format* :din-a4-hoch) 
;; :din-a4-hoch oder :din-a4-quer moeglich
(defparameter *postscript-scale*  1.0)          ;; scalierungs-factor


;; override empty method of standard GINA to install Postscript printing
(defginamethod postscript-print ((doc document) &aux print-command)
  (description "hardcopy document on postscript printer"
	       :called-by-gina "in reaction to the postscript-print menu entry"
	       :called-by-application :rarely)
  (with-slots (main-view) doc
    (when (not (and main-view 
		    (eq (gina-class-of main-view)
			(find-gina-class 'view))))
      (warning-dialog "Document does not have a printable main view!"
		      :document doc)
      (return-from postscript-print)))
  (setq print-command 
    (question-dialog "Print command to be submitted:"
		     :document doc
		     :default *postscript-print-string*))
  (when print-command
    (with-clock-cursor
	(let ((ps-file (concatenate 'string "/tmp/" (name doc) ".ps")))
	  (with-slots (main-view) doc
	    (with-open-file
		(ps-stream ps-file :direction :output :if-exists :supersede)
	      (unwind-protect
		  (progn 
		    (setf (printing main-view) ps-stream)
		    (ps-header main-view)
		    (ps-background main-view)
		    (draw main-view 0 0 0 
			  (width main-view) (height main-view)))
		;; cleanup:
		(ps-end ps-stream)	
		(setf (printing main-view) nil)))		  
	    (shell-command
	     (concatenate 'string "cat " (namestring (truename ps-file))
			  " " print-command))
	    (shell-command
	     (concatenate 'string "rm "
			  (namestring (truename ps-file)))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; :after methods for drawing primitives do the printing
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defginamethod draw-point :after ((view view) x y)
  (description "after deamon for postscript printing")
  (when (printing view)
    (ps-point x y view)))

(defginamethod draw-points :after ((view view)
			    points "a flat list of alternating x and y values"
			    &optional relative-p)
  (description "after deamon for postscript printing")
  (declare (ignore relative-p))
  (when (printing view)
    (ps-points points view)))

(defginamethod draw-line :after ((view view) x1 y1 x2 y2 
					     &optional (relative-p nil))
  (description "after deamon for postscript printing")
  (declare (ignore relative-p))
  (when (printing view)
    (ps-line x1 y1 x2 y2 view)))


(defginamethod draw-lines :after ((view view)
			   points "a flat list of alternating x and y values"
			   &key relative-p fill-p (shape :complex))
   (description "after deamon for postscript printing")
   (declare (ignore shape relative-p))
   (when (printing view)
     (ps-lines points fill-p view)))

(defginamethod draw-segments :after ((view view) segments)
  (description "after deamon for postscript printing")
  (when (printing view)
    (ps-segments segments view)))

(defginamethod draw-rectangle :after ((view view) 
			       x y width height &optional (fill-p nil))
  (description "after deamon for postscript printing")
  (when (printing view)
    (ps-rectangle x y width height fill-p view)))

(defginamethod draw-rectangles :after ((view view) 
				rectangles &optional (fill-p nil))
  (description "after deamon for postscript printing")
  (when (printing view)
    (ps-rectangles rectangles fill-p view)))

(defginamethod draw-arc :after ((view view) 
			x y width height angle1 angle2 &optional (fill-p nil))
  (description "after deamon for postscript printing")
  (when (printing view)
    (ps-arc x y (/ width 2) (/ height 2) angle1 angle2
	    fill-p (xlib::gcontext-arc-mode (gcontext view)) view)))

(defginamethod draw-arcs :after ((view view) arcs &optional (fill-p nil))
  (description "after deamon for postscript printing")
  (when (printing view)
    (ps-arcs arcs fill-p (xlib::gcontext-arc-mode (gcontext view)) view)))

(defginamethod draw-glyph :after ((view view)  x y elt
					&key translate width (size :default))
  (description "after deamon for postscript printing")
  (declare (ignore size width translate))
  (when (printing view)
    (ps-glyph x y elt view)))

(defginamethod draw-glyphs :after ((view view)  x y sequence
			 &key (start 0) end translate width (size :default))
  (description "after deamon for postscript printing")
  (declare (ignore size width translate end start))
  (when (printing view)
    (ps-glyphs x y sequence view)))

(defginamethod copy-plane :after ((view view) pixmap depth source-x source-y
				       source-width source-height 
				       dest-x dest-y)
  (description "after deamon for postscript printing")
  (declare (ignore depth source-x source-y 
		   source-width source-height))
  (when (printing view)
    (ps-bitmap  pixmap view dest-x dest-y)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; functions for postscript output
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *pi/2* (/ pi 2))

(defun make-ps-color-or-pattern (view)
  (let ((gc (gcontext view))(arr nil) (z 0)
	(stream (printing view))
	(window (x-window view)))
    (let ((stipp (xlib::gcontext-stipple gc))
	  (fcol (xlib::gcontext-foreground gc))
	  (bcol (xlib::gcontext-background gc)))
      (if stipp 
	  (progn 
	    (let ((wi (xlib::drawable-width stipp))
		  (he (xlib::drawable-height stipp)))
	      (format stream "% a pattern ~%")
	      (format stream "/pattern (\\~%")
	      (setf arr (xlib::image-z-pixarray
			 (xlib::get-image stipp 
					  :x (xlib::drawable-x stipp)
					  :y (xlib::drawable-y stipp)
					  :width  wi
					  :height he
					  :format :z-pixmap)))
	      (loop for i from (1- wi) downto 0 do
		    (loop for j from (1- he) downto 0 do
			  (format stream "~d" (aref arr i j))
			  (when (eql 1 (aref arr i j)) (incf z)))
		    (format stream "\\~%"))
	      (format stream ") def~%")
	      (cond
		((eql z (* wi he));; pattern filled , *p-black*
		 (format stream "  ~d ~d ~d srgb ~%"
			 (red-value   fcol window)
			 (green-value fcol window)
			 (blue-value  fcol window)))
		((eql z 0);; pattern empty , *p-white*
		 (format stream "  ~d ~d ~d srgb ~%"
			 (red-value   bcol window)
			 (green-value bcol window)
			 (blue-value  bcol window)))
		(T (format stream "    pattern 32 0 sms~%")))))
	  (format stream "  ~d ~d ~d srgb ~%"
			 (red-value   fcol window)
			 (green-value fcol window)
			 (blue-value  fcol window)))
      
      )))

(defun make-ps-line-style (gc stream)
  (format stream "  ~d slw ~%" (xlib::gcontext-line-width gc))
  (format stream "  ~d setlinejoin ~%"
	  (cond 
	    ((eql (xlib::gcontext-join-style gc) :bevel) 2)
	    ((eql (xlib::gcontext-join-style gc) :round) 1)
	    (T 0)))
  (format stream "  ~d setlinecap ~%"
	  (cond 
	    ((eql (xlib::gcontext-cap-style gc) :projecting) 2)
	    ((eql (xlib::gcontext-cap-style gc) :round) 1)
	    (T 0)))
  (cond 
    ((eql (xlib::gcontext-line-style gc) :dash)        (format stream "  [2 2] 0 setdash ~%" ))
    ((eql (xlib::gcontext-line-style gc) :double-dash) (format stream "  [1 1] 0 setdash ~%" ))
    (T nil))
  )
  
(defun ps-end (stream)
  (format stream " ~%")
  (format stream "showpage grestore ~%")
  )

(defun ps-point (x y view)
  (let ((stream (printing view)))
    (format stream "% a point ~%")
    (format stream "  n~%")
    (format stream "  ~d ~d ~d ~d ~d ~d ellipse~%" x y 1 1 0 (* 2 pi))
    (make-ps-color-or-pattern view)
    (format stream "  c f~%~%")
    )
  )

(defun ps-points (points view)
  (let ((stream (printing view)))
    (format stream "% some points ~%")
    (make-ps-color-or-pattern view)
    (loop for i from 0 to (- (/ (length points) 2) 1)
	  do
	  (let ((x (nth (* i 6) points))
		(y (nth (+ (* i 6) 1) points)))
	    (format stream "  n~%")
	    (format stream "  ~d ~d ~d ~d ~d ~d ellipse~%~%" x y 0 0 0 (* 2 pi))))
    (format stream "  f~%~%")
    ))

(defun ps-line (x0 y0 x1 y1 view)
  (let ((gc (gcontext view))
	(stream (printing view))
	)
    (format stream "% a line ~%")
    (format stream "  n ~d ~d m ~d ~d l ~%" x0 y0 x1 y1)
    (make-ps-color-or-pattern view)
    (make-ps-line-style gc stream)
    (format stream "  s~%~%")
    ))

(defun ps-lines (points fill view)
  (let ((gc (gcontext view))
	(stream (printing view))
	)
    (format stream "% a polyline~%")
    (format stream "  n~%")
    (format stream "  ~d ~d m~%" (first points) (second points))
    (loop for l from 2 to (/ (length points) 2)
	  do (format stream "  ~d ~d l~%"
		     (nth (- (* 2 l) 2) points) (nth (- (* 2 l) 1) points)))

    (make-ps-color-or-pattern view)
    (if fill
	(format stream "  c f~%")
	(progn
	  (make-ps-line-style gc stream)
	  (format stream "  s~%~%")
	  ))
    ))

(defun ps-segments (points view)
  (let ((gc (gcontext view))
	(stream (printing view))
	)
    (format stream "% some segments~%")
    (make-ps-color-or-pattern view)
    (make-ps-line-style gc stream)
    (loop for l from 0 to (1- (/ (length points) 4))
	  do
	  (format stream "  n~%")
	  (format stream "  ~d ~d m~%" (nth (+ (* 4 l) 0) points) (nth (+ (* 4 l) 1) points))
	  (format stream "  ~d ~d l~%" (nth (+ (* 4 l) 2) points) (nth (+ (* 4 l) 3) points))
	  (format stream "  s~%~%"))
    ))

(defun ps-rectangle(x y width height fill view)
  (let ((gc (gcontext view))
	(stream (printing view))
	)
    (format stream "% a rectangle~%")
    (format stream "  n~%")
    (format stream "  ~d ~d m~%" x y)
    (format stream "  ~d ~d r~%" width 0)
    (format stream "  ~d ~d r~%" 0 height)
    (format stream "  ~d ~d r~%" (- 0 width) 0)
    (make-ps-color-or-pattern view)
    (if fill
	(format stream "  c f~%")
	(progn
	  (make-ps-line-style gc stream)
	  (format stream "  s~%~%")
	  ))
    ))

(defun ps-rectangles (rects fill view)
  (let ((gc (gcontext view))
	(stream (printing view))
	)
    (format stream "% some rectangles~%")
    (make-ps-color-or-pattern view)
    (let (x y  width height)
      (loop for l from 0 to (1- (/ (length rects) 4))
	    do
	    (setf x      (nth (+ (* 4 l) 0) rects)
		  y      (nth (+ (* 4 l) 1) rects)
		  width  (nth (+ (* 4 l) 2) rects)
		  height (nth (+ (* 4 l) 3) rects))
	    (format stream "  n~%")
	    (format stream "  ~d ~d m~%" x y)
	    (format stream "  ~d ~d r~%" width 0)
	    (format stream "  ~d ~d r~%" 0 height)
	    (format stream "  ~d ~d r~%" (- 0 width) 0)
	    (if fill
		(format stream "  c f~%")
		(progn
		  (make-ps-line-style gc stream)
		  (format stream "  s~%~%")
		  ))))
    ))


(defun ps-arc (x y x-rad y-rad start-angle angle fill arc-mode view)
  (let ((gc (gcontext view))
	(stream (printing view))
	(sa (round (* start-angle 180) pi))
	(se (round (* (+ start-angle angle) 180) pi)))
    (format stream "% an arc ~%")
    (format stream "  n~%")
    (format stream "  ~d ~d ~d ~d ~d ~d ellipse~%"
	    (round (+ x x-rad)) (round (+ y y-rad)) (round x-rad) (round y-rad)
	    (if (plusp angle) sa se) (if (plusp angle) se sa))
    (when (and fill (not (eql sa (mod se 360))) (eq arc-mode :PIE-SLICE))
      (format stream "  ~d ~d  l ~%" (round (+ x x-rad)) (round (+ y y-rad))))
    (make-ps-color-or-pattern view)
    (if fill
	(format stream "  c f~%")
	(progn
	  (make-ps-line-style gc stream)
	  (format stream "  s~%~%")
	  ))
    ))
   


(defun ps-arcs (arcs fill arc-mode view)
  (let ((gc (gcontext view))
	(stream (printing view))
	)
    (format stream "% some arcs ~%")
    (make-ps-color-or-pattern view)
    (loop for i from 0 to (- (/ (length arcs) 6) 1)
	  do
	  (let ((x       (nth (* i 6) arcs))
		(y       (nth (+ (* i 6) 1) arcs))
		(x-rad   (/ (nth (+ (* i 6) 2) arcs) 2))
		(y-rad   (/ (nth (+ (* i 6) 3) arcs) 2))
		(start-angle (nth (+ (* i 6) 4) arcs))
		(angle   (nth (+ (* i 6) 5) arcs)))
	       (format stream "  n~%")
	    (format stream "  ~d ~d ~d ~d ~d ~d ellipse~%"
		    (round (+ x x-rad))
		    (round (+ y y-rad))
		    (round x-rad) (round y-rad)
		    ;; exchange start and angle
		    (if (plusp angle)
			(round  (* start-angle 180) pi)
			(round  (* (+ start-angle angle) 180) pi))
		    (if (plusp angle)
			(round  (* (+ start-angle angle) 180) pi)
			(round  (* start-angle 180) pi))
		    )
	    (when (and fill (> (mod angle (* 2 pi)) 0.000001) (eq arc-mode :PIE-SLICE))
	      (format stream "  ~d ~d  l ~%" (round (+ x x-rad)) (round (+ y y-rad))))
	    (if fill
		(format stream "  c f~%~%")
		(progn
		  (make-ps-line-style gc stream)
		  (format stream "  s~%~%")
		  ))))
    ))


(defun format-text (string)
  ;;  runde Klammern werden mit \( geschrieben
  (if (or (position #\( string)(position #\) string))
      (loop
       with new      = nil
       with str      = string
       with pos      = 0
       until (not (setf pos (or (position #\( str)(position #\) str))))
       do
       (setf new (concatenate 'string
			      new
			      (subseq str 0 pos)
			      "\\" ;; was: (string (user::int-char 92))
			      (subseq str pos (1+ pos))
			      ))
       (setf str (subseq str (1+ pos)))
       finally (return (concatenate 'string new str)))
      string))


(defun ps-glyph (x0 y0 ele view)
  (let ((font-size 14)
	(font-name "Helvetica")
	(stream (printing view))
	(font (xlib::font-name (xlib::gcontext-font (gcontext view)))))
    (format stream "% a glyph ~%")
;    (format t "font = ~s   " font)
    (make-ps-color-or-pattern view)
    (let ((font-descr (parse-xlib-font-string font)))
;      (format t " computes to ~a~%" font-descr)
      (setf font-name (find-ps-font-name font-descr))
      (when (integerp (third font-descr))(setf font-size (third font-descr)))
      (format stream "  /~a findfont ~d scalefont setfont~%" font-name font-size)
      (format stream "  ~d ~d m gsave 1 -1 scale~%" x0 y0)
      (format stream "  (~a) show grestore~%~%" ele))
  ))

(defun ps-glyphs (x0 y0 string view)
  (let ((font-size 14)
	(font-name "Helvetica")
	(stream (printing view))
	(font (xlib::font-name (xlib::gcontext-font (gcontext view)))))
    (format stream "% a text ~%")
    (make-ps-color-or-pattern view)
;    (format t "font = ~s computes to " font)
    (let ((font-descr (parse-xlib-font-string font)))
;      (format t "~a ~%" font-descr)
      (setf font-name (find-ps-font-name font-descr))
      (when (third font-descr) (setf font-size (third font-descr))))
    (format stream "  /~a findfont ~d scalefont setfont~%" font-name font-size)
    (format stream "  ~d ~d moveto gsave 1 -1 scale~%" x0 y0)
    (format stream "  (~a) show grestore~%~%" (format-text string))
    ))

(defun ps-bitmap (pixmap view x-pos y-pos)
;;; noch nicht ganz fertig; aber so aehnlich gehts.
  (let ((gc (gcontext view))
	(stream (printing view))
	(window (x-window view))
	(image (xlib::get-image pixmap
				:x (xlib::drawable-x pixmap)
				:y (xlib::drawable-y pixmap)
				:width  (xlib::drawable-width pixmap)
				:height (xlib::drawable-height pixmap)
				:format :z-pixmap)))
    (let (
	  (arr (xlib::image-z-pixarray image))
	  (wi  (xlib::image-width  image))
	  (he  (xlib::image-height image))
	  (fcol (xlib::gcontext-foreground gc))
	  ;;(bcol (xlib::gcontext-background gc))
	  )
;      (print "a bitmap")
      (format stream "% a bitmap ~%")
      (format stream "  ~d ~d ~d srgb ~%"
	      (red-value   fcol window)
	      (green-value fcol window)
	      (blue-value  fcol window))
      (format stream "   ~d ~d ~%" wi he)
      (format stream "   true ~%")
;      (print x-pos)(print y-pos)
      (format stream "   [1 0 0 1 -~d -~d]~%" x-pos y-pos)
      (format stream "{<~%")   ;; hier folgt das Bitmap als Hex-String
      (loop for i from 0 to (1- he) do
	    (let ((z 0))
	      (loop for j from 0 to (1- wi) do
		    (case (mod j 4)
		      ((0) (incf z (* 8 (aref arr i j))))
		      ((1) (incf z (* 4 (aref arr i j))))
		      ((2) (incf z (* 2 (aref arr i j))))
		      ((3) (incf z (aref arr i j))
		       (format stream "~x" z)
		       (setf z 0)))))
	    (format stream "~%")) 
      (format stream ">} imagemask~%~%")
      )))
 
(defun red-value (pixel window)
  (xlib::color-red
   (first (xlib::query-colors (xlib::window-colormap window)
		       (list pixel)))))

(defun green-value (pixel window)
  (xlib::color-green
   (first (xlib::query-colors (xlib::window-colormap window)
		       (list pixel)))))

(defun blue-value (pixel window)
  (xlib::color-blue
   (first (xlib::query-colors (xlib::window-colormap window)
		       (list pixel)))))

(defun parse-xlib-font-string (xlib-font-string)
;; "-adobe-times-bold-i-normal--17-100-..."
  (loop for char
	    across 
	    ;;#-Symbolics being #-Symbolics the #-Symbolics array-elements #-Symbolics of
	    xlib-font-string
	;; When will there be Common Lisp on other Machines than Symbolics?
	with count      = 0
	with foundry    = nil
	with family     = nil
	with weight     = nil
	with slant      = nil
	with pixel-size = nil
	with char-list  = nil
	until   (= count 8)
	finally
	  (return (process-font-descriptor-components foundry family weight slant pixel-size))
	do
    (push char char-list)
    (when (char-equal char #\-) (incf count)
	  (when (= count 2)
	    (setq foundry   (concatenate 'string (butlast (cdr (reverse char-list))))
		  char-list nil))
	  (when (= count 3)
	    (setq family    (concatenate 'string
					 (substitute #\- #\space (butlast (reverse char-list))
						     :test #'char=))
		  char-list nil))
	  (when (= count 4)
	    (setq weight    (concatenate 'string (butlast (reverse char-list)))
		  char-list nil))
	  (when (= count 5)
	    (setq slant     (concatenate 'string (butlast (reverse char-list)))
		  char-list nil))
	  (when (= count 7) (setq char-list nil))
	  (when (= count 8)
	    (setq pixel-size
		  (read-from-string
		    (concatenate 'string (butlast (reverse char-list)))))))))

(defun process-font-descriptor-components
       (foundry family-name weight-name slant pixel-size)
  (let (family face)
    (setq family (concatenate 'string
		     foundry
		     (if (not (or (string-equal "" foundry)
				  (string-equal "" family-name))) "-" "")
		     family-name)
	  face
	(concatenate 'string
		     weight-name
		     (if (not (or (string-equal "" weight-name)
				  (string-equal "" slant))) "-" "")
		     (when (not (string-equal "" slant))
		       (cond ((string-equal "i" slant) "italic")
			     ((string-equal "r" slant) "roman")
			     ((string-equal "o" slant) "oblique")
			     ((string-equal "ri" slant) "reverse-italic")
			     ((string-equal "ro" slant) "reverse-oblique")
			     ((string-equal "ot" slant) "other")
			     (t slant)))))
    (list family
	  (if (string-equal face "") "normal" face) ;; Face may be empty.
	  pixel-size)))

(defun find-ps-font-name (font-descr)
  (let ((family (string (first font-descr)))
	(face   (string (second font-descr))))
    (cond     
      ((< (length family) 6) "Courier")

      ((string-equal family "courier" :start1 6 :end1 12 :end2 6)
       (if  (string-equal face "bold" :start1 0 :end1 3 :end2 3)
	    (if (string-equal face "oblique" :start1 5 :end1 11 :end2 6)
		"Courier-BoldOblique"
		"Courier-Bold")
	    (if (string-equal face "oblique" :start1 7 :end1 13 :end2 6)
		"Courier-Oblique"
		"Courier")))
	  
      ((string-equal family "times" :start1 6 :end1 10 :end2 4)
       (if  (string-equal face "bold" :start1 0 :end1 3 :end2 3)
	    (if (string-equal face "italic" :start1 5 :end1 10 :end2 5)
		"Times-BoldItalic"
		"Times-Bold")
	    (if (string-equal face "italic" :start1 7 :end1 12 :end2 5)
		"Times-Italic"
		"Times-Roman")))
      
      ((string-equal family "helvetica" :start1 6 :end1 14 :end2 8) 
       (if  (string-equal face "bold" :start1 0 :end1 3 :end2 3)
	    (if (string-equal face "oblique" :start1 5 :end1 11 :end2 6)
		"Helvetica-BoldOblique"
		"Helvetica-Bold")
	    (if (string-equal face "oblique" :start1 7 :end1 13 :end2 6)
		"Helvetica-Oblique"
		"Helvetica")))

      ((string-equal family "new-century-schoolbook" :start1 6 :end1 27 :end2 21)
       (if  (string-equal face "bold" :start1 0 :end1 3 :end2 3 :end2 3)
	    (if (string-equal face "italic" :start1 5 :end1 10 :end2 5)
		"NewCenturySchlbk-BoldItalic"
		"NewCenturySchlbk-Bold")
	    (if (string-equal face "italic" :start1 7 :end1 12 :end2 5)
		"NewCenturySchlbk-Italic"
		"NewCenturySchlbk-Roman")))
      
      ((string-equal family "palatino")
       (if  (string-equal face "bold" :start1 0 :end1 3 :end2 3 :end2 3)
	    (if (string-equal face "italic" :start1 5 :end1 10 :end2 5)
		"Palatino-BoldItalic"
		"Palatino-Bold")
	    (if (string-equal face "italic" :start1 7 :end1 12 :end2 5)
		"Palatino-Italic"
		"Palatino-Roman")))
    
      ((string-equal family "symbol" :start1 6 :end1 11 :end2 5) "Symbol")
      ((string-equal family "zapfdingbats") "ZapfDingBats")

      (T "Courier"))
    ))

(defun ps-header (view)
  (let ((stream (printing view)))
    (with-slots (x-pos y-pos width height) view
	(format stream "%!PS-Adobe-2.0 EPSF-2.0~%")
	(format stream "%%Bounding Box: 0 0 ~d ~d ~%" width height)
	(format stream "%%Title: Postscript-Output fuer GINA
%%Creator: Fritz Leugner , W.Deisel, Fa. S.E.P.P.
%%End Comments 
gsave 

   a4
   /m /moveto load def /r /rlineto load def /l /lineto load def 
   /s /stroke load def /n /newpath load def /c /closepath load def
   /srgb /setrgbcolor load def    /f /fill load def
   /a /add load def /mu /mul load def /slw /setlinewidth load def

% CTM berechnen
% clippath verschieben nach Nullpunkt
  /llx clippath pathbbox pop pop pop def
  /lly clippath pathbbox pop pop exch pop def
  /urx clippath pathbbox pop exch pop exch pop def
  /ury clippath pathbbox exch pop exch pop exch pop def
  llx lly translate   
% matrixelemente der ctm berechnen
  /m13 0 0 transform pop def
  /m11 1 0 transform pop m13 sub def
  /m23 0 0 transform exch pop def 
  /m22 0 1 transform exch pop m23 sub def 
% y-wertebereich bestimmen
  /cy0  llx lly transform exch pop def
  /cy1  urx ury transform exch pop def
  /dy  cy1 cy0 sub abs def
% 
  0 dy m22 div neg translate ~%")
	
	(when (eql *postscript-format* :din-a4-quer)
	    (format stream "% dia-a4-quer ~%")
	    (format stream "  0 ury neg translate  90 rotate ~%"))
        (when (numberp *postscript-scale*)
	  (format stream "% scaling with factor ~%")
	  (format stream "  ~f ~f scale ~%" *postscript-scale* *postscript-scale*))
	
	(format stream "  0.86 -0.86 scale

%  Arc-definition x y r a1 a2
/ellipsedict 8 dict def
ellipsedict /mtrx matrix put

/ellipse
  { ellipsedict begin
    /endangle exch def
    /startangle exch def
    /yrad exch def
    /xrad exch def
    /y exch def
    /x exch def

  /savematrix mtrx currentmatrix def
  x y translate
  xrad yrad neg scale
  0 0 1 startangle endangle arc
  savematrix setmatrix
  end
 } def

  2 setlinecap

% ****************** here begins pattern part ********

/sms { % SetMusterScreen
   /winkel exch def   % gewuenscter winkel
   /breite exch def   % gewuenschte breite in Punkt
   /muster exch def   % Musterstring
   /samplebreite muster length sqrt cvi def
   /aufl matrix defaultmatrix 3 get abs 72 mu def
   gsave
      winkel rotate
      samplebreite dup scale
      1 0 dtransform
   grestore
   2 copy	% Entfernungsberechnung
   dup mu	% Ergebnis in gbreite speichern
   exch
   dup mu
   a sqrt
   /gbreite exch def
   exch atan	% Winkel berechnen
   /winkel exch def
   /aus 0 def	% ausgeschaltete Punkte
   /an  0 def	% eingeschaltete Punkte
   /freq aufl breite div def
   freq winkel {SF} setscreen
   aus aus an a div  setgray
   {} settransfer
} bind def

/SF { % hier steht die eigentliche Screenfunktion 
   /Y exch def	% Koordinaten der Halbtonzelle
   /X exch def	% zwischen -1 und 1
   /XX X 1 a 2 div  samplebreite mu cvi def
   /YY Y 1 a 2 div  samplebreite mu cvi def
   /zeiger XX YY samplebreite mu a def
   muster zeiger 1 getinterval (1) eq {
      /an an 1 a def 1
   }{
      /aus aus 1 a def 0.
   } ifelse
} bind def

% ****************** here begins the variable part ********
")))
  )

(defun ps-background (view)
   (let ((color (first (xlib::query-colors (xlib::window-colormap (x-window view))
		      (get-motif-resources view :background))))
	 (stream (printing view)))
    (with-slots (x-pos y-pos width height) view
      (format stream "% the background ~%")
      (format stream "  n~%")
      (format stream "  ~d ~d m~%" 0 0)   ;; x-pos und und y-pos sind manchmal unbound
      (format stream "  ~d ~d r~%" width 0)
      (format stream "  ~d ~d r~%" 0 height)
      (format stream "  ~d ~d r~%" (- 0 width) 0)
      (format stream "  ~f ~f ~f srgb ~%"
	      (xlib::color-red   color)
              (xlib::color-green color)
              (xlib::color-blue  color))
      (format stream "  c f~%~%")
    ))
   )
   
  

