;;; -*- Mode: Lisp; Syntax: ANSI-Common-Lisp; Package: CLIM-USER; Base: 10; Lowercase: Yes -*-

(in-package :clim-user)

;;; The conceptual structure of the input editor

;; The overall control flow of the input editor
(catch 'rescan                  ;thrown to when a rescan is invoked
  (reset-scan-pointer stream)   ;sets STREAM-RESCANNING-P to T
  (loop
     (funcall continuation stream)))

;; The control flow while reading a gesture in READ-GESTURE
;; This loop runs inside of the (catch 'rescan ...) above
(progn
  (rescan-if-necessary stream)
  (loop
    ;; If SP is less than FP
    ;;   Then get the next gesture from the input editor buffer at SP
    ;;   and increment SP
    ;;   Else read the next gesture from the encapsulated stream
    ;;   and insert it into the buffer at IP
    ;; Set the "rescan in progress" flag to false
    ;; Call STREAM-PROCESS-GESTURE on the gesture
    ;;   If it was a "real" gesture
    ;;     Then exit with the gesture as the result
    ;;     Else it was an input editing command (which has already been
    ;;     processed), so continue looping
    ))


(define-presentation-method accept ((type pathname) stream (view textual-view) &key default)
  (when (null default)
    (setq default *default-pathname-defaults*))
  (when (and merge-default default-type)
    (setq default (make-pathname :host (pathname-host default)
				 :device (pathname-device default)
				 :directory (pathname-directory default)
				 :name (pathname-name default)
				 :type default-type
				 :version default-version)))
  ;; This is where we will insert the completed pathname
  (let ((buffer-start (stream-scan-pointer stream)))
    (multiple-value-bind (pathname success string)
	;; Complete the pathname by consulting the file system
	(complete-input stream #'(lambda (string action)
				   (pathname-complete string action default))
			:allow-any-input t :help-displays-possibilities nil)
      (declare (ignore success))
      (handler-bind ((error
		       #'(lambda (error)
			   (simple-parse-error "Error parsing pathname string ~A" string))))
        (unless pathname
	  (setq pathname (parse-namestring string nil default)))
	(when merge-default
	  (setq pathname (merge-pathnames pathname default default-version))))
      ;; If we are rescanning, we don't want to replace anything, otherwise
      ;; the pathname will "grow" by black magic
      (unless (stream-rescanning-p stream)
	;; Replace the textual representation of the completed pathname
	;; into the input buffer
	(presentation-replace-input stream pathname type view
				    :buffer-start buffer-start))
      (values pathname type))))


;;; Compare the next three ways of doing completion

(defun generator-complete (&optional (possibilities '(("One" 1) ("Two" 2) ("Three" 3)))
				     (stream *standard-input*))
  (with-input-editing (stream)			;the completer needs the input editor
    (with-activation-gestures (#\Newline)	;tell the completer when to finish
      (flet ((generator (string suggester)
	       (declare (ignore string))
	       (dolist (possibility possibilities)
		 (funcall suggester (first possibility) (second possibility)))))
	(clim:complete-input 
	  stream 
	  #'(lambda (string action)
	      (clim:complete-from-generator
		string #'generator nil
		:action action)))))))

(defun possibilities-complete (&optional (possibilities '(("One" 1) ("Two" 2) ("Three" 3)))
					 (stream *standard-input*))
  (with-input-editing (stream)			;the completer needs the input editor
    (with-activation-gestures (#\Newline)	;tell the completer when to finish
      (clim:complete-input
	stream
	#'(lambda (string action)
	    (clim:complete-from-possibilities
	      string possibilities nil
	      :action action))))))

(defun suggestion-complete (&optional (possibilities '(("One" 1) ("Two" 2) ("Three" 3)))
				      (stream *standard-input*))
  (with-input-editing (stream)			;the completer needs the input editor
    (with-activation-gestures (#\Newline)	;tell the completer when to finish
      (clim:completing-from-suggestions (stream)
	(map nil
	     #'(lambda(x)
		 (clim:suggest (first x) (second x)))
	     possibilities)))))
