;;; -*- Mode: Lisp; Syntax: ANSI-Common-Lisp; Package: CLIM-USER; Base: 10; Lowercase: Yes -*-

(in-package :clim-user)


;; This pane will serve as a "color swatch"
(defclass color-chooser-pane (clim:clim-stream-pane) ())

;; Repainting this pane entails displaying the current color
(defmethod clim:handle-repaint :after ((stream color-chooser-pane) region)
  (declare (ignore region))
  (display-color (clim:pane-frame stream) stream))

(clim:define-application-frame color-chooser ()
    ((color :accessor color :initform clim:+black+)
     red blue green
     intensity hue saturation)
  (:menu-bar nil)
  (:panes 
    (display (clim:make-clim-stream-pane 
	       :type 'color-chooser-pane
	       :scroll-bars nil
	       :display-function 'display-color
	       ;; Make sure we don't have a useless cursor blinking away...
	       :initial-cursor-visibility nil))
    ;; Pressing the "Exit" button exits from the frame
    (exit clim:push-button 
	  :label "Exit"
	  :activate-callback #'(lambda (button)
				 (clim:frame-exit (clim:pane-frame button))))
    ;; Three sliders for red, green, and blue
    (rgb (with-slots (red green blue) clim:*application-frame*
	   (clim:outlining ()
	     (clim:horizontally ()
	       (setq red (clim:make-pane 'clim:slider
			   :label "Red" :foreground clim:+red+
			   :orientation :vertical
			   :min-value 0.0 :max-value 1.0 
			   :show-value-p t :decimal-places 3
			   :client 'color :id 'red))
	       (setq green (clim:make-pane 'clim:slider
			     :label "Green" :foreground clim:+green+
			     :orientation :vertical
			     :min-value 0.0 :max-value 1.0 
			     :show-value-p t :decimal-places 3
			     :client 'color :id 'green))
	       (setq blue (clim:make-pane 'clim:slider
			    :label "Blue" :foreground clim:+blue+
			    :orientation :vertical
			    :min-value 0.0 :max-value 1.0 
			    :show-value-p t :decimal-places 3
			    :client 'color :id 'blue))))))
    ;; Three sliders for intensity, hue, and saturation
    (ihs (with-slots (intensity hue saturation) clim:*application-frame*
	   (clim:outlining ()
	     (clim:horizontally ()
	       (setq intensity (clim:make-pane 'clim:slider
				 :label "Intensity"
				 :orientation :vertical
				 :min-value 0.0 :max-value (sqrt 3)
				 :show-value-p t :decimal-places 3
				 :client 'color :id 'intensity))
	       (setq hue (clim:make-pane 'clim:slider
			   :label "Hue"
			   :orientation :vertical
			   :min-value 0.0 :max-value 1.0 
			   :show-value-p t :decimal-places 3
			   :client 'color :id 'hue))
	       (setq saturation (clim:make-pane 'clim:slider
				  :label "Saturation"
				  :orientation :vertical
				  :min-value 0.0 :max-value 1.0 
				  :show-value-p t :decimal-places 3
				  :client 'color :id 'saturation)))))))
  (:layouts
    (default 
      (clim:horizontally ()
	(clim:outlining ()
	  (clim:vertically () display exit))
	rgb ihs))))

(defmethod display-color ((frame color-chooser) stream)
  (clim:with-bounding-rectangle* (left top right bottom) 
      (clim:window-viewport stream)
    (clim:with-output-recording-options (stream :record nil)
      ;; Fill the entire viewport with a block of the chosen color
      (clim:draw-rectangle* stream left top right bottom
			    :filled t :ink (color frame)))))

;; The RGB callbacks continuously update the RGB components of the color,
;; and simultaneously cause the IHS components to be updated by changing
;; the IHS sliders.
(defmacro define-rgb-callbacks (color)
  (check-type color (member red green blue))
  (let* ((rgb '(red green blue))
	 (new-rgb (substitute 'value color rgb)))
    `(progn
       (defmethod clim:value-changed-callback
		  ((slider clim:slider) (client (eql 'color)) (id (eql ',color)) value)
	 (let ((frame (clim:pane-frame slider)))
	   (multiple-value-bind (,@rgb) (clim:color-rgb (color frame))
	     (declare (ignore ,color))
	     (setf (color frame) (clim:make-rgb-color ,@new-rgb)))
	   (update-ihs frame)))
       (defmethod clim:drag-callback
		  ((slider clim:slider) (client (eql 'color)) (id (eql ',color)) value)
	 (let ((frame (clim:pane-frame slider)))
	   (multiple-value-bind (,@rgb) (clim:color-rgb (color frame))
	     (declare (ignore ,color))
	     (setf (color frame) (clim:make-rgb-color ,@new-rgb)))
	   (update-ihs frame))))))

(define-rgb-callbacks red)
(define-rgb-callbacks green)
(define-rgb-callbacks blue)

;; Update the IHS sliders themselves.  Doing it this way means that the
;; sets of sliders track each other dynamically.
(defmethod update-ihs ((frame color-chooser))
  (with-slots (intensity hue saturation) frame
    (multiple-value-bind (ii hh ss) (clim:color-ihs (color frame))
      (setf (clim:gadget-value intensity  :invoke-callback nil) ii)
      (setf (clim:gadget-value hue	  :invoke-callback nil) hh)
      (setf (clim:gadget-value saturation :invoke-callback nil) ss))))

;; Like the RGB callbacks...
(defmacro define-ihs-callbacks (color)
  (check-type color (member intensity hue saturation))
  (let* ((ihs '(intensity hue saturation))
	 (new-ihs (substitute 'value color ihs)))
    `(progn
       (defmethod clim:value-changed-callback
		  ((slider clim:slider) (client (eql 'color)) (id (eql ',color)) value)
	 (let ((frame (clim:pane-frame slider)))
	   (multiple-value-bind (,@ihs) (clim:color-ihs (color frame))
	     (declare (ignore ,color))
	     (setf (color frame) (clim:make-ihs-color ,@new-ihs)))
	   (update-rgb frame)))
       (defmethod clim:drag-callback
		  ((slider clim:slider) (client (eql 'color)) (id (eql ',color)) value)
	 (let ((frame (clim:pane-frame slider)))
	   (multiple-value-bind (,@ihs) (clim:color-ihs (color frame))
	     (declare (ignore ,color))
	     (setf (color frame) (clim:make-ihs-color ,@new-ihs)))
	   (update-rgb frame))))))

(define-ihs-callbacks intensity)
(define-ihs-callbacks hue)
(define-ihs-callbacks saturation)

(defmethod update-rgb ((frame color-chooser))
  (with-slots (red green blue) frame
    (multiple-value-bind (rr gg bb) (clim:color-rgb (color frame))
      (setf (gadget-value red   :invoke-callback nil) rr)
      (setf (gadget-value green :invoke-callback nil) gg)
      (setf (gadget-value blue  :invoke-callback nil) bb))))

;; After the slider has had its value changed, we update the state of
;; the frame.  This means that the swatch doesn't change until the user
;; stops sliding the slider.
(defmethod clim:value-changed-callback :after
	   ((slider clim:slider) (client (eql 'color)) id value)
  (declare (ignore id value))
  (clim:redisplay-frame-pane (clim:pane-frame slider) 'display))


(defvar *color-choosers* nil)

(defun do-color-chooser (&key (port (clim:find-port)) (force nil))
  (let* ((framem (clim:find-frame-manager :port port))
	 (frame 
	   (let* ((entry (assoc port *color-choosers*))
		  (frame (cdr entry)))
	     (when (or force (null frame))
	       (setq frame (clim:make-application-frame 'color-chooser
			     :frame-manager framem)))
	     (if entry 
		 (setf (cdr entry) frame)
		 (push (cons port frame) *color-choosers*))
	     frame)))
    (clim:run-frame-top-level frame)
    (color frame)))
