;;; -*- Mode: Lisp; Syntax: ANSI-Common-Lisp; Package: CLIM-USER; Base: 10; Lowercase: Yes -*-

(in-package :clim-user)

;;; Class browser nodes

;; OBJECT is the Lisp object associated with the node (i.e., a class object)
;; INFERIORS is a list of child nodes
;; SUPERIORS is a list of parent nodes
;; TICK is incremented whenever this node should be redisplayed
(defclass class-browser-node ()
    ((object :reader node-object :initarg :object)
     (inferiors :accessor node-inferiors :initform nil)
     (superiors :accessor node-superiors :initform nil)
     (tick :accessor node-tick :initform 0)))

(defun make-class-browser-node (object)
  (make-instance 'class-browser-node :object object))

(defmethod node-object-name ((node class-browser-node))
  (class-name (node-object node)))

;; Note the use of NODE-TICK as the redisplay cache value
(defmethod display-node ((node class-browser-node) stream)
  (clim:updating-output (stream :unique-id node
				:cache-value (node-tick node))
    (let ((class (node-object node)))
      (clim:with-output-as-presentation (stream node 'class-browser-node)
        (clim:with-output-as-presentation (stream class 'class-name)
          (write (node-object-name node) :stream stream))))))

;; When we increment the redisplay tick for this node, we propagate it up
;; the tree so that we get the proper redisplay for the entire tree
(defmethod tick-node ((node class-browser-node))
  (labels ((tick (node)
             (incf (node-tick node))
             (dolist (superior (node-superiors node))
               (tick superior))))
    (declare (dynamic-extent #'tick))
    (tick node)))

(defun make-class-browser-root (object)
  (typecase object
    (clos:class
      (make-class-browser-node object))
    (symbol
      (let ((class (clos:find-class object nil)))
        (when class
          (make-class-browser-node class))))))

(defmethod node-generate-inferior-objects ((node class-browser-node))
  (clos:class-direct-subclasses (node-object node)))

(defmethod node-any-inferior-objects-p ((node class-browser-node))
  (not (null (clos:class-direct-subclasses (node-object node)))))
  
(defun node-eql (n1 n2)
  (eql (node-object n1) (node-object n2)))


;;; The CLASS presentation type

(clim:define-presentation-type class-name ()
  :history t)

(clim:define-presentation-method clim:accept
    ((type class-name) stream (view clim:textual-view) &key default)
  (let* ((class-name (clim:accept 'symbol :stream stream :view view
				  :default (and default (clos:class-name default))
				  :prompt nil))
         (class (clos:find-class class-name nil)))
    (unless class
      (clim:input-not-of-required-type class-name type))
    class))

(clim:define-presentation-method clim:present
    (class (type class-name) stream (view clim:textual-view) &key)
  (prin1 (clos:class-name class) stream))


;;; The class browser itself

(clim:define-application-frame class-browser ()
    ((tree-depth :initform 1)
     (root-nodes :initform nil)
     (all-nodes :initform nil))
  (:command-definer t)
  (:command-table (class-browser :inherit-from (clim:accept-values-pane)))
  (:panes
    (graph :application
           :display-function 'display-graph-pane
           :display-after-commands t
           :incremental-redisplay t
           :scroll-bars :both
           :end-of-page-action :allow
           :end-of-line-action :allow)
    (interactor :interactor :height '(5 :line)))
  (:layouts
   (default
     (clim:vertically ()
       (4/5 graph)
       (1/5 interactor)))))

(defmethod display-graph-pane ((browser class-browser) stream)
  (let ((root-nodes (slot-value browser 'root-nodes)))
    (when root-nodes
      (clim:updating-output (stream :unique-id root-nodes)
        (clim:format-graph-from-roots
	  root-nodes #'display-node #'node-inferiors
	  :graph-type :dag
	  :stream stream
	  :orientation :horizontal
	  :merge-duplicates t)))))

(defmethod generate-class-graph ((browser class-browser) nodes
				 &optional (depth (slot-value browser 'tree-depth)))
  (when nodes
    (let ((generated nil))
      (labels 
        ((collect-inferiors (node parent-node depth)
           (when (and (plusp depth)
                      (not (eql node parent-node)))
             (let ((inferior-objects
                     (node-generate-inferior-objects node)))
               (when inferior-objects
                 (setq generated t)             ;we generated something
                 (dolist (object inferior-objects)
                   (let ((inferior-node
                           (find-node-for-object browser object)))
                     (unless (member node (node-superiors inferior-node))
                       (setf (node-superiors inferior-node)
                             (nconc (node-superiors inferior-node) (list node))))
                     (unless (member inferior-node (node-inferiors node)
                                     :test #'node-eql)
                       (setf (node-inferiors node)
                             (nconc (node-inferiors node) (list inferior-node))))
                     ;; Recursively collect inferiors for these nodes
                     (collect-inferiors inferior-node node (1- depth)))))))))
        (declare (dynamic-extent #'collect-inferiors))
        (dolist (node nodes)
          (collect-inferiors node nil depth)))
      generated)))

;; Find or intern a new node.
(defmethod find-node-for-object ((browser class-browser) object &key (test #'eql))
  (with-slots (all-nodes) browser
    (dolist (node all-nodes)
      (when (funcall test object (node-object node))
        (return-from find-node-for-object node)))
    (let ((node (make-class-browser-node object)))
      (setq all-nodes (nconc all-nodes (list node)))
      node)))

(define-class-browser-command (com-show-graph :name t :menu t)
    ((objects '(sequence class)
              :prompt "some class names"
              :default nil))
  (clim:with-application-frame (frame)
    (with-slots (root-nodes all-nodes) frame
      (setq root-nodes (mapcar #'make-class-browser-root objects))
      ;; ALL-NODES and ROOT-NODES must not be EQ lists...
      (setq all-nodes (copy-list root-nodes))
      (window-clear (clim:get-frame-pane frame 'graph))
      (generate-class-graph frame root-nodes)
      (clim:redisplay-frame-pane frame 'graph :force-p t))))

(clim:define-gesture-name :show-graph :pointer-button (:left :shift))

(clim:define-presentation-to-command-translator show-graph
    (class-browser-node com-show-graph class-browser
     :gesture :show-graph)
    (object)
  (list (list (node-object object))))

(define-class-browser-command com-show-node-inferiors
    ((node 'class-browser-node :prompt "node to show inferiors for"))
  (when (generate-class-graph clim:*application-frame* (list node) 1)
    (tick-node node)))

(clim:define-presentation-to-command-translator show-node-inferiors
   (class-browser-node com-show-node-inferiors class-browser
    :gesture :select
    :tester ((object)
             (node-any-inferior-objects-p object)))
   (object)
  (list object))

(define-class-browser-command com-hide-node-inferiors
    ((node 'class-browser-node :prompt "node to hide inferiors of"))
  (when (node-inferiors node)
    (setf (node-inferiors node) nil)
    (tick-node node)))

(clim:define-presentation-to-command-translator hide-node-inferiors
    (class-browser-node com-hide-node-inferiors class-browser
     :gesture :describe
     :tester ((object)
              (not (null (node-inferiors object)))))
    (object)
  (list object))

(define-class-browser-command com-delete-node
    ((node 'class-browser-node :prompt "node to delete"))
  (when (node-superiors node)
    (dolist (superior (node-superiors node))
      (setf (node-inferiors superior) (delete node (node-inferiors superior))))
    (tick-node node)))

(clim:define-presentation-to-command-translator delete-node
    (class-browser-node com-delete-node class-browser
     :gesture :delete
     :tester ((object) (and (null (node-inferiors object))
                            (not (null (node-superiors object))))))
    (object)
  (list object))

(define-class-browser-command (com-set-depth :name t :menu "Depth")
    ((depth '(integer 1 10) 
	    :prompt "depth" :default (slot-value clim:*application-frame* 'tree-depth)))
  (setf (slot-value clim:*application-frame* 'tree-depth) depth))

(define-class-browser-command (com-redisplay-graph :name t :menu "Redisplay") ()
  (clim:redisplay-frame-pane clim:*application-frame* 'graph :force-p t))

(define-class-browser-command (com-quit-browser :name "Quit" :menu "Quit") ()
  (clim:frame-exit clim:*application-frame*))


(defun do-class-browser (&key (port (clim:find-port)) (force nil))
  (clim:find-application-frame 'class-browser
    :frame-manager (clim:find-frame-manager :port port)
    :own-process nil :create (if force :force t)
    :width 700 :height 600))
