;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: PS; Base:10 -*-

 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ;                                                                         ; ;
; ;                           postscript.lisp                               ; ;
; ;                                                                         ; ;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;  Programmer:  Erik Eilerts   (with help from David Throop)

#|*****************************************************************************
 |
 |  Copyright 1991 University of Texas at Austin
 |
 |  Permission to use, copy, modify, distribute, and sell this software and its
 |  documentation for any purpose is hereby granted without fee, provided that
 |  the above copyright notice appear in all copies and that both that
 |  copyright notice and this permission notice appear in supporting
 |  documentation, and that the name of UT-Austin not be used in advertising
 |  or publicity pertaining to distribution of the software without specific
 |  specific, written prior permission.  UT-Austin makes no representations 
 |  about the suitability of this software for any purpose.  It is provided 
 |  "as is" without express or implied warranty.
 |
 |  UT-AUSTIN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 |  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL 
 |  UT-AUSTIN BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 |  OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, 
 |  WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, 
 |  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 |  SOFTWARE.
 |
 |*****************************************************************************
 |#

(in-package 'ps)

(defmacro with-postscript-sizing (&rest body)
  `(unwind-protect
     (progn
       (setf ps:*use-postscript-sizing* t)
       ,@body)
     (setf ps:*use-postscript-sizing* nil)))



(defmethod get-postscript-parameters ((postscripter postscript-handler) parent)
  (choose-values " Postscript File Configuration Menu"
      `(""
	(output-path ,(namestring (output-path postscripter)) 
		     " Postscript Output Path "
		     :string-or-nil)
	""
	(destination ,(destination postscripter) " Image Destination "
		     :choose (:laser-printing :document-inclusion))
	""
	(orientation ,(orientation postscripter) "Image Orientation "
		     :choose (:portrait :landscape))
	""
	(number-of-pages ,(number-of-pages postscripter)
			 " Number of Image Output Pages " 
			 :choose (:one :multiple))
	(multi-page-scale ,(multi-page-scale postscripter)
			  " Multi-page scaling factor " :number)
	""
	""
	(plot-label ,(plot-label postscripter)
		    " Text label to display (nil = none)" :string-or-nil)
	""
	(display-date ,(display-date postscripter)
		      " Display date on plot " :choose (t nil))
	(label-location ,(label-location postscripter)
			" Location to display label " :choose (:top :bottom))
	(label-scale  ,(label-scale postscripter)
		      " Scaling factor for the label " :number)
	"")
      :associated-window parent))


(defmethod setup-output-configuration ((postscripter postscript-handler) parent)
  "Set varibles for PostScript Output."
  (with-slots (number-of-pages destination output-path orientation
	       display-date plot-label label-location label-scale
	       multi-page-scale) postscripter
    (let ((input-result nil))

     (setf input-result (get-postscript-parameters postscripter parent))

     (when input-result
       (setf number-of-pages (get-choose-value input-result 'number-of-pages))
       (setf destination (get-choose-value input-result 'destination))

       ;; a postscript document for inclusion in other documents 
       ;; cannot have multiple pages.
       (when (and (equal number-of-pages :multiple)
		  (eq destination :document-inclusion))
	 (setf number-of-pages :one))
       
       (setf output-path (get-choose-value input-result 'output-path))
       (setf orientation (get-choose-value input-result 'orientation))
       (setf display-date (get-choose-value input-result 'display-date))
       (setf plot-label (get-choose-value input-result 'plot-label))
       (setf label-location (get-choose-value input-result 'label-location))
       (setf label-scale (get-choose-value input-result 'label-scale))
       (setf multi-page-scale (get-choose-value input-result 'multi-page-scale))
       t
       ))))


;; the fast mode of printing is used only when the bounding box is known 
;; and the user is printing everything to one page.

(defmethod open-stream ((postscripter postscript-handler)
			&optional (pane nil))
  (with-slots (current-window compute-boundingbox number-of-pages
	       postscript-output-file output-path destination
	       temp-output-path current-window) postscripter

    (initialize-postscript-variables postscripter)
    (init-labels postscripter)
    (setf current-window pane)
    (if (and (not compute-boundingbox)
	     (eq number-of-pages :one))
	(progn
	  (setf postscript-output-file
		(open output-path :direction :output 
		      :if-exists :overwrite :if-does-not-exist :create))
	  (write-postscript-header postscripter)
	  (case destination
	    (:laser-printing
	     (setup-onepage-coordinates postscripter))))
	(progn
	  (setf temp-output-path
		(concatenate 'string output-path "1temp"))
	  (setf postscript-output-file
		(open temp-output-path :direction :output
		      :if-exists :overwrite :if-does-not-exist :create))))))

(defun get-current-short-date ()
  (multiple-value-bind (ignore1 ignore2 ignore3 date month yr)  ;;sec min hr
                       (get-decoded-time)
    (declare (ignore ignore1 ignore2 ignore3))
    (format nil "~D/~2,'0D/~2,'0D" month date (mod yr 100))))

(defun get-current-date ()
  (multiple-value-bind (sec min hr date month yr)
		       (get-decoded-time)
    (format nil "~D/~2,'0D/~2,'0D  ~D:~2,'0D:~2,'0D" 
	    month date (mod yr 100) hr min sec)))

(defmethod write-postscript-header ((postscripter postscript-handler))
  (with-slots (postscript-output-file bounding-box) postscripter
    (format postscript-output-file "%! -*- Mode: Text -*-~%~%")
    (format postscript-output-file "%%Title: Postscript Document~%")
    (format postscript-output-file "%%Creator: ~a~%" (get-author))
    (format postscript-output-file "%%CreationDate: ~a~%" (get-current-date))
    (format postscript-output-file "%%For: ~a~%" (get-author))
;  (format postscript-output-file "%%DocumentFonts: atend~%")
    (format postscript-output-file "%%Pages: 1~%")
    (format postscript-output-file "%%BoundingBox: ~,2f ~,2f ~,2f ~,2f~%"
	    (first bounding-box) (second bounding-box)
	    (third bounding-box) (fourth bounding-box))
    (format postscript-output-file "~%")
    (format postscript-output-file "statusdict /waittimeout 30 put~%")
    (format postscript-output-file "/fontarray 30 array def~%")
    (format postscript-output-file "/f {fontarray exch get setfont} def~%")
    (format postscript-output-file "/estfont {findfont exch scalefont fontarray 3 1 roll put} def~%")
    (format postscript-output-file "/m {moveto} def~%")
    (format postscript-output-file "/format-rotation 0 def~%")
    (format postscript-output-file "/format-y-translation 0 def~%")
    (format postscript-output-file "/new-matrix {0 format-y-translation translate~%")
    (format postscript-output-file "	      format-rotation rotate} def~%")
    (format postscript-output-file "/new-page {showpage new-matrix} def~%")
    (format postscript-output-file "%%EndProlog~%~%")

    (format postscript-output-file "new-matrix~%")
    )
)
;;&/inch {72 mul} def~&"))


(defmethod set-bounding-box ((postscripter postscript-handler) l tp r b)
  (with-slots (bounding-box compute-boundingbox) postscripter
    (setf bounding-box (list l (- b) r (- tp)))
    (setf compute-boundingbox nil)))

;; Note:  the bounding box gets screwed up because all y coords are
;;       converted to negative numbers.  So, the top part of the box
;;       is really (- (fourth boundingbox))

(defmethod compute-bounding-box ((postscripter postscript-handler))
  (with-slots (bounding-box) postscripter
    (values (first bounding-box) (- (fourth bounding-box))
	    (third bounding-box) (- (second bounding-box)))))

(defmethod check-bounds ((postscripter postscript-handler) bounding-box)
  (check-x-bounds postscripter (first bounding-box))
  (check-y-bounds postscripter (second bounding-box))
  (check-x-bounds postscripter (third bounding-box))
  (check-y-bounds postscripter (fourth bounding-box)))

(defmethod check-x-bounds ((postscripter postscript-handler) x)
  (with-slots (bounding-box) postscripter
    (let ((low-x (first bounding-box))
          (high-x (third bounding-box)))
      (when (< x low-x)
	(setf (first bounding-box) x))
      (when (> x high-x)
        (setf (third bounding-box) x)))))

(defmethod check-y-bounds ((postscripter postscript-handler) y)
  (with-slots (bounding-box) postscripter
    (let ((low-y (second bounding-box))
          (high-y (fourth bounding-box)))
      (when (< y low-y)
	(setf (second bounding-box) y))
      (when (> y high-y)
        (setf (fourth bounding-box) y)))))

(defmethod arrange-output-file ((postscripter postscript-handler))
  (with-slots (compute-boundingbox number-of-pages postscript-output-file
	       output-path destination temp-output-path) postscripter
    (unless (and (not compute-boundingbox)
		 (equal number-of-pages :one))
      (close postscript-output-file))

    (when (probe-file output-path)
      (delete-file output-path))
     (setf postscript-output-file
	  (open output-path :direction :output :if-exists :overwrite
		            :if-does-not-exist :create))
    (write-postscript-header postscripter)

    (case destination
      (:laser-printing
         (case number-of-pages
	   (:one
	     (setup-onepage-coordinates postscripter)
             (close postscript-output-file)
             (copy-file temp-output-path output-path
			:if-exists :append :characters t)
	     (delete-file temp-output-path))
	   (:multiple
	    (process-multipage-setup postscripter))))
      (:document-inclusion
        (close postscript-output-file)
	(copy-file temp-output-path output-path
		   :if-exists :append :characters t)
	(delete-file temp-output-path)))))

;; Postscript page setup
;; ---------------------
;;
;; 
;;      The apple laserwriter has a self imposed clipping boundary that has been
;;   determined to be:
;;
;;              Top
;;              784
;;     Left 22       593 Right
;;               8
;;             Bottom
;;
;;   for an 8.5 X 11 sheet of paper.  So, any postscript printing must first 
;;   move the image within these boundaries, otherwise it will get clipped.
;;       The postscript coordinates are arranged on a 8.5 x 11 page as follows:
;;
;;       y
;;
;;       ^
;;       |
;;       |
;;
;;       0
;;        0   ---> x  
;; 
;;   with (0,0) being in the lower left hand corner.
;;       Therefore the normal plotting coordinates of a postscript page are:
;;
;;               784
;;             22   593            Normal 8.5 x 11 page
;;                8
;;
;;               -22
;;             8    784            8.5 x 11 page rotated 90 degrees
;;              -593
;;
;;
;;   Setup Onepage Coordinates
;;   -------------------------
;;       This is a shrink to fit postscript method.  Basically, what happens is
;;   that a scaling factor is computed for both the x and y axis of the plot
;;   which can be applied to the axies to make them reduce the image along that
;;   axis enough to fit on one page.  The axis with the smallest scaling factor
;;   is chosen and then used for both axies.  (This prevents distortion, which
;;   occurrs when one axis is scaled differently than the other)  The
;;   coordinates for the center of the image are then scaled and used to
;;   determine a translation vector that will move the center of the image to
;;   the center of the output device page.
;;       Please note that when the image is rotated 90 degrees, the output
;;   device coordinates become negative on the y-axis.

(defmethod setup-onepage-coordinates ((postscripter postscript-handler))
  (let* (
         (output-device-boundaries (output-device-boundaries postscripter))
	 (orientation (orientation postscripter))
         ;; setup coordinates for the output device.
	 (left-x (if (eq orientation :landscape)
		     (fourth output-device-boundaries)
		     (first output-device-boundaries)))
         (top-y (if (eq orientation :landscape)
		    (- (first output-device-boundaries))
		    (second output-device-boundaries)))
         (right-x (if (eq orientation :landscape)
		      (second output-device-boundaries)
		      (third output-device-boundaries)))
         (bottom-y (if (eq orientation :landscape)
		       (- (third output-device-boundaries))
		       (fourth output-device-boundaries)))
         (dev-x-dist (- right-x left-x))
         (dev-y-dist (- top-y bottom-y))
         (dev-x-center (+ left-x (/ dev-x-dist 2)))
         (dev-y-center (+ bottom-y (/ dev-y-dist 2)))

         ;; setup coordinates of the image.
         (llx (first (bounding-box postscripter)))
         (lly (second (bounding-box postscripter)))
         (urx (third (bounding-box postscripter)))
         (ury (fourth (bounding-box postscripter)))
         (plot-x-dist (- urx llx))                   ;; length and width of the image
         (plot-y-dist (- ury lly))
         (plot-x-center (+ llx (/ plot-x-dist 2)))   ;; center of the image
         (plot-y-center (+ lly (/ plot-y-dist 2)))

         (xscale 1.0)                                ;; scaling necessary to fit
         (yscale 1.0)                                ;;  the image on the page

         (xtrans nil)                                ;; translation necessary to 
         (ytrans nil)                                ;;  center the image on the page
        )

    ;; scale the image to fit on the page.  But, make sure the scale
    ;; is the same for both axes of the image.  Otherwise, the image
    ;; will be skewed.  Also, don't enlarge the image by keeping the
    ;; scale from exceeding 1.0
    (setq xscale (min (/ dev-x-dist plot-x-dist)
		      (/ dev-y-dist plot-y-dist)
		      1.0))
    (setq yscale xscale)

    ;; center the image by translating the drawing window to the center of
    ;; the page, dev-x-center.  Then, move the image to the center of the
    ;; page, (- (* xscale plot-x-center))  Note:  the point needs to be
    ;; scale here, since the scaling is done after the point is added.

    (setq xtrans  (- dev-x-center (* xscale plot-x-center)))
    (if (equal orientation :landscape)
        (setq ytrans (- dev-y-center (* yscale plot-y-center)))
        (setq ytrans (- dev-y-center (* yscale plot-y-center))))

    (format (postscript-output-file postscripter) 
      "~2&% 			New page~2&")

    (format (postscript-output-file postscripter) "gsave~%")
    (if (equal orientation :landscape)
        (format (postscript-output-file postscripter) "90.0 rotate~%")
        (format (postscript-output-file postscripter) "0.0 rotate~%"))

    (format (postscript-output-file postscripter)
	    "~,1f ~,1f translate~%" xtrans ytrans)
    (format (postscript-output-file postscripter)
	    "~,3f ~,3f scale~%" xscale yscale)
 
    (format (postscript-output-file postscripter) "newpath~%~%")
   ))
    

;;   Process Multipage Setup
;;   -----------------------
;;       This is a spread the wealth method.  It first determines how many pages
;;   will be necessary for printing the whole image.  It then computes the space
;;   that will be left over on the topmost page and the rightmost page if the
;;   image were to be printed starting at the bottom left corner of the multiple
;;   pages.  This space is divided in half for both the axises, which makes it
;;   the space that will border the image if the image is printed centered on the
;;   multiple pages.  Now, all that is necessary is to translate the image, so 
;;   that when it starts printing on the bottom left page, it will print while
;;   maintaining this border space.
;;       The image is then printed across the multiple pages by treating the
;;   image as a virtual image and moving it so that different sections of it
;;   lie over the output device window.  Then the copypage command is issued,
;;   which causes that part of the image to be printed on the output device.
;;       Unfortunately, it is necessary to send the entire image to the printer
;;   each time a page is printed, since its not possible to reset the printer's
;;   clipping window after it has received input.

(defmethod process-multipage-setup ((postscripter postscript-handler))
  (let* (
         (output-device-boundaries (output-device-boundaries postscripter))
	 (output-device-original-boundaries
	         (output-device-original-boundaries postscripter))
	 (orientation (orientation postscripter))
         ;; setup coordinates for the output device.
	 (left-x (if (equal orientation :landscape)
		     (fourth output-device-boundaries)
		     (first output-device-boundaries)))
         (top-y (if (equal orientation :landscape)
		    (- (first output-device-boundaries))
		    (second output-device-boundaries)))
         (right-x (if (equal orientation :landscape)
		      (second output-device-boundaries)
		      (third output-device-boundaries)))
         (bottom-y (if (equal orientation :landscape)
		        (- (third output-device-boundaries))
		        (fourth output-device-boundaries)))
	 (org-left-x (if (equal orientation :landscape)
		         (fourth output-device-original-boundaries)
		         (first output-device-original-boundaries)))
         (org-top-y (if (equal orientation :landscape)
		        (- (first output-device-original-boundaries))
	 	        (second output-device-original-boundaries)))
         (org-right-x (if (equal orientation :landscape)
		          (second output-device-original-boundaries)
		          (third output-device-original-boundaries)))
         (org-bottom-y (if (equal orientation :landscape)
		       (- (third output-device-original-boundaries))
		       (fourth output-device-original-boundaries)))
         ;; reduce its size by 10% so the image on the next page will
	 ;; overlap the image from the previous page.
         (dev-x-dist (* (- right-x left-x) 0.9))
         (dev-y-dist (* (- top-y bottom-y) 0.9))

         ;; setup coordinates for the image.
         (scale (multi-page-scale postscripter))
         (llx (first (bounding-box postscripter)))
         (lly (second (bounding-box postscripter)))
         (urx (third (bounding-box postscripter)))
         (ury (fourth (bounding-box postscripter)))
         (plot-x-dist (* (- urx llx) scale))
         (plot-y-dist (* (- ury lly) scale))

         ;; determine number of pages necessary to print image.
         (plot-x-pages (ceiling (/ plot-x-dist dev-x-dist)))
         (plot-y-pages (ceiling (/ plot-y-dist dev-y-dist)))
 
         ;; determine the blank space left over after the image is printed
	 ;; across multiple pages.  Then divide this amount in 2 to get the
	 ;; distance the plot needs to be from the initial boundaries of the
	 ;; first page (left and bottom margins) so that the whole image will
	 ;; end up centered on the multiple pages.
         (plot-x-fill (/ (mod (- dev-x-dist (rem plot-x-dist dev-x-dist))
			      dev-x-dist) 2))
         (plot-y-fill (/ (mod (- dev-y-dist (rem plot-y-dist dev-y-dist))
			      dev-y-dist) 2))

         ;; initial translation necessary to center plot on the multiple pages
         (xtrans (+ plot-x-fill (* (- llx) scale)))
         (ytrans nil)

         ;; distances the clipping window needs to move to display other parts
	 ;; of the image.
         (xtrans-inc dev-x-dist)
         (ytrans-inc dev-y-dist)

         ;; initial translation that starts the printing at the bounding edge
	 ;; of the output device.
         (xtransfix left-x)
         (ytransfix nil)

         (line-count 0)                ;lines read in so far
         (graphic-blocks 1)            ;number of blocks of images
         (postscript-input-file nil)   ;file of drawing commands
         (postscript-output-file nil)
        )

    (if (equal orientation :landscape)        ;; compute y translation
        (progn
          (setq ytrans (+ (* (- lly) scale) plot-y-fill))
          (setq ytransfix bottom-y))
        (progn
          (setq ytrans (+ (* (- lly) scale) plot-y-fill))
          (setq ytransfix bottom-y)))

    (close (postscript-output-file postscripter))
    (setq  postscript-input-file
	  (open (temp-output-path postscripter) :direction :input))
    (setf (postscript-output-file postscripter)
	  (open (output-path postscripter) :direction :output 
		:if-exists :append :if-does-not-exist :create))

    (setq postscript-output-file (postscript-output-file postscripter))

    (format postscript-output-file "~%~%/graphic-block~a  {~%" graphic-blocks)

    (do ((input-block nil))
        ((eq 'eof (setq input-block (read-line postscript-input-file nil 'eof))))
      (incf line-count)
      (format postscript-output-file "   ~a~%" input-block)
      (when (> line-count 75)
        (setq line-count 1)
        (format postscript-output-file "  } def~%~%")
        (incf graphic-blocks)
        (format postscript-output-file "/graphic-block~a  {~%" graphic-blocks)
        ))

    (format postscript-output-file "  } def~%~%~%")

    (close postscript-input-file)
    (delete-file (temp-output-path postscripter))

    (format postscript-output-file "%% routine to print each page of the image.~%")
    (format postscript-output-file "    /rows ~a def                    ~%" plot-y-pages)
    (format postscript-output-file "    /columns ~a def                 ~%" plot-x-pages)
    (format postscript-output-file "                                    ~%")
    (if (equal orientation :landscape)
        (format postscript-output-file "    90.0 rotate                     ~%")
        (format postscript-output-file "    0.0 rotate                      ~%"))
    (format postscript-output-file "                                    ~%")
    (format postscript-output-file "    newpath                         ~%")
    (format postscript-output-file "      ~,2f ~,2f m                  ~%" org-left-x  org-bottom-y)
    (format postscript-output-file "      ~,2f ~,2f lineto                  ~%" org-left-x  org-top-y)
    (format postscript-output-file "      ~,2f ~,2f lineto                  ~%" org-right-x org-top-y)
    (format postscript-output-file "      ~,2f ~,2f lineto                  ~%" org-right-x org-bottom-y)
    (format postscript-output-file "                                    ~%")
    (format postscript-output-file "    closepath                       ~%")
    (format postscript-output-file "    clip                            ~%")
    (format postscript-output-file "    newpath                         ~%")
    (format postscript-output-file "                                    ~%")
    (format postscript-output-file "    0 1 rows 1 sub                  ~%")
    (format postscript-output-file "      {  /rowcount exch def         ~%")
    (format postscript-output-file "        0 1 columns 1 sub           ~%")
    (format postscript-output-file "          { /colcount exch def      ~%")
    (format postscript-output-file "            gsave                   ~%")
    (draw-labels postscripter t)
    (format postscript-output-file "              newpath               ~%")
    (format postscript-output-file "              ~,2f ~,2f m      ~%" left-x  bottom-y)
    (format postscript-output-file "              ~,2f ~,2f lineto      ~%" left-x  top-y)
    (format postscript-output-file "              ~,2f ~,2f lineto      ~%" right-x top-y)
    (format postscript-output-file "              ~,2f ~,2f lineto      ~%" right-x bottom-y)
    (format postscript-output-file "                                    ~%")
    (format postscript-output-file "              closepath             ~%")
    (format postscript-output-file "              clip                  ~%")
    (format postscript-output-file "              newpath               ~%")
    (format postscript-output-file "                                    ~%")
    (format postscript-output-file "              ~,3f ~,3f translate   ~%" xtransfix ytransfix)
    (format postscript-output-file "              ~,2f ~,2f colcount mul sub ~%" xtrans xtrans-inc)
    (format postscript-output-file "                ~,2f ~,2f rowcount mul sub ~%" ytrans ytrans-inc)
    (format postscript-output-file "                 translate          ~%")
    (format postscript-output-file "              ~,3f ~,3f scale       ~%" scale scale)
    (format postscript-output-file "                                    ~%")
    (do ((i 1 (1+ i)))
        ((> i graphic-blocks))
      (format postscript-output-file "              graphic-block~a       ~%" i))
    (format postscript-output-file "                                    ~%")
    (format postscript-output-file "              copypage              ~%")
    (format postscript-output-file "              erasepage             ~%")
    (format postscript-output-file "            grestore                ~%")
    (format postscript-output-file "          } for                     ~%")
    (format postscript-output-file "      } for                         ~%")
    (format postscript-output-file "                                    ~%")
    (format postscript-output-file "                                    ~%")
    (format postscript-output-file "    newpath                         ~%~%")
    (close postscript-output-file)
   ))
    


;; This function has not really been implemented yet.
(defmethod next-page ((postscripter postscript-handler))
  (with-slots (postscript-output-file) postscripter
    (case (destination postscripter) 
      (:document-inclusion    ;No action need be taken for TEXT-FORMAT images
       nil)  
      (:laser-printing
       (format postscript-output-file "~&next-page~&")))))

(defmethod close-stream ((postscripter postscript-handler))
  (when (clip-region-active postscripter)
    (setf (clip-region-active postscripter) nil)
    (format (postscript-output-file postscripter) "  grestore~%"))

  (case (destination postscripter)
    (:document-inclusion
       (if (equal (number-of-pages postscripter) :one)
           (draw-labels postscripter)))
    (:laser-printing
      (when (equal (number-of-pages postscripter) :one)
        (format (postscript-output-file postscripter) "grestore~%")
        (draw-labels postscripter)
        (format (postscript-output-file postscripter) "~&showpage~&"))))

  (close (postscript-output-file postscripter))

  (if (compute-boundingbox postscripter)
      (arrange-output-file postscripter)
      (setf (compute-boundingbox postscripter) t))
  t)

(defmethod init-labels ((postscripter postscript-handler))
  (when (or (display-date postscripter) (plot-label postscripter))
    (let* ((top-y (if (equal (orientation postscripter) :landscape)
		      (- (first (output-device-boundaries postscripter)))
		      (second (output-device-boundaries postscripter))))
           (bottom-y (if (equal (orientation postscripter) :landscape)
		         (- (third (output-device-boundaries postscripter)))
		         (fourth (output-device-boundaries postscripter))))
           (offset (ceiling (+ 10 (* 12 (label-scale postscripter))))))

       (case (label-location postscripter)
         (:bottom
           (setq bottom-y (+ bottom-y offset))
           (if (equal (orientation postscripter) :landscape)
               (setf (third (output-device-boundaries postscripter)) (- bottom-y))
               (setf (fourth (output-device-boundaries postscripter)) bottom-y)))
         (:top
           (setq top-y (- top-y offset))
           (if (equal (orientation postscripter) :landscape)
               (setf (first (output-device-boundaries postscripter)) (- top-y))
               (setf (second (output-device-boundaries postscripter)) top-y)))))))

(defmethod draw-labels ((postscripter postscript-handler) &optional (multi-page nil))
  (when (or (display-date postscripter) (stringp (plot-label postscripter)))
    (let* ((top-y (if (equal (orientation postscripter) :landscape)
		      (- (first (output-device-boundaries postscripter)))
		      (second (output-device-boundaries postscripter))))
           (bottom-y (if (equal (orientation postscripter) :landscape)
		         (- (third (output-device-boundaries postscripter)))
		         (fourth (output-device-boundaries postscripter))))
  	   (left-x (if (equal (orientation postscripter) :landscape)
		       (fourth (output-device-boundaries postscripter))
		       (first (output-device-boundaries postscripter))))
           (indent (if multi-page "              " "  "))
           (xdate-string (when (display-date postscripter) (get-current-short-date)))
           (xstart (+ left-x 20))
           (ystart nil)
           (scale (label-scale postscripter))
           (inverse-scale (/ 1 scale))
           (display-string nil))

    (cond ((and (stringp (plot-label postscripter)) (display-date postscripter))
           (setq display-string
		 (concatenate 'string xdate-string "      " (plot-label postscripter))))
          ((stringp (plot-label postscripter))
           (setq display-string (plot-label postscripter)))
          ((display-date postscripter)
           (setq display-string xdate-string)))

    (case (label-location postscripter)
      (:top     (setq ystart (+ top-y 10)))
      (:bottom  (setq ystart (- bottom-y (ceiling (+ 10 (* 12 scale)))))))

    (setq xstart (/ xstart scale))
    (setq ystart (/ ystart scale))

    (unless multi-page
      (format (postscript-output-file postscripter) "~agsave~%" indent))
    (when (equal (number-of-pages postscripter) :one)
      (if (equal (orientation postscripter) :landscape)
          (format (postscript-output-file postscripter) "~a90.0 rotate~%" indent)
          (format (postscript-output-file postscripter) "~a0.0 rotate~%" indent)))
    (format (postscript-output-file postscripter) "~a~,3f ~,3f scale~%" indent scale scale) 
    (format (postscript-output-file postscripter) "~a/Courier-Bold findfont 12 scalefont setfont~%" indent)
    (format (postscript-output-file postscripter) "~a~,2f ~,2f m (~a) show~%" indent xstart ystart display-string)
    (format (postscript-output-file postscripter) "~a~,3f ~,3f scale~%" indent inverse-scale inverse-scale)
    (unless multi-page
      (format (postscript-output-file postscripter) "~agrestore~%" indent))
    )))

;;; helpful references: sample of Symbol typeface p 251 Red PS manual.
;;; The correspondence between the Symbol typeface and the (octal) ASCII
;;; collating sequence is on p256 of the blue PS manual.
;;; ASCII character set p206 blue PS manual

;; Given a character code, print that character at that point on the
;; page, in the given pointsize.

(defmethod special-char ((postscripter postscript-handler) x y ascii
			 &key (size 12)(ink *black*))
  (let* ((font :|Symbol|)
	 (pair (list font size))
	 (familiar-font (assoc pair (fonts-in-use postscripter) :test #'equal))
	 (gatom (if familiar-font
		    (second familiar-font)
		    (gentemp (format nil "~:@(~a-~)" font)))))
    (enforce-font postscripter font familiar-font size gatom pair))
  (check-grayscale postscripter ink)
  (format (postscript-output-file postscripter)
	  "~&~,1f ~,1f m (\\~o) show~&" x (- y) ascii))



(defmethod text-style-ascent ((postscripter postscript-handler) font)
  (let ((ps-font (decode-font postscripter font))
	(scale-factor (get-ps-scale-factor font)))
    (if ps-font
	(* (font-ascent ps-font) scale-factor)
        40)))

(defmethod text-style-descent ((postscripter postscript-handler) font)
  (let ((ps-font (decode-font postscripter font))
	(scale-factor (get-ps-scale-factor font)))
    (if ps-font
	(* (font-descent ps-font) scale-factor)
        20)))

(defmethod text-style-height ((postscripter postscript-handler) font)
  (let ((ps-font (decode-font postscripter font))
	(scale-factor (get-ps-scale-factor font)))
    (if ps-font
	(* (font-height ps-font) scale-factor)
        60)))


(defmethod text-size ((postscripter postscript-handler) text 
		      &key text-style (start 0) (end nil))
  (let ((ps-font (decode-font postscripter text-style))
	(scale-factor (get-ps-scale-factor text-style)))
    (if (null ps-font)
	(values (* (length text) 6) 60)
        (let* ((array (font-array ps-font))
	       (total-width 0)
	       (char-width nil)
	       (space-width (aref array (char-code #\space))))
	  (unless end
	    (setf end (length text)))
	  (do ((i start (1+ i)))
	      ((>= i end))
	    (setf char-width (or (aref array (char-code (char text i)))
				 space-width))
	    (incf total-width (* char-width scale-factor)))
	  (values total-width (* (font-height ps-font) scale-factor))))))


(defun get-ps-scale-factor (font)
  (/ (case (clim:text-style-size font)
       (:tiny              4)
       (:very-small        6)
       (:small             8)
       (:normal           10)
       (:large            12)
       (:very-large       16)
       (:huge             20)
       (otherwise         10))
     1000  ;; scales afm numbers into printer coordinates
     ))


;;; Everytime a new font is introduced into a postscript output file,
;;; the font must be loaded and scaled and named.  Keep track of what
;;; fonts have already been used in *used-fonts*.   *Used-fonts* is an
;;; alist of the form (((fontface fontsize) name)((fontface fontsize)
;;; name)...).

;;; The fonts and linewidths stay the same in PostScript until they are
;;; changed.   Keep track of their current values in *linewidth* and
;;; *current-font*.

;;; Given a string, determine to what PostScript typeface it
;;; corresponds.  Do this by looking up the family and face in a large,
;;; nested CASE statement.  Then determine the size in another CASE
;;; statement.  If this is a new font, we will have to execute
;;; PostScript FINDFONT and SCALEFONT commands, (which is expensive, so
;;; we only want to do it once.)  Keep track of what fonts we have
;;; already seen as an Alist on *used-fonts*.  Associate each new font
;;; with a GENTEMP atom in the Alist.     If the font has been seen
;;; before (if we find it when we look it up in *used-fonts*), check to
;;; see if it is the *current-font*.  If so, do nothing.  If not, tell
;;; PostScript to SETFONT to this font.

(defun decode-afm-font-name (font)
  (let ((style (clim:text-style-face font)))
    (case (clim:text-style-family font)
      ((:fix :courier)
       (cond ((eq style :bold)          :|Courier-Bold|)
	     ((eq style :italic)        :|Courier-Oblique|)
	     ((eq style :oblique)       :|Courier-Oblique|)
	     ((eq style :bold-italic)   :|Courier-BoldOblique|)
	     ((eq style :bold-oblique)  :|Courier-BoldOblique|)
	     ((equal style '(:bold :italic)) :|Courier-BoldOblique|)
	     (t                         :|Courier|)))
      ((:sans-serif :helvetica)
       (cond ((eq style :bold)          :|Helvetica-Bold|)
	     ((eq style :italic)        :|Helvetica-Oblique|)
	     ((eq style :oblique)       :|Helvetica-Oblique|)
	     ((eq style :bold-italic)   :|Helvetica-BoldOblique|)
	     ((eq style :bold-oblique)  :|Helvetica-BoldOblique|)
	     ((equal style '(:bold :italic))  :|Helvetica-BoldOblique|)
	     (t                         :|Helvetica|)))
      ((:serif :times-roman)
       (cond ((eq style :bold)          :|Times-Bold|)
	     ((eq style :italic)        :|Times-Italic|)
	     ((eq style :oblique)       :|Times-Italic|)
	     ((eq style :bold-italic)   :|Times-BoldItalic|)
	     ((eq style :bold-oblique)  :|Times-BoldItalic|)
	     ((equal style '(:bold :italic)) :|Times-BoldItalic|)
	     (t                         :|Times-Roman|)))
      (otherwise  :|Courier|))))
  

(defmethod decode-font ((postscripter postscript-handler) 
                       &optional (font *default-postscript-font*))
  (with-slots (font-list) postscripter
    (let* ((fname (decode-afm-font-name font))
	   (font-obj (find fname font-list
			   :test #'(lambda (fn fobj)
				     (eq fn (font-name fobj))))))

      (unless font-obj
	(format *terminal-io* "~%ERROR:  the font ~a is unknown." fname))

      font-obj)))


(defun dump-fonts ()
  (let ((font-filenames
	 (list "Courier" "Courier-Bold" "Courier-Oblique" "Courier-BoldOblique"
	       "Helvetica" "Helvetica-Bold" "Helvetica-Oblique"
	       "Helvetica-BoldOblique"
	       "Times-Roman" "Times-Bold" "Times-Italic" "Times-BoldItalic"
	       "Symbol")))
    (setf (font-list *postscripter*) nil)
    (dolist (font font-filenames)
      (load-ps-font *postscripter* font
		    (concatenate 'string *postscript-afm-directory*
				 font ".afm")))

    (with-open-file (outstream "font-info.ps" :direction :output
			       :if-does-not-exist :create)
	(dolist (font-rec (reverse (font-list *postscripter*)))
	  (let ((font-height (font-height font-rec))
		(font-ascent (font-ascent font-rec))
		(font-descent (font-descent font-rec))
		(font-name (font-name font-rec))
		(font-array (font-array font-rec))
		(index 9))
	    (format outstream "~%(:~s ~a ~a ~a" (intern font-name)
		    font-height font-ascent font-descent)
	    (dotimes (i 9)
	      (format outstream " ~2d" (aref font-array i)))
	    (dotimes (j 13)
	      (format outstream "~%")
	      (dotimes (i 19)
		(format outstream " ~3d" (aref font-array index))
		(incf index)))
	    (format outstream ")")
	    )))))

(defun parse-bbox (fontbbox)
  (let (pos asc desc ignore)
    (when (null fontbbox) (return-from parse-bbox (values nil nil)))
    (when (string= fontbbox "") (return-from parse-bbox (values nil nil)))
    (multiple-value-setq (ignore pos) (read-from-string fontbbox))
    (setf fontbbox (subseq fontbbox pos))
    (when (string= fontbbox "") (return-from parse-bbox (values nil nil)))
    (multiple-value-setq (desc pos) (read-from-string fontbbox))
    (setf fontbbox (subseq fontbbox pos))
    (when (string= fontbbox "") (return-from parse-bbox (values nil nil)))
    (multiple-value-setq (ignore pos) (read-from-string fontbbox))
    (setf fontbbox (subseq fontbbox pos))
    (when (string= fontbbox "") (return-from parse-bbox (values nil nil)))
    (multiple-value-setq (asc pos) (read-from-string fontbbox))
    (values asc desc)))


    
 
(defmethod load-ps-font ((postscripter postscript-handler) 
			 font-name font-file)
  (with-slots (font-list) postscripter
    (unless (probe-file font-file)
      (format *terminal-io* "~%ERROR:  unable to file file ~a" font-file)
      (return-from load-ps-font nil))

    (with-open-file (instream font-file :direction :input)
      (let ((abort nil)
	    (array (make-array 256 :initial-element nil))
	    (char-found nil)
	    (ascender nil)
	    (descender nil)
	    (fontbbox nil))
	(format *terminal-io* "~%;;;Loading font file ~a" font-file)
        (do ((line (read-line instream nil 'eof)
		   (read-line instream nil 'eof)))
	    ((or (and (eq line 'eof) (setf abort t))
		 (and (>= (length line) 16)
		      (equal (subseq line 0 16) "StartCharMetrics"))))
	  (cond ((and (> (length line) 8)
		      (equal (subseq line 0 8) "Ascender"))
		 (setf ascender
		       (get-input-width 
			(find-substring-after-string "Ascender" line))))
		((and (> (length line) 8)
		      (equal (subseq line 0 8) "FontBBox"))
		 (setf fontbbox
		       (find-substring-after-string "FontBBox" line)))
		((and (> (length line) 9)
		      (equal (subseq line 0 9) "Descender"))
		 (setf descender
		       (get-input-width 
			(find-substring-after-string "Descender" line))))))

	(setf ascender nil)  ;; force it to use the FontBBox for now.
	
	(when (or abort (null ascender) (null descender))
	  (multiple-value-bind (asc desc) (parse-bbox fontbbox)
	    (if (or (null asc) (null desc))
		(progn
		  (format *terminal-io* "~%ERROR: unable to process file ~a"
			  font-file)
		  (return-from load-ps-font nil))
		(progn
		  (setf ascender asc)
		  (setf descender desc)))))

        (do ((line (read-line instream nil 'eof)
		   (read-line instream nil 'eof)))
	    ((or (eq line 'eof)
		 (and (>= (length line) 14)
		      (equal (subseq line 0 14) "EndCharMetrics"))))
	  (when (and (> (length line) 0) (char= (char line 0) #\C))
	    (let* ((wx1 (find-substring-after-string "WX" line))
		   (wx (and wx1 (find-substring-before-string ";" wx1)))
		   (c1 (find-substring-after-string "C" line))
		   (C (and c1 (find-substring-before-string ";" c1)))
		   (width (get-input-width wx))
		   (char (get-input-char c)))
	      (when (and char width (>= char 32) (<= char 255))
		(setf (aref array char) width)
		(setf char-found t)))))
	(unless char-found
	  (format *terminal-io* "~%ERROR: no chars processed in ~a" font-file)
	  (return-from load-ps-font nil))

        (let ((space-size (aref array (char-code #\space))))
          (unless space-size
            (setf space-size 0)
            (dotimes (x 255)
              (when (> (aref array x) space-size)
                (setf space-size (aref array x))))
	    (setf (aref array (char-code #\space)) space-size)))
	(let ((font-obj (make-instance 'postscript-font
				       :font-height (- ascender descender)
				       :font-ascent ascender
				       :font-descent (abs descender)
				       :font-name font-name
				       :font-array array)))
	  (push font-obj font-list)
	  font-obj)))))

(defun get-input-width (x)
  (unless (blank-string? x)
    (read-from-string x)))

(defun get-input-char (char-id)
  (unless (blank-string? char-id)
    (read-from-string char-id)))
 
(defmethod set-font ((postscripter postscript-handler) 
	             &optional (font *default-postscript-font*))
  (check-font postscripter font))

(defun decode-font-name (font)
  (let ((style (clim:text-style-face font)))
    (case (clim:text-style-family font)
      ((:fix :courier)
       (cond ((eq style :bold)          :|Courier-Bold|)
	     ((eq style :italic)        :|Courier-Italic|)
	     ((eq style :oblique)       :|Courier-Oblique|)
	     ((eq style :bold-italic)   :|Courier-BoldItalic|)
	     ((eq style :bold-oblique)  :|Courier-BoldOblique|)
	     ((equal style '(:bold :italic)) :|Courier-BoldItalic|)
	     (t                         :|Courier|)))
      ((:sans-serif :helvetica)
       (cond ((eq style :bold)          :|Helvetica-Bold|)
	     ((eq style :italic)        :|Helvetica-Italic|)
	     ((eq style :oblique)       :|Helvetica-Oblique|)
	     ((eq style :bold-italic)   :|Helvetica-BoldItalic|)
	     ((eq style :bold-oblique)  :|Helvetica-BoldOblique|)
	     ((equal style '(:bold :italic)) 
	      :|Helvetica-BoldItalic|)
	     (t                         :|Helvetica|)))
      ((:serif :times-roman)
       (cond ((eq style :bold)          :|Times-Bold|)
	     ((eq style :italic)        :|Times-Italic|)
	     ((eq style :oblique)       :|Times-Oblique|)
	     ((eq style :bold-italic)   :|Times-BoldItalic|)
	     ((eq style :bold-oblique)  :|Times-BoldOblique|)
	     ((equal style '(:bold :italic))  :|Times-BoldItalic|)
	     (t                         :|Times-Roman|)))
      (otherwise  :|Courier|))))
  

(defmethod check-font ((postscripter postscript-handler) 
                       &optional (font *default-postscript-font*))
  (let* ((points 
              (case (clim:text-style-size font)
 		(:tiny              4)
		(:very-small        6)
		(:small             8)
		(:normal           10)
		(:large            12)
		(:very-large       16)
		(:huge             20)
		(otherwise         10)))
	 (font-name (decode-font-name font))
	 (pair (list font-name points))        ;pair is a list of (font size)
	 (familiar-font (assoc pair (fonts-in-use postscripter) :test #'equal))
	 (gatom (if familiar-font
		    (second familiar-font)
		    (progn
		      (font-index postscripter)
;;		      (incf (font-index postscripter))
		      (gentemp (format nil "~:@(~a-~)" (car pair)))
		      ))))
    (enforce-font postscripter (car pair) familiar-font 
		  (cadr pair) gatom pair)))

(defmethod symbol-font ((postscripter postscript-handler) 
                       &optional (size 12))
  (let* ((font :|Symbol|)
	 (pair (list font size))
	 (familiar-font (assoc pair (fonts-in-use postscripter) :test #'equal))
	 (gatom (if familiar-font
		    (second familiar-font)
		    (progn
		      (font-index postscripter)
		      (gentemp (format nil "~:@(~a-~)" (car pair)))
		      ))))
    (enforce-font postscripter (car pair) familiar-font 
		  (cadr pair) gatom pair)))



(defmethod enforce-font ((postscripter postscript-handler)
			 font familiar-font size gatom pair)
  (cond ((not familiar-font)
	 (format (postscript-output-file postscripter)
		 "~& /~s ~&   /~a findfont ~s scalefont def~&~s setfont~&"
		 gatom font size gatom)
;	 (format (postscript-output-file postscripter)
;		 "~& ~d ~d ~s estfont~%" gatom size font)
;	 (format (postscript-output-file postscripter)
;		 "~& ~d f~%" gatom)
	 (push (list pair gatom) (fonts-in-use postscripter))
	 (setf (current-font postscripter) pair))

	((not (equal pair (current-font postscripter)))
;	 (format (postscript-output-file postscripter)
;		 "~& ~d f~%" gatom)
	 (format (postscript-output-file postscripter) "~&~s setfont~&" gatom)
	 (setf (current-font postscripter) pair))

	(t nil)))  ;; font is already active



;;; In PostScript, the linewidth is a global parameter.  Every time we draw 
;;; a line, we check to make sure the linewidth we want is the same as the
;;; last line we drew.  If not, we must change it.

(defmethod check-linewidth ((postscripter postscript-handler) thickness)
  (with-slots (linewidth postscript-output-file) postscripter
    (unless (= linewidth thickness)
      (format postscript-output-file "~&~,3f setlinewidth~&" thickness)
    (setf linewidth thickness))))

;;; Anytime we draw anything, we must check to see if the current gray level
;;; is unchanged from last the last stroke.  If it has changed, we must
;;; "setgray" the new level.  
;;; In postscript, "setgray 0" is black and "setgray 1" is white.

(defmethod decode-clim-color ((postscripter postscript-handler) color)
  (with-slots (current-window) postscripter
    (let ((return-color nil))
      (when (eq color clim:+background+)
	(if current-window
	    (setf color (clim:medium-background current-window))
	    ;; the user didn't bother to give us a window, so we just guess
	    (progn
	      (setf color nil)
	      (setf return-color *white*))))
      (when (eq color clim:+foreground+)
	(if current-window
	    (setf color (clim:medium-foreground current-window))
	    ;; the user didn't bother to give us a window, so we just guess
	    (progn
	      (setf color nil)
	      (setf return-color *black*))))
      
      (when (and color (typep color 'clim:color))
	(case (type-of color)
	  (clim-utils:gray-color
	    (setf return-color (slot-value color 'clim-utils::luminance)))
	  ;; NTSC formula:   luminosity = .299 red + .587 green + .114 blue
	  (clim-utils:rgb-color
	    (multiple-value-bind (red green blue) (clim:color-rgb color)
	      (setf return-color
		    (+ (* red .299) (* green .587) (* blue .114)))))
	  (clim-utils::ihs-color
	    (multiple-value-bind (intens hue sat) (clim:color-ihs color)
	      (declare (ignore hue sat))
	      (setf return-color intens)))))

      return-color)))

(defmethod check-grayscale ((postscripter postscript-handler) 
			    &optional (color *black*))
  (with-slots (grayval postscript-output-file current-window
		       current-color) postscripter
   (unless (eq color current-color)
    (setf current-color color)
    (unless color
      (setq color *black*))

    (unless (numberp color)
      ;; then it must be a CLIM color.  Let's try to decode it.
      (unless (setf color (decode-clim-color postscripter color))
	;; It wasn't even a CLIM color.  So, just guess.
	(setf color *black*)))

    (unless (= grayval color)
      (setf grayval color)
      (format postscript-output-file "~%~d setgray~%" color)))))

;; the dash array is:  [inked gap ...] offset setdash

(defmethod check-line-dashes ((postscripter postscript-handler) 
			      &optional (line-dashes nil) &aux dash-pattern)
  (with-slots (current-line-dashes postscript-output-file
				   current-window) postscripter
    (unless (eq current-line-dashes line-dashes)
      (setf current-line-dashes line-dashes)
      (cond ((null line-dashes)
	     (setf dash-pattern "[] 0"))
	    ((typep line-dashes 'array)
	     (let ((len (length line-dashes)))
	       (setf dash-pattern "[")
	       (dotimes (i len)
		 (setf dash-pattern
		       (concatenate 'string dash-pattern
				    (format nil "~a " (aref line-dashes i)))))
	       (setf dash-pattern
		     (concatenate 'string dash-pattern "] 0"))))
	    (t
             ;; this is the default dash pattern
	     (setf dash-pattern "[4 4] 2")))
      (format postscript-output-file "~&~a setdash~&" dash-pattern))))


;;; Special handling to get strings output with the right escapes for 
;;; postscript.  Fortunately, postscript uses the same escape character,
;;; the backslash, as does Common-Lisp.  However, a different set of
;;; characters need to be slashified.  Some things that Lisp slashifies
;;; are turned to alphanumeric, (by giving them the same characteristics
;;; as #\a) and some that it treats as other things need to be slashified
;;; (by giving them the same characteristics as #\").

;;; The alphanumeric chars are:  / " |
;;  The slashified chars are:    \ ( ) %

;; Given a string, insert the proper slashes in front of the left and
;; right parens.  Otherwise, postscript will misinterpret the parens.

(defmethod ps-string ((postscripter postscript-handler) string)
 (let ((print-string "")
       (current-char nil))
   ;; make sure string's not a symbol
   (unless (stringp string)
     (setf string (format nil "~s" string)))
   (do ((i (1- (length string)) (1- i)))
       ((< i 0))
    (setf current-char (char string i))
    (if (or (char= current-char #\()
	    (char= current-char #\))
	    (char= current-char #\\))
	(setf print-string
	      (concatenate 'string "\\" (string current-char) print-string))
	(setf print-string
	      (concatenate 'string (string current-char) print-string))))
  print-string))




(defmethod start-region ((postscripter postscript-handler))
  (with-slots (postscript-output-file) postscripter
    (format postscript-output-file "   gsave~%")))

(defmethod end-region ((postscripter postscript-handler))
  (with-slots (postscript-output-file linewidth 
	       current-color grayval current-font) postscripter
    (format postscript-output-file "   grestore~%")
    (setf linewidth -1)
    (setf grayval -1)
    (setf current-color nil)
    (setf current-font nil)))

(defmethod translate ((postscripter postscript-handler) x y)
  (with-slots (postscript-output-file) postscripter
    (format postscript-output-file "   ~,1f ~,1f translate~%" x (- y))))

(defmethod scale ((postscripter postscript-handler) x y)
  (with-slots (postscript-output-file) postscripter
    (format postscript-output-file "   ~,3f ~,3f scale~%" x y)))

(defmethod rotate ((postscripter postscript-handler) deg)
  (with-slots (postscript-output-file) postscripter
    (format postscript-output-file "   ~,1f rotate~%" deg)))


;; Note: only one clipping box can be active at a time, otherwise they
;; cancel themselves out.  For that reason, clipping boxes have their
;; own regions.

(defmethod set-clipping-box ((postscripter postscript-handler)
			     left-x top-y right-x bottom-y)
  (with-slots (postscript-output-file clip-region-active) postscripter
    
    (if clip-region-active
	;; force an automatic restore here, otherwise the clip regions
	;; will cancel themselves out.
	(progn
	  (format postscript-output-file "  grestore~%")
	  (format postscript-output-file "  gsave~%"))
	(progn
	  (setf clip-region-active t)
	  (format postscript-output-file "  gsave~%")))

    (format postscript-output-file "  newpath~%")
    (format postscript-output-file "  ~,2f ~,2f m~%" 
	    left-x  (- bottom-y))
    (format postscript-output-file "  ~,2f ~,2f lineto~%" 
	    left-x  (- top-y))
    (format postscript-output-file "  ~,2f ~,2f lineto~%"
	    right-x (- top-y))
    (format postscript-output-file "  ~,2f ~,2f lineto~%"
	    right-x (- bottom-y))
    (format postscript-output-file "~%")
    (format postscript-output-file "  closepath~%")
    (format postscript-output-file "  clip~%")
    (format postscript-output-file "  newpath~%")))

(defmethod set-super-clipping-box ((postscripter postscript-handler) 
				   list-of-points)
  (with-slots (postscript-output-file clip-region-active) postscripter
    (if clip-region-active
	;; force an automatic restore here, otherwise the clip regions
	;; will cancel themselves out.
	(progn
	  (format postscript-output-file "  grestore~%")
	  (format postscript-output-file "  gsave~%"))
	(progn
	  (setf clip-region-active t)
	  (format postscript-output-file "  gsave~%")))

    (let ((first-point (car list-of-points)))
      (format postscript-output-file "  newpath~%")
      (format postscript-output-file "  ~,2f ~,2f m~%"
	      (first first-point) (- (second first-point)))
      (dolist (point (cdr list-of-points))
	(format postscript-output-file "  ~,2f ~,2f lineto~%" 
		(first point) (- (second point))))
      (format postscript-output-file "~%")
      (format postscript-output-file "  closepath~%")
      (format postscript-output-file "  clip~%")
      (format postscript-output-file "  newpath~%"))))


(defmethod end-clipping-box ((postscripter postscript-handler))
  (with-slots (postscript-output-file linewidth grayval 
	       current-color current-font clip-region-active) postscripter
    (format postscript-output-file "  grestore~%")
    (setf linewidth -1)
    (setf grayval -1)
    (setf current-color nil)
    (setf current-font nil)
    (setf clip-region-active nil)))

(defmethod comment ((postscripter postscript-handler) comment)
  (with-slots (postscript-output-file) postscripter
    (format postscript-output-file "~%%% ~a~%" comment)))


;;;  BASIC DRAWING FUNCTIONS - all of these functions take the same arguments
;;;  as does the corresponding CLIM methods.

(defmethod draw-text ((postscripter postscript-handler) string x y
			&key (text-style *default-postscript-font*)
			     (line-thickness 1) (ink *black*)
			     (bounding-box nil)
		             (align-x :left) (align-y :baseline))

  (with-slots (postscript-output-file compute-boundingbox) postscripter
    (unless (and (eq align-x :left) (eq align-y :baseline))
      (let ((xoff 0)
	    (yoff 0))
	(case align-x
	  (:left )
	  (:center (setf xoff (- (/ (text-size postscripter string
					       :text-style text-style) 2))))
	  (:right  (setf xoff (- (text-size postscripter string
					    :text-style text-style)))))
	(case align-y
	  (:baseline )
	  (:top (setf yoff (text-style-ascent postscripter text-style)))
	  (:center (setf yoff
			 (/ (- (text-style-ascent postscripter text-style)
			       (text-style-descent postscripter text-style))
			    2)))
	  (:bottom (setf yoff
			 (- (text-style-descent postscripter text-style)))))
	(incf x xoff)
	(incf y yoff)))

    (check-linewidth postscripter line-thickness)
    (check-grayscale postscripter ink)
    (check-font postscripter text-style)
    (when compute-boundingbox
      (if bounding-box
	  (check-bounds postscripter bounding-box)
	  (let* ((string-len (text-size postscripter string
					:text-style text-style))
		 (string-ascent (text-style-ascent postscripter text-style))
		 (string-descent (text-style-descent postscripter text-style))
		 (x-end (+ x string-len))
		 (y-top (- y string-ascent))
		 (y-bottom (+ y string-descent)))
	    (check-x-bounds postscripter x)
	    (check-x-bounds postscripter x-end)
	    (check-y-bounds postscripter (- y-bottom))
	    (check-y-bounds postscripter (- y-top)))))
    (format postscript-output-file
	    "~&~,1f ~,1f m (~a) show~&" x (- y)
	    (ps-string postscripter string)      ;Slashify #\\, #\( and #\)
	    )))


(defmethod draw-line ((postscripter postscript-handler) x1 y1 x2 y2
		      &key (line-thickness 1) (ink *black*) 
		           (bounding-box nil) (line-dashes nil))
  (with-slots (postscript-output-file compute-boundingbox) postscripter
    (when compute-boundingbox
      (if bounding-box
	  (check-bounds postscripter bounding-box)
	  (progn
	    (check-x-bounds postscripter x1)
	    (check-x-bounds postscripter x2)
	    (check-y-bounds postscripter (- y1))
	    (check-y-bounds postscripter (- y2)))))
    (check-linewidth postscripter line-thickness)
    (check-grayscale postscripter ink)
    (check-line-dashes postscripter line-dashes)
    (format postscript-output-file
	    "~&~,1f ~,1f m~&~,1f ~,1f lineto stroke~&"
	    x1 (- y1) x2 (- y2))))



(defmethod draw-vector ((postscripter postscript-handler) x1 y1 x2 y2
			&key (arrow-head-length 10) (arrow-base-width 5)	
			     (ink *black*) (filled t) (line-thickness 1))
  (multiple-value-bind (xy-points xbas ybas)
      (triangle-point-translation
		  x1 y1 x2 y2 arrow-base-width arrow-head-length)
    (draw-line postscripter x1 y1 xbas ybas 
	       :line-thickness line-thickness :ink ink)
    (draw-polygon postscripter xy-points
		  :closed t :filled filled :ink ink
		  :line-thickness 1)))


(defun triangle-point-translation (from-x from-y to-x to-y
				   arrow-base-width arrow-head-length
				   &aux (halfbase (/ arrow-base-width 2.0)))
  (let* ((dy (- to-y from-y))
	 (dx (float (- to-x from-x)))
	 (alpha (atan dy dx))
	 (beta (atan halfbase arrow-head-length))
	 (len (sqrt (float (+ (* arrow-head-length arrow-head-length)
			      (* halfbase halfbase)))))
	 (xt (- to-x (* len (cos (- alpha beta)))))
	 (yt (- to-y (* len (sin (- alpha beta)))))	
	 (xb (- to-x (* len (cos (+ alpha beta)))))
	 (yb (- to-y (* len (sin (+ alpha beta)))))
	 (xbas (- to-x (* len (cos alpha))))
	 (ybas (- to-y (* len (sin alpha)))))
    (values (list to-x to-y xt yt xb yb)
	    xbas ybas)))

(defmethod draw-circle ((postscripter postscript-handler) x y rad
			&key (line-thickness 1) (ink *black*)
			     (bounding-box nil) (filled nil)
			     (line-dashes nil)
			     (start-angle 0) (end-angle 360))
  (with-slots (postscript-output-file compute-boundingbox) postscripter
    (when compute-boundingbox
      (if bounding-box
	  (check-bounds postscripter bounding-box)
	  (progn
	    (check-x-bounds postscripter (- x rad))
	    (check-x-bounds postscripter (+ x rad))
	    (check-y-bounds postscripter (- (- y) rad))
	    (check-y-bounds postscripter (+ (- y) rad)))))
    (check-linewidth postscripter line-thickness)
    (check-grayscale postscripter ink)
    (check-line-dashes postscripter line-dashes)
    
    (if filled
	;draw a filled in circle connecting line from current point.
	(format postscript-output-file    
		"~&newpath ~,2f ~,2f ~,2f ~a ~a arc fill~&"	
		x (- y) rad start-angle end-angle)
	(format postscript-output-file
		"~&newpath ~,2f ~,2f ~,2f ~a ~a arc stroke~&"
		x (- y) rad start-angle end-angle))))

(defun point-dist (x1 y1 x2 y2)
  "This calculates the distance between two points (x1,y1) and (x2,y2)."
  
  (let ((dx (- x2 x1))
	(dy (- y2 y1)))
    (sqrt (+ (* dx dx) (* dy dy)))))


;; This is probably handled by draw-circle
(defmethod draw-arc ((postscripter postscript-handler) xc yc xsp ysp deg
		     &key (line-thickness 1) (ink *black*) (filled nil)
		          (bounding-box nil) (line-dashes nil))
 (with-slots (postscript-output-file compute-boundingbox) postscripter
  (unless (or (null ink) (zerop deg))

    (setq deg (* (signum deg)  
		 (let ((arc-angle (rem (abs deg) 360)))
		    (if (zerop arc-angle)
			360
			arc-angle))))

    (let* ((radians-per-degree 0.0174532925199433)
	   (dx                 (- xsp xc))
	   (dy                 (- (- ysp yc)))
	   (radius             (point-dist xc yc xsp ysp))
	   (start-angle (/ (atan dy dx) radians-per-degree))
           (end-angle nil))
      (if (minusp start-angle)
          (setq start-angle (+ start-angle 360)))
      (if (minusp deg)
          (progn
            (setq end-angle start-angle)
            (setq start-angle (+ end-angle deg))
            (if (minusp start-angle)
                (setq start-angle (+ start-angle 360))))
          (progn
            (setq end-angle (rem (+ start-angle deg) 360))))

      (when compute-boundingbox
        (if bounding-box
	    (check-bounds postscripter bounding-box)
	    (progn
	      (check-x-bounds postscripter (- xc radius))
	      (check-x-bounds postscripter (+ xc radius))
	      (check-y-bounds postscripter (- (- yc) radius))
	      (check-y-bounds postscripter (+ (- yc) radius)))))
      (check-linewidth postscripter line-thickness)
      (check-grayscale postscripter ink)
      (check-line-dashes postscripter line-dashes)

      (if filled
          ;draw a filled in circle connecting line from current point.
          (format postscript-output-file
	          "~&newpath ~,2f ~,2f ~,2f ~,4f ~,4f arc fill~&"
	          xc (- yc) radius start-angle end-angle)
          (format (postscript-output-file postscripter)
	          "~&newpath ~,2f ~,2f ~,2f ~,4f ~,4f arc stroke~&"
	          xc (- yc) radius start-angle end-angle))))))


(defmethod draw-symbol ((postscripter postscript-handler) x y char-id
		       &key (ink *black*) (line-thickness 1))
  (with-slots (postscript-output-file) postscripter
    (check-linewidth postscripter line-thickness)
    (check-grayscale postscripter ink)
    (check-x-bounds postscripter x)
    (check-y-bounds postscripter (- y))
    (symbol-font postscripter)
    (format postscript-output-file
	    "~&~,1f ~,1f m (\\~o) show~&" x (- y) char-id)))
    
(defmethod draw-point ((postscripter postscript-handler) x y 
		       &key (ink *black*) (line-thickness 1))
  ;; CLIM's way, but it's not too snazy
  ;; (format postscript-output-file
  ;;	     "~&~,1f ~,1f m  0 0 rlineto  stroke~&" x (- y))
    
  ;; David Throop's Way
  ;; Draw a point in the symbol font centered around (x y).
  ;; It's supposed to be faster than drawing a small circle.  (56 octal = 46)
  ;; (draw-symbol postscripter (- x 1.5) (+ y 1) #8r56 :ink ink
  ;;	       :line-thickness line-thickness)

  ;; This way is as good as any
  (draw-circle postscripter x y line-thickness :filled t :ink ink))

(defmethod draw-rectangle ((postscripter postscript-handler)
			   left top right bottom
			   &key (line-thickness 1)(ink *black*)(filled t)
			        (bounding-box nil) (line-dashes nil))
  (with-slots (postscript-output-file compute-boundingbox) postscripter
    (when compute-boundingbox
      (if bounding-box
	  (check-bounds postscripter bounding-box)
	  (progn
	    (check-x-bounds postscripter left)
	    (check-x-bounds postscripter right)
	    (check-y-bounds postscripter (- top))
	    (check-y-bounds postscripter (- bottom)))))
    (check-linewidth postscripter line-thickness)
    (check-grayscale postscripter ink)
    (check-line-dashes postscripter line-dashes)
    
    (if (and (= left right) (= top bottom))
	(draw-point postscripter left right)
	(progn
	  (format postscript-output-file
		  "~&~,1f ~,1f m       % drawing a box~&~
                   ~,1f ~,1f lineto~&~
                   ~,1f ~,1f lineto~&~
                   ~,1f ~,1f lineto~&  closepath"
		  left (- top) left (- bottom)
		  right (- bottom) right (- top))
	  (if filled
	      (format postscript-output-file "  fill~%")
	      (format postscript-output-file "  stroke~%"))))))


(defmethod draw-segment ((postscripter postscript-handler) segment
			 &key (line-thickness 1)(ink *black*)(filled nil)
			      (bounding-box nil))
 (with-slots (postscript-output-file compute-boundingbox) postscripter
  (let* ((first-point (car segment))
         (fx (car first-point))
         (fy (- (cadr first-point))))
    (check-linewidth postscripter line-thickness)
    (check-grayscale postscripter ink)
    (when compute-boundingbox
      (if bounding-box
	  (check-bounds postscripter bounding-box)
          (dolist (point segment)
            (check-x-bounds postscripter (first point))
            (check-y-bounds postscripter (- (second point))))))

    (format postscript-output-file "  ~,1f ~,1f m~%" fx fy)
    (dolist (point (cdr segment))
      (let ((x (car point))
            (y (- (cadr point))))
        (format postscript-output-file "  ~,1f ~,1f lineto~%" x y)))

    (when (equal first-point (car (last segment)))
      (format postscript-output-file "  closepath~%"))
    (if filled
	(format postscript-output-file "  fill~%")
	(format postscript-output-file "  stroke~%")))))


(defmethod draw-lines ((postscripter postscript-handler) coord-seq
		       &key (line-thickness 1) (ink *black*)
		            (bounding-box nil) (line-dashes nil))
  (with-slots (compute-boundingbox postscript-output-file) postscripter
    (check-linewidth postscripter line-thickness)
    (check-grayscale postscripter ink)
    (check-line-dashes postscripter line-dashes)
      
    (when compute-boundingbox
      (if bounding-box
	  (check-bounds postscripter bounding-box)
	  (dolist (pair coord-seq)
	    (let ((p1 (car pair))
		  (p2 (cadr pair)))
	      (check-x-bounds postscripter (car p1))
	      (check-x-bounds postscripter (car p2))
	      (check-y-bounds postscripter (- (cdr p1)))
	      (check-y-bounds postscripter (- (cdr p2)))))))

    (dolist (pair coord-seq)
      (let ((p1 (car pair))
	    (p2 (cadr pair)))
	(format postscript-output-file
	    "~&~,1f ~,1f m~&~,1f ~,1f lineto stroke~&"
	    (car p1) (- (cdr p1)) (car p2) (- (cdr p2)))))))



;;; PS-DRAW-POLYGON draws a polygon, POINTS is a list of its vertices.
;;; By default, the interior of the polygon is filled - this is the
;;; extension over PS-DRAW-LINES. 
 
(defmethod draw-polygon ((postscripter postscript-handler) xy-points
			 &key (closed nil) (filled nil) (ink *black*)
			      (line-thickness 1) (bounding-box nil)
			      (line-dashes nil))
  (with-slots (compute-boundingbox postscript-output-file) postscripter

    (when (oddp (length xy-points))
      (format *terminal-io* "~% ERROR:  draw-polygon called with an ~
                             odd number of points.  The points are:~%  ~a"
	      xy-points)
      (return-from draw-polygon nil))
	      
    (check-linewidth postscripter line-thickness)
    (check-grayscale postscripter ink)
    (check-line-dashes postscripter line-dashes)

    (when compute-boundingbox
      (if bounding-box
	  (check-bounds postscripter bounding-box)
	  (do ((points-list xy-points (cddr points-list)))
	      ((null points-list))
	    (check-x-bounds postscripter (car points-list))
	    (check-y-bounds postscripter (- (cadr points-list))))))

     (format postscript-output-file
	    "~&newpath ~,1f ~,1f m~&" 
	    (car xy-points) (- (cadr xy-points)))
    
    (do ((points-list (cddr xy-points) (cddr points-list)))
	((null points-list))
      (format postscript-output-file
	      "~&~,1f ~,1f lineto~&" (car points-list) (- (cadr points-list))))

    (when closed
      (format postscript-output-file "closepath "))

    (if filled
	(format postscript-output-file "fill~&")
        (format postscript-output-file "stroke~&"))))

(defmethod draw-ring ((postscripter postscript-handler) x y minor-r major-r 
		      &key (ink *black*) (line-thickness 1)
		           (bounding-box nil) (line-dashes nil))
  (with-slots (compute-boundingbox postscript-output-file) postscripter
    (let ((radius (/ (+ major-r minor-r) 2.0)))
      (check-grayscale postscripter ink)
      (check-linewidth postscripter line-thickness)
      (check-line-dashes postscripter line-dashes)
      
      (when compute-boundingbox
         (if bounding-box
	     (check-bounds postscripter bounding-box)
	     (progn
	       (check-x-bounds postscripter (+ x radius))
	       (check-x-bounds postscripter (- x radius))
	       (check-y-bounds postscripter (+ y radius))
	       (check-y-bounds postscripter (- y radius)))))
      ;The NEWPATH has to be output to stop connecting line from current point.
      (format postscript-output-file
	    " newpath ~,1f ~,1f ~,1f 0 360 arc stroke~&"	
	     x (- y) radius))))


(defun find-num-bits (num-designs)
  (let ((size-limits '((2 . 1) (4 . 2) (16 . 4) (256 . 8)))
	(num-bits nil))
    (dolist (sizes size-limits)
      (when (<= num-designs (car sizes))
	(setf num-bits (cdr sizes))
	(return)))
    (or num-bits 16)))

(defmethod draw-icon ((postscripter postscript-handler) icon x y)
  (with-slots (postscript-output-file) postscripter
    (unless (typep icon 'CLIM-UTILS:PATTERN)
      (format *terminal-io* "~% ERROR:  in draw-icon, ~a is not an icon." icon)
      (return-from draw-icon nil))
    
    (let* ((array (slot-value icon 'clim::array))
	   (height (array-dimension array 0))
	   (width (array-dimension array 1))
	   (designs (slot-value icon 'clim-utils::designs))
	   (num-designs (array-total-size designs))
	   (grayscale-array (make-array num-designs :element-type 'integer))
	   (bits/sample (find-num-bits num-designs))
	   (grayscale-map (1- (expt 2 bits/sample)))
	   hex-string
	   (finished nil)
	   output-string)

      (when (> bits/sample 8)
	(format *terminal-io*
		"~% ERROR:  in draw-icon, too many colors are being used (~a)."
		num-designs)
	(return-from draw-icon nil))
    
      (check-x-bounds postscripter x)
      (check-x-bounds postscripter (+ x width))
      (check-y-bounds postscripter (- y))
      (check-y-bounds postscripter (- (+ y height)))
      
      ;; convert the design colors into a grayscale color.  This is a cheap
      ;; way of doing it since it maps the colors into a small number of
      ;; grayscale colors.  So, it basically means that you should only use
      ;; grayscale colors when making icons.
      (dotimes (i num-designs)
	(setf (aref grayscale-array i)
	      (round (* (decode-clim-color postscripter (aref designs i))
			grayscale-map))))
      
      (setf hex-string (convert-array-to-hex array grayscale-array bits/sample))
      
      (format postscript-output-file "  gsave~%")
      (format postscript-output-file "  ~,1f ~,1f translate~%"
	      x (- (+ y height)))
      (format postscript-output-file "  ~d ~d scale~%" width height)
      (format postscript-output-file "  ~d ~d ~d~%" width height bits/sample)
      (format postscript-output-file "  [~d 0 0 -~d 0 ~d]~%"
	      width height height)
      (format postscript-output-file "  {<~%")
      (do ()
	  (finished)
	(setf output-string "")
	(if (= bits/sample 8)  ;; for 8 bits/samp, hex-string has 8-bit nums
	    (dotimes (i 37)  ;; try to make lines a little less than 80 chars
	      (if hex-string
		  ;; this is probably a bad way to do this
		  (setf output-string
			(concatenate 'string output-string
				     (format nil "~2,'0X" (pop hex-string))))
		  (progn
		    (setf finished t)
		    (return))))
	    (dotimes (i 75)     ;; try to make lines a little less than 80 chars
	      (if hex-string
		  ;; this is probably a bad way to do this
		  (setf output-string
			(concatenate 'string output-string
				     (format nil "~X" (pop hex-string))))
		  (progn
		    (setf finished t)
		    (return)))))
	(unless (string= output-string "")
	  (format postscript-output-file "   ~a~%" output-string)))
      (format postscript-output-file "   >}~%")
      (format postscript-output-file "  image~%")
      (format postscript-output-file "  grestore~%"))))

      
    

    
(defun convert-array-to-hex (array garray bits/sample)    
  (let* ((sample/word (/ 8 bits/sample)))
    (case sample/word
      (1 (convert-8bit-array-to-hex array garray))
      (2 (convert-4bit-array-to-hex array garray))
      (4 (convert-2bit-array-to-hex array garray))
      (8 (convert-1bit-array-to-hex array garray)))))

(defun convert-8bit-array-to-hex (array garray)
  (let* ((height (array-dimension array 0))
	 (width (array-dimension array 1))
	 (finished nil)
	 (col 0) (row 0)
	 (hex-list nil))

    ;; each word is 8-bits, so we don't need to do any combining, just
    ;; read off the words
    (do ()
	(finished)
      (push (aref garray (aref array row col)) hex-list)
      (incf col)
      (when (>= col width)
	(setf col 0)
	(incf row)
	(when (>= row height)
	  (setf finished t))))

    (nreverse hex-list)))

(defun convert-4bit-array-to-hex (array garray)
  (let* ((height (array-dimension array 0))
	 (width (array-dimension array 1))
	 (finished nil)
	 (add-extra-nibble (oddp width))
	 (col 0) (row 0)
	 (hex-list nil))

    ;; each word is 4-bits, so we don't need to do any combining, just
    ;; read off the words
    (do ()
	(finished)
      (push (aref garray (aref array row col)) hex-list)
      (incf col)
      (when (>= col width)
	(setf col 0)
	(incf row)
	(when (>= row height)
	  (setf finished t))
	(when add-extra-nibble
	  (push 0 hex-list))))

    (nreverse hex-list)))


(defun convert-2bit-array-to-hex (array garray)
  (let* ((height (array-dimension array 0))
	 (width (array-dimension array 1))
	 (finished nil)
	 (col 0) (row 0)
	 position val
	 (extra-bits (mod width 4))
	 (add-extra-nibble (or (= extra-bits 1) (= extra-bits 2)))
	 (hex-list nil)
	 (bit-array
	    (make-array 16 :initial-contents
	      '(0 16 32 48 64 80 96 112 128 144 160 176 192 208 224 240))))

    ;; combine the parts to make an 4-bit nibble.
    (do ()
	(finished)
      (setf position :high)
      (dotimes (k 2)
	(if (eq position :high)
	    (progn
	      (setf val (aref bit-array (aref garray (aref array row col))))
	      (setf position :low))
	    (incf val (aref garray (aref array row col))))
	(incf col)
	(when (>= col width)
	  (setf col 0)
	  (incf row)
	  (when (>= row height)
	    (setf finished t))
	  (when add-extra-nibble
	    (push val hex-list)
	    (setf val 0))
	  (return)))
      (push val hex-list))
    
    (nreverse hex-list)))


(defun convert-1bit-array-to-hex (array garray)
  (let* ((height (array-dimension array 0))
	 (width (array-dimension array 1))
	 (finished nil)
	 (col 0) (row 0)
	 val
	 (extra-bits (mod width 8))
	 (add-extra-nibble (and (>= extra-bits 1) (<= extra-bits 4) t))
	 (hex-list nil)
	 (bit-array (make-array 4 :initial-contents '(8 4 2 1))))

    ;; combine the parts to make a 4-bit nibble.
    (do ()
	(finished)
      (setf val 0)
      (dotimes (k 4)
	(unless (zerop (aref garray (aref array row col)))
	  (incf val (aref bit-array k)))
	(incf col)
	(when (>= col width)
	  (setf col 0)
	  (incf row)
	  (when (>= row height)
	    (setf finished t))
	  (when add-extra-nibble
	    ;; we have to do this because postscript reads in data 8-bits at a
	    ;; time.  If the end of the line has 4 or less extra bits, then
	    ;; another nibble needs to be added.  Otherwise, postscript will
	    ;; take that nibble from the following line.
	    (push val hex-list)
	    (setf val 0))       ;; this is a dummy value.
	  (return)))
      (push val hex-list))
    
    (nreverse hex-list)))

  
    
;;--------------------------------------------------------------------------
;;                            Auxillary Functions
;;--------------------------------------------------------------------------

(defun copy-file (from-file to-file &key (if-exists :append) (characters t))
  (declare (ignore characters))
  (if (not (probe-file to-file))
      (rename-file from-file to-file)
      (let ((to-stream (open to-file :if-exists if-exists
			             :if-does-not-exist :create
				     :direction :output))
	    (from-stream (open from-file :if-does-not-exist :error
			                 :direction :input))
	    (line nil))
	(do ()
	    ((eq line :eof))
	  (setf line (read-line from-stream nil :eof))
	  (unless (eq line :eof)
	    (write-line line to-stream)))
	(close to-stream)
	(close from-stream)
	t
	)))


;-----------------------------------------------------------------------
;
;  CHOOSE-VALUES
;
;     Displays a list of prompts and default values for the prompts.  The
;   user must either choose the default values or enter in new values.
;     The arguments are:
;
;       label        - a label for the menu
;       option-list  - a list of prompts, default values, and id's
;
;     option-list is a list of:
;       (id-name default-value prompt [read-function-id])
;     where [] indicates optional.
;
;     Example:
; 
;       (setq *foo*
;             (choose-values "Enter an address and value"
;                            `((address ,(novalue) "  Address" :objects-string)
;                              (value   ,(novalue) "  Value"   :objects-string)))
;
;       The user is prompted and enters:
;         Address:  (plant parts)
;         Value:  stem
;
;       ->  ((address (plant parts))
;            (value stem))
;
;       (get-choose-value *foo* 'address)
;       -> (plant parts)
;
;-----------------------------------------------------------------------

(defun choose-values (label option-list &key (associated-window nil))
   (let ((return-values nil)
	 (stream associated-window)
	 (index 0)
	 (menu-id (gensym "MENU"))
	 (menu-window nil))

    (unless associated-window
     (format *terminal-io*
	     "~% ERROR:  no associated window was specified in choose-values.")
     (return-from choose-values nil))

    (dolist (option (remove-if #'stringp option-list))
      (setf (get menu-id index) (cadr option))
      (incf index))

    (multiple-value-bind (x y) (clim:stream-pointer-position* stream)
      (with-simple-restart (abort nil)
        (clim:accepting-values
	 (stream :own-window '(:right-margin 200 :bottom-margin 30)
		 :label label
		 :initially-select-query-identifier 'start
		 :resynchronize-every-pass nil
		 :x-position (max 0 (- x 50)) :y-position (max 0 (- y 50)))
	 (setf menu-window stream)
	 (setf index 0)
	 (clim:terpri stream)
	 (clim:terpri stream)
	 (dolist (option-segment option-list)
	   (if (stringp option-segment)
		(clim:write-string option-segment stream)
		(progn
		  (setf (get menu-id index)
		    (case (fourth option-segment)
		     (:choose
			(clim:accept (cons 'member (fifth option-segment))
				     :stream stream
				     :prompt (third option-segment)
				     :default (get menu-id index)))
		     (:set 
			(clim:accept (cons 'clim:subset (fifth option-segment))
				     :stream stream
				     :prompt (third option-segment)
				     :default (get menu-id index)))
		     (:number
			(clim:accept 'number
				     :stream stream
				     :prompt (third option-segment)
				     :default (get menu-id index)))
		     (:number-or-nil
			(clim:accept '(clim:null-or-type number)
				     :stream stream
				     :prompt (third option-segment)
				     :default (get menu-id index)))
		     (:string-or-nil
			(clim:accept '(clim:null-or-type string)
				     :stream stream
				     :prompt (third option-segment)
				     :default (get menu-id index)))
		     (:pathname
			(clim:accept 'clim:pathname
				     :stream stream
				     :prompt (third option-segment)
				     :default (get menu-id index)))
		     (otherwise
			(clim:accept 'symbol
				     :stream stream
				     :prompt (third option-segment)
				     :default (get menu-id index)))
		     ))
		    (incf index)))
	    (clim:terpri stream))
	  (clim:terpri stream))

	(setf index 0)
	(setf return-values
	      (mapcar #'(lambda (opt-rec)
			  (prog1
			    (list (car opt-rec) (get menu-id index))
			    (incf index)))
		      (remove-if #'stringp option-list)))
	(force-output menu-window)
	(return-from choose-values return-values)))

    (force-output menu-window)
    nil))


(defun get-choose-value (value-rec value-variable)
  (cadr (assoc value-variable value-rec)))

