;;; -*- Mode: LISP; Syntax: Ansi-common-lisp; Package: CLIM-USER; Base: 10 -*-

;;;; Fast display tables in CLIM using home-grown incremental redisplay.
;;;
;;;
;;; Copyright (c) 1993, Philip L. Stubblefield, Rockwell International.
;;;
;;; This software is in the public domain, and may be freely copied,
;;; distributed, and/or modified.  Rockwell International disclaims all
;;; warranties regarding the performance of this software, its use for
;;; any purpose, and any results obtained thereby.
;;;
;;; Please feel free to report any bugs, bug fixes, or enhancements to
;;; phil@rpal.rockwell.com, although the author makes no promises to
;;; release updates in a timely fashion!
;;;
;;; This software has been tested using CLIM 1.1 under Genera 8.1.1 on a
;;; Symbolics MacIvory 3, Allegro CL 4.1 on a SparcStation 10 Model 40,
;;; and MCL 2.0p2 on a Macintosh IIfx.
;;;
;;; *** Note:  This file uses quite a few internal CLIM functions in
;;; order to manage incremental redisplay efficiently.  Search for the
;;; strings "CLIM::" and "CLIM-UTILS:" to find CLIM internals.  Also,
;;; some of the names and behavior of CLIM functions, both internal and
;;; external, will change in CLIM 2.0.  A new version of this software
;;; will probably be released at that time.
;;;
;;; Release date:  March 3, 1993.
;;;
;;;
;;;; Overview:
;;;
;;; These tables are designed around a particular paradigm.  Each cell
;;; displays a particular attribute of some client object.  Cells in the
;;; same row represent the same client object, and cells in the same
;;; column represent the same attribute.  (The first row contains column
;;; titles.)  The table adjusts its dimensions automatically in response
;;; to changes in the size of the output in each cell.
;;;
;;; For example, an inventory application might display the available
;;; stock as a table:
;;;
;;;   Part No.  Description                  Price   Stock
;;;   BS21x-2   Bifurcated Snorkle           75.50   34
;;;   B375-R0   Retractable Muffler Bearing  39.99   0
;;;   QP22i     Anodized Clavitracle         249.00  2
;;;   G72222z   Brushed Umphagoggin (Beige)  1.97    14
;;;
;;; In this table, each row represents a particular class of parts, and
;;; each column a particular attribute.
;;;
;;; Implementationally, each row contains a pointer to its client
;;; object.  Each column contains the following items:  the reader
;;; function used to access a particular attribute of each client
;;; object; the presentation type used to display the value returned by
;;; the reader function; a unique ID that serves as a handle to the
;;; client program; and a second presentation type used to display this
;;; ID as the title of the column.
;;;
;;; In addition, a column may contain a writer function that, when
;;; applied to a new value and a client object, updates the same
;;; attribute accessed by the associated reader function.  When this
;;; writer function is supplied, and when appropriate translators are
;;; defined, each cell in the column may be used as an input device as
;;; well as an output device.
;;;
;;;
;;;; Interface:
;;;
;;; The following classes are provided.  The user may specialize any or
;;; all of the classes as necessary.  Each class has an associated
;;; presentation type of the same name.
;;;
;;;   STANDARD-TABLE
;;;   STANDARD-ROW
;;;   STANDARD-COLUMN
;;;   STANDARD-CELL
;;;
;;; In addition, the presentation type STANDARD-COLUMN-TITLE is used as
;;; the default presentation type for column titles.
;;;
;;;
;;; The following functions are used for creating tables and their
;;; components.  Cells are created internally, but the class of cell
;;; created may be controlled by writing a method for the generic
;;; function TABLE-CELL-CLASS that specializes on an appropriate
;;; subclass of STANDARD-TABLE.  See each function definition for more
;;; documentation.
;;;
;;;   MAKE-TABLE &key (class 'standard-table)
;;;                   (rows '()) (columns '())
;;;                   (x-spacing "  ") (y-spacing 0)
;;;                   sort-predicate
;;;              &allow-other-keys
;;;
;;;   MAKE-ROW object &key (class 'standard-row) &allow-other-keys
;;;
;;;   MAKE-COLUMN id attribute-reader cell-presentation-type
;;;               &key (class 'standard-column)
;;;                    attribute-writer
;;;                    title-presentation-type
;;;               &allow-other-keys
;;;
;;;
;;; These functions add or delete components from a table:
;;;
;;;   ADD-ROW row table
;;;   ADD-COLUMN column table
;;;   DELETE-ROW row-or-object table
;;;   DELETE-COLUMN column-or-id table
;;;
;;;
;;; These functions are used to hide a row or column or to make it
;;; visible again:
;;;
;;;   HIDE-ROW row-or-object table
;;;   HIDE-COLUMN column-or-id table
;;;   SHOW-ROW row-or-object table
;;;   SHOW-COLUMN column-or-id table
;;;
;;;
;;; These functions allow the user to locate a table component given the
;;; objects available to a client program:
;;;
;;;   FIND-ROW row-object table &optional errorp
;;;   FIND-COLUMN column-id table &optional errorp
;;;   FIND-CELL column-id row-object table &optional errorp
;;;
;;;
;;; These functions provide editing capability for cells whose columns
;;; possess an attribute writer function:
;;;
;;;   EDIT-CELL cell &key own-window insert-default
;;;   EDIT-ATTRIBUTE object column-id table &key own-window insert-default
;;;
;;;
;;; The function REDISPLAY-TABLE displays or redisplays a table on a
;;; given stream.  The function INVALIDATE-CELL marks the contents of a
;;; given cell dirty so that its output will be updated during the next
;;; call to REDISPLAY-TABLE.
;;;
;;;
;;; The command table TABLE contains general commands for manipulating
;;; tables, and must be inherited by the command table associated with
;;; any frame in which tables are to be manipulated.  It also contains
;;; several presentation-to-command translators that make these commands
;;; available on the menu popped-up by the :MENU gesture.

(cl:in-package "CLIM-USER")

(declaim (declaration values))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Utilities.

(declaim (inline funcallablep))

(defun funcallablep (thing)
  (or (functionp thing)
      (and (symbolp thing)
	   (fboundp thing))))


;;; In March, 1991, X3J13 voted to remove GET-SETF-METHOD from the
;;; language, and to rename GET-SETF-METHOD-MULTIPLE-VALUE to
;;; GET-SETF-EXPANSION.  Not all Lisps have made the change yet.

(eval-when (:compile-toplevel :load-toplevel :execute)
  (unless (fboundp 'get-setf-expansion)
    (setf (symbol-function 'get-setf-expansion)
	  ;; *** Aarrggh!  The Franz implementation has screwed up the
	  ;; package structure.
	  #-ExCL #'get-setf-method
	  #+ExCL #'cltl1:get-setf-method)))


;;; Generalized variable reference version of DELETE.  Note that we take
;;; care to evaluate the subforms from left to right.

(defmacro deletef (item sequence &rest keys
		   &key (test #'eql) test-not (key #'identity)
                   from-end (start 0) end count)
  (declare (dynamic-extent keys))
  (declare (ignore test test-not key from-end start end count))
  (multiple-value-bind (temps vals stores store-form access-form)
      (get-setf-expansion sequence)
    (let ((i (gensym)))
      `(let* ((,i ,item)
	      ,@(mapcar #'list temps vals)
	      (,(first stores) (delete ,i ,access-form ,@keys)))
	 ,store-form))))


;;; Destructively modifies the list in order to delete the item at the
;;; given index, which must be a non-negative integer less than or equal
;;; to the length of the list.

(defmacro deletef-nth (list index)
  (multiple-value-bind (temps vals stores store-form access-form)
      (get-setf-expansion list)
    (let ((n (gensym))
	  (length (gensym)))
      `(let* (,@(mapcar #'list temps vals)
	      (,n ,index)
	      (,length (length ,access-form)))
	 (cond ((not (<= 0 ,n ,length))
		(error "~S is not an index within ~S." ,n ,access-form))
	       ((zerop ,n)
		(let ((,(first stores) (rest ,access-form)))
		  ,store-form))
	       (t
		(let ((tail (nthcdr (1- ,n) ,access-form)))
		  (setf (cdr tail) (cddr tail))
		  ,access-form)))))))


;;; Destructively modifies the list in order to insert the item in the
;;; proper position with respect to the given predicate, which should
;;; return non-NIL iff the first argument is strictly less than the
;;; second in an appropriate sense, or NIL otherwise.

(defmacro insertf (item list predicate &key (key nil keyp))
  (multiple-value-bind (temps vals stores store-form access-form)
      (get-setf-expansion list)
    (let ((i (gensym "ITEM-"))
	  (p (gensym "PREDICATE-"))
	  (k (gensym "KEY-"))
	  (v (gensym "VALUE-")))
      `(let* ((,i ,item)
	      ,@(mapcar #'list temps vals)
	      (,p ,predicate)
	      ,@(and keyp `((,k ,key)
			    (,v (funcall ,k ,i)))))
	 (labels ((test (list)
		    (or (endp list)
			,(if keyp
			     `(funcall ,p ,v (funcall ,k (first list)))
			     `(funcall ,p ,i (first list)))))
		  (insert (leading trailing)
		    (if (test leading)
			(setf (cdr trailing) (cons ,i leading))
			(insert (cdr leading) leading))))
	   (declare (dynamic-extent #'test #'insert))
	   (cond ((test ,access-form)
		  (let ((,(first stores) (cons ,i ,access-form)))
		    ,store-form))
		 (t
		  (insert (rest ,access-form) ,access-form)
		  ,access-form)))))))


;;; Destructively modifies the list in order to insert the item at the
;;; given index, which must be a non-negative integer less than or equal
;;; to the length of the list.

(defmacro insertf-nth (item list index)
  (multiple-value-bind (temps vals stores store-form access-form)
      (get-setf-expansion list)
    (let ((i (gensym))
	  (n (gensym))
	  (length (gensym)))
      `(let* ((,i ,item)
	      ,@(mapcar #'list temps vals)
	      (,n ,index)
	      (,length (length ,access-form)))
	 (cond ((not (<= 0 ,n ,length))
		(error "~S is not an index within ~S." ,n ,access-form))
	       ((zerop ,n)
		(let ((,(first stores) (cons ,i ,access-form)))
		  ,store-form))
	       (t
		(let ((tail (nthcdr (1- ,n) ,access-form)))
		  (setf (cdr tail) (cons ,i (cdr tail)))
		  ,access-form)))))))


;;; Note that push-end and pushnew-end destructively modify the list in
;;; order to avoid consing.

(defmacro push-end (item list)
  (multiple-value-bind (temps vals stores store-form access-form)
      (get-setf-expansion list)
    (let ((i (gensym)))
      `(let* ((,i ,item)
	      ,@(mapcar #'list temps vals))
	 (cond ((endp ,access-form)
		(let ((,(first stores) (cons ,i nil)))
		  ,store-form))
	       (t
		(setf (cdr (last ,access-form)) (cons ,i nil))
		,access-form))))))


;;; Destructively sorts the sequence in place.

(defmacro sortf (sequence predicate &key key)
  (multiple-value-bind (temps vals stores store-form access-form)
      (get-setf-expansion sequence)
    `(let* (,@(mapcar #'list temps vals)
	    (,(first stores)
	     (sort ,access-form ,predicate ,@(and key `(:key ,key)))))
       ,store-form)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; CLIM utilities.


;;; Defines the named command table and defines an appropriately named
;;; command-defining macro.  For example, if the command table is named
;;; WINDOW, then the macro is named DEFINE-WINDOW-COMMAND.

(defmacro define-command-table-and-definer (command-table-name
					    &key inherit-from)
  (let ((definer
	  (let ((*print-case* :upcase))
	    (intern (format nil "DEFINE-~A-COMMAND" command-table-name)))))
    #+ExCL (when (member excl:*current-case-mode*
			 '(:case-sensitive-lower :case-insensitive-lower)
			 :test #'eq)
	     (setq definer (string-downcase definer)))
    `(progn
       (unless (find-command-table ',command-table-name :errorp nil)
	 (define-command-table ,command-table-name
	   :inherit-from ,inherit-from))
       (unless (and (fboundp ',definer)
		    (macro-function ',definer))
	 (defmacro ,definer (command-name-and-options arguments &body body)
	   #+:Genera (declare (zwei:indentation 1 3 2 1))
	   (setq command-name-and-options
		 (if (listp command-name-and-options)
		     (copy-list command-name-and-options)
		     (list command-name-and-options)))
	   (setf (getf (rest command-name-and-options) :command-table)
		 ',command-table-name)
	   `(define-command ,command-name-and-options ,arguments ,@body)))
       #+Genera (scl:defprop ,definer
			     define-command
			     zwei:definition-function-spec-type)
       #+Genera (scl:defprop ,definer
			     remove-command
			     zwei:kill-definition)
       #+Genera (scl:defprop ,definer
			     zwei:defselect-function-spec-finder
			     zwei:definition-function-spec-finder))))


;;; Cursor utilities.

(defmacro with-stream-cursor-position ((stream absolute-x absolute-y) &body body)
  "Sets the cursor position to the given absolute coordinates on STREAM,
   executes the body, and then restores the old cursor position."
  (check-type stream symbol)
  (let ((old-x (gensym "OLD-X-"))
	(old-y (gensym "OLD-Y-")))
    `(multiple-value-bind (,old-x ,old-y)
	 (stream-cursor-position* ,stream)
       (unwind-protect
	   (progn
	     (stream-set-cursor-position* ,stream ,absolute-x ,absolute-y)
	     ,@body)
	 (stream-set-cursor-position* ,stream ,old-x ,old-y)))))


(defmacro with-stream-cursor-position-at-record ((stream record) &body body)
  "Sets the cursor position to the beginning of the output record RECORD
   on STREAM, executes the body, and then restores the old cursor position."
  (check-type stream symbol)
  (check-type record symbol)
  (let ((left (gensym "LEFT-"))
	(top (gensym "TOP-"))
	(x-offset (gensym "X-OFFSET-"))
	(y-offset (gensym "Y-OFFSET-")))
    `(multiple-value-bind (,left ,top) (output-record-position* ,record)
       (declare (fixnum ,left ,top))
       (multiple-value-bind (,x-offset ,y-offset)
	   (convert-from-relative-to-absolute-coordinates
	     ,stream
	     (output-record-parent ,record))
	 (declare (fixnum ,x-offset ,y-offset))
	 (with-stream-cursor-position (,stream
				       (the fixnum (+ ,left ,x-offset))
				       (the fixnum (+ ,top ,y-offset)))
	   ,@body)))))


;;; Keeping the cursor at the origin of the stream's output history
;;; prevents the start positions of child output records from doing
;;; strange things (at least, *I* think they're strange).

(defmacro with-stream-cursor-position-at-origin ((stream) &body body)
  "Sets the cursor position to the origin of the output history of STREAM,
   executes the body, and then restores the old cursor position."
  ;; Concept snarfed from WITH-NEW-OUTPUT-RECORD-INTERNAL.
  (check-type stream symbol)
  (let ((origin-x (gensym "ORIGIN-X-"))
	(origin-y (gensym "ORIGIN-Y-")))
    `(multiple-value-bind (,origin-x ,origin-y)
	 (point-position*
	   (clim::output-recording-stream-output-record-absolute-position ,stream))
       (with-stream-cursor-position (,stream ,origin-x ,origin-y)
	 ,@body))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Tables.
;;;
;;; Each table object incorporates all of the information needed to
;;; display a table on a given stream.
;;;
;;; Note that the table is a presentation output record.  Its children
;;; are the columns of the table, its grandchidren are the cells of each
;;; column, and its presentation object is the table itself.  Tables are
;;; column-oriented so that columns can be moved quickly when a new
;;; value for a cell changes the horizontal spacing of the table.

(eval-when (:compile-toplevel :load-toplevel :execute)

(defclass standard-table (standard-presentation)
    ((name
       :accessor table-name
       :initarg :name
       :documentation "A string or symbol used only for display purposes.")
     (rows
       :accessor table-rows
       :initform (list (make-title-row))
       :type list
       :documentation "The list of rows in the table, including the title row.")
     (stream
       :reader table-stream
       :initform nil
       :documentation "The stream upon which the table is displayed.")
     (x-spacing
       :reader table-x-spacing			;SETF method below
       :initarg :x-spacing
       :documentation "A specification for the spacing between table columns.")
     (y-spacing
       :reader table-y-spacing			;SETF method below
       :initarg :y-spacing
       :documentation "A specification for the spacing between table rows.")
     (real-x-spacing
       :reader table-real-x-spacing
       :initform 0
       :type fixnum
       :documentation "The number of pixels between table columns.")
     (real-y-spacing
       :reader table-real-y-spacing
       :initform 0
       :type fixnum
       :documentation "The number of pixels between table rows.")
     (sort-predicate
       :reader table-sort-predicate		;SETF method below
       :initarg :sort-predicate
       :documentation "A predicate applied to pairs of row objects, or NIL.")
     (needs-redisplay-p
       :accessor table-needs-redisplay-p
       :initform t
       :documentation "Non-NIL iff the table requires redisplay."))
  (:default-initargs :name "Anonymous"
		     :x-spacing "  "
		     :y-spacing 0
		     :sort-predicate nil
		     :modifier nil
		     :single-box t
		     :allow-sensitive-inferiors t))

)						;end of EVAL-WHEN

(declaim (inline tablep standard-table-p))

(defun tablep (thing)
  (typep thing 'standard-table))

(defun standard-table-p (thing)
  (typep thing 'standard-table))

(defmethod print-object ((table standard-table) stream)
  (print-unreadable-object (table stream :type t :identity t)
    (with-bounding-rectangle* (left top right bottom) table
      (format stream "~A /x ~D:~D y ~D:~D/"
	      (table-name table) left right top bottom))))

(defmethod initialize-instance :before ((table standard-table)
					&key &allow-other-keys)
  ;; A table is its own presentation object and presentation type.
  (with-slots (clim::object clim::type) table
    (setq clim::object table)
    (setq clim::type (presentation-type-of table))))

(defmethod initialize-instance :after ((table standard-table)
				       &key rows columns &allow-other-keys)
  (with-slots (sort-predicate) table
    (check-type sort-predicate
		(or null function (and symbol (satisfies fboundp)))
		"NIL, a function, or a symbol with a functional value"))
  ;; The order of the following three forms is crucial!
  (dolist (column columns)
    (add-output-record-element table column))
  (add-cells (table-title-row table) table)
  (dolist (row rows)
    (add-row row table)))

(defun make-table (&rest initargs
		   &key
		   (class 'standard-table)
		   (rows '()) (columns '())
		   (x-spacing "  ") (y-spacing 0)
		   sort-predicate
		   &allow-other-keys)
  ;; *** :X-SPACING and :Y-SPACING are called :INTER-COLUMN-SPACING and
  ;; :INTER-ROW-SPACING in CLIM 1.1.
  "Creates and returns a new table of class CLASS.  The value of the
   keyword :ROWS, which must be a list of row objects, determines the
   initial rows of the table, and similarly for :COLUMNS.  Both values
   default to the empty list.

   The values of the keywords :X-SPACING and :Y-SPACING respectively
   determine the amount of whitespace between columns and between rows,
   and default to \"  \" and 0 respectively. Valid values are the same as
   for FORMATTING-TABLE.

   The value of the keyword :SORT-PREDICATE must be either NIL, the
   default, or a function of two arguments.  If NIL, then the rows are
   displayed in the same order in which they are added to the table.
   Otherwise, the predicate is used to determine their ordering.  It
   will be applied to pairs of client objects, and must return a non-NIL
   value if and only if the first is strictly less than the second in
   some appropriate ordering."
  ;; MAKE-INSTANCE is often faster when given a constant class.
  (if (eq class 'standard-table)
      (make-instance 'standard-table
		     :rows rows
		     :columns columns
		     :x-spacing x-spacing
		     :y-spacing y-spacing
		     :sort-predicate sort-predicate)
      (apply #'make-instance class :allow-other-keys t initargs)))


;;; CLIM internals.

(defmethod clim::elements-never-overlap-p ((record standard-table))
  t)


;;; Adds the table to the stream's output history, then processes the
;;; spacing specifications, which may depend on the stream.

(defmethod (setf table-stream) (new-stream (table standard-table))
  (unless (extended-output-stream-p new-stream)
    (error "For ~S, ~S is not an extended-output stream."
	   table new-stream))
  (with-slots (stream x-spacing y-spacing real-x-spacing real-y-spacing) table
    (setq stream new-stream)
    (add-output-record-element (output-recording-stream-output-record stream) table)
    (setq real-x-spacing (translate-spacing-specification x-spacing stream))
    (setq real-y-spacing (translate-spacing-specification y-spacing stream))))


;;; This function should be called in case of damage to the display, or
;;; in case the table is moved to a different stream.

(defmethod reinitialize-table ((table standard-table) stream)
  (unless (null (table-stream table))
    (erase-table table))
  (setf (table-stream table) stream)
  (dolist (row (table-rows table))
    (setf (row-needs-redisplay-p row) t))
  (setf (table-needs-redisplay-p table) t))


;;; Simple table accessors.

(defmethod table-columns ((table standard-table))
  (clim::output-record-elements table))

(defmethod table-has-hidden-columns-p ((table standard-table))
  (notevery #'column-visible-p (table-columns table)))

(defmethod table-title-row ((table standard-table))
  (first (table-rows table)))

(defmethod (setf table-title-row) (new-value (table standard-table))
  (setf (first (table-rows table)) new-value))

(defmethod table-object-rows ((table standard-table))
  (rest (table-rows table)))

(defmethod (setf table-object-rows) (new-value (table standard-table))
  (setf (rest (table-rows table)) new-value))

(defmethod table-has-hidden-rows-p ((table standard-table))
  (notevery #'row-visible-p (table-object-rows table)))


;;; Changing the X- or Y-spacing causes the new spacing specification to
;;; be parsed and the table to be redisplayed.

(defmethod (setf table-x-spacing) (new-spacing (table standard-table))
  (with-slots (stream x-spacing real-x-spacing) table
    (unless (equal new-spacing x-spacing)
      (setq x-spacing new-spacing)
      (setq real-x-spacing (translate-spacing-specification x-spacing stream))
      (setf (table-needs-redisplay-p table) t))
    new-spacing))

(defmethod (setf table-y-spacing) (new-spacing (table standard-table))
  (with-slots (stream y-spacing real-y-spacing) table
    (unless (equal new-spacing y-spacing)
      (setq y-spacing new-spacing)
      (setq real-y-spacing (translate-spacing-specification y-spacing stream))
      (setf (table-needs-redisplay-p table) t))
    new-spacing))


;;; Translates a spacing specification into the appropriate number of
;;; pixels.  Each specification can be one of:
;;;
;;;   1) NIL, meaning no extra spacing;
;;;   2) a non-negative integer, meaning that number of pixels;
;;;   3) a character, meaning the width of that character; or
;;;   4) a string, meaning the width of that string.

(defun translate-spacing-specification (specification stream)
  (cond ((or (null specification)
	     (null stream))
	 0)
	((and (integerp specification)
	      (not (minusp specification)))
	 specification)
	((characterp specification)
	 (stream-character-width stream specification))
	((stringp specification)
	 (stream-string-width stream specification))
	(t
	 (error "The spacing specifier ~S is invalid." specification))))


;;; Changing the table sort predicate causes the table to be resorted
;;; and redisplayed.

(defmethod (setf table-sort-predicate) (new-value (table standard-table))
  (with-slots (sort-predicate) table
    (unless (eq new-value sort-predicate)
      (setq sort-predicate new-value)
      (table-sort-rows table)
      (unless (null (table-stream table))
	(erase-table table nil))
      (setf (table-needs-redisplay-p table) t))))

(defmethod table-sort-rows ((table standard-table))
  (let ((sort-predicate (table-sort-predicate table)))
    (unless (null sort-predicate)
      (let ((columns (table-columns table)))
	(sortf (table-object-rows table) sort-predicate :key #'row-object)
	(mapc #'clear-output-record columns)
	(dolist (row (table-rows table))
	  (mapc #'add-output-record-element columns (row-cells row)))))))


;;; Returns the class of cell to instantiate for this class of table.
;;; More specific classes may with to override this default method.

(defmethod table-cell-class ((table standard-table))
  'standard-cell)


;;; Deletes all the (non-title) rows of the table.

(defmethod clear-table ((table standard-table))
  (let ((stream (table-stream table)))
    (unless (null stream)
      (erase-table table nil)
      (dolist (row (table-object-rows table))
	(delete-row row table))
      (window-set-viewport-position* stream 0 0))
    (setf (table-needs-redisplay-p table) t)))
	

;;; Erases the table from the stream.

(defmethod erase-table ((table standard-table) &optional (erase-title-row-p t))
  (let ((stream (table-stream table)))
    (cond (erase-title-row-p
	   ;; Erase entire table.
	   (erase-region table stream))
	  ((not (null (table-object-rows table)))
	   ;; Erase all non-title rows.
	   (let ((title-height (row-height (table-title-row table))))
	     (declare (fixnum title-height))
	     (with-bounding-rectangle* (left top right bottom) table
	       (setq top (the fixnum (+ top title-height)))
	       (erase-region table stream left top right bottom)))))))


;;; Erases the region bounded by LEFT, TOP, RIGHT, and BOTTOM, if
;;; supplied, or the region corresponding to the given output record.
;;; Differs from ERASE-OUTPUT-RECORD in that it does not delete the
;;; output record from its parent, nor does it attempt to replay any
;;; overlapping presentations.

(defmethod erase-region ((record output-record) stream
			 &optional
			 (left 0 edges-supplied-p) (top 0) (right 0) (bottom 0))
  (declare (fixnum left top right bottom))
  (unless edges-supplied-p
    (multiple-value-setq (left top right bottom)
      (bounding-rectangle* record)))
  (multiple-value-bind (x-offset y-offset)
      (convert-from-relative-to-absolute-coordinates
	stream
	(output-record-parent record))
    (declare (fixnum x-offset y-offset))
    (with-output-recording-options (stream :record-p nil)
      (draw-rectangle* stream
		       (the fixnum (+ left x-offset))
		       (the fixnum (+ top y-offset))
		       (the fixnum (+ right x-offset))
		       (the fixnum (+ bottom y-offset))
		       :ink +background+))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Table rows.
;;;
;;; Unlike the other table-related classes, rows are *not* themselves
;;; output records.  Besides storing pertinent information about the
;;; row, they maintain a list of pointers to the cell output records so
;;; that the table may be accessed in a row-oriented fashion if needed.

(eval-when (:compile-toplevel :load-toplevel :execute)

(defclass standard-row ()
    ((object
       :reader row-object
       :initarg :object
       :documentation "The object displayed in this row.")
     (cells
       :accessor row-cells
       :initform '()
       :type list
       :documentation "The list of cells in this row.")
     (y-position
       :accessor row-y-position
       :initform 0
       :type fixnum
       :documentation "The Y-coordinate of the cells in this row.")
     (width
       :accessor row-width
       :initform 0
       :type fixnum
       :documentation "The horizontal extent of the cells in this row.")
     (height
       :accessor row-height
       :initform 0
       :type fixnum
       :documentation "The vertical extent of the cells in this row.")
     (visible
       :accessor row-visible-p
       :initform t
       :documentation "True iff this row should be visible.")))

)						;end of EVAL-WHEN

(declaim (inline rowp standard-row-p))

(defun rowp (thing)
  (typep thing 'standard-row))

(defun standard-row-p (thing)
  (typep thing 'standard-row))

(defmethod print-object ((row standard-row) stream)
  (print-unreadable-object (row stream :type t :identity t)
    (let ((object (row-object row)))
      (present object (presentation-type-of object) :stream stream))))

(defmethod initialize-instance :after ((row standard-row)
				       &key &allow-other-keys)
  (unless (slot-boundp row 'object)
    (error "No ~S was supplied for ~S." :object row)))

(defun make-row (object &rest initargs
		 &key (class 'standard-row) &allow-other-keys)
  "Creates and returns a row (of class CLASS) associated with the given
   client object."
  ;; MAKE-INSTANCE is often faster when given a constant class.
  (if (eq class 'standard-row)
      (make-instance 'standard-row :object object)
      (apply #'make-instance class :object object :allow-other-keys t initargs)))


;;; Simple row accessors.

(defmethod row-position ((row standard-row))
  (declare (values left top))
  (values 0
	  (row-y-position row)))

(defmethod row-size ((row standard-row))
  (declare (values width height))
  (values (row-width row)
	  (row-height row)))

(defmethod (setf row-y-position) :after (new-y (row standard-row))
  (dolist (cell (row-cells row))
    (setf (cell-y-position cell) new-y)))

(defmethod row-needs-redisplay-p ((row standard-row))
  (some #'cell-needs-redisplay-p (row-cells row)))

(defmethod (setf row-needs-redisplay-p) (new-value (row standard-row))
  (dolist (cell (row-cells row))
    (setf (cell-needs-redisplay-p cell) new-value)))


;;; Simple row operations.

(defmethod erase-row ((row standard-row) stream)
  (unless (null (row-cells row))
    (let* ((left 0)
	   (top (row-y-position row))
	   (right (row-width row))
	   (bottom (the fixnum (+ (the fixnum top)
				  (the fixnum (row-height row)))))
	   (cell (first (row-cells row))))
      (erase-region cell stream left top right bottom))))


;;; Finding rows from other pointers.

(defgeneric find-row (object table &optional errorp)
  (:documentation "Returns the row associated with the given client object."))

(defmethod find-row (object (table standard-table) &optional (errorp t))
  (or (find object (table-rows table) :key #'row-object :test #'eq)
      (and errorp
	   (error "The table ~S contains no row for the object ~S."
		  table object))))


;;; The title row is marked by a special row object.

(defconstant +title-row+ '+title-row+
  "The marker object for the row of column titles.")

(defun make-title-row ()
  (make-row +title-row+))

(defmethod title-row-p ((row standard-row))
  (eq (row-object row) +title-row+))


;;; Title cells are a special class.

(defclass title-cell (standard-cell)
    ())

(defun title-cell-p (thing)
  (typep thing 'title-cell))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Table columns.
;;;
;;; Each column is a presentation output record whose parent is the
;;; table, whose children are the cell output records, and whose
;;; presentation object is the column itself.

(eval-when (:compile-toplevel :load-toplevel :execute)

(defclass standard-column (standard-presentation)
    ((id
       :accessor column-id
       :initarg :id
       :documentation "A unique (via #'EQUAL) object identifying the column.")
     (attribute-reader
       :accessor column-attribute-reader
       :initarg :attribute-reader
       :documentation "A function that, given a row object, returns the
		       attribute displayed in this column.")
     (attribute-writer
       :accessor column-attribute-writer
       :initarg :attribute-writer
       :documentation "NIL, or a function that, given a new value and a row
		       object, sets the attribute displayed in this column.")
     (cell-presentation-type
       :accessor column-cell-presentation-type
       :initarg :cell-presentation-type
       :documentation "The presentation type used for cells in this column.")
     (title-presentation-type
       :accessor column-title-presentation-type
       :initarg :title-presentation-type
       :documentation "The presentation type used for the title of this column.")
     (visible
       :accessor column-visible-p
       :initform t
       :documentation "True iff this column should be visible.")
     (displayed-p
       :accessor column-displayed-p
       :initform nil
       :documentation "Non-NIL iff the column contains visible output.")
     (needs-redisplay-p
       :accessor column-needs-redisplay-p
       :initform t
       :documentation "Non-NIL iff the column requires redisplay."))
  (:default-initargs :attribute-writer nil
		     :cell-presentation-type nil
		     :title-presentation-type 'standard-column-title
		     :modifier nil
		     :single-box t
		     :allow-sensitive-inferiors t))

)						;end of EVAL-WHEN

(declaim (inline columnp standard-column-p))

(defun columnp (thing)
  (typep thing 'standard-column))

(defun standard-column-p (thing)
  (typep thing 'standard-column))

(defmethod print-object ((column standard-column) stream)
  (print-unreadable-object (column stream :type t :identity t)
    (with-bounding-rectangle* (left top right bottom) column
      (present (column-id column) (column-title-presentation-type column)
	       :stream stream)
      (format stream " /x ~D:~D y ~D:~D/" left right top bottom))))

(defmethod initialize-instance :before ((column standard-column)
					&key &allow-other-keys)
  ;; A column is its own presentation object and type.
  (with-slots (clim::object clim::type) column
    (setq clim::object column)
    (setq clim::type (presentation-type-of column))))

(defmethod initialize-instance :after ((column standard-column)
				       &key &allow-other-keys)
  (unless (slot-boundp column 'id)
    (error "No ~S was supplied for ~S." ':id column))
  (unless (slot-boundp column 'attribute-reader)
    (error "No ~S was supplied for ~S." ':attribute-reader column))
  (with-slots (attribute-reader attribute-writer) column
    (unless (funcallablep attribute-reader)
      (error "The value of ~S, ~S,~@
	      was neither a function nor a symbol with a functional definition."
	     ':attribute-reader attribute-reader))
    (unless (or (null attribute-writer)
		(funcallablep attribute-writer))
      (error "The value of ~S, ~S,~@
	      was neither a function nor a symbol with a functional definition."
	     ':attribute-writer attribute-writer)))
  ;; EXPAND-PRESENTATION-TYPE-ABBREVIATION will signal an error given an
  ;; invalid specifier.
  (with-slots (cell-presentation-type title-presentation-type) column
    (expand-presentation-type-abbreviation cell-presentation-type)
    (expand-presentation-type-abbreviation title-presentation-type)))

(defun make-column (id attribute-reader cell-presentation-type
		    &rest initargs
		    &key
		    (class 'standard-column)
		    attribute-writer
		    title-presentation-type
		    &allow-other-keys)
   "Creates and returns a column of class CLASS.  ID may be any object,
   but must be unique (via EQUAL) for this table.  ATTRIBUTE-READER must
   be a function of one argument that, when applied to a client object,
   returns the value of the attribute to be displayed in this column.
   CELL-PRESENTATION-TYPE, which must be a valid presentation-type
   specifier, is used to display each attribute value.

   If supplied, the value of the keyword :ATTRIBUTE-WRITER must be a
   function of two arguments, a new value and a client object, that
   updates the same attribute accessed by the associated attribute
   reader function.

   The value of the keyword :TITLE-PRESENTATION-TYPE, which must be a
   valid presentation-type specifier, is used to present the column ID
   as the column title.  The default value, the presentation type
   STANDARD-COLUMN-TITLE, behaves as follows:  if the column ID is a
   string, it is displayed as the title; if a symbol, its print name is
   displayed as if processed by STRING-CAPITALIZE; otherwise, the ID is
   presented using the presentation type returned by the function
   PRESENTATION-TYPE-OF.  All presentations are displayed using an
   italic text face."
  (when (null title-presentation-type)
    (setq title-presentation-type 'standard-column-title))
  ;; MAKE-INSTANCE is often faster when given a constant.
  (if (eq class 'standard-column)
      (make-instance 'standard-column
		     :id id
		     :attribute-reader attribute-reader
		     :attribute-writer attribute-writer
		     :cell-presentation-type cell-presentation-type
		     :title-presentation-type title-presentation-type)
      (apply #'make-instance class
	     :id id
	     :attribute-reader attribute-reader
	     :cell-presentation-type cell-presentation-type
	     :allow-other-keys t
	     initargs)))


;;; CLIM internals.

(defmethod clim::elements-never-overlap-p ((record standard-column))
  t)


;;; Simple column accessors.

(defmethod column-table ((column standard-column))
  (output-record-parent column))

(defmethod column-cells ((column standard-column))
  (clim::output-record-elements column))

(defmethod column-x-position ((column standard-column))
  (multiple-value-bind (x y) (output-record-position* column)
    (declare (ignore y))
    x))

(defmethod (setf column-x-position) (new-x (column standard-column))
  ;; NEW-X is already relative to the coordinate system of the table,
  ;; which is the parent output record of the column, so no conversion
  ;; of coordinate systems is necessary.  The column is always at
  ;; Y-position zero.
  (output-record-set-position* column new-x 0))

(defmethod column-width ((column standard-column))
  (bounding-rectangle-width column))

(defmethod (setf column-width) (new-width (column standard-column))
  (multiple-value-bind (width height)
      (bounding-rectangle-size column)
    (declare (ignore width))
    (clim-utils:bounding-rectangle-set-size column new-width height)
    new-width))

(defmethod column-height ((column standard-column))
  (bounding-rectangle-height column))

(defmethod (setf column-height) (new-height (column standard-column))
  (multiple-value-bind (width height)
      (bounding-rectangle-size column)
    (declare (ignore height))
    (clim-utils:bounding-rectangle-set-size column width new-height)
    new-height))


;;; Simple column operations.

(defmethod erase-column ((column standard-column) stream)
  (when (column-displayed-p column)
    (erase-region column stream)
    (setf (column-displayed-p column) nil)))

(defgeneric find-column (column-id table &optional errorp)
  (:documentation "Returns the column having the given ID."))

(defmethod find-column (column-id (table standard-table) &optional (errorp t))
  (or (find column-id (table-columns table) :key #'column-id :test #'equal)
      (and errorp
	   (error "The table ~S contains no column whose ID is ~S."
		  table column-id))))


;;; Iterates over the visible cells of a given column and over the
;;; corresponding rows.

(defmacro do-visible-cells-and-rows ((cell-variable row-variable column table
				      &optional return-form)
				     &body body)
  (check-type cell-variable symbol)
  (check-type row-variable symbol)
  (let ((cells (gensym "CELLS-"))
	(rows (gensym "ROWS-")))
    `(do ((,cells (column-cells ,column) (rest ,cells))
	  (,rows (table-rows ,table) (rest ,rows)))
	 ((null ,cells)
	  #+Ignore (check-type ,rows null)
	  ,return-form)
       (let ((,cell-variable (first ,cells))
	     (,row-variable (first ,rows)))
	 (loop
	   (when (row-visible-p ,row-variable)
	     (return))
	   ;; Skip hidden rows, because the corresponding cells have
	   ;; been removed from the column.
	   (setq ,rows (rest ,rows))
	   (setq ,row-variable (first ,rows)))
	 ,@body))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Table cells.
;;;
;;; Each cell is a presentation output record whose parent is the column
;;; (an output record) to which the cell belongs, whose presentation
;;; type is CELL, and whose presentation object is the object associated
;;; with the row to which the cell belongs.

(eval-when (:compile-toplevel :load-toplevel :execute)

(defclass standard-cell (standard-presentation)
    ((displayed-p
       :accessor cell-displayed-p
       :initform nil
       :documentation "Non-NIL iff the cell contains visible output.")
     (needs-redisplay-p
       :accessor cell-needs-redisplay-p
       :initform t
       :documentation "Non-NIL iff the cell requires redisplay."))
  (:default-initargs :modifier nil
		     :single-box t
		     :allow-sensitive-inferiors t))

)						;end of EVAL-WHEN

(declaim (inline cellp standard-cell-p))

(defun cellp (thing)
  (typep thing 'standard-cell))

(defun standard-cell-p (thing)
  (typep thing 'standard-cell))

(defmethod print-object ((cell standard-cell) stream)
  (print-unreadable-object (cell stream :type t :identity t)
    (let ((object (cell-object cell)))
      (present object (presentation-type-of object) :stream stream))
    (with-bounding-rectangle* (left top right bottom) cell
      (format stream " /x ~D:~D y ~D:~D/" left right top bottom))))

(defmethod initialize-instance :before ((cell standard-cell)
					&key &allow-other-keys)
  ;; A cell is its own presentation type.
  (with-slots (clim::type) cell
    (setq clim::type (presentation-type-of cell))))

(defun make-cell (object &rest initargs
		  &key (class 'standard-cell) &allow-other-keys)
  ;; MAKE-INSTANCE is often faster when given a constant class.
  (if (eq class 'standard-cell)
      (make-instance 'standard-cell :object object)
      (apply #'make-instance class :object object :allow-other-keys t initargs)))


;;; Simple cell accessors.

(defmethod cell-size ((cell standard-cell))
  (declare (values width height))
  (bounding-rectangle-size cell))

(defmethod cell-column ((cell standard-cell))
  (output-record-parent cell))

(defmethod cell-column-id ((cell standard-cell))
  (column-id (cell-column cell)))

(defmethod cell-table ((cell standard-cell))
  (column-table (cell-column cell)))

(defmethod cell-row ((cell standard-cell))
  ;; This is a trifle slow, but the alternative is for every cell to
  ;; keep a pointer to the row, which I thought unnecessary.  This
  ;; function is not used during redisplay, so the lack of speed
  ;; shouldn't really matter.
  (let* ((column (cell-column cell))
	 (table (column-table column)))
    (do-visible-cells-and-rows (this-cell row column table)
      (when (eq this-cell cell)
	(return-from cell-row row)))
    ;; Maybe the row isn't visible.
    (dolist (row (table-rows table))
      (unless (row-visible-p row)
	(when (find cell (row-cells row) :test #'eq)
	  (return-from cell-row row)))))
  ;; Should never get here!
  (error "The cell ~S could not be found in any row!" cell))

(defmethod cell-object ((cell standard-cell))
  (presentation-object cell))

(defmethod cell-y-position ((cell standard-cell))
  (multiple-value-bind (x y)
      (output-record-position* cell)
    (declare (ignore x))
    y))

(defmethod (setf cell-y-position) (new-y (cell standard-cell))
  ;; NEW-Y is relative to the coordinate system of the table, but since
  ;; we know that the column is always at Y-position zero with respect
  ;; to the table, we don't need to add a Y-offset.  The cell itself is
  ;; always at X-position zero.
  (output-record-set-position* cell 0 new-y))


;;; Simple cell operations.

(defmethod erase-cell ((cell standard-cell) stream)
  (when (cell-displayed-p cell)
    (erase-region cell stream)
    (setf (cell-displayed-p cell) nil)))


;;; Finding cells from other pointers.

(defgeneric find-cell (object column-id table &optional errorp)
  (:documentation
    "Returns the cell at the intersection of the row associated with the
     given client object and the column having the given ID."))

(defmethod find-cell (object column-id (table standard-table)
		      &optional (errorp t))
  (find-cell (find-row object table errorp) column-id table errorp))

(defmethod find-cell (object (column standard-column) (table standard-table)
		      &optional (errorp t))
  (find-cell object (column-id column) table errorp))

(defmethod find-cell ((row standard-row) column-id (table standard-table)
		      &optional (errorp t))
  (or (find column-id (row-cells row) :key #'cell-column-id :test #'equal)
      (and errorp
	   (error "The table row ~S contains no cell whose ID is ~S."
		  row column-id))))


;;; Mapping upwards from a presentation to the cell in which it appears.

(defun presentation-cell (record)
  "If the given record is a descendant of a table cell, returns the cell,
   else returns NIL."
  (typecase record
    (standard-cell
      record)
    (output-record
      (presentation-cell (output-record-parent record)))
    (otherwise
      nil)))

(defun cell-presentation-p (record)
  "Returns true iff the given output record is part of the presentation of
   a table cell."
  (not (null (presentation-cell record))))

(defun cell-presentation-object (record)
  (let ((cell (presentation-cell record)))
    (if (null cell)
	(error "~S is not the descendant of a table cell." record)
	(cell-object cell))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Manipulating rows and columns.
;;;
;;; Note that the relative ordering of columns and rows is always
;;; maintained when a table is modified.


;;; Stops displaying the given row without removing it from the table.

(defgeneric hide-row (row-or-object table)
  (:documentation "Prohibits the given row from being displayed."))

(defmethod hide-row (row-object (table standard-table))
  (hide-row (find-row row-object table) table))

(defmethod hide-row ((row standard-row) (table standard-table))
  (when (row-visible-p row)
    ;; Can't hide the title row!
    (when (member row (table-object-rows table) :test #'eq)
      (erase-row row (table-stream table))
      (dolist (cell (row-cells row))
	(let ((column (output-record-parent cell)))
	  ;; Because one of our design goals is to allow the columns to
	  ;; move quickly in the horizontal direction, we must be able
	  ;; to call REPLAY on the column.  Therefore, we must actually
	  ;; remove the cells (output records) from their columns in
	  ;; order to hide them.  We save the parent pointer.
	  (delete-output-record-element column cell)
	  (setf (output-record-parent cell) column)))
      (setf (row-visible-p row) nil)
      (setf (table-needs-redisplay-p table) t))))


;;; Resumes displaying a previously hidden row.

(defgeneric show-row (row-or-object table)
  (:documentation "Resumes displaying a previously hidden row."))

(defmethod show-row (row-object (table standard-table))
  (show-row (find-row row-object table) table))

(defmethod show-row ((row standard-row) (table standard-table))
  (unless (row-visible-p row)
    (let ((index (position row (table-rows table) :test #'eq)))
      (dolist (cell (row-cells row))
	(add-nth-output-record (output-record-parent cell) cell index))
      (setf (row-visible-p row) t)
      (setf (row-needs-redisplay-p row) t)
      (setf (table-needs-redisplay-p table) t))))


;;; Stops displaying the given column without removing it from the
;;; table.

(defgeneric hide-column (column-or-id table)
  (:documentation "Prohibits the given column from being displayed."))

(defmethod hide-column (column-id (table standard-table))
  (hide-column (find-column column-id table) table))

(defmethod hide-column ((column standard-column) (table standard-table))
  (when (column-visible-p column)
    (when (member column (table-columns table) :test #'eq)
      (erase-column column (table-stream table))
      (setf (column-visible-p column) nil)
      (setf (table-needs-redisplay-p table) t))))


;;; Resumes displaying a previously hidden column.

(defgeneric show-column (column-or-id table)
  (:documentation "Resumes displaying a previously hidden column."))

(defmethod show-column (column-id (table standard-table))
  (show-column (find-column column-id table) table))

(defmethod show-column ((column standard-column) (table standard-table))
  (unless (column-visible-p column)
    (when (member column (table-columns table) :test #'eq)
      (setf (column-visible-p column) t)
      (dolist (cell (column-cells column))
	(setf (cell-needs-redisplay-p cell) t))
      (setf (column-needs-redisplay-p column) t)
      (setf (table-needs-redisplay-p table) t))))


;;; Adds the given row to the table.

(defgeneric add-row (row table)
  (:documentation "Adds the given row to the table."))

(defmethod add-row ((row standard-row) (table standard-table))
  (unless (member row (table-rows table) :test #'eq)
    (let ((predicate (table-sort-predicate table)))
      (register-object (row-object row) table row)
      (if (null predicate)
	  (push-end row (table-object-rows table))
	  (insertf row (table-object-rows table) predicate :key #'row-object))
      (add-cells row table)
      (setf (table-needs-redisplay-p table) t))))

(defmethod add-cells ((row standard-row) (table standard-table))
  (let ((object (row-object row))
	(row-visible-p (row-visible-p row))
	(index (position row (table-rows table)))
	(class (if (title-row-p row)
		   'title-cell
		   (table-cell-class table)))
	(cells '()))
    (when (null index)
      (error "The row ~S is not part of the table ~S!" row table))
    (dolist (column (table-columns table))
      (when (column-visible-p column)
	(let ((cell (make-cell object :class class)))
	  (when row-visible-p
	    (add-nth-output-record column cell index))
	  (push cell cells))))
    (setf (row-cells row) (nreverse cells))))


;;; Adds the given column to the table.

(defgeneric add-column (column table &key after)
  (:documentation
  "Adds the given column to the table.  If the column is currently part
   of the table, it is moved to the new position.

   The value of the keyword :AFTER determines where the new column is
   positioned with respect to existing columns.  The new column is added
   to the end of the list if the value is either :END or NIL, the
   default; to the beginning if the value is :START; or after a given
   column if the value is either an existing column or the ID of an
   existing column."))

(defmethod add-column (column-id (table standard-table) &key (after ':end))
  (add-column (find-column column-id table) table :after after))

(defmethod add-column ((column standard-column) (table standard-table)
		       &key (after ':end))
  (when (member column (table-columns table) :test #'eq)
    (delete-column column table))
  (cond ((eq after ':start)
	 (add-nth-output-record table column 0))
	((or (eq after ':end)
	     (null after))
	 (add-output-record-element table column))	;adds at end
	((member after (table-columns table) :test #'eq)
	 (add-nth-output-record
	   table
	   column
	   (1+ (position after (table-columns table) :test #'eq))))
	(t
	 (setq after (find-column after table t))	;ERRORP = T
	 ;; We can only get here if the column exists.
	 (add-nth-output-record
	   table
	   column
	   (1+ (position after (table-columns table) :test #'eq)))))
  (add-cells column table)
  (setf (table-needs-redisplay-p table) t))

(defmethod add-cells ((column standard-column) (table standard-table))
  (let ((class (table-cell-class table))
	(index (position column (table-columns table))))
    (when (null index)
      (error "The column ~S is not part of the table ~S!" column table))
    (dolist (row (table-rows table))
      (let ((cell (make-cell (row-object row) :class class)))
	(when (row-visible-p row)
	  (add-output-record-element column cell))	;adds at end!
	(insertf-nth cell (row-cells row) index)))))


;;; Adds the given child to the given record at the given index,
;;; allowing the children to be ordered.
;;;
;;; This definition depends upon internal details of the structure of
;;; the class CLIM::LINEAR-OUTPUT-RECORD.  It was developed from the
;;; source code of CLIM 1.1 under Genera 8.1, but also works under those
;;; implementations listed at the top of this file.

(defmethod add-nth-output-record ((record clim::linear-output-record) child index)
  "Adds CHILD to RECORD as the INDEX'th child."
  (declare (fixnum index))
  (unless (<= 0 index (the fixnum (clim::output-record-element-count record)))
    (error "~S is not a valid index within ~S." index record))
  ;; We must call ADD-OUTPUT-RECORD-ELEMENT so that all the :BEFORE and
  ;; :AFTER daemons will fire, and then fix up the ordering ourselves.
  ;; Ah, the wonders of object-oriented design!
  (add-output-record-element record child)
  (with-slots (clim::elements clim::fill-pointer) record
    ;; ELEMENTS is either the child itself or an array of children.  Of
    ;; course, a single child by itself doesn't need to be ordered.
    (when (arrayp clim::elements)
      ;; FILL-POINTER points just past the final child, which is the one
      ;; that was just added.  Move everybody one slot towards the end
      ;; to make room for the INDEX'th element.
      (let ((vector clim::elements))
	(declare (type simple-vector vector))
	#+Genera (declare (sys:array-register vector))
	(do ((i (the fixnum (1- (the fixnum clim::fill-pointer))) (1- i)))
	    ((= i index))
	  (declare (fixnum i) (cl:optimize (speed 3) (safety 0)))
	  (setf (svref vector i) (svref vector (1- i))))
	(setf (svref vector index) child)))))


;;; Removes the given row from the table.

(defgeneric delete-row (row-or-object table)
  (:documentation "Deletes the given row from the table."))

(defmethod delete-row (object (table standard-table))
  (delete-row (find-row object table) table))

(defmethod delete-row ((row standard-row) (table standard-table))
  ;; Can't delete the title row!
  (let ((rows (table-object-rows table)))
    (when (member row rows :test #'eq)
      (when (row-visible-p row)
	(erase-row row (table-stream table))
	(dolist (cell (row-cells row))
	  (delete-output-record-element (output-record-parent cell) cell)))
      (unregister-object (row-object row) table)
      (setf (table-object-rows table) (delete row rows))
      (setf (table-needs-redisplay-p table) t))))


;;; Removes the given column from the table.

(defgeneric delete-column (column-or-id table)
  (:documentation "Deletes the given column from the table."))

(defmethod delete-column (column-id (table standard-table))
  (delete-column (find-column column-id table) table))

(defmethod delete-column ((column standard-column) (table standard-table))
  (let ((index (position column (table-columns table))))
    (unless (null index)
      (erase-column column (table-stream table))
      (delete-output-record-element table column)
      (clear-output-record column)
      (dolist (row (table-rows table))
	(deletef-nth (row-cells row) index))
      (setf (table-needs-redisplay-p table) t))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Presentation types and commands.
;;;
;;; We don't use DEFINE-PRESENTATION-TO-COMMAND-TRANSLATOR for many of
;;; the following commands because that macro doesn't allow you to
;;; return multiple values, which we use so that we can include the
;;; option '(:ECHO NIL) as the third value returned.


;;; A table is presented using its name.

(define-presentation-type standard-table ()
  :history nil)

(define-presentation-method present ((table standard-table) (type standard-table)
				     stream view
				     &key &allow-other-keys)
  (declare (ignore view))
  (write-string (table-name table) stream))


;;; A row is presented using its associated object.

(define-presentation-type standard-row ()
  :history nil)

(define-presentation-method present ((row standard-row) (type standard-row)
				     stream view
				     &key acceptably for-context-type)
  (let ((object (row-object row)))
    (present object (presentation-type-of object)
	     :stream stream
	     :view view
	     :acceptably acceptably
	     :for-context-type for-context-type)))


;;; A column is presented using its ID and title presentation type.

(define-presentation-type standard-column ()
  :history nil)

(define-presentation-method present ((column standard-column)
				     (type standard-column)
				     stream view
				     &key acceptably for-context-type)
  (present (column-id column) (column-title-presentation-type column)
	   :stream stream
	   :view view
	   :acceptably acceptably
	   :for-context-type for-context-type))


;;; A cell is presented using the object of the row in which it appears
;;; and the presentation type of the column in which it appears.

(define-presentation-type standard-cell ()
  :history nil)

(define-presentation-method present ((cell standard-cell) (type standard-cell)
				     stream view
				     &key acceptably for-context-type)
  (present (cell-object cell)
	   (column-cell-presentation-type (cell-column cell))
	   :stream stream
	   :view view
	   :acceptably acceptably
	   :for-context-type for-context-type))


;;; By default, column titles are presented using this presentation
;;; type.

(define-presentation-type standard-column-title ())

(define-presentation-method present (object (type standard-column-title)
				     stream view
				     &key &allow-other-keys)
  (declare (ignore view))
  (with-text-face (:italic stream)
    (typecase object
      (string
	(write-string object stream))
      (symbol
	(format stream "~:(~A~)" object))
      (otherwise
	(present object (presentation-type-of object) :stream stream)))))

(define-presentation-method presentation-typep (object (type standard-column-title))
  (declare (ignore object))
  t)


;;; The command table TABLE contains commands and translators for
;;; table-related operations.  If you wish to use these commands, your
;;; top-level command table must inherit from TABLE.

(eval-when (:load-toplevel :execute :compile-toplevel)
  (define-command-table-and-definer table :inherit-from (global-command-table)))


;;; Hide the indicated row.

(define-table-command com-hide-row
    ((row 'standard-row)
     (table 'standard-table))
  (hide-row row table))

(define-presentation-translator hide-row
    (standard-table command table
      :gesture nil				;only in menu
      :documentation ((object stream)
		      (write-string "Hide a row of " stream)
		      (present object (presentation-type-of object)
			       :stream stream
			       :sensitive nil)))
    (object window)
  (let ((row (menu-choose-row object
			      :test #'row-visible-p
			      :stream window)))
    (values `(com-hide-row ,row ,object)
	    '(command :command-table table)
	    '(:echo nil))))


;;; Show one or more previously-hidden rows.

(define-table-command com-show-row
    ((row 'standard-row)
     (table 'standard-table))
  (show-row row table))

(define-presentation-translator show-row
    (standard-table command table
      :gesture nil				;only in menu
      :tester ((object)
	       (and (tablep object)
		    (table-has-hidden-rows-p object)))
      :tester-definitive t
      :documentation ((object stream)
		      (write-string "Show a hidden row of " stream)
		      (present object (presentation-type-of object)
			       :stream stream
			       :sensitive nil)))
    (object window)
  (let ((row (menu-choose-row object
			      :test-not #'row-visible-p
			      :stream window)))
    (values `(com-show-row ,row ,object)
	    '(command :command-table table)
	    '(:echo nil))))

(define-table-command com-show-all-rows
    ((table 'standard-table))
  (dolist (row (table-object-rows table))
    (unless (row-visible-p row)
      (show-row row table))))

(define-presentation-translator show-all-rows
    (standard-table command table
      :gesture nil				;only in menu
      :tester ((object)
	       (and (tablep object)
		    (table-has-hidden-rows-p object)))
      :tester-definitive t
      :documentation ((object stream)
		      (write-string "Show all hidden rows of " stream)
		      (present object (presentation-type-of object)
			       :stream stream
			       :sensitive nil)))
    (object)
  (values `(com-show-all-rows ,object)
	  '(command :command-table table)
	  '(:echo nil)))


;;; Hide the indicated column.

(define-table-command com-hide-column
    ((column 'standard-column)
     (table 'standard-table))
  (hide-column column table))

(define-presentation-translator hide-column
    (standard-column command table
      :gesture nil				;only in menu
      :tester ((object)
	       ;; Can't hide the last column!
	       (let ((table (column-table object)))
		 (> (length (table-columns table)) 1)))
      :tester-definitive t
      :documentation ((object stream)
		      (write-string "Hide column " stream)
		      (present object (presentation-type-of object)
			       :stream stream
			       :sensitive nil)))
    (object)
  (values `(com-hide-column ,object ,(column-table object))
	  '(command :command-table table)
	  '(:echo nil)))

(define-presentation-translator menu-hide-column
    (standard-table command table
      :gesture nil				;only in menu
      :tester ((object)
	       ;; Can't hide the last column!
	       (> (length (table-columns object)) 1))
      :tester-definitive t
      :documentation ((object stream)
		      (write-string "Hide a column of " stream)
		      (present object (presentation-type-of object)
			       :stream stream
			       :sensitive nil)))
    (object window)
  (let ((column (menu-choose-column object
				    :test #'column-visible-p
				    :stream window)))
    (values `(com-hide-column ,column ,object)
	    '(command :command-table table)
	    '(:echo nil))))


;;; Show one or more previously-hidden columns.

(define-table-command com-show-column
    ((column 'standard-column)
     (table 'standard-table))
  (show-column column table))

(define-presentation-translator show-column
    (standard-table command table
      :gesture nil				;only in menu
      :tester ((object)
	       (and (tablep object)
		    (table-has-hidden-columns-p object)))
      :tester-definitive t
      :documentation ((object stream)
		      (write-string "Show a hidden column of " stream)
		      (present object (presentation-type-of object)
			       :stream stream
			       :sensitive nil)))
    (object window)
  (let ((column (menu-choose-column object
				    :test-not #'column-visible-p
				    :stream window)))
    (values `(com-show-column ,column ,object)
	    '(command :command-table table)
	    '(:echo nil))))

(define-table-command com-show-all-columns
    ((table 'standard-table))
  (dolist (column (table-columns table))
    (unless (column-visible-p column)
      (show-column column table))))

(define-presentation-translator show-all-columns
    (standard-table command table
      :gesture nil				;only in menu
      :tester ((object)
	       (and (tablep object)
		    (table-has-hidden-columns-p object)))
      :tester-definitive t
      :documentation ((object stream)
		      (write-string "Show all hidden columns of " stream)
		      (present object (presentation-type-of object)
			       :stream stream
			       :sensitive nil)))
    (object)
  (values `(com-show-all-columns ,object)
	  '(command :command-table table)
	  '(:echo nil)))


;;; Choosing table components from a dialog.

(defun menu-choose-column (table &key test test-not (stream *standard-output*))
  (let* ((columns (cond (test-not
			 (remove-if test-not (table-columns table)))
			(test
			 (remove-if-not test (table-columns table)))
			(t
			 (table-columns table))))
	 (column (and columns
		      (menu-choose columns
				   :label "Choose a column:"
				   :presentation-type 'standard-column
				   :associated-window stream))))
    (if (null column)
	(abort)
	column)))

(defun menu-choose-row (table &key test test-not (stream *standard-output*))
  (let* ((rows (cond (test-not
		      (remove-if test-not (table-object-rows table)))
		     (test
		      (remove-if-not test (table-object-rows table)))
		     (t
		      (table-object-rows table))))
	 (row (and rows
		   (menu-choose rows
				:label "Choose a row:"
				:presentation-type 'standard-row
				:associated-window stream))))
    (if (null row)
	(abort)
	row)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; (Re)displaying tables.


;;; Redisplays the table upon the given stream.  We take care to avoid
;;; errors even if the table has been moved to a different stream or has
;;; been damaged somehow.

(defgeneric redisplay-table (table stream)
  (:documentation "Displays or redisplays TABLE on the given stream."))

(defmethod redisplay-table ((table standard-table) stream)
  (let ((output-history (output-recording-stream-output-record stream)))
    (cond ((not (eq stream (table-stream table)))
	   (let ((parent (output-record-parent table)))
	     (cond ((null parent)
		    ;; The table has not yet been displayed.
		    (setf (table-stream table) stream))
		   (t
		    ;; The table has been moved to a different stream.
		    (delete-output-record-element parent table)
		    (reinitialize-table table stream)))))
	  ((output-record-contains-child-p output-history table)
	   ;; Things are fine:  do nothing.
	   )
	  ((eq (output-record-parent table) output-history)
	   ;; Something (possibly WINDOW-CLEAR) has deleted the table
	   ;; from the stream, but we can just reattach and replay it,
	   ;; since we know there's nothing between the table and the
	   ;; history.
	   (add-output-record-element output-history table)
	   (replay table stream))
	  (t
	   ;; If all else fails, rebuild the table.
	   (reinitialize-table table stream)))
    (when (table-needs-redisplay-p table)
      (recompute-table-size table)
      (replay-table table)
      (clim::redisplay-decorations (table-stream table))	;scroll bars
      (setf (table-needs-redisplay-p table) nil))))


;;; Returns non-NIL iff RECORD contains the CHILD output record at some
;;; level.  (We have to map downward instead of upward because the
;;; function WINDOW-CLEAR removes the downward pointer but *not* the
;;; upward pointer.)

(defun output-record-contains-child-p (record child)
  (labels ((find-child (candidate)
	     (if (eq candidate child)
		 (return-from output-record-contains-child-p child)
		 (map-over-output-record-elements candidate #'find-child))))
    (declare (dynamic-extent #'find-child))
    (find-child record)
    nil))					;search failed


;;; Traverses the table, creating new presentations for all changed
;;; table cells, and computing the new size of the table.

(defmethod recompute-table-size ((table standard-table))
  (multiple-value-bind (left top) (output-record-position* table)
    (let* ((width (recompute-table-width table left))
	   (height (recompute-table-height table top width)))
      (table-set-size table width height))))


;;; Sets the size of the table's output record, then propagates the new
;;; size up through ancestor output records.

(defmethod table-set-size ((table standard-table) width height)
  (let ((parent (output-record-parent table)))
    (with-bounding-rectangle* (old-left old-top old-right old-bottom) table
      (clim-utils:bounding-rectangle-set-size table width height)
      (unless (null parent)
	(multiple-value-bind (x-offset y-offset)
	    (convert-from-descendant-to-ancestor-coordinates table parent)
	  (declare (fixnum x-offset y-offset))
	  (clim::recompute-extent-for-changed-element
	    parent table
	    (the fixnum (+ old-left x-offset))
	    (the fixnum (+ old-top y-offset))
	    (the fixnum (+ old-right x-offset))
	    (the fixnum (+ old-bottom y-offset))))))))


;;; Iterates over the columns of the table, calculating the new extent
;;; of each column.

(defmethod recompute-table-width ((table standard-table) left)
  (let ((right left)
	(x-spacing (table-real-x-spacing table))
	(stream (table-stream table)))
    (declare (fixnum right x-spacing))
    (dolist (row (table-rows table))
      (setf (row-height row) 0))
    (dolist (column (table-columns table))
      (when (column-visible-p column)
	(unless (= (the fixnum (column-x-position column)) right)
	  (erase-column column stream)
	  (setf (column-x-position column) right)
	  (setf (column-needs-redisplay-p column) t))
	(let ((column-width (recompute-column-width column table)))
	  (declare (fixnum column-width))
	  (setf (column-width column) column-width)
	  (setq right (the fixnum (+ right column-width x-spacing))))))
    ;; Return the final X coordinate, corrected for overshoot.
    (the fixnum (- right x-spacing))))


;;; Iterates over the cells of the column, creating new presentations
;;; when necessary, and computing column width and row height.

(defmethod recompute-column-width ((column standard-column) (table standard-table))
  (let ((column-width 0)
	(stream (table-stream table)))
    (declare (fixnum column-width))
 (unless (= (length (table-rows table))
	    (length (column-cells column))))
    (do-visible-cells-and-rows (cell row column table)
      (when (cell-needs-redisplay-p cell)
	(erase-cell cell stream)
	(present-cell cell row column stream)
	(unless (= (the fixnum (cell-y-position cell))
		   (the fixnum (row-y-position row)))
	  (setf (cell-y-position cell) (row-y-position row))))
      (multiple-value-bind (cell-width cell-height)
	  (cell-size cell)
	(declare (fixnum cell-height cell-width))
	(when (> cell-width column-width)
	  (setq column-width cell-width))
	(when (> cell-height (the fixnum (row-height row)))
	  (setf (row-height row) cell-height))))
    column-width))


;;; Invisibly presents the given cell on the given stream.

(defmethod present-cell ((cell standard-cell) (row standard-row)
			 (column standard-column) stream)
  (let ((value (if (title-row-p row)
		   (column-id column)
		   (funcall (column-attribute-reader column) (row-object row))))
	(presentation-type (if (title-row-p row)
			       (column-title-presentation-type column)
			       (column-cell-presentation-type column))))
    (clear-output-record cell)
    (with-end-of-line-action (:allow stream)
      (with-end-of-page-action (:allow stream)
	(with-output-recording-options (stream :draw-p nil)
	  ;; This prevents the start positions of various output records
	  ;; from getting out of synch.
	  (with-stream-cursor-position-at-origin (stream)
	    ;; This is equivalent to calling PRESENT, except that the
	    ;; macro WITH-OUTPUT-AS-PRESENTATION accepts the keyword
	    ;; :PARENT, which PRESENT does not.  Specifying :PARENT CELL
	    ;; causes the new presentation to be added as a child of the
	    ;; cell output record.
	    (with-output-as-presentation (:object value
					  :type presentation-type
					  :stream stream
					  :parent cell
					  :single-box t)
	      (call-presentation-generic-function present
		value
		presentation-type
		stream
		(stream-default-view stream)
		:acceptably nil
		:for-context-type presentation-type))))))))


;;; Iterates over the cells in each row, creating new presentations when
;;; necessary, and calculating the new extent of each row.

(defmethod recompute-table-height ((table standard-table) top width)
  (declare (fixnum top width))
  (let ((bottom top)
	(y-spacing (table-real-y-spacing table))
	(stream (table-stream table)))
    (declare (fixnum bottom y-spacing))
    (dolist (row (table-rows table))
      (when (row-visible-p row)
	(unless (= (the fixnum (row-y-position row)) bottom)
	  (erase-row row stream)
	  (setf (row-y-position row) bottom)
	  (setf (row-needs-redisplay-p row) t))
	(setf (row-width row) width)
	(setq bottom (the fixnum (+ bottom
				    (the fixnum (row-height row))
				    y-spacing)))))
    ;; Correct for overshoot.
    (setq bottom (the fixnum (- bottom y-spacing)))
    (let ((height (the fixnum (- bottom top))))
      (dolist (column (table-columns table))
	(setf (column-height column) height)))
    bottom))


;;; Replays the changed or moved output records of the table.

(defmethod replay-table ((table standard-table))
  (let ((stream (table-stream table)))
    (dolist (column (table-columns table))
      (when (column-visible-p column)
	(if (column-needs-redisplay-p column)
	    ;; Redisplay the entire column.
	    (replay-column column stream)
	    ;; Redisplay changed cells only.
	    (dolist (cell (column-cells column))
	      (replay-cell cell stream)))))))

(defmethod replay-column ((column standard-column) stream)
  (replay column stream)
  (dolist (cell (column-cells column))
    (setf (cell-displayed-p cell) t)
    (setf (cell-needs-redisplay-p cell) nil))
  (setf (column-displayed-p column) t)
  (setf (column-needs-redisplay-p column) nil))

(defmethod replay-cell ((cell standard-cell) stream)
  (replay cell stream)
  (setf (cell-displayed-p cell) t)
  (setf (cell-needs-redisplay-p cell) nil))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Editing cells.
;;;
;;; The function EDIT-CELL and its associated command COM-EDIT-CELL
;;; allow a table to be used as an input device.

(defgeneric edit-cell (cell &key own-window insert-default)
  (declare (values new-value object)))

(defmethod edit-cell ((cell standard-cell) &key own-window insert-default)
  "Edits the value displayed in CELL, modifying the associated client
   object to contain the new value.

   If :OWN-WINDOW is true, then an ACCEPTING-VALUES dialog is used;
   otherwise the cell is edited in place.

   If :INSERT-DEFAULT is true, then the current value is inserted into
   the text field before editing begins so that it may be modified."
  (declare (values new-value object))
  (let* ((object (cell-object cell))
	 (column (cell-column cell))
	 (presentation-type (column-cell-presentation-type column))
	 (table (column-table column))
	 (stream (table-stream table))
	 (old-value (funcall (column-attribute-reader column) object))
	 (new-value
	   (if own-window
	       (accept-cell-value-dialog old-value presentation-type stream)
	       (accept-cell-value cell old-value insert-default
				  presentation-type stream))))
    (funcall (column-attribute-writer column) new-value object)
    (values new-value object)))

(defmethod accept-cell-value ((cell standard-cell) default insert-default
			      presentation-type stream)
  "Accepts a cell value in place."
  (with-stream-cursor-position-at-record (stream cell)
    (erase-region cell stream)
    (with-input-focus (stream)
      (with-output-recording-options (stream :record-p nil)
	(accept presentation-type
		:stream stream
		:default default
		:insert-default insert-default
		:prompt nil)))))

(defun accept-cell-value-dialog (default presentation-type stream)
  "Accepts a cell value within an ACCEPTING-VALUES dialog."
  (accepting-values (stream :own-window t :label "Edit cell")
    (fresh-line stream)
    (setq default (accept presentation-type
			  :stream stream
			  :default default
			  :prompt "New value")))
  default)


;;; An alternate syntax for invoking editing, this function uses the
;;; hooks visible to the client program, i.e., the column ID, the row
;;; object, and the table itself.

(defgeneric edit-attribute (object column-id table
			    &key own-window insert-default)
  (declare (values new-value object))
  (:documentation
    "Edits the value displayed in the cell at the intersection of the
     row associated with OBJECT and the column having the given ID,
     modifying the object to contain the new value.
    
     For more information, see the documentation for EDIT-CELL."))

(defmethod edit-attribute (object column-id (table standard-table)
			   &key own-window insert-default)
  (declare (values new-value object))
  (edit-cell (find-cell object column-id table)
	     :own-window own-window
	     :insert-default insert-default))


;;; Command interface.
;;;
;;; We don't actually enable this by default, because some editable
;;; cells may need special treatment.  However, specialized translators
;;; may be based on this template.

(define-command (com-edit-cell :command-table table)
    ((cell 'standard-cell)
     &key
     (own-window 'boolean :default nil)
     (insert-default 'boolean :default nil))
  (edit-cell cell :own-window own-window :insert-default insert-default))

#||
(define-presentation-translator edit-cell
    (standard-cell command table
      :gesture :select
      :menu t
      :tester ((presentation)
	       (and (cell-presentation-p presentation)
		    (column-attribute-writer
		      (cell-column (presentation-cell presentation)))))
      :tester-definitive t
      :documentation "Edit cell"
      :pointer-documentation "Edit cell")
    (presentation)
  (values `(com-edit-cell ,(presentation-cell presentation))
	  '(command :command-table table)
	  '(:echo nil)))
||#



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Mapping objects to the tables and rows in which they appear.
;;;
;;; Whenever a new row is created, a mapping is created from the user
;;; object to the table and row.  This mapping is used by SETF :AFTER
;;; daemons to invalidate the relevant table cell displays whenever the
;;; the user object is modified.

(defvar *table-object-registry* (make-hash-table)
  "Maps each object to the list of table/row pairs in which it appears.")

(defun clear-object-registry ()
  (clrhash *table-object-registry*))

(defun register-object (object table row)
  (let ((alist (gethash object *table-object-registry*)))
    (unless (assoc table alist)
      (setf (gethash object *table-object-registry*)
	    (acons table row alist)))))

(defun unregister-object (object table)
  (let ((alist (gethash object *table-object-registry*)))
    (when (assoc table alist)
      (setq alist (delete table alist :key #'car))
      (if (null alist)
	  (remhash object *table-object-registry*)
	  (setf (gethash object *table-object-registry*) alist)))))

(defun object-table-row-alist (object)
  (values (gethash object *table-object-registry*)))


;;; Invalidates the relevant table cell whenever the given object is
;;; modified.

(defun invalidate-cell (object column-id)
  "Iterates over all tables in which OBJECT appears, invalidating the
   appropriate table cells so that their output will update during the
   next call to REDISPLAY-TABLE."
  (dolist (entry (object-table-row-alist object))
    (let ((table (car entry))
          (row (cdr entry)))
      (setf (cell-needs-redisplay-p (find-cell row column-id table)) t)
      (setf (table-needs-redisplay-p table) t))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Object caching tables.

;;; This subclass of tables caches a copy of the list of row objects
;;; being displayed.  Passing a list of row objects to the method
;;; UPDATE-TABLE-OBJECT-CACHE causes the cache to be updated, adding and
;;; deleting rows as necessary.  This approach is slower than using
;;; daemons to drive the methods ADD-ROW and DELETE-ROW, but is
;;; sometimes necessitated by the design of the client program.
;;;
;;; IMPORTANT:  When using a table of this type, never call ADD-ROW or
;;; DELETE-ROW directly!  Doing so will break the table.  Instead, call
;;; UPDATE-TABLE-OBJECT-CACHE, passing it the new list of row objects.
;;; (Calling CLEAR-TABLE is still valid.)

(defclass object-caching-table (standard-table)
    ((object-cache
       :accessor table-object-cache
       :type list
       :documentation "The cached list of row objects.")
     (cache-test
       :reader table-cache-test			;immutable
       :initarg :cache-test
       :documentation "The function used to compare new and cached objects.")
     (cache-equal
       :reader table-cache-equal			;immutable
       :documentation "The function used to compare entire cached lists."))
  (:default-initargs :cache-test #'eql))

(defmethod initialize-instance :after ((table object-caching-table)
				       &key &allow-other-keys)
  (with-slots (object-cache cache-test cache-equal) table
    (check-type cache-test (or function (and symbol (satisfies fboundp)))
		"a function or a symbol with a functional value")
    (if (eq cache-test #'eql)
	(setq cache-equal #'equal)
	(flet ((cache-equal (old-list new-list)
		 (and (= (length old-list) (length new-list))
		      (every cache-test old-list new-list))))
	  (setq cache-equal #'cache-equal)))
    (setq object-cache (mapcar #'row-object (table-object-rows table)))))

(defmethod update-table-object-cache ((table object-caching-table) new-objects)
  (let ((old-objects (table-object-cache table)))
    (when (null (funcall (table-cache-equal table) old-objects new-objects))
      (let ((cache-test (table-cache-test table)))
	(setf (table-object-cache table) (copy-list new-objects))
	(dolist (old-object old-objects)
	  (unless (find old-object new-objects :test cache-test)
	    (delete-row (find-row old-object table) table)))
	(dolist (new-object new-objects)
	  (unless (find new-object old-objects :test cache-test)
	    (add-row (make-row new-object) table)))
	(setf (table-needs-redisplay-p table) t)))))

(defmethod clear-table :after ((table object-caching-table))
  (setf (table-object-cache table) '()))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Miscellany.


;;; Writes the table to the given stream with the columns delimited by
;;; the given separator.  When written to a file, this allows the user
;;; to import data into spreadsheets and word processors.

(defmethod write-table ((table standard-table)
			&key (stream *standard-output*)
			     (separator #\Tab))
  (flet ((present-safely (object presentation-type)
	   ;; Use a default printer if the presentation method dies.
	   (handler-case (present object presentation-type :stream stream)
	     (error ()
		    (format stream "~:(~A~)" object)))))
    (declare (dynamic-extent #'present-safely))
    (let ((columns (table-columns table))
	  (need-separator-p nil))
      (fresh-line stream)
      (dolist (column columns)
	(if need-separator-p
	    (write-char separator stream)
	    (setq need-separator-p t))
	(present-safely (column-id column)
			(column-title-presentation-type column)))
      (dolist (row (table-object-rows table))
	(let ((object (row-object row)))
	  (setq need-separator-p nil)
	  (terpri stream)
	  (dolist (column columns)
	    (if need-separator-p
		(write-char separator stream)
		(setq need-separator-p t))
	    (present-safely (funcall (column-attribute-reader column) object)
			    (column-cell-presentation-type column)))))
      (values))))
