;;; -*- Syntax: Common-lisp; Mode: Lisp; Package: User -*-

(in-package :user)

#|
This package provides the ability to define the layout of menu items
in a command menu pane.  Defining a layout spec of the form:

   (("col1row1" "col1row2" ...) ("col2row1" "col2row2" ...) ...)

produces a menu item layout of the form:

col1row1    col2row1
col1row2    col2row2

Each enty in the layout is the menu items "name" as defined by the
:menu option to clim:define-command or the command definer macro produced
by clim:define-application-frame.

See the example application frame included in this file.

This package has been tested on:

Allegro CL 4.1
CLIM 1.1
SunOS 4.1.1

To use this package:

  1.  Include ordered-menu-item-mixin as a superclass of your
      application frame.
  2.  Provide a place to store the definition of the menu layout
      and sorted list of menu items.  I usually do this as
      slots in my application frame.
  3.  Add code in the top level of the application frame to compute
      a sorted list of menu items per the layout specification.
  4.  You're done.

Ideas on enhancements.

  o  extend layout syntax to allow style specifications (such as
     the menu items text style, highlighting mechanisms, etc.)
  o  define column (row?) "headings" that are not mouse sensitive
  o  deal with "dividers"
  o  define a mechanism for "icon" menu items

Provided for your enjoyment by:

Brian H. Anderson
Boeing Commercial Airplane
(206) 234-0881
bha@gumby.boeing.com

No warrenty, expressed or implied.  No commitments by the Boeing Company.
|#

;;; ----------------------------------------------------------------------
;;;
;;; Hack at being able to specify the layout of command menu items.
;;;
;;; ----------------------------------------------------------------------

;;; Specialized command menu item displayer.

(defclass ordered-menu-item-mixin
    ()
    
    ()
  )

;;; This takes a layout spec:
;;;
;;;   ((col1row1 col1row2 ...) (col2row1 col2row2 ...) ...)
;;;
;;; and computes a likewise ordered list of menu items for the displayer.

(defmethod layout-menu-items ((frame ordered-menu-item-mixin) menu-layout command-table)
  ;; This needs to check for unspecified commands and place them in a column!
  (let ((menu (slot-value command-table 'clim::menu)))
    
    (order-menu-items frame menu-layout menu)))

(defmethod order-menu-items ((frame ordered-menu-item-mixin) menu-layout menu)
  "Returns an ordered list of menu items from \"menu\" in column major format\
according to \"menu-layout\".\
(Ignores bogus menu item names in menu-layout)."
  (let ((remaining-menu-items (loop for menu-item across menu collect menu-item))
	ordered-menu-items)
    
    (setf ordered-menu-items
      (loop for column-name-list in menu-layout
	  for column-menu-item = (loop for row-name in column-name-list
				     for row-menu-item = (find row-name menu
							       :test #'(lambda (item test-item)
									 (string-equal (first test-item) item)))
				     unless row-menu-item
				     do
				       (format t "~&Unknown row menu name: ~s (ignoring).~&" row-name)
				       
				     ;; Remove processed menu item.
				     when row-menu-item
				     do
				       (setf remaining-menu-items
					 (delete row-menu-item remaining-menu-items :test #'equal))
				       
				     ;; Don't collect null row items (from invalid row names).
				     when row-menu-item collect row-menu-item)

	  ;; Don't collect null columns (from invalid column name lists).
	  when column-menu-item collect column-menu-item)
    )

    (cond (remaining-menu-items
	   ;; Append the menu items not found in the layout specification so that
	   ;; they are placed in the last column.
	   (append ordered-menu-items (list remaining-menu-items)))
	  (t
	   ordered-menu-items))))
  
;;; A display function suitable for :command-menu panes.
;;; Assumes that menu items have been ordered by order-menu-items in column major format.
(defmethod display-ordered-menu ((frame ordered-menu-item-mixin) stream
								 &rest keys
								 &key
								 ordered-menu-items
								 command-table
								 (cell-align-x ':left) (cell-align-y ':top)
								 (equalize-column-widths nil)
								 &allow-other-keys)
  (declare (ignore keys))
  
  (let ((the-command-table
	 (or command-table
	     (clim:find-command-table (clim:frame-command-table frame))))

	menu)

    ;; The menu descriptions from the command table.
    (setf menu (slot-value the-command-table 'clim::menu))
    
    ;; The list of menu items.
    ;; Maybe a form to eval.
    (when (listp ordered-menu-items)
      (setf ordered-menu-items (eval ordered-menu-items)))
    
    (cond ((zerop (count-if #'(lambda (x) (not (null (first x)))) menu))
	   (with-text-face (:italic stream)
	     (write-string "[No menu items]" stream)))

	  (t
	   
	   (clim:indenting-output (stream '(2 :character))
	      (clim:formatting-table (stream
				      :equalize-column-widths equalize-column-widths
				      ;; This needs to be a function of the available width
				      ;; while insuring some minimum spacing.
				      :inter-column-spacing '(2 :character)
				      )

		 (loop for column in ordered-menu-items
		     do
		       (clim:formatting-column (stream)
			  (loop for row in column
			      do
				(clim:formatting-cell (stream
						       :align-x cell-align-x
						       :align-y cell-align-y)
				   (clim:with-text-face (:bold stream)
				     (clim:present row
						   'clim::command-menu-element
						   :stream stream
						   :single-box t))))))))))))

;;; ----------------------------------------------------------------------

;;;
;;; Now an example...
;;;

(clim:define-application-frame test-menu

    ;; Inherits from...
    (ordered-menu-item-mixin
     clim:application-frame)

  ;; Application slots.
  ((menu-layout
    :documentation "Layout specification for menu items."
    :accessor ordered-menu-layout
    :initform '(("Clear" "Quit")
		("Redisplay")
		("Graph Class")))

   (menu-items
    :documentation "Holder of ordered list of menu items for \"display-ordered-menu\"."
    :accessor ordered-menu-items
    :initform nil)
   )

  (:panes (
	   (menu :command-menu
		 :display-function
		 '(display-ordered-menu
		   :ordered-menu-items (ordered-menu-items clim:*application-frame*))
		 :stream-background clim:+antique-white+
		 :stream-foreground clim:+black+)
	   (interactor :interactor
		       :scroll-bars :both
		       :stream-background clim:+antique-white+
		       :stream-foreground clim:+black+)
	   (mouse-line :pointer-documentation
		       :stream-background clim:+antique-white+
		       :stream-foreground clim:+black+)
	   ))
  (:layout
   ((default
	(:column 1
		 (menu :compute)
		 (interactor :rest)
		 (mouse-line :compute)
		 ))
    ))					;end :layout


  (:command-table (test-menu-command-table :inherit-from (clim:user-command-table)))
  (:command-definer t)
  (:top-level (test-menu-top-level))
  )


;;; ----------

;;;
;;; Commands.
;;;

(define-test-menu-command (com-clear
			   :name t
			   :menu t)

    ()

  (clim:window-clear (interactor-pane clim:*application-frame*)))

(define-test-menu-command (com-redisplay
			   :name t
			   :menu t)
    ()

  (clim:redisplay-frame-panes clim:*application-frame* :force-p t))

(define-test-menu-command (com-quit
			   :name t
			   :menu t)
    ()

  (clim:frame-exit clim:*application-frame*))

(define-test-menu-command (com-graph-class
			   :name t
			   :menu t)
    
    ((class '(or symbol clos::standard-class)
	    :prompt "A Class"
	    :provide-default t)
     &key
     (kind '(clim:member-alist
	     (("Superclasses" :value :supers)
	      ("Subclasses" :value :subs)))
	   :default :supers
	   :prompt "Class superclasses or subclasses"
	   :provide-default t)
     (orientation '(clim:member-alist
		    (("Horizontal" :value :horizontal)
		     ("Vertical" :value :vertical)))
		  :default :horizontal
		  :provide-default t
		  :prompt "Graph Orientation"))

  
  (when (symbolp class)
    (setf class (clos::find-class class)))

  (let ((stream *standard-output*))
    (clim:format-graph-from-root class
				 #'(lambda (class stream)
				     (clim:surrounding-output-with-border
				      (stream :shape :rectangle)
				    
				      (clim:with-output-as-presentation (:object class
										 :stream stream
										 :single-box t)
					(format stream "~s" (slot-value class 'clos::name)))))
				 (ecase kind
				   (:subs #'clos::class-direct-subclasses)
				   (:supers #'clos::class-direct-superclasses))
				 :orientation orientation
				 :stream stream)))
  
;;; ----------------------------------------------------------------------

;;;
;;; Frame methods.
;;;

(defmethod interactor-pane ((frame test-menu))
  (clim:get-frame-pane frame 'interactor))

;;; Specialized application top level for ordered menu items.
(defmethod test-menu-top-level ((frame test-menu) &rest args)

  ;; Compute user layout of menu items from the layout specification.
  (setf (ordered-menu-items frame)
    (layout-menu-items frame
		       (ordered-menu-layout frame)
		       (clim:find-command-table (clim:frame-command-table frame))))

  ;; This always seems to be necessary(?).
  (clim:layout-frame-panes frame (clim:frame-top-level-window frame))
  
  (unwind-protect
      (loop
	  with clim:*command-previewers* = '(#\meta-?)
	  with clim:*help-characters* = '(#\control-meta-?)
	  with clim::*abort-characters* = '(#\control-c)
	  with clim:*complete-characters* = '(#\tab)
	  with clim:*possibilities-characters* = '(#\control-?)

	  do

	    (clim:default-frame-top-level frame :prompt "test-menu> ")
	    )
    
    ;; Cleanup...
    ;;(format t "Goodbye.~&")
    )
  )

;;; ----------------------------------------------------------------------

(defvar *test-menu-frame-root* (clim:open-root-window :clx))

(defvar *current-test-menu-frame* nil)
(defvar *current-test-menu-process* nil)

(defun run-test-menu-frame (frame)

  (unwind-protect
      (clim:run-frame-top-level frame)
    
    (setf *current-test-menu-frame* nil)
    (setf *current-test-menu-process* nil)))

(defun make-test-menu-frame-1 (&key (parent *test-menu-frame-root*)
				    (width 860) (height 600))
  (let (frame process)
      
    (setf frame (clim:make-application-frame 'test-menu
					     :width width
					     :height height
					     :parent parent))
      
    (setf process (mp:process-run-function "TEST-MENU-FRAME"
					   #'run-test-menu-frame
					   frame))
    (values frame process)))

(top-level:alias ("test-menu-frame" 5 :case-sensitive) (&key force)
   "Make a new test frame."

   (cond ((or force (null *current-test-menu-frame*))
	  (multiple-value-setq (*current-test-menu-frame* *current-test-menu-process*)
	    (make-test-menu-frame-1)))
	 (t
	  ;; There already is one.
	  ;; Raise to the top of the window stack.
	  (let ((window (clim:frame-top-level-window *current-test-menu-frame*)))
	    (clim:window-stack-on-top window)
	    (force-output window))
	  ))
   )

(top-level:alias ("reset-test-menu-frame" 6 :case-sensitive) ()
   "Reset test frame."

   (setf *current-test-menu-frame* nil)
   (setf *current-test-menu-process* nil))
