;;; -*- Package: TRANSLISP; Mode: LISP; Syntax: Common-lisp; Base: 10 -*-
;;;_________________________________________________________________________________
;;;
;;;                       System: Translisp
;;;                       Module: RULE-MACROS-COMPILE
;;;                       (Version 1.0)
;;;
;;; Copyright (c): Forschungsgruppe INFORM, Matthias Ressel, Andreas Girgensohn
;;;                Universitaet Stuttgart
;;;
;;; File: ODIN:>matthias>translisp>rules>rule-macros-compile.lisp
;;; File Creation Date: 1/18/88 22:25:05
;;; Last Modification Time: 1/18/88 22:25:05
;;; Last Modification By: matthias
;;;
;;;
;;; Changes (worth to be mentioned):
;;; ================================
;;;
;;;________________________________________________________________________________ 

(in-package 'translisp)

(export '(map-on-second map-on-first protocol-format debug-format TransformForm
	  XTransformNonForm deftrigger defrule defruleset ==> |...| IT
	  NoProtocol TriggerForm TransformNonForm People author
	  define-multiple-rulesets define-ruleset-hierarchie))

(defparameter *default-trigger-name* 'TriggerForm
  "ruleset name for triggers")

(defmacro count-symbol (string-or-integer)
  `(intern (symbol-name (gensym ,string-or-integer))))

(defmacro define-ruleset-hierarchie (alist)
  `(dolist (pair ',alist)
      (setf (get (car pair) '$supersets$)
	    (reachable-nodes (list (car pair))
			     #'(lambda (x) 
				 (cadr (assoc x ',alist)))))))
		    

(defmacro define-multiple-rulesets (list-of-names parameter &rest keyargs
							    &key (loop :once)
								 (else (car parameter)))
  (declare (ignore loop else))
  `(progn ,@(mapcar #'(lambda (name)
			`(defruleset-1 ,name ,parameter nil ,@keyargs))
		     list-of-names)))

(defmacro defruleset (name parameter &rest docu-and-keywordargs)
  "defines a new ruleset incl. the function of the same name"
  (let ((docu nil))
    (when (stringp (car docu-and-keywordargs))
      (setq docu (list (car docu-and-keywordargs)))
      (setq docu-and-keywordargs (cdr docu-and-keywordargs)))
    `(defruleset-1 ,name ,parameter ,docu ,.docu-and-keywordargs)))

(defmacro defruleset-1 (name parameter docu &key (loop :once)
						 (else (car parameter))
						 special
						 used-by)
  `(progn ,@(make-ruleset-function
	      name parameter docu loop else (if (listp special)	; :special *return*
						special	;               =
						(list special))	; :special (*return*)
	      used-by)))

(defun make-ruleset-function (name parameters docu-list
			      use-until? else-expr &optional *specials* used-by)
  (declare (special *rulesets-defined*))
;  (dolist (var *specials*)		; the global-parameters must be
;    (unless (get var 'global)			; declared special at compile-time
;      (proclaim `(special ,var))		; -- Andreas
;      (setf (get var 'global) t)))
  (pushnew name *rulesets-defined*)		; this is meant for the compiler
  (setf (get name :global-parameters) *specials*)
  `((pushnew ',name *rulesets-defined*)		; and this for loading
    (pushnew ',name (get ',name '$supersets$))  
    (setf (get ',name ':global-parameters) ', *specials*)
    ,@(if used-by (list `(setf (get ',name 'used-by) ',used-by)))
;    ,@(mapcar #'(lambda (par)
;		  `(define-global ,par))	; CCC this should be changed
;	       global-parameters)		; in locally specials
    (defun ,name ,parameters
       ,@docu-list
      ,@(if *specials*
	    `((declare (special ,@*specials*))))
      (let ((rule-bdgs-gen (choose-first ,(car parameters) ',name)))
	(cond (rule-bdgs-gen
	       ,(cond ((or (null use-until?)
			   (member use-until? '(forever :forever)))
		       `(do-forever ',name ,(car parameters) rule-bdgs-gen))
		      ((member use-until? '(t once :once) :test #'eq)
		       `(do-once ',name ,(car parameters) rule-bdgs-gen))
		      ((member use-until? '(until-equal :until-equal))
		       `(do-until-equal ',name ,(car parameters) rule-bdgs-gen))))
	      (t ,else-expr))))))

(defmacro defrule (&rest macro-args)
  (let* ((name (car macro-args))
	 (ruleset (cadr macro-args))
	 (priority (caddr macro-args))
	 (pattern (cadddr macro-args))
	 (restargs (cddddr macro-args))
	 (condition (if (eq (car restargs) '==>)
			t
			(pop restargs)))
	 (replacement  (cadr restargs))
	 (restargs (cddr restargs))
	 (action (if (cdddr restargs)
		     (pop restargs)))
	 (optimization-goals (car restargs))
	 (protocol-flag (cadr restargs))
	 (attributes (caddr restargs)))
    (define-rule name ruleset priority 
		 pattern condition replacement action 
		 optimization-goals protocol-flag attributes)))

(defmacro deftrigger (trigger-name priority trigger-form
		      &key (goal '(People))
		      (protocol '(NoProtocol))
		      (attributes '((author "Matthias Ressel"))))
  (let ((foo-name (car trigger-form))
	(restvar) (keyword-ruleset-name nil))
    (multiple-value-bind (pattern replacement restform last-var)
	(parse-trigger-form trigger-form)
      (when restform
	(setq restvar (CreateNamSegm (intern "*REST")))
	(let ((pattern-var
		(if last-var
		    (CreateNamSegmWithRestrictElems (intern "*REST")
						    (CreateRestriction `(progn ,last-var)))
		    restvar)))
	  (nconc pattern (list pattern-var))
	  (ecase (car restform)
	    (&rest
	      (nconc replacement
		     (list
		       (CreateToBeEvaluatedAndThenSplicedEXpression
			 (CreateEvalForm
			   `(TransformNonForm ',(cadr restform) ,restvar)))))
	      (when (cddr restform) (warn "no more forms allowed after '&rest body'")))
	    (&key
	      (setq keyword-ruleset-name
		    (intern (concatenate 'string
					 "TRIGGER-"
					 (symbol-name foo-name)
					 "-KEYS")))
	      (nconc replacement
		     (list (CreateToBeEvaluatedAndThenSplicedExpression
			     (CreateEvalForm `(,keyword-ruleset-name ,restvar)))))))))      
      `(progn (defrule ,trigger-name , *default-trigger-name* ,priority
		       ,pattern
		       ==>
		       ,replacement
		       ,goal
		       ,protocol
		       ,attributes)
	      ,@(if keyword-ruleset-name 
		    (cons
		      `(defruleset ,keyword-ruleset-name (keywordlist)
			 ,(concatenate 'string
				       "Trigger for keyword parameters of function "
				       (symbol-name foo-name))
			 :loop :once
			 :else keywordlist
			 :used-by (,trigger-name))
		      (make-keyword-parameter-trigger
			trigger-name restform keyword-ruleset-name goal
			attributes)))))))

(defun concat-Transform (sym)
  (intern (concatenate 'string
	       (symbol-name 'Transform)
	       (symbol-name sym))))

(defun nonformp (item)
  (and (listp item)
       (eq (car item) 'nonform)))

(defun evalp (item)
  (and (listp item)
       (eq (car item) 'eval)))

(defun transform-function (transformation-type)
  (if (nonformp transformation-type)
      'XTransformNonForm			; switched arguments
      'TransformForm))

(defun XTransformNonForm (nonform type)
  (TransformNonForm type nonform))

(defun normalize-trafo-type (trafo-type)
  (if (nonformp trafo-type)
      `',(cadr trafo-type)
      (if (evalp trafo-type)
	  (cadr trafo-type)
	  `',trafo-type)))

(defun parse-trigger-form (trigger-form)
  (let ((pattern (list (car trigger-form)))
	(replacement (list (car trigger-form)))
	(restlist) (last-var))
    (count-symbol 0)
    (cond ((member '|...| (cdr trigger-form))
	   (when (or (member '&optional (cdr trigger-form))
		     (member '&optional (cdr trigger-form))
		     (member '&optional (cdr trigger-form)))
	     (error "Keywords (&optional ...) and Ellipsis not allowed in same pattern"))
	   (do* ((restform (cdr trigger-form))
		 (expr-type) (segm-found nil))
		((null restform))
	     (setq expr-type (car restform))
	     (cond ((eq (cadr restform) '|...|)
		    (setq restform (cdr restform))
		    (setq segm-found t)
		    (let ((var (CReateNamSegm (count-symbol "*VAR"))))
		      (push var pattern)
		      (push (CreateToBeEvaluatedAndThenSplicedExpression
			      (CreateEvalForm
				`(map-on-first ',(transform-function expr-type)
					       ,var
					       ,(normalize-trafo-type expr-type))))
			    replacement)))
		   (t (let ((var (CreateNamElem (count-symbol "VAR"))))
	(push var pattern)
	(push (CreateToBeEvaluatedAndThenInsertedEXpression
		(CreateEvalForm
		  `(,(transform-function expr-type)
		    ,var
		    ,(normalize-trafo-type expr-type))))
	      replacement))))
	     (setq restform (cdr restform))))
	  
	  (t 
    (do* ((restform (cdr trigger-form) (cdr restform))
	  (expr-type (car restform) (car restform)))
	 ((null restform))
      (when (member expr-type '(&optional &rest &key))
	(setq restlist restform) (return))
      (let ((var (CreateNamElem (count-symbol "VAR"))))
	(push var pattern)
	(push (CreateToBeEvaluatedAndThenInsertedEXpression
		(CreateEvalForm
		  `(,(transform-function expr-type)
		    ,var
		    ,(normalize-trafo-type expr-type))))
	      replacement)))
    (when (eql (car restlist) '&optional)
      (do* ((restform (cdr restlist) (cdr restform))
	    (var-name (count-symbol "*VAR")
		      (count-symbol "*VAR"))
	    (repl-var (CreateNamSegm var-name)
		      (CreateNamSegm var-name))
	    (pattern-var (CreateRestrictNamSegm
			   var-name
			   (CreateRestriction '(null (cdr IT))))
			 (CreateREstrictNamSegmWithRestrictElems
			   var-name
			   (CreateRestriction `(progn ,var-before))
			   (CreateREstriction '(null (cdr IT)))))			     
	    (var-before nil))
	   ((null restform) (setq restlist '()))
	(when (member (car restform) '(&rest &key))
	  (setq restlist restform)
	  (setq last-var var-before) (return))
	(push pattern-var pattern)
	(push (CreateToBeEvaluatedAndThenSplicedEXpression
		(CreateEvalForm
		  `(if ,repl-var
		       (list (,(transform-function (car restform))
			      (car ,repl-var) ,(normalize-trafo-type
						 (car restform)))))))
	      replacement)
	(setq var-before repl-var)))))
;    (setq pattern (nreverse pattern))
;    (setq replacement (nreverse replacement))
    (values (nreverse pattern) (nreverse replacement) restlist last-var)))

(defun make-keyword-parameter-trigger (main-trigger-name keyword-form ruleset-name goal
				       attributes)
  (do ((keyword-form (cdr keyword-form) (cddr keyword-form))
		      (result-rules))
		     ((null keyword-form) (nreverse result-rules))
    (push
      (let ((keyvar (CreateNamElem 'key))
	    (restvar (CreateNamSegm (intern "*REST"))))
	`(defrule ,(intern (concatenate 'string
				  (symbol-name main-trigger-name)
				  "-KEY-"
				  (symbol-name (car keyword-form))))
		    ,ruleset-name 1		; priority = 1
		    (,(car keyword-form) ,keyvar ,restvar)
		    ==>
		    (,(car keyword-form)
		     ,(CreateToBeEvaluatedAndThenInsertedEXpression
			(CreateEvalForm `(,(transform-function (cadr keyword-form))
					  ,keyvar ,(normalize-trafo-type
						     (cadr keyword-form)))))
		     ,(CreateToBeEvaluatedAndThenSplicedEXpression
			(CreateEvalForm `(,ruleset-name ,restvar))))
		    ,goal
		    (NoProtocol)
		    ,attributes))
      result-rules)))

; Protocol Functions
; ==================

(unless (boundp '*debug-output*) (setq *debug-output* *debug-io*))

(defmacro debug-format (format &rest args)
  `(locally (declare (special *debug-output*))
	    (format *debug-output* "~&~@?~%" ,format ,.args)))

;;; CCC (protocol-format ("format-zeile-1" args) ("format-zeile-2" args))

(defmacro protocol-format (&rest args)
  `(when (protocol?)
     (terpri *protocolport*)
     (tab *lmar* *protocolport*)
     (format (or *protocolport* t) ,.args)))

; Macro fuer Trigger
; ==================

(defmacro map-on-first (foo first &rest args)
  `(mapcar #'(lambda (item)
	       (funcall ,foo item ,.args))
	       ,first))

(defmacro map-on-second (foo first second)
  `(mapcar #'(lambda (item) (funcall ,foo ,first item))
	   ,second))


(defun map-get-unique (list1 property)
  (do ((rlist list1 (cdr rlist))
       (result ()))
      ((null rlist) result)
    (setq result (union result (get (car rlist) property)))))

(defun define-rule (name ruleset priority pattern cond replacement action
                    optimization-goals protocol-flag attributes)
  (declare (special *debug-output* *rulesets-defined*))
  (let ((ruleset-list (if (consp ruleset)
			  ruleset
			  (list ruleset))))
    (multiple-value-bind (match-func-name functions)
	(transform-pattern pattern cond name
			   (map-get-unique ruleset-list :global-parameters))
      (multiple-value-bind (repl-func-name repl-func)
	  (transform-replacement replacement action name
				 (map-get-unique
				   ruleset-list :global-parameters))
	(mapc #'(lambda (item)
		  (unless (member item *rulesets-defined*)
		    (format *debug-output* "~&%Note: Declaration for ~
                                       ruleset ~a should be done: " item)
		    (format *debug-output* "(defruleset ~a (expr))~%" item)
		    (push item *rulesets-defined*)))
	      ruleset-list)
	`(progn 
	   ;; CCC hier pattern transformieren:
	   ;; dies liefert:
	   ;; 0. pretty-pattern und condition
	   ;; 1. eine Funktion, die das matchen uebernimmt (mit cond!!)
	   ;; 2. die Funktiondefinition
	   ;; 3. die Restriktionfunktionen fuer match1
	   ,.functions
	   ,repl-func
	   ;; ,.(compile-replacement replacement name nil)
	   (apply #'InstallRule
		  '(,name
		    ,ruleset
		    ,priority
		    ,match-func-name
		    ,pattern
		    ,cond
		    ,repl-func-name
		    ,replacement
		    ,action
		    ,optimization-goals
		    ,protocol-flag
		    ,attributes)))))))

(defun compile-pattern (pattern rulename &optional def-list)
  (declare (special *specials*))
  (cond ((atom pattern) def-list)
	((Var? pattern)
	 (cond ((RestrictElem? pattern)
		(when (consp (TheElemPredicate pattern))
		  (let ((pred-name
			  (gentemp
			    (concatenate 'string
					 (string rulename)
					 "-PRED-"))))
		    (push `(defun ,pred-name (IT)	; CCC IT ????
			     ,.(if *specials*
				   (list `(declare (special ,.*specials*))))
			     ,.(cddr (TheElemPredicate pattern))
			     ;; predicate = (lambda (IT) . rest)
			     )
			  def-list)
		    (setf (TheElemPredicate pattern) pred-name))))
	       ((RestrictSegm? pattern)
		(when (consp (TheSegmPredicate pattern))
		  (let ((pred-name
			  (gentemp
			    (concatenate 'string
					 (string rulename)
					 "-PRED-"))))
		    (push `(defun ,pred-name (IT)
			     ,.(if *specials*
				   (list `(declare (special ,.*specials*))))
			     ,.(cddr (TheSegmPredicate pattern)))
			  def-list)
		    (setf (TheSegmPredicate pattern) pred-name)))))
	 def-list)
	(t (let ((def-list (compile-pattern (car pattern)
					    rulename
					    def-list)))
	     (compile-pattern (cdr pattern) rulename def-list)))))

(defun compile-replacement (replacement rulename def-list)
  (declare (special *specials*))
  (cond ((atom replacement) def-list)
        ((or (IsEvaluateAndSplice? replacement)
             (IsEvaluateAndInsert? replacement))
         (if (not (and (consp (ExpressionToBeEvaled replacement))
                       (null (cdr (ExpressionToBeEvaled replacement)))))
             (let ((eval-foo-name
                     (gentemp
                       (concatenate 'string
				    (string rulename)
				    "-EVAL-"))))
               (push `(defun ,eval-foo-name ()
			     ,.(if *specials*
				   (list `(declare (special ,.*specials*))))
			     ,.(cddr (ExpressionToBeEvaled replacement)))
                     def-list)
               (setf (ExpressionToBeEvaled replacement) eval-foo-name)
               def-list)))
        ((Var? replacement) def-list)
        (t (let ((def-list (compile-replacement (car replacement)
                                                rulename
                                                def-list)))
             (compile-replacement (cdr replacement) rulename def-list)))))

;;;------------------------------------------------------------------------------------
;;; CCC

(defun transform-pattern (pattern condition rulename *specials*)
  (declare (special *specials*))
  (let ((*match-pattern* (find-match-pattern pattern))
	(*match-selector* nil))
    (declare (special *match-pattern* *match-selector*))
    (multiple-value-bind (exprs checks variables)
	(transform-subpattern pattern 'dat nil t)	
      (declare (ignore variables))
      (when *match-pattern*
	(push `(let ((match-result
		       (match1 ', *match-pattern* , *match-selector* *bindings*)))
		 ,(make-test 'match-result)
		 (setq *bindings* (car match-result))
		 match-result)
	      exprs))
      ;; match darf nur an einer Stelle aufgerufen werden, da sonst das
      ;; backtracking nicht funktioniert
      (let ((func-name (gentemp (concatenate 'string (string rulename) "-MATCH-"))))
	(values func-name
		(cons `(defun ,func-name (dat)
			 ,.(if *specials* (list `(declare (special ,.*specials*))))
			 (block pattern-body
			   (let ((*bindings* (list (cons '$left$ dat))))
			     (declare (special *bindings*))
			     ,.(when checks
				 (list (make-test (if (cdr checks)
						      (cons 'and (nreverse checks))
						      (car checks)))))
			     (prog1 (progn ,.(nreverse exprs))
				    (unless ,condition (return-from pattern-body nil))))))
		      (compile-pattern *match-pattern* rulename)))))))

;;; Sucht die Stelle, an der Match1 aufgerufen werden muss

(defun find-match-pattern (pattern)
  (let ((*match-pattern* nil))
    (declare (special *match-pattern*))
    (find-match-pattern-1 pattern)
    *match-pattern*))

(defun find-match-pattern-1 (pattern)
  (declare (special *match-pattern*))
  (cond ((or (atom pattern) (Var? pattern)) nil)
	((or (atom (car pattern)) (Elem? (car pattern)))
	 (find-match-pattern-1 (cdr pattern)))
	((Segm? (car pattern))
	 (when (cdr pattern)
	   (setq *match-pattern* pattern)
	   t))
	((find-match-pattern-1 (car pattern))
	 (when (find-match-pattern-1 (cdr pattern))
	   (setq *match-pattern* pattern))
	 t)
	(t (find-match-pattern-1 (cdr pattern)))))

(defun transform-subpattern (pattern dat variables &optional top)
  (declare (special *match-pattern* *match-selector*))
  (do ((rest pattern (cdr rest))
       (count 0 (1+ count))
       (max-length-p t)
       (exprs nil)
       (checks nil))
      ((or (Var? rest)
	   (atom rest))
       (cond ((null rest))
	     ((Var? rest)
	      (setq max-length-p nil
		    exprs (nconc (match-selector-with-elemvar
				   (make-nthcdr count dat) rest variables)
				 exprs)))
	     (t (setq max-length-p nil)
		(push `(eql ,(make-nthcdr count dat) ',rest) checks)))
       (when (and top (null *match-pattern*))
	 (push '(list *bindings*) exprs))
       (values exprs (add-length-check dat checks count max-length-p) variables))
    (cond ((eq rest *match-pattern*)
	   (setq *match-selector* (make-nthcdr count dat))
	   (return (values exprs (add-length-check dat checks count nil) variables)))
	  ((Elem? (car rest))
	   (setq exprs (nconc (match-selector-with-elemvar
				(make-nth count dat) (car rest) variables)
			      exprs))
	   (let ((varname (Name (car rest))))
	     (when varname
	       (pushnew varname variables))))
	  ((not (Segm? (car rest)))
	   (cond ((null (car rest)) (push `(null ,(make-nth count dat)) checks))
		 ((atom (car rest))
		  (push `(eql ,(make-nth count dat) ',(car rest)) checks))
		 (t (multiple-value-bind (sub-exprs sub-checks new-variables)
			;; sub-exprs: (invertierte) Liste von Ausdruecken, die
			;;            auszufuehren sind, i.d.R. Variablen-Bindungen
			;; sub-checks: Praedikate, die zuallererst die Struktur des Datums
			;;             ueberpruefen und gegebenenfalls zu FAIL fuehren
			;; new-variables: alle bisher gefundenen Variablen
			(transform-subpattern (car rest) (make-nth count dat) variables)
		      (setq variables new-variables
			    exprs (nconc sub-exprs exprs)
			    checks (nconc sub-checks checks))))))
	  (t (let ((selector (make-nthcdr count dat)))	; (cdr rest) = nil !!
	       (setq exprs (nconc (match-selector-with-segmvar selector (car rest) variables)
				  exprs))
	       (push `(real-listp ,selector) checks)
	       (let ((varname (Name (car rest))))
		 (when varname
		   (pushnew varname variables))))
	     (when (and top (null *match-pattern*))
	       (push '(list *bindings*) exprs))
	     (return (values exprs (add-length-check dat checks count nil) variables))))))

(defun make-nth (index dat)
  (case index
    (0 `(car ,dat))
    (1 `(cadr ,dat))
    (2 `(caddr ,dat))
    (3 `(cadddr ,dat))
    (t `(nth ,index ,dat))))

(defun make-nthcdr (index dat)
  (case index
    (0 dat)
    (1 `(cdr ,dat))
    (2 `(cddr ,dat))
    (3 `(cdddr ,dat))
    (4 `(cddddr ,dat))
    (t `(nthcdr ,index ,dat))))

(defun make-test (test)
  `(unless ,test
     (return-from pattern-body nil)))

;;; Der Test der Restriktionen koennte an den Anfang gezogen werden (in die Liste checks),
;;; wenn dort nicht auf andere Variablen zugegriffen wird (-> Aenderungsmoeglichkeit)

(defun match-selector-with-elemvar (selector elemvar variables)
  ;; selector: Ausdruck, der aus gesamtem Datum den Teil selektiert, der der
  ;;           augenblicklichen Variable matchen muss.
  ;; elemvar: aktuelle Pattern-Variable
  ;; variables: bis jetzt im Pattern angetroffene Variablen. Das sind jene, die zu diesem
  ;;            Zeitpunkt bereits gebunden sind.
  (let ((exprs nil)
	(local-selector selector)
	(varname (Name elemvar)))
    ;; expr: ?
    ;; local-selector: ?
    ;; varname: Name der aktuellen Patternvariablen
    (when (or varname (RestrictElem? elemvar)) ; multiple using of selector
      (setq local-selector 'value))		; CCC what is value
    (push (make-test `(test-comma-splice ,local-selector)) exprs)
    (when varname
      (if (member varname variables :test #'eq)
	  (push (make-test `(equal ,local-selector
				   (cdr (assoc ',varname *bindings* :test #'eq))))
		exprs)
	  (push `(push (cons ',varname ,local-selector) *bindings*) exprs)))
    (when (RestrictElem? elemvar)
      (push (make-test `(,(TheElemPredicate elemvar) ,local-selector)) exprs))
    (if (eq local-selector selector)
	exprs
	(list `(let ((,local-selector ,selector)) ,.(nreverse exprs))))))

(defun match-selector-with-segmvar (selector segmvar variables)
  (let ((exprs nil)
	(local-selector selector)
	(varname (Name segmvar)))
    (when varname
      (when (or (RestrictElem? segmvar) (RestrictSegm? segmvar))
	(setq local-selector 'value))
      (if (member varname variables :test #'eq)
	  (push (make-test `(equal ,local-selector
				   (cdr (assoc ',varname *bindings* :test #'eq))))
		exprs)
	  (push `(push (cons ',varname ,local-selector) *bindings*) exprs)))
    (when (RestrictElem? segmvar)
      (let ((loop-var 'elem))
	(push `(dolist (,loop-var ,local-selector)
		 ,(make-test `(,(TheElemPredicate segmvar) ,loop-var)))
	      exprs)))
    (when (RestrictSegm? segmvar)
      (push (make-test `(,(TheSegmPredicate segmvar) ,local-selector)) exprs))
    (if (eq local-selector selector)
	exprs
	(list `(let ((,local-selector ,selector)) ,.(nreverse exprs))))))

(defun add-length-check (dat checks min-length max-length-p)
  (if (zerop min-length)
      (if max-length-p
	  (list `(null ,dat))
	  checks)
      (nconc checks (list (if max-length-p
			      `(oflengthp ,dat ,min-length)
			      `(minlengthp ,dat ,min-length))))))

;;;----------------------------------------------------------------------------
;;; Funktionen fuer die Laufzeit

(defun real-listp (list)
  (do ((rest list (cdr rest)))
      ((atom rest) (null rest))))

(defun minlengthp (list length)
  (do ((count length (1- count))
       (rest list (cdr rest)))
      ((or (atom rest)
	   (zerop count))
       (zerop count))))

(defun oflengthp (list length)
  (do ((count length (1- count))
       (rest list (cdr rest)))
      ((or (atom rest)
	   (zerop count))
       (and (null rest) (zerop count)))))

;;;----------------------------------------------------------------------------
;;; Replacement

(defun transform-replacement (replacement action rulename *specials*)
  (declare (special *specials*))
  (let ((expr (transform-replacement-1 replacement))
	(func-name (gentemp (concatenate 'string (string rulename) "-REPLACE-"))))
    (values func-name
	    `(defun ,func-name (*bindings*)
	       (declare (special *bindings* ,.*specials*))
	       ,(if action
		    `(let ((result ,expr))
		       (push (cons '$result$ result) *bindings*)
		       ,action
		       result)
		    expr)))))

(defun transform-replacement-1 (pattern)
  (cond ((null pattern) nil)
        ((atom pattern) `',pattern)
        ((NamElem? pattern) pattern)
        ((IsEvaluateAndInsert? pattern)
         (ShowExpressionToBeEvaled pattern))
        ((IsEvaluateAndSplice? pattern)
         `(CreateToBeSplicedExpression ,(ShowExpressionToBeEvaled pattern)))
        ((NamSegm? pattern) `(CreateToBeSplicedExpression ,pattern))
	((IsEvaluateAndInsert? (car pattern))
	 `(let ((value ,(ShowExpressionToBeEvaled (car pattern))))
	    (if (ToBeSpliced? value)
		,(merge-replacements 'non-copy-append
				     '(ExpressionToSplice value)
				     (transform-replacement-1 (cdr pattern)))
		,(merge-replacements
		   'cons 'value (transform-replacement-1 (cdr pattern))))))
	((IsEvaluateAndSplice? (car pattern))
	 (merge-replacements 'nconc
			     `(mapcan #'ListNonSplice
				      ,(ShowExpressionToBeEvaled (car pattern)))
			     (transform-replacement-1 (cdr pattern))))
	((IsVariable? (car pattern))
	 (if (IsSpliceVariable? (car pattern))
	     (merge-replacements
	       'non-copy-append
	       (car pattern)
	       (transform-replacement-1 (cdr pattern)))
	     (merge-replacements
	       'cons
	       (car pattern)
	       (transform-replacement-1 (cdr pattern)))))
	(t (merge-replacements 'cons
			       (transform-replacement-1 (car pattern))
			       (transform-replacement-1 (cdr pattern))))))

(defun merge-replacements (func car-replacement cdr-replacement)
  (cond ((null cdr-replacement)
	 (case func
	   ((nconc non-copy-append) car-replacement)
	   (cons `(list ,car-replacement))))
	((not (and (eq func 'cons)
		   (consp cdr-replacement)
		   (member (car cdr-replacement) '(cons list* list) :test #'eq)))
	 (list func car-replacement cdr-replacement))
	((eq (car cdr-replacement) 'list)
	 `(list ,car-replacement . ,(cdr cdr-replacement)))
	(t `(list* ,car-replacement . ,(cdr cdr-replacement)))))

(setf (get 'rule-macros-compile 'version) 11.0)