;;; -*- Package: (FRANZLISP-READER CL); Mode: LISP; Syntax: Common-lisp; Base: 10 -*-

; The package flr is for the internal representation of some sophisticated
; constructs as readtime evaluations, character macros etc.
; It is named franzlisp-reader because the representation of e.g. backquoted
; expressions is much like it is in FranzLisp

(in-package 'franzlisp-reader :nicknames '(flr))

(export '(set-franz-input-syntax reset-franz-input-syntax
	  set-franz-backquote-syntax reset-franz-backquote-syntax
	  backquote-reader comma-reader
	  qu* |,| |,@| |,.| |,!| |,?| |,$| |,#| |,*|))


; Readmacros for #+ #- #/ #\ #. ` ,
; ==============================

;;; #+lispm expr

(defun |#+/#-reader| (stream subchar arg)
  (declare (ignore arg))
  (make-conditional-inclusion
    :test subchar
    :feature (read stream t nil t)
    :form    (read stream t nil t)))

(defstruct (conditional-inclusion
	     (:print-function print-conditional-inclusion))
  "Internal representation of conditional inclusions
   #+feature form   or  #-feature form"
  test
  feature
  form)

(defun print-conditional-inclusion (struct stream depth)
  (declare (ignore depth))
  (format stream "#~C~S ~S"
	  (conditional-inclusion-test struct)
	  (conditional-inclusion-feature struct)
	  (conditional-inclusion-form struct)))

;;; #/c

(defun franz-character-reader (stream subchar arg)
  (declare (ignore subchar arg))
  (make-franz-char :char (read-char stream t nil t)))

;;; #^C

(defun franz-control-character-reader (stream subchar arg)
  (declare (ignore subchar arg))
  (make-franz-char
   :char (int-char (logand 31 (char-int (read-char stream t nil t))))))

(defstruct (franz-char
	     (:print-function print-franz-char))
  char)

(defun print-franz-char (struct stream depth)
  (declare (ignore depth))
  (prin1 (franz-char-char struct) stream))

;;; #\esc bzw. #!esc

(defun franz-special-character-reader (stream subchar arg)
  (declare (ignore subchar arg))
  (make-franz-special-char
    :char (read-preserving-whitespace stream t nil t)))

(defstruct (franz-special-char
	     (:print-function print-franz-special-char))
  char)

(defun print-franz-special-char (struct stream depth)
  (declare (ignore depth))
  (format stream "#\\~S" (franz-special-char-char struct)))

(defun readtime-eval-reader (stream subchar arg)
  (declare (ignore subchar arg))
  (make-readtime-eval
    :expression (read stream t nil t)))

(defstruct (readtime-eval
	     (:print-function print-readtime-eval))
  expression)

(defun print-readtime-eval (struct stream depth)
  (declare (ignore depth))
  (format stream "#.~S"
	  (readtime-eval-expression struct)))

;;; Backquote

(defun backquote-reader (str char)
  (declare (ignore char))
  (list 'qu* (read str t nil t)))

(defun comma-reader (str char)
  (declare (ignore char))
  (case (peek-char nil str t nil t)
    (#\@ (read-char str)
     (cons '|,@| (read str t nil t)))
    (#\. (read-char str)
     (cons '|,.| (read str t nil t)))
    (#\! (read-char str)
     (cons '|,!| (read str t nil t)))
    (#\$ (read-char str)
     (cons '|,$| (read str t nil t)))
    (#\? (read-char str)
     (cons '|,?| (filter-read str)))
    (#\* (read-char str)
     (cons '|,*| (filter-read str)))
    (#\# (read-char str)
     (cons '|,#| (filter-read str)))
    (otherwise
      (cons '|,| (read str t nil t)))))

(defun filter-read (stream)
  (let ((st (string (read stream t nil t)))
	(lastpos 0)
	(symbols nil))
    (dotimes (pos (length st))
      (when (eql (char st pos) #\:)
	(push (intern (subseq st lastpos pos)) symbols)
	(setq lastpos (1+ pos))))
    (push (intern (subseq st lastpos)) symbols)
    (nreverse symbols)))

;;; Die folgende Funktion gestattet die Auswertung von Backquote-Ausdruecken,
;;; die wie in Inform-Franzlisp eingelesen wurden (` -> qu*  , -> |,|  ,@ -> |,@| usw.)
;;; Um auf die bereits im Package SYSTEM-INTERNALS definierte Umwandlung
;;; einfach zurueckgreifen zu koennen, werden die dort eingefuehrten globalen
;;; Variablen **BACKQUOTE-...** umgebunden

#+lispm

(defmacro qu* (backquoted-item)
  (let ((flag nil)
	(thing nil)
	(SI::**BACKQUOTE-COUNT** (1+ SI::**BACKQUOTE-COUNT**))
	(SI::|**BACKQUOTE-,-FLAG**| '|,|)
	(SI::|**BACKQUOTE-,@-FLAG**| '|,@|)
	(SI::|**BACKQUOTE-,.-FLAG**| '|,.|))
    (multiple-value-setq (flag thing)
		    (SI::BACKQUOTIFY backquoted-item))
    (and (eq flag SI::|**BACKQUOTE-,@-FLAG**|)
	 (error " \",@\" right after a \"`\": `,@~s" thing))
    (and (eq flag SI::|**BACKQUOTE-,.-FLAG**|)
	 (error " \",.\" right after a \"`\": `,.~s" thing))
    (values (SI::BACKQUOTIFY-1 flag thing) 'list)))

#-lispm

(progn

(defvar *backquote-count* 0)
(defvar *|backquote-,-flag|* '|,|)
(defvar *|backquote-,@-flag|* '|,@|)
(defvar *|backquote-,.-flag|* '|,.|)

(defmacro qu* (backquoted-item)
  (let ((flag nil)
	(thing nil)
	(*backquote-count* (1+ *backquote-count*)))
    (multiple-value-setq (flag thing)
		    (backquotify backquoted-item))
    (and (eq flag *|backquote-,@-flag|*)
	 (error " \",@\" right after a \"`\": `,@~s" thing))
    (and (eq flag *|backquote-,.-flag|*)
	 (error " \",.\" right after a \"`\": `,.~s" thing))
    (values (backquotify-1 flag thing) 'list)))
  
;;expansions of backquotes actually use these five functions
;;so that one can recognize what came from backquote and what did not.
;
;(defmacro xr-bq-cons (car cdr)
;  `(cons ,car ,cdr))
;
;(defmacro xr-bq-list (&rest elements)
;  `(list . ,elements))
;
;(defmacro xr-bq-list* (&rest elements)
;  `(list* . ,elements))
;
;(defmacro xr-bq-append (&rest elements)
;  `(append . ,elements))
;
;(defmacro xr-bq-nconc (&rest elements)
;  `(nconc . ,elements))

;;; Feature: #= may not function within a backquote-expr or vice versa

(defun backquotify (code)
  (prog (aflag a dflag d)
	(cond ((atom code)
	       (cond ((null code) (return (values nil nil)))
		     ((or (numberp code)
			  (eq code t))
		      (return (values t code)))
		     (t (return (values 'quote code)))))
	      ((eq (car code) *|backquote-,-flag|*)
	       (setq code (cdr code))
	       (go comma))
	      ((eq (car code) *|backquote-,@-flag|*)
	       (return (values *|backquote-,@-flag|* (cdr code))))
	      ((eq (car code) *|backquote-,.-flag|*)
	       (return (values *|backquote-,.-flag|* (cdr code)))))
	(multiple-value-setq (aflag a) (backquotify (car code)))
	(multiple-value-setq (dflag d) (backquotify (cdr code)))
	(and (eq dflag *|backquote-,@-flag|*)
	     (error " \",@\" after a \".\": .,@~s in ~s" d code))
	(and (eq dflag *|backquote-,.-flag|*)
	     (error " \",.\" after a \".\": .,.~s in ~s" d code))
	(cond ((eq aflag *|backquote-,@-flag|*)
	       (cond ((null dflag)
		      (setq code a)
		      (go comma)))
	       (return (values
			 'append
			 (cond ((eq dflag 'append)
				(cons a d))
			       (t (list a (backquotify-1 dflag d)))))))
	      ((eq aflag *|backquote-,.-flag|*)
	       (cond ((null dflag)
		      (setq code a)
		      (go comma)))
	       (return (values
			 'nconc
			 (cond ((eq dflag 'nconc)
				(cons a d))
			       (t (list a (backquotify-1 dflag d)))))))
	      ((null dflag)
	       (cond ((member aflag '(quote t nil))
		      (return (values 'quote (list a))))
		     (t (return (values
				  'list
				  (list (backquotify-1 aflag a)))))))
	      ((member dflag '(quote t))
	       (cond ((member aflag '(quote t nil))
		      (return (values 'quote (cons a d))))
		     (t (return (values 'list*
					(list
					  (backquotify-1 aflag a)
					  (backquotify-1 dflag d))))))))
	(setq a (backquotify-1 aflag a))
	(and (member dflag '(list list*))
	     (return (values dflag (cons a d))))
	(return (values 'list* (list a (backquotify-1 dflag d))))
     comma (cond ((atom code)
		  (cond ((null code)
			 (return (values nil nil)))
			((or (numberp code)
			     (eq code 't))
			 (return (values t code)))
			(t (return (values *|backquote-,-flag|* code)))))
		 ((eq (car code) 'quote)
		  (return (values (car code) (cadr code))))
		 ((member (car code) '(append list list* nconc))
		  (return (values (car code) (cdr code))))
		 ((eq (car code) 'cons)
		  (return (values 'list* (cdr code))))
		 (t (return (values *|backquote-,-flag|* code))))))

(defun backquotify-1 (flag thing)
  (cond ((or (eq flag *|backquote-,-flag|*)
	     (member flag '(t nil)))
	 thing)
	((eq flag 'quote)
	 (list 'quote thing))
	((eq flag 'list*)
	 (cond ((null (cddr thing))
		(cons 'cons thing))
	       (t (cons 'list* thing))))
	(t (cons flag
;		 (cdr (assoc flag `((cons . xr-bq-cons)
;				   (list . xr-bq-list)
;				   (append . xr-bq-append)
;				   (nconc . xr-bq-nconc))
;			     :test #'eq))
		 thing
		 ))))

) ; #-lispm (progn

;;; 

(defun set-franz-input-syntax ()
  (set-dispatch-macro-character #\# #\+ '|#+/#-reader|)
  (set-dispatch-macro-character #\# #\- '|#+/#-reader|)
  (set-dispatch-macro-character #\# #\/ 'franz-character-reader)
  ; the following is for character-input
  (set-dispatch-macro-character #\# #\^ 'franz-control-character-reader)
  (set-dispatch-macro-character #\# #\! 'franz-special-character-reader)
  (set-dispatch-macro-character #\# #\. 'readtime-eval-reader)
  (set-macro-character #\` 'backquote-reader)
  (set-macro-character #\, 'comma-reader)
 ; (set-syntax-from-char #\: #\A)
  )

(defun set-franz-backquote-syntax ()
  (set-macro-character #\` 'backquote-reader)
  (set-macro-character #\, 'comma-reader))

(defun reset-franz-backquote-syntax ()
  (set-syntax-from-char #\` #\`)
  (set-syntax-from-char #\, #\,))

(defun reset-franz-input-syntax ()
  (let ((standard-readtable (copy-readtable nil)))
    (set-dispatch-macro-character #\# #\+
	    (get-dispatch-macro-character #\# #\+ standard-readtable))
      (set-dispatch-macro-character #\# #\-
 	    (get-dispatch-macro-character #\# #\- standard-readtable))
      (set-dispatch-macro-character #\# #\/
	    (get-dispatch-macro-character #\# #\/ standard-readtable))
      (set-dispatch-macro-character #\# #\\
	    (get-dispatch-macro-character #\# #\\ standard-readtable))
      (set-dispatch-macro-character #\# #\.
	    (get-dispatch-macro-character #\# #\. standard-readtable))
      ;; CCC seems to be the same as (set-syntax-from-char #\` #\`)
      (set-macro-character #\`
	    (get-macro-character #\` standard-readtable))
      (set-macro-character #\,
	    (get-macro-character #\, standard-readtable))
      (set-syntax-from-char #\: #\:)))

