;;; -*- Mode: LISP; Syntax: Common-lisp; Package: User; Base: 10 -*-

;;;----------------------------------------------------------------------
;;; Package
;;;----------------------------------------------------------------------

;(in-package 'franz-to-common)

;(export '(help-convert convert-franz-to-common))

;;;----------------------------------------------------------------------
;;; Filter Functions that preserve Case Sensitivity of Franz Lisp
;;;----------------------------------------------------------------------

(defun convert-case (file &key (output-file (merge-pathnames ".flisp" file)))
  "escapes all small letters and colon"
  (with-open-file (instream file :direction :input)
    (with-open-file (outstream output-file :direction :output)
      (convert-case-stream instream outstream))))
      
(defun convert-case-stream (instream outstream)
  (do ((ch (read-char instream nil nil) (read-char instream nil nil)))
      ((eq ch nil)  "case-converted")
    (cond ((member ch '(#\" #\|))
	   (unread-char ch instream)
	   (prin1 (read-preserving-whitespace instream) outstream))
	  ((eql ch #\\)
	   (write-char #\\ outstream)
	   (write-char (read-char instream nil nil) outstream))
	  ((eql ch #\;)
	   (write-char #\; outstream)
	   (write-line (read-line instream nil nil) outstream))
	  ((or (lower-case-p ch)
	       (member ch '(#\:)))
	   (write-char #\\ outstream)
	   (write-char ch outstream))
	  ((eql ch #\#)
	   (sharp-convert instream outstream))
	  (t (write-char ch outstream)))))

(defun sharp-convert (instr outstr)
  (let ((ch (read-char instr nil nil)))
    (cond ((member ch '(#\/ #\^))
	   (format outstr "#~A~A" ch (read-char instr nil nil)))
	  ;; Bug!? The Reader does not like a Redefinitiond of #\, it works,
	  ;; however, with #! (is not used yet) cmp. function set-franz-input-syntax
	  ((eql ch #\\) (format outstr "#!"))
	  (t (format outstr "#~A" ch)))))

(defun normalize (file &key (output-file (merge-pathnames ".flisp" file)))
  "makes the output from convert-case more readable"
  (with-open-file (instream file :direction :input)
    (with-open-file (outstream output-file :direction :output)
      (normalize-stream instream outstream))))


(defun normalize-stream (instream outstream)
  (do ((ch (peek-char nil instream nil nil) (peek-char nil instream nil nil)))
      ((eq ch nil) "normalized")
    (cond ((eql ch #\;)
	   (read-char instream)
	   (write-char #\; outstream)
	   (write-line (read-line instream nil nil) outstream))
	  ((eql ch #\#)
	   (read-char instream)
	   (sharp-convert instream outstream))
	  ((member ch '(#\\ #\" #\|))
	   (print-with-franz-filter
	     (read-preserving-whitespace instream) outstream))
	  (t (write-char (read-char instream) outstream)))))

(defun print-with-franz-filter (form outstr)
  (cond ((eq form '|nil|) (format outstr "()"))
	(t (prin1 form outstr))))

(defun prepare-syntax (file &key (output-file (merge-pathnames ".flisp" file)))
  "makes a readable version of a franzlisp file that can serve as input for common lisp"
  (let ((tmp-file (merge-pathnames ".convert-tmp" file)))
    (when (probe-file tmp-file)
      (delete-file tmp-file))
    (convert-case file :output-file tmp-file)
    (normalize tmp-file :output-file output-file)
    (delete-file tmp-file)
    output-file))


;;;------------------------------------------------------------------------------------------
;;; User Functions
;;;------------------------------------------------------------------------------------------

(defvar *default-franz-file* ".l")

(defun convert-franz-to-common (franz-file &optional (common-file ".lsp")
				(protocol-file ".PROT"))
  "Converts FranzLisp file into CommonLisp file while protocoling"
  ;; Assuming (not (equal (pathname-type franz-file) "flisp"))
  (let ((*read-into-package* (find-package "FL"))
	(source-file (merge-pathnames
		       franz-file *default-franz-file*)))
    (declare (special *read-into-package*))
    (assert (probe-file source-file) (source-file) "File ~s not found" source-file)
    (setq *default-franz-file* source-file)
    (let ((flisp-file (prepare-syntax source-file)))
      (prog1 (tf flisp-file common-file protocol-file)
	(delete-file flisp-file)))))
  
  )

#+kcl
(defun help-convert ()
  (declare (special *fc-pathname-defaults*))
  (with-open-file (instream (merge-pathnames *fc-pathname-defaults*
					     "doc/help-text.text")
			    :direction :input)
    (do ((line (read-line instream nil) (read-line instream nil)))
	((null line))
      (write-line line))))

