;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:ZWEI; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B CPTFONTB); Vsp:0; Patch-file:T -*-

;1;; File "3ZWEI-METHOD-DOCUMENTATION*"*
;1;; Improving Document Function, Document Variable, and Edit Definition in innumerable ways.*
;1;; Finally they all get along in a reasonable way with Flavors and CLOS methods.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;     7 Mar 87*	1Jamie Zawinski*	1Created.*
;1;;5    *9 Mar 87*	1Jamie Zawinski*	1Added special-casing of SELF.*
;1;; 5  *10 Mar 87*	1Jamie Zawinski*	1Defined 5find-flavor-that-handles*; now this code 2always* finds 2something*...*
;1;;    17 Mar 87*	1Jamie Zawinski*	1Added functionality to 5com-describe-variable-at-point* and 5com-edit-definition.**
;1;;5   *19 Mar 87*	1Jamie Zawinski*	1Added 5*functions-meaning-send** variable.*
;1;; 5   *1 May 87*	1Jamie Zawinski*	1Added a hack to 5document-relevant-method* special-casing 5funcall-self.**
;1;;*				1Added better error messages when flavors or messages can't be parsed, and made it*
;1;;*				1  bug out faster if the form is sending to 3SELF* and 3SELF* can't be determined.*
;1;;5   *14 May 87*	1Jamie Zawinski*	1Implemented 5com-quick-arglist*.*
;1;;*				1made 5com-edit-definition* make the buffer read-only only if it didn't already exist.*
;1;;5    *6 Jun 87*	1Jamie Zawinski*	1changed 5document-function-call* to show the source file as well.*
;1;;*				1Made 5describe-variable-internal* say whether it's the name of a defstruct.*
;1;;5   *24 Jun 87*	1Jamie Zawinski*	1Made 5document-function-call* say so when the function called is defined in the current buffer.*
;1;;5    *7 Jun 87*	1Jamie Zawinski*	1Ported to 3.0, maintaining 2.1 compatibility.*
;1;;5   *25 Sep 87*	1Jamie Zawinski*	1Added support for 5defstruct* descriptions under 3.0.*
;1;;5    *5 Oct 87*	1Jamie Zawinski*	1Fixed bug in describing defstruct functions: was taking 3cdr* of a symbol...*
;1;;*				1Changed things so that instead of always (or always not) searching 5*all-flavor-names**, it*
;1;;*				1 3Y-OR-N-P*s the user.*
;1;;*				1Made 5C-Sh-V* show the default value of instance variables.*
;1;;5   *24 Jun 88*	1Jamie Zawinski*	1Cleaned up.  Renamed 5document-relevant-method* to be 5relevant-method-info*.*
;1;;*				1Made some error messages more sane.*
;1;;*				1Made it possible to type in the name of a flavor when it can't be determined.*
;1;;5    *5 Jul 88*	1Jamie Zawinski*	1Made it recognise microcoded functions.*
;1;;5   *13 Jul 88*	1Jamie Zawinski*	1Fixed 5current-form-self* - it was not coping with buffers with no defun intervals.*
;1;;5   *23 Sep 88*	1Jamie Zawinski*	1Made 5document-function-call* better handle microcode functions and defstruct BOA-constructors.*
;1;;   14 Jun 89*	1Jamie Zawinski *	1Modified 5document-function-call* to describe each sub-method of combined methods.*
;1;;   24 Aug 89*	1Jamie Zawinski *	1Made 5document-function-call-1* better handle lexical closures, including methods and daemon methods*
;1;;*				1 which are lexical closures.  Made it better handle instances of 5(defmethod (foo :bar) function-name)*.*
;1;;*  116 Nov 89*	1Jamie Zawinski *	1It didn't work for combined-methods that included 5:wrapper* methods (which are really macros).  Fixed.*
;1;;*				1Added conditionally-compiled CLOS support, which is as verbose as the flavors support.*
;1;;*  117 Nov 89*	1Jamie Zawinski *	1Made Edit Definition default to a consed-up function spec even if it can't compose methods (or whatever).*
;1;;*				1Cleaned up error messages some.*
;1;;*   12 Dec 89*	1Jamie Zawinski *	1Made it say what the method combination is (as in 5:or :base-flavor-last*).*
;1;;*

;1;; Synopsis: What this File Does and Why it Does It.*
;1;;*
;1;; *ZWEI::COM-QUICK-DOCUMENTATION1, *ZWEI::COM-QUICK-ARGLIST1, *ZWEI::DESCRIBE-VARIABLE-INTERNAL1, and *ZWEI::COM-EDIT-DEFINITION
;1;; are redefined.   These functions now understand Flavors much better.  *
;1;;*
;1;;   5Control-Shift-D* will describe the method call if the current form is a 5SEND*.  If it is not apparrent what*
;1;;     kind of a flavor the target of the send is (ie, it is not a declared varible and it is not 5SELF*) then, the*
;1;;     user may type in the name of the flavor, or have us search 5*ALL-FLAVOR-NAMES** until a suitable flavor is found.*
;1;;     Note that both 6(FUNCALL TV:MAIN-SCREEN :SIZE)* and 6(FUNCALL #'CAR '(A S D))* are documented properly.*
;1;;     If the function being described is a 5DEFSTRUCT* accessor, copier, or constructor, you are told that.*
;1;;     If the function is microcoded, you are told that.  If the function is SETFable, you are told that.*
;1;;     If the function being described is really a 3:COMBINED* method, then each element of the combined method is described as well.*
;1;;     If the function being described is lexically closed over some variables, the variables and their values are printed.*
;1;;     The source file of the function is printed.*
;1;;     If the function being described is a CLOS Generic Function, then the arglist (and argument types) of each of its methods are*
;1;;     shown.  The documentation and source file for each CLOS method is shown as well.*
;1;;*
;1;;   5Control-Shift-A* is just like 5Control-Shift-D*, except that it does not print the documentation string, and it*
;1;;     uses the Echo Area for output.  If a numeric argument is provided, and the function being described is a 3:COMBINED* method,*
;1;;     or the function is a CLOS Generic Function, then the arglists for all of the elements of the combined method are shown in a*
;1;;     random-typeout window.*
;1;;*
;1;;   5Control-Shift-V* now shows you the value of the variable you are describing.*
;1;;     If you are within a 5DEFMETHOD* (Flavors syntax), and the variable you are describing is an instance variable of the appropriate*
;1;;     flavor, it tells you what it knows about that instance variable (which unfortunately isn't much, since all it can find out is what type*
;1;;     of flavor you are dealing with, not which particular instance).  This isn't possible for CLOS defmethods, since CLOS has no concept*
;1;;     of SELF, and instance variables are not automatically bound within a method's scope.*
;1;;     It also tells you whether the symbol in question is the name of a flavor or defstruct.*
;1;;*
;1;;   5Meta-.* will default to the method-call at point, if there is one.  If the flavor type cannot be determined,*
;1;;     behavior is as for 5Control-Shift-D*: the user is queried as to whether to search 5*ALL-FLAVOR-NAMES**, or type*
;1;;     in the name of a flavor.*
;1;;     This command treats 5*EDIT-DEFINITION-IS-READ-ONLY** more reasonably.  The buffer is made read-only only if*
;1;;     it didn't already exist.  If it already exists and is not read only, the user probably 2wants* it that way.*
;1;;     Currently, nothing special is done for generic functions, but you should be given a choice of which method to edit.*
;1;;*
;1;;*

(defvar 4*functions-meaning-send** '(send funcall apply lexpr-funcall lexpr-send funcall-self)
  "2A list of those functions/macros which *ZWEI:COM-QUICK-DOCUMENTATION2 and *ZWEI:COM-EDIT-DEFINITION2 treat as Method calls.*")


(defun 4current-form-self* (point)
  "2If the form in which POINT lies is a DEFMETHOD, this returns a flavor-object corresponding to SELF.
What it returns is the flavor-object named in the CAADR of the current top-level form.
If the current top-level form is not of the form* ( 2<sexpr>* ( 2<flavor-name-symbol> ...* ) 2...* )2 this function returns NIL.*"
  (let* ((defun (defun-interval point 1 t nil))
	 (first-bp (if defun (interval-first-bp defun) (interval-last-bp *interval*)))
	 (bp (copy-bp first-bp)))
    (move-bp bp (forward-atom bp 1 t))                          ;1 move over the *"(DEFMETHOD"
    (move-bp bp (forward-over '(#\Space #\Tab #\Page #\() bp))  ;1 move to beginning of the first element of the name spec.*
    (let* ((string (string-interval bp (forward-atom bp 1 t)))  ;1 *STRING1 is the first elt of the list.*
	   (symbol (ignore-errors (read-from-string string))))  ;1 *SYMBOL1 is *NIL1, or the symbol that *STRING1 represents.*
      (and symbol
	   (values (get symbol 'SI::FLAVOR) symbol)))))


(defun 4find-flavor-that-handles* (message)
  "2Iterate through all of the flavors until one is found that handles MESSAGE.  This could take some time...*"
  (check-type message keyword)
  (dolist (symbol *all-flavor-names* nil)
    (declare (symbol symbol)
	     (optimize speed (safety 0) (space 0))
	     (inline GET MEMBER))
    (let* ((flavor (get symbol 'SI::FLAVOR)))
      (declare (type si:flavor flavor))
      (when ;(member (the keyword message) (the list (si:flavor-method-alist flavor)) :test #'eq :key #'car)
	    ;1; ## The compiler should perform this optimization, but it doesn't.  5sys:memq* is microcoded, but the above*
	    ;1; ## 5member* call is transformed into 5sys:member**, which is not as efficient.*
	    (sys:memq (the keyword message) (the list (si:flavor-method-alist flavor)))
	(return flavor)))))

(defvar 4*cant-determine-flavor-fquery-choices**
	'(:type :tyi
	  :choices (((:SEARCH "3Search Flavor List*") #\S)
		    ((:READ   "3Read a Flavor Name*") #\R)
		    ((NIL     "3Give Up*") #\N))))


(defun 4relevant-method-info* (point)
  "2Returns five values:  the function specification list, documentation string, argument list, the returned values list of
the method-call at POINT, and a string describing any problems that occurred getting this information.  Any of these values
may be NIL.
If the method-call is to SELF and we are inside of a DEFMETHOD, then the type of SELF is determined by what we are 
defining a method for.
If no method could be found by any other means and the user answers says its ok, then *ALL-FLAVOR-NAMES* is iterated through
until a flavor is found that handles the message at point.  This could take a while.*"
  (declare (values function-spec doc-string arglist return-list
		   problem-description-string problem-description-keyword bogus-fspec))
  (setq point (copy-bp point))
  (let* ((start-point (forward-up-list-or-string point -1))
	 (funcall-self-p nil))
    (when start-point (move-bp point start-point))
    (move-bp point (forward-char point 1 t)) ;1 Get into the list*
    ;1; ## HACK to take care of the different syntax of FUNCALL-SELF.*
    (if (string-equal (string-trim '(#\Space #\Tab #\Return) (string-interval point (forward-sexp point 1)))
		      "3FUNCALL-SELF*")
	(setq funcall-self-p t)
	(move-bp point (forward-sexp point 1 t)) ;1 skip over SEND*
	)
    (let* ((had-to-search-p nil)  ;1 T if we had to search **ALL-FLAVOR-NAMES*
	   (doing-self-p nil)     ;1 T if the flavor we're using is *SELF
	   temp
	   (message-string (string-interval (setq temp (forward-sexp point 1 t))
					    (forward-sexp temp 1 t) t t))
	   (message (ignore-errors (read-from-string message-string nil nil)))
	   (flavor-string (if funcall-self-p
			      "3SELF*"
			      (string-trim '(#\Space #\Tab #\Return)
					   (string-interval point (forward-sexp point 1 t) t t))))
	   flavor function-object function-spec)
      (cond ((string-equal flavor-string "3SELF*")
	     (setq doing-self-p t)
	     (setq flavor (current-form-self point)))
	    (t (ignore-errors (setq flavor (eval (read-from-string flavor-string nil nil))))))  ;1 Ick.  Calling EVAL.*
      (unless flavor
	(when (and (keywordp message) (not doing-self-p))
	  (case (fquery *cant-determine-flavor-fquery-choices*
		3 *"2Can't determine ~:@(~A~)'s flavor type.  Search flavor list, Read a flavor name, or give up? *"
		  flavor-string)
	    (:SEARCH (setq had-to-search-p t)
		     (setq flavor (find-flavor-that-handles message)))
	    (:READ   (setq flavor (read-flavor-name "2Flavor Name*"
				    (format nil "2Type the name of a flavor (with completion) which handles the ~S message.*" message)))
		     (setq flavor (get flavor 'SI::FLAVOR)))
	    (t nil))))
      (when (and flavor message)
	(setq function-object (if (typep flavor 'INSTANCE)
				  (get-handler-for flavor message)
				  ;1; This might signal an error if the flavor is not instantiable/can't be composed; ignore it.*
				  (ignore-errors (si:get-flavor-handler-for (si:flavor-name flavor) message)))))
      (when (functionp function-object)
	(setq function-spec (or (function-name function-object) function-object)))
      (let* ((documentation (if (listp function-object)
				(when (stringp (fourth function-object))
				  (fourth function-object))
				;1; Sometimes #'documentation craps out.*
				(ignore-errors (documentation function-object 'function))))
	     (arglist (if (listp function-object)
			  (third function-object)
			  ;1; Sometimes #'arglist craps out.*
			  (ignore-errors (cdr (arglist function-spec)))))
	     (return-list (unless (listp function-object)
			    ;1; Sometimes #'arglist craps out.*
			    (ignore-errors (multiple-value-bind (arglist return-list) (arglist function-spec)
					     (declare (ignore arglist))
					     return-list))))
	     (problem-key (cond (function-spec nil)  ;1 If we have this, we're winning.*
				((not (keywordp message))		:NO-MESSAGE)
				((and (null flavor) had-to-search-p)	:NO-HANDLING-FLAVOR)
				((null flavor)				:NO-FLAVOR-TYPE)
				((null function-spec)
				 (typecase flavor
				   (SI::FLAVOR
				    (if (and (sys:flavor-method-hash-table flavor)
					     (sys:flavor-depends-on-all flavor))
					:NO-MESSAGE-HANDLER
					:COMPOSITION-ERROR))
				   (SI::INSTANCE			:NO-MESSAGE-HANDLER)
				   (t					:INVALID-FLAVOR)))
				(t 					:UNKNOWN-PROBLEM)))
	     (bogus-fspec nil)
	     (problems    (case problem-key
			    (:no-message "2Could not parse a message.*")
			    (:no-handling-flavor
			     (format nil "2Invalid Flavor or Method: Could not find any flavor that handles ~S*" message))
			    (:no-flavor-type (format nil "2Couldn't determine ~:@(~A~)'s flavor-type.*" flavor-string))
			    (:no-message-handler
			     (let* ((name (if (typep flavor 'si:flavor) (si:flavor-name flavor) (type-of flavor))))
			       (setq bogus-fspec `(:method ,name ,message))
			       (format nil "2Flavor ~S does not handle the ~S message.*" name message)))
			    (:composition-error
			     (setq bogus-fspec `(:method ,(si:flavor-name flavor) ,message))
			     (format nil "2Could not compose ~S's methods.*" (si:flavor-name flavor)))
			    (:invalid-flavor (format nil "2Invalid Flavor: ~S is not a flavor or instance.*" flavor))
			    (:unknown-problem "2Couldn't find a method... don't know why...*")
			    (t (when problem-key (error "2Internal error: unknown problem keyword ~S*" problem-key))))))
	(values function-spec documentation arglist return-list problems problem-key bogus-fspec)))))


;1;;; Documenting Function or Method Calls*

(defcom 4com-quick-documentation* 2"Prints documentation for the function at point, special-casing SEND to document the method."* ()
  (document-function-call t)
  dis-none)

(defcom 4com-quick-arglist* 2"Prints the arglist for the function at point, special-casing SEND to use the method.  
 With a numeric argument, show the arglists for all *:COMBINED2 methods as well."* ()
  (document-function-call nil)
  dis-none)

(defun 4document-function-call* (&optional (doc-string-p t))
  "2Prints the documentation and arglist of the function at point, special-casing SEND to document the method.
If DOC-STRING-P is NIL, then only the arglist is printed.*"
  (let* ((name (relevant-function-name (point))))
    (cond ;1;*
          ;1; Describe methods.*
          ;1;*
          ((member name *functions-meaning-send*)
	   (multiple-value-bind (fspec doc arglist return-list problem-string)
				(relevant-method-info (point))
	     (let* ((stream (if doc-string-p *standard-output* *query-io*)))
	       (cond (fspec
		      (document-function-call-1 fspec doc arglist return-list doc-string-p stream))
		     (t (fresh-line stream)
			(princ problem-string stream)))
	       (send stream :send-if-handles :typeout-stays))))
	  ;1;*
	  ;1; Describe normal function calls.*
	  ;1;*
	  ((null name) (barf "3No valid function at point.*"))
	  
	  (t  (unless (fdefinedp name) (barf "3~S is not defined.*" name))
	      (let* ((doc (when doc-string-p (documentation name 'function)))
		     (stream (if doc-string-p *standard-output* *query-io*)))
		(multiple-value-bind (arglist return-list) (arglist name)
		  (document-function-call-1 name doc arglist return-list doc-string-p stream))
		(send stream :send-if-handles :typeout-stays))))))


(defun get-closure-values 4(*lcl4)*
  (when (symbolp lcl) (setq lcl (symbol-function lcl)))
  (let* ((c (sys:convert-closure-to-list lcl))
	 (env (cdr c))
	 (env-vec (sys:%make-pointer sys:dtp-list env))
	 (dbi (sys:get-debug-info-struct (car c)))
	 (parent-info (sys:get-debug-info-field dbi :lexical-parent-debug-info))
	 (result '()))
    (do* ((i 0 (1+ i))
	  (levels (sys:closure-levels? env-vec))
	  (env-vec env-vec (sys:%make-pointer sys:dtp-list (car env-vec)))
	  (pnt-info parent-info (sys:get-debug-info-field pnt-info :lexical-parent-debug-info))
	  (vars (sys:get-debug-info-field pnt-info :variables-used-in-lexical-closures)
		(sys:get-debug-info-field pnt-info :variables-used-in-lexical-closures)))
	 ((> i levels))
      (push (list (sys:get-debug-info-field pnt-info :name))
	    result)
      (do* ((j 0 (1+ j))
	    (vals (cdr env-vec) (cdr vals))
	    (val (car vals) (car vals))
	    (vrs vars (cdr vrs))

	    (var (car vrs) (car vrs)))
	    ((null vrs))
	(push (cons var val) (cdr (car result)))))
    (nreverse result)))


(defun 4document-function-call-1 *(fspec doc arglist return-list doc-string-p stream &optional recursive-p type)
  ;1;*
  ;1; If the thing passed in is a function-object, grab its name.*
  ;1; If the thing passed in is a macro (as in 5(MACRO . #<function ... >)*), then grab the macro-function object.*
  ;1; If the thing is a closure object and the doc-string is nil, then try to get a doc-string out of the closure's function.*
  (when (and (consp fspec) (eq (car fspec) 'MACRO) (functionp (cdr fspec)))
    (setq fspec (cdr fspec)
	  type (or type :MACRO)))
  
  (cond #+CLOS
	((typep fspec 'ticlos:generic-function)
	 (let* ((function (send fspec :function)))
	   (setq fspec (function-name function t)
		 doc (or doc (documentation function 'function))
		 type (or type :GENERIC-FUNCTION))))
	
	((compiled-function-p fspec)
	 (setq fspec (function-name fspec t)))

	((or (closurep fspec)
	     (and (symbolp fspec) (closurep (symbol-function fspec))))
	 (let* ((function (closure-function (if (symbolp fspec) (symbol-function fspec) fspec))))
	   (setq fspec (function-name function t)
		 doc (or doc (documentation function 'function))
		 type :CLOSURE)))
	)
  (when (and (null type) (symbolp fspec))
    (cond ((macro-function fspec) (setq type :MACRO))
	  ((special-form-p fspec) (setq type :SPECIAL-FORM))
	  #+CLOS ((ticlos:generic-function-p fspec) (setq type :GENERIC-FUNCTION))
	  ))

  (document-function-call-print-arglist-internal fspec arglist return-list type stream recursive-p)
  
  (unless recursive-p
    (document-function-call-print-docstrings-internal fspec doc doc-string-p stream recursive-p))
  
  (let* ((print-other-stuff
	   (cond #+CLOS ((eq recursive-p :CLOS) t)
		 #+CLOS ((ticlos:generic-function-p fspec)
		  (document-function-call-generic-internal fspec doc arglist return-list doc-string-p stream recursive-p))
		 ((symbolp fspec)	;1 Magic info that doesn't apply for flavors-methods.*
		  (document-function-call-functions-internal fspec doc arglist return-list doc-string-p stream recursive-p))
		 (t			;1 Magic info that applies only to methods.*
		  (document-function-call-methods-internal fspec doc arglist return-list doc-string-p stream recursive-p)))))
    (when recursive-p
      (document-function-call-print-docstrings-internal fspec doc (and doc-string-p print-other-stuff) stream recursive-p))
    ))


(defun document-function-call-print-arglist-internal (fspec arglist return-list type stream recursive-p)
  (cond #+CLOS
	((and recursive-p (consp fspec) (eq (car fspec) 'TICLOS:METHOD))
	 (let* ((name (second fspec)))
	   (when (typep name 'ticlos:generic-function)
	     (setq name (function-name (ticlos:generic-function-discriminator-code name) t)))
	   (prin1 name stream))
	 (when (and (null type) (symbolp (third fspec))) (setq type (third fspec)))
	 (when type (format stream "3 (~(~A~))*" type))
	 (princ "3: *" stream)
	 (if (listp arglist)
	     (let* ((new-arglist '())
		    (types (car (last fspec))))
	       (dolist (arg arglist)
		 (if (member arg LAMBDA-LIST-KEYWORDS :test #'eq)
		     (push arg new-arglist)
		     (let* ((type (pop types)))
		       (if (or (eq type T) (null types) (consp arg))
			   (push arg new-arglist)
			   (push (list arg type) new-arglist)))))
	       (setq new-arglist (nreverse new-arglist))
	       (print-arglist-internal new-arglist stream))
	     (princ arglist stream)))
	(t
	 (prin1 fspec stream)
	 (when type (format stream "3 (~(~A~))*" type))
	 (princ "3: *" stream)
	 (when (and recursive-p (consp arglist)) (pop arglist))	;1 Take off the 3SYS:.OPERATION.* argument.*
	 (if (listp arglist)
	     (print-arglist-internal arglist stream)
	     (princ arglist stream))))
  (when return-list
    (format stream "3  ~:A*" return-list)))


(defun document-function-call-print-docstrings-internal4 *(fspec doc doc-string-p stream recursive-p)
  (when doc-string-p
    (unless recursive-p
      (when (and (symbolp fspec) (get fspec 'sys:setf-method))
	(format stream "3~&SETFable.*")))
    (if doc
	(format stream "3~2&~A*" doc)
	(unless recursive-p
	  (if (consp fspec)
	      (format stream "3~&~{~S ~}is not documented.*" (cddr fspec))
	      (format stream "3~&~S is not documented.*" fspec)))))
  
  (when (closurep (fdefinition fspec)) (document-function-calls-closure-internal fspec))
  (let* ((file (si::get-source-file-name fspec 'defun))
	 (buffer-file (buffer-pathname *interval*))
	 (this-one-p (and file
			  (equal (pathname-name file) (pathname-name buffer-file))
			  (equal (pathname-directory file) (pathname-directory file))
			  (equal (pathname-host file) (pathname-host file)))))
    (when file (if this-one-p
		   (format stream "3~&Defined in this buffer, ~A~%*" file)
		   (format stream "3~&Defined from ~A~%*" file)))))



(defun 4document-function-call-functions-internal* (fspec doc arglist return-list doc-string-p stream recursive-p)
  (declare (ignore arglist return-list doc doc-string-p recursive-p))  ;1 Duty now for the future.  Maybe we'll need them someday.*
  (let* ((mcr-p (or (sys:microcode-function-p fspec)
		    (get fspec 'COMPILER::OPCODE)
		    (get fspec 'COMPILER::MISC-VAL)))
	 (parent (car (sys:get-debug-info-field (sys:get-debug-info-struct (symbol-function fspec))
						:function-parent)))
	 )
    (cond (mcr-p
	   (format stream "3~&This function is implemented in microcode.*"))
	  (parent
	   (let* ((description (and parent (get parent 'sys::defstruct-description)))
		  (fields (and description (sys:defstruct-description-slot-alist description)))
		  (field (find-if #'(lambda (x) (member fspec x :test #'eq)) fields))
		  (slot-name (car field)))
	     (cond (slot-name
		    (format stream "3~&This is a DEFSTRUCT accessor for the ~S slot of ~S structures.*"
			    slot-name parent))
		   
		   ((eq fspec (sys:defstruct-description-predicate description))
		    (format stream "3~&This is a DEFSTRUCT type-predicate for ~S structures.*" parent))
		   
		   ((eq fspec (sys::defstruct-description-copier description))
		    (format stream "3~&This is a DEFSTRUCT copier for ~S structures.*" parent))
		   
		   ((member fspec (sys::defstruct-description-constructors description)
			    :key #'(lambda (x) (if (consp x) (car x) x))
			    :test #'eq)
		    (format stream "3~&This is a DEFSTRUCT constructor function for ~S structures.*"
			    parent))
		   
		   (t (format stream "3~&This function's parent is ~S.*" parent)))))
	  ))
  t  ;1 return T meaning do the rest.*
  )


#+CLOS
(defun 4document-function-call-generic-internal* (fspec doc arglist return-list doc-string-p stream recursive-p)
  (declare (ignore arglist return-list doc recursive-p))  ;1 Duty now for the future.  Maybe we'll need them someday.*
  (when (or doc-string-p *numeric-arg-p*)
    (setq stream *standard-output*)		;1 Always use a random-typeout window here - many lines will be printed.*
    (if doc-string-p
	(format stream "3~2&This is a Generic Function.  CLOS Methods:*")
	(format stream "3~2&This is a Generic Function.  CLOS Method arglists:*"))
    (dolist (clos-method (ticlos:generic-function-methods (fdefinition fspec)))
      (format stream "3~2&----------~%*")
      (let* ((fdef (send clos-method :function)))
	(multiple-value-bind (arg ret) (arglist fdef)
	  (document-function-call-1 fdef (documentation fdef 'function) arg ret doc-string-p stream :CLOS)))))
  t  ;1 return T meaning do the rest.*
  )


(defun 4document-function-call-methods-internal* (fspec doc arglist return-list doc-string-p stream recursive-p)
  (declare (ignore arglist return-list doc))  ;1 Duty now for the future.  Maybe we'll need them someday.*
  (when (and (or doc-string-p *numeric-arg-p*) (not recursive-p))
    (let* ((fef (fdefinition fspec))
	   (debug-info (sys:get-debug-info-struct fef))
	   (combined-derivation (and debug-info (getf (sys:get-debug-info-field debug-info :plist)
						      'sys:combined-method-derivation)))
	   (combined-name (car combined-derivation))
	   (daemon-type (second combined-derivation))
	   (daemon-order (third combined-derivation))
	   (combined-info (cdddr combined-derivation)))
      (setq combined-info
	    ;1;*
	    ;1; Sort the combined info into the order of execution - wrapper, around, before, primary, and after.*
	    ;1; Others (others?) go at the end.*
	    (nconc (reverse   (cdr (assoc :wrapper combined-info :test #'eq)))
		   (reverse   (cdr (assoc :around  combined-info :test #'eq)))
		   (reverse   (cdr (assoc :before  combined-info :test #'eq)))
		   (reverse   (cdr (assoc nil      combined-info :test #'eq)))
		   (copy-list (cdr (assoc :after   combined-info :test #'eq)))
		   (mapcan #'(lambda (x)
			       (and (not (member (car x) '(:wrapper :around :before nil :after) :test #'eq))
				    (reverse (cdr x))))
			   combined-info)))
      (when combined-derivation
	(setq stream *standard-output*)		;1 Always use a random-typeout window here - many lines will be printed.*
	(cond ((and daemon-type daemon-order)
	       (format t "3~2&Method combination for ~S is ~S ~S*" combined-name daemon-type daemon-order))
	      (daemon-type  (format t "3~2&Method combination for ~S is ~S*" combined-name daemon-type))
	      (daemon-order (format t "3~2&Method order for ~S is ~S*" combined-name daemon-order)))
	(dolist (x combined-info)
	  (format stream "3~2&----------~%*")
	  ;1;*
	  ;1; Get the function definition of this method and document that.  This is necessary for methods which are lexical closures;*
	  ;1; if we don't pass in the function object (actually, the closure object) we won't get the docstring.*
	  ;1;*
	  (let* ((fdef (or (ignore-errors (fdefinition x)) x))
		 (fdef-is-symbolic-function (or (symbolp fdef)
						(and (compiled-function-p fdef) (symbolp (function-name fdef))))))
	    ;1;*
	    ;1; If the name of the real definition of this method is not a list, then this method is a named-function, as produced*
	    ;1; by 5(defmethod (foo :bar) function-name)*. * 1We pass in the name of the position which this function occupied, so*
	    ;1; the* 1user will see something like* 5FUNCTION-NAME (before) (ARG ARG)1. ** 1Without this, one would only know what*
	    ;1; kind* 1of method this was by its* 1position.*
	    ;1;*
	    (multiple-value-bind (arg ret) (arglist fdef)
	      (document-function-call-1 fdef (documentation fdef 'function) arg ret doc-string-p stream
					(if fdef-is-symbolic-function nil :RECURSIVE)
					(if fdef-is-symbolic-function
					    (if (cdddr x) (third x) :PRIMARY)
					    nil))
	      )))
	nil  ;1 return NIL meaning bug out.*
	))))


(defun document-function-calls-closure-internal (fspec)
  (let* ((lex-closure-vars (get-closure-values (fdefinition fspec))))
    (when lex-closure-vars
      (terpri) (terpri)
      (let* ((i 0))
	(dolist (list lex-closure-vars)
	  (format t "3~&Lexically scoped variables at level ~D (from function ~S):*" i (car list))
	  (incf i)
	  (dolist (pair (cdr list))
	    (format t "3~&  ~20S ~S*" (car pair) (cdr pair)))))
      (terpri) (terpri))))


;1;;; Documenting Variables*

(defun 4describe-variable-internal* (var)
  (let* ((decl (getl var '(special compiler:system-constant)))
	 (bound (boundp var))
	 (doc (documentation var 'variable))
	 (stream (if (or doc bound) *standard-output* *query-io*))
	 (flavor-p (get var 'sys::flavor))
	 (struct-p (get var 'si::defstruct-description))
	 (current-flavor (current-form-self (point)))
	 (relevant-instance-vars (and current-flavor
				      (ignore-errors (si::flavor-all-instance-variables current-flavor))))
	 (know-something (or bound doc flavor-p struct-p)))
    ;1;*
    ;1; Describe instance variables of the current method.*
    ;1;*
    (when (member (the symbol var) (the list relevant-instance-vars) :test #'eq)
      (setq know-something t)
      (let* ((specp (member var (the list (si::flavor-special-instance-variables current-flavor))  :test #'eq))
	     (getp  (member var (the list (si::flavor-gettable-instance-variables current-flavor)) :test #'eq))
	     (setp  (member var (the list (si::flavor-settable-instance-variables current-flavor)) :test #'eq))
	     (all-vars (sys:flavor-all-instance-variables current-flavor))
	     (all-inits (sys:flavor-instance-variable-initializations current-flavor))
	     (default (cdr (assoc (position var all-vars :test #'eq) all-inits :test #'=))))
	(format stream "3~&~S is ~A~A instance variable of ~S.*"
		var
		(if specp
		    "3a Special*"
		    (if (or getp setp) "3a*" "3an*"))
		(cond ((and getp setp) "3 gettable and settable*")
		      (getp "3 gettable*")
		      (setp "3 settable*")
		      (t ""))
		(si::flavor-name current-flavor))
	(when default
	  (let* ((*print-pretty* t)
		 (*print-circle* t)
		 (*print-length* 2)
		 (*print-level* 2))
	    (format stream "3  It's default value is ~S.*" (car default))))))
    (when (or decl bound doc flavor-p struct-p)
      (setq know-something t)
      ;1;*
      ;1; Say whether this symbol names a flavor or a structure.*
      ;1;*
      (when (or flavor-p struct-p)
	(format stream "3~&~S is the name of ~A*" var
		(cond ((and flavor-p struct-p) "3a flavor and a defstruct.*")
		      (flavor-p "3a flavor.*")
		      (struct-p "3a defstruct.*"))))
      ;1;*
      ;1; Describe it's value.*
      ;1;*
      (if bound
	  (let* ((*print-pretty* t)
		 (*print-circle* t)
		 (*print-length* 5)
		 (*print-level* 5)
		 (val-str (prin1-to-string (symbol-value var))))
	    (format stream "3~&~S has a ~S as its value*" var (type-of (symbol-value var)))
	    (if (or (> (length val-str) 25) (position #\Return val-str))
		(format stream "3:~% ~A~%*" val-str)
		(format stream "3: ~A *" val-str)))
	  (format stream "3~&~S has no value*" var bound))
      ;1;*
      ;1; Say where it came from.*
      ;1;*
      (when (eq (car decl) 'special)
	(format stream "3 and is declared special~:[ by file ~A~]*"
		(eq (cadr decl) t) (cadr decl)))
      (when (eq (car decl) 'compiler:system-constant)
	(format stream "3 and is a system-constant*"))
      (princ #\. stream)
      ;1;*
      ;1; Show documentation.*
      ;1;*
      (if doc
	  (format stream "3~&Documentation:~%~A*" doc))
      know-something)))


;1;; Editing Definitions*

(defcom 4com-edit-definition*
	2"Go to the definition of a specified function or method.
The name of the function is read from the mini-buffer, defaulting to the function or method call at point.
Relevant variables:
  ZWEI::*EDIT-DEFINITION-IS-READ-ONLY*,  ZWEI::*FIND-PATCH-DEFINITIONS-TOO*,  ZWEI::*FUNCTIONS-MEANING-SEND*
The buffer will be made read-only only if *EDIT-DEFINITIONS-IS-READ-ONLY* is T and the buffer did not already exist."* ()
  (if *numeric-arg-p*
      (edit-next-definition)
      (let* ((function-spec (relevant-function-name (point)))
	     (buffers-that-existed (copy-list *zmacs-buffer-list*)))  ;1 Check to see if any buffers get created here.*
	(when (member function-spec *functions-meaning-send*)
	  (multiple-value-bind (fspec ignore ignore ignore ignore ignore bogus-fspec)
			       (relevant-method-info (point))
	    (setq function-spec (or fspec bogus-fspec function-spec))))
	(multiple-value-bind (spec string explicit-package-p)
			     (read-function-name "3Edit definition*" function-spec 'aarray-ok 'multiple-ok)
	  (or (and (consp spec)
		   (neq (car spec) :method))
	      (setq spec (list spec)))
	  ;1; If there's only one entry in the aarray, and its for a different package,*
	  ;1; but the symbol in the current package has some sort of definition in a file,*
	  ;1; include them both.*
	  (if (and (not explicit-package-p) (symbolp (car spec)))
	      (multiple-value-bind (this-pkg-symbol foundp)
		  (find-symbol (string-upcase (string (car spec))))
		(if (and foundp
			 (not (equal this-pkg-symbol (car spec)))
			 (get this-pkg-symbol :source-file-name))
		    (push this-pkg-symbol spec))))
	  (edit-definition-1 (car spec) (if explicit-package-p t spec) string))
	;1; Make the Edit Definition Buffer be Read-Only ONLY if it didn't already exist.*
	(when (and *edit-definition-is-read-only* (not (member *interval* buffers-that-existed)))
	  (make-buffer-read-only *interval*))))
  DIS-TEXT)
