;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:UUMAP; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B) -*-

;1;; File "3UUMAPS*"*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;    14 Jun 89*	1Jamie Zawinski*	1 Created.*
;1;;*


(in-package "3UUMAP*")


(eval-when (load eval compile)
 (defconstant 4VALID-FIELD-NAMES *'(#\N #\S #\O #\C #\E #\T #\P #\L #\R #\U #\W #\F))
 )

(defconstant 4LOCAL*	425*	"2Cost of local area network.*")
(defconstant 4DEDICATED*	495*	"2Cost of high speed dedicated.*")
(defconstant 4DIRECT*	4200*	"2Cost of local call.*")
(defconstant 4DEMAND*	4300*	"2Cost of normal call (long distance, anytime).*")
(defconstant 4HOURLY*	4500*	"2Cost of hourly poll.*")
(defconstant 4EVENING*	41800*	"2Cost of time restricted call.*")
(defconstant 4DAILY*	45000*	"2Cost of daily poll.*")
(defconstant 4WEEKLY*	430000*	"2Cost of irregular poll.*")
(defconstant 4DEAD*	4MOST-POSITIVE-FIXNUM*	"2Cost of an unusable path.*")

(defconstant 4POLLED*	4DAILY*	"2Cost of a nightly poll.*")

(defconstant 4HIGH*	4-5)*
(defconstant 4LOW*	4+5)*
(defconstant 4FAST*	4-80)*
(defconstant 4SLOW*	4 *804)*

(defconstant 4DEFAULT*	4000	4"2The cost to use if no other was specified.**")
(defconstant 4NETWORK*	40*	4"2The cost of connecting to a network.**")


(defstruct 4(connection *(:print-function %print-connection))
  (name nil :type symbol)
  (cost 0   :type fixnum)
  (char #\! :type string-char)
  (symbolic-cost nil :type list)
  (terminal-p nil :type (member T NIL))
  )

(defstruct 4(site *(:print-function %print-site))
  (name nil :type symbol)	;1 3#N**	1UUCP name of site.*
  (aliases nil :type list)	;	1A continuation of the 3#N* field.*
  machine			;1 3#S**	1manufacturer machine model; operating system & version.*
  organization			;1 3#O**	1organization name.*
  contact-name			;1 3#C**	1contact person's name.*
  contact-email			;1 3#E**	1contact person's electronic mail address.*
  contact-telno			;1 3#T**	1contact person's telephone number.*
  address			;1 3#P**	1organization's postal address.*
  lat/long			;1 3#L**	1latitude / longitude in string format.*
  remarks			;1 3#R**	1remarks.*
  netnews-neighbors		;1 3#U**	1netnews neighbors.*
  last-edit			;1 3#W**	1who last edited the entry ; date edited.*
  )


(defun 4%print-connection *(struct stream ignore)
  (format stream "3#<~S ~A ~A*" (type-of struct)
	  (connection-name struct)
	  (connection-symbolic-cost struct))
  (when (connection-terminal-p struct) (princ "3 term*" stream))
  (princ #\> stream))

(defun 4%print-site *(struct stream ignore)
  (princ "3#<*" stream)
  (prin1 (type-of struct) stream)
  (princ #\Space stream)
  (princ (or (site-name struct) "3??*") stream)
  (when (site-aliases struct)
    (princ #\Space stream)
    (princ (site-aliases struct) stream))
  (princ "3>*" stream))


(defun 4describe-site *(site &optional (verbose t))
  (when (symbolp site) (setq site (or (get site 'SITE) site)))
  (check-type site site)
  (format t "3~2&Site:      ~A*" (site-name site))
  (when (site-aliases site) (format t "3 (~A~{, ~A~})*" (car (site-aliases site)) (cdr (site-aliases site))))
  (terpri)
  (when (site-machine site)         (format t "3~&Machine:   ~A*" (site-machine site)))
  (when (site-organization site)    (format t "3~&Org:       ~A*" (site-organization site)))
  (when verbose
    (when (site-contact-name site)    (format t "3~&Contact:   ~A*" (site-contact-name site)))
    (when (site-contact-email site)
      (if (site-contact-name site)
	3  *(format t "3 (email: ~A)*" (site-contact-email site))
	3  *(format t "3~&Contact:     ~A*" (site-contact-email site))))
    (when (site-contact-telno site)   (format t "3~&Telephone:* 3~A*" (site-contact-telno site))))
  (when (site-address site)         (format t "3~&Address:  * 3~A*" (site-address site)))
  (when verbose
    (when (site-lat/long site)        (format t "3~&Lat/Long: * 3~A*" (site-lat/long site))))
  (when (site-remarks site)         (format t "3~&Remarks:   ~A*" (site-remarks site)))
  (when verbose
    (when (site-netnews-neighbors site)
      (format t "~&3Netnews:* 3 *~{~<~%3~10t*~13,110*:; ~3A*~>~^,~}." (site-netnews-neighbors site))))
  (terpri)
  (when verbose
    (let* ((conns (get (site-name site) 'connections))
	   (*read-default-float-format* 'short-float)) ;1 Don't print as 1.0s0*
      (when conns (format t "3~&Mail:*"))
      (dolist (conn conns)
	(format t "3~11t~A~30t~A~55t~D~%*"
		(connection-name conn) (connection-symbolic-cost conn)
		(if (>= (connection-cost conn) most-positive-fixnum)
		    "3N/A*"
		    (connection-cost conn))))))
  (values))


(defmacro 4site-field-access-1 *(site field-char &optional (newval nil newval-supplied-p))
  (check-type field-char string-char)
  (let* ((name (case field-char
		 (#\N 'name)
		 (#\S 'machine)
		 (#\O 'organization)
		 (#\C 'contact-name)
		 (#\E 'contact-email)
		 (#\T 'contact-telno)
		 (#\P 'address)
		 (#\L 'lat/long)
		 (#\R 'remarks)
		 (#\W 'last-edit)
		 (#\U 'netnews-neighbors)
		 (t nil)))
	 (accessor (and name (intern (string-append "3SITE-*" (string name))))))
    (when accessor
      (if newval-supplied-p
	  `(setf (,accessor ,site) ,newval)
	  `(,accessor ,site)))))

(defsetf 4site-field-access-1 *site-field-access-1)

(defmacro 4site-field-access *(site field-char &optional (newval nil newval-supplied-p))
  (let* ((body '()))
    (dolist (c VALID-FIELD-NAMES)
      (push (list c (if newval-supplied-p
			`(setf (site-field-access-1 ,site ,c) ,newval)
			`(site-field-access-1 ,site ,c)))
	    body))
    `(case ,field-char
       ,@(nreverse body)
       (t ,(when newval-supplied-p
	     `(warn "3Line ~D: Unknown field ~S*" line-number ,field-char))))))

(defsetf 4site-field-access *site-field-access)


(defun 4add-field *(site field-char field-line line-number)
  (cond ((char= field-char #\N)
	 (let* ((new (tokenize field-line 3)))
	   (unless (site-name site)
	     (setf (site-name site) (pop new))
	     (remprop (site-name site) 'CONNECTIONS)
	     (remprop (site-name site) 'ALIAS-OF)
	     )
	   (when new
	     (setf (site-aliases site)
		   (nconc (site-aliases site) new)))))
	(t
	 (let* ((old (site-field-access site field-char))
		(value (cond ((char= field-char #\U)
			      (append old (tokenize field-line 3)))
			     (t
			      (let* ((nonwhite (or (position-if-not #'(lambda (x) (member x '(#\Space #\Tab) :test #'char=))
								    field-line :start 3)
						   (length field-line)))
				     (new-str (subseq field-line (or nonwhite 3))))
				(if (string= "" new-str)
				    (setq new-str nil)
				    (nsubstitute #\Space #\Tab new-str))
				(if old
				    (string-append old #\Newline #\Tab (or new-str ""))
				    new-str))))))
	   (setf (site-field-access site field-char) value)))))


(defun 4tokenize* (string &optional (start 0) end &optional (intern-p t))
  (unless end (setq end (length string)))
  (macrolet ((make-token (string start end)
	       `(if intern-p
		    (intern (subseq ,string ,start ,end) "3UUMAP*")
		    (subseq ,string ,start ,end))))
    (let* ((tokens '())
	   (last-nonwhite-pos nil))
      (do* ((pos start (1+ pos)))
	   ((>= pos end)
	    (when last-nonwhite-pos
	      (push (make-token string last-nonwhite-pos end) tokens)))
	(let* ((c (char string pos)))
	  (cond ((and (graphic-char-p c) (char/= c #\Space) (char/= c #\,))
		 (unless last-nonwhite-pos (setq last-nonwhite-pos pos)))
		(last-nonwhite-pos                            ;1 we've encountered whitespace after nonwhite.*
		 (push (make-token string last-nonwhite-pos pos) tokens)
		 (setq last-nonwhite-pos nil)
		 (setq start pos)))))
      (nreverse tokens))))


(defun 4valid-site-line-p *(line)
  (let* ((length (length line)))
    (when (and (> length 1)
	       (char= (char line 0) #\#)
	       (member (char line 1) VALID-FIELD-NAMES :test #'char=)
	       (or (= length 2) (member (char line 2) '(#\Space #\Tab))))
      (char line 1))))


(defvar 4*site-names* *'())


(defun 4parse-site-file *(pathname)
  (with-open-file (stream pathname :direction :input :characters t)
    (let* ((state nil)
	   (site nil)
	   (line-number -1))
      (loop
	(incf line-number)
	(block LINE
	  (let* ((line (or (read-line stream nil nil)	;1 At EOF, return, unless we haven't reached the final state, in which case*
			   (if site "" (return))))	;1 pretend we read one more blank line.*
		 (length (length line))
		 (starts-with-# (and (> length 0) (char= #\# (char line 0))))
		 (valid-p (and starts-with-# (valid-site-line-p line)))
		 (blank-p (and (not (or valid-p starts-with-#))
			       (string= "" (string-trim '(#\Space #\Tab) line))))
		 (comment-p (and starts-with-#				;1 First char is #.*
				 (or (and (> length 1)
					  (or (char= #\Tab (char line 1))
					      (char= #\Space (char line 1))))
				     (or (< length 3)			;1 No Third char, or third is not Tab.*
					 (char/= #\Tab (char line 2))))))
		 )
	    
;	    (format t "3~& ~s ~s ~s ~s ~s~60t ~s*" valid-p blank-p comment-p state line)
	    
	    (cond (blank-p (when site
			     (setf (get (site-name site) 'site) site)
			     (pushnew (site-name site) *site-names*))
			   (setq state 'BETWEEN)
			   (setq site nil))
		  
		  ;1; An invalid line at the front of the file.  Ignore it.*
	          ((and (not valid-p) (null state))
		   nil)
		  
		  ;1; We have read a field, and this line begins with tabs/spaces, or has the same code as the previous line.*
		  ;1; Append it to the previous field.*
		  ((and (characterp state)
			(> (length line) 2)
			(or (member (char line 0) '(#\Space #\Tab) :test #'char=)
			    (and valid-p
				 (char= state (char line 1)))))
		   (add-field site state line line-number))
		  
		  (valid-p
		   (case state
		     (SITE    (add-field site valid-p line line-number))	;1 A normal field, after #N.*
		     (BETWEEN (cond ((char/= valid-p #\N)
				     ;1; When coming out of BETWEEN state, the field must be #N.*
				     ;1; If it isn't, we give the user the opportunity to go to the NEMO state, meaning ignore this block.*
				     (unless (eq state 'NEMO)
				       (warn "3Line ~D: The first field was not #N.  Ignoring this host entry.*"
					     line-number)
				       (setq state 'NEMO)))
				    (t
				     (setq site (make-site))
				     (add-field site valid-p line line-number) ;1 Add the #N field to the site we've just created.*
				     (setq state 'SITE)
				     )))
		     (NEMO    nil) ;1 In this state, the user has said ``ignore all following fields until blank line.''*
		     ))
		  
		  (t
		   (when (and (eq state 'SITE) (not starts-with-#))
		     (setq state 'BETWEEN))
		   (unless comment-p
		     (case state
		       ((NEMO BETWEEN) (parse-connection-line line line-number))
		       (t (warn "3Line ~D: Unknown field, ~S in ~S.*" line-number (char line 1) line)))))
		  )))))))


(defun 4char-pos *(chars string &optional (start 0) end from-end)
  "2Returns the earliest position of one of the CHARS in STRING.*"
  (position-if #'(lambda (char)
		   (member (the string-char char)
			   (the list chars)
			   :test #'char=))
	       (the string string)
	       :start start :end end :from-end from-end))


;1;; To express a host with aliases:*
;1;;*
;1;;*	3host-name = alias-1, alias-2, alias-3 ...*
;1;;*
;1;; To express a named network, whose members are fully connected:*
;1;;*
;1;;*	3network-name = { host-1, host-2, host-3, ... }*
;1;;*
;1;; To express an unnamed network, whose members are fully connected:*
;1;;*	3= { host-1, host-2, host-3, ... }*
;1;;*
;1;; There may be a routing-character before the 3{* or after the 3}* as in:*
;1;;*
;1;;*	3= @{ host-1, host-2, host-3, ... }*
;1;;*	3= { host-1, host-2, host-3, ... }!*
;1;;*
;1;; There may be a cost expression after the 3}* and optional routing character, as in:*
;1;;*
;1;;*	3= { host-1, host-2, host-3, ... }(DEDICATED)*
;1;;*	3= { host-1, host-2, host-3, ... }!(LOCAL)*
;1;;*
;1;; The network is a "pseudo-host" - a bidirectional connection is assumed between each listed host and the network, rather than*
;1;; assuming a connection between each host and every other host.  The cost from a host to the network pseudo-host is zero;*
;1;; The cost from the network to each host is the given cost-expression (after the 3}*).*
;1;;*
;1;;*
;1;; To declare that some hosts or connections are dead:*
;1;;*
;1;;*	3dead { conn-desc, conn-desc, ... }*
;1;;*
;1;; Where 3conn-desc* is either a host name (meaning that host is dead) or of the form 3host!host* meaning that the connection between*
;1;; the two hosts is dead.*
;1;;*
;1;; To forget all info about the given host or connections:*
;1;;*
;1;;*	3delete { conn-desc, conn-desc, ... }*
;1;;*
;1;; To increment or decrement the value of all connections to a given host:*
;1;;*
;1;;*	3adjust { host-and-cost, host-and-cost, host-and-cost, ... }*
;1;; as in*
;1;;*	3adjust { somehost(+5), otherhost(LOW), nohost(DEAD) }*
;1;;*
;1;; The 3file* declaration changes the name to be used in error-messages.  This is silly noise to accomodate the use of piping.*
;1;;*
;1;; The 3private* declaration means that it is preferred that we not generate paths to these hosts, though we may generate paths*
;1;; to other hosts that pass through them. (yeah right...)*
;1;;*

;1;; A host which has an explicit connection to a network pseudo-host may be used as a gateway to the hosts on that network in the*
;1;; obvious way.  That host does not need to be a member of the network.*
;1;;*
;1;; Network pseudo-hosts whose names begin with a dot are really ``domains.''  This is big hairy magic.*
;1;; When generating a path through a domain pseudo-host, we do not use the usual method of pushing the connection on the end.*
;1;; Rather, we append the domain to the name of the last host.  So, when going from '3foo*' to '3bar*' through the '3.EDU*' pseudo-host,*
;1;; we generate*
;1;;*
;1;;*	3foo ! bar.EDU*		1instead of*		3foo ! .EDU ! bar*
;1;;*	3foo ! bar.CMU.EDU*	1or*			3foo ! .EDU ! .CMU ! bar*
;1;;*
;1;;*


(defun 4clearem *()
  (do-local-symbols (s (find-package "3UUMAP*"))
    (remprop s 'connections)
    (remprop s 'alias-of)
    (remprop s 'network-p)
    ))

(defun 4parse-connection-line *(string &optional line-number)
  (multiple-value-bind (keyword host list conn-char cost-expr)
		       (parse-connection-line-1 string line-number)
    (case keyword
      (:NETWORK
       (let* ((host-name (if (string= host "")
			     (gensym)
			     (intern host "3UUMAP*")))
	      (conns '()))
	 (dolist (string list)
	   (check-type string string)
	   (setq string (intern string "3UUMAP*"))
	   (let* ((conn (make-connection :name string :symbolic-cost cost-expr :cost (or (eval cost-expr) 0))))
	     (when conn-char (setf (connection-char conn) conn-char))
	     (push conn conns)))
	 (let* ((old (get host-name 'CONNECTIONS)))
	   (setf (get host-name 'CONNECTIONS) (nconc old conns)))
	 (setf (get host-name 'NETWORK-P) t)
;	 (format t "3~&Network ~A: *" host-name)
	 (dolist (c conns)
;	   (format t "3~&   ~s*" c)
	   (let* ((name (connection-name c)))
	     (push (make-connection :name host-name :symbolic-cost 'NETWORK :cost 0)
		   (get name 'CONNECTIONS)))
	   )
;	 (format t "3~3%*")
	 ))
      
      (:ALIAS
       (let* ((host-name (intern host "3UUMAP*")))
	 (dolist (alias-host list)
	   (setf (get (intern (string alias-host) "3UUMAP*") 'ALIAS-OF) host-name)))
;       (format t "3~&Host ~A: aliases ~A*" host list)
       )
      
      (:CONNECTIONS
       (let* ((host-name (intern host "3UUMAP*")))
;	 (dolist (conn list)
;	   (check-type conn connection)
	   ;(when conn-char (setf (connection-char conn) conn-char))
	   ;(push conn conns)
;	   )
	 (let* ((old (get host-name 'CONNECTIONS)))
	   (setf (get host-name 'CONNECTIONS) (nconc list old)))
;	 (format t "3~&Host ~A: *" host-name)
;	 (dolist (c list) (format t "3~&   ~s*" c))
;	 (format t "3~3%*")
	 ))
      
      (t
       (format t "3~&Noise at line ~S: ~S ~S*" line-number keyword host)
       )))
  (values))


(defvar 4*parse-connection-line-previous-line-state* *nil)
(defvar 4*parse-connection-line-previous-line-host* *nil)

;1 *(setq 4*parse-connection-line-previous-line-host** nil 4*parse-connection-line-previous-line-state** nil)

(defun 4parse-connection-line-1 *(string &optional line-number)
  (let* ((comment (position #\# string :test #'char=)))
    (when (or (eql comment 0) (string= "" string)) (return-from 4PARSE-CONNECTION-LINE*-1 nil))
    (cond ((member (char string 0) '(#\Space #\Tab) :test #'char=)
	   (case *parse-connection-line-previous-line-state*
	     (NIL (error "3invalid cont line.*"))
	     (:NETWORK
	      (let* ((close-curly (position #\} string))
		     (conn-char-pos (and close-curly
					 (char-pos '(#\@ #\! #\% #\:) string close-curly)))
		     (conn-char (and conn-char-pos (char string conn-char-pos)))
		     
		     (paren-pos (and close-curly (position #\( string :start close-curly)))
		     (close-paren (and paren-pos (position #\) string :start paren-pos)))
		     (cost-expression (and close-paren
					   (connection-value-tokenize string (1+ paren-pos) close-paren))))
		(when close-curly
		  (setq *parse-connection-line-previous-line-state* nil))
		(let* ((old-name (car 4*parse-connection-line-previous-line-host**))
		       (old-conn-char (cdr 4*parse-connection-line-previous-line-host**)))
		  (values :NETWORK old-name
			  (tokenize string 1 close-curly nil)
			  (or conn-char old-conn-char)
			  cost-expression
			  ))))
	     (:CONNECTIONS
	      (values :CONNECTIONS 4*parse-connection-line-previous-line-host**
		      (4parse-connection-line-connections* string 0 nil line-number)))
	     (t
	      (values 4*parse-connection-line-previous-line-*state4**
		      4*parse-connection-line-previous-line-host**
		      (tokenize string 0 nil nil)))))
	  
	  (t
	   (let* ((alias-line-p nil)
		  (word nil)
		  (after-word nil)
		  (length (length string)))
	     (do* ((i 0 (1+ i))
		   (past-word-p nil))
		  ((>= i length))
	       (let* ((c (char string i)))
		 (cond ((char= c #\=)
			(setq alias-line-p t
			      word (subseq string 0 (or past-word-p i))
			      after-word (1+ i))
			(return))
		       ((or (char= c #\Space)
			    (char= c #\Tab))
			(unless past-word-p
			  (setq past-word-p i)))
		       (t
			(when past-word-p
			  (setq word (subseq string 0 past-word-p)
				after-word i)
			  (return))))))
	     (setq 4*parse-connection-line-previous-line-host** word)
	     (cond (alias-line-p
		    (let* ((open-curly  (position #\{ string :start after-word))
			   (close-curly (and open-curly (position #\} string :start open-curly))))
		      (cond (open-curly
			     ;(assert close-curly () "3unmatched curlies.*")
			     (let* ((conn-char-pos (or (char-pos '(#\@ #\! #\% #\:) string after-word open-curly)
						       (and close-curly
							    (char-pos '(#\@ #\! #\% #\:) string close-curly))))
				    (conn-char (and conn-char-pos (char string conn-char-pos)))
				    
				    (paren-pos (and close-curly (position #\( string :start close-curly)))
				    (close-paren (and paren-pos (position #\) string :start paren-pos)))
				    (cost-expression (and close-paren
							  (connection-value-tokenize string (1+ paren-pos) close-paren))))
			       (setq *parse-connection-line-previous-line-state* (if close-curly NIL :NETWORK))
			       (setq 4*parse-connection-line-previous-line-host** (cons word conn-char))
			       (values :network word
				       (tokenize string (1+ open-curly) close-curly nil)
				       conn-char cost-expression
				       ;(null close-curly)
				       )))
			    (t
			     (setq *parse-connection-line-previous-line-state* :ALIAS)
			     (let* ((aliases (tokenize string after-word nil nil)))
			       (values :alias word aliases))))))
		   (t
		    (let* ((key (or (find word '(:PRIVATE :DEAD :DELETE :FILE :ADJUST) :test #'string-equal)
				    :CONNECTIONS)))
		      (setq *parse-connection-line-previous-line-state* key)
		      (case key
			(:CONNECTIONS (values :CONNECTIONS word
					      (and after-word
						   (4parse-connection-line-connections* string after-word nil line-number))))
			(:FILE (values :file (subseq string after-word)))
			(t
			 (string-subst-char #\Space #\{ string nil nil)
			 (string-subst-char #\Space #\} string nil nil)
			 (values key (tokenize string after-word nil nil))))))))))))



(defun 4parse-connection-line-connections *(string &optional (start 0) end line-number)
  (declare (string string))
  (let* ((length (or end
		     (position #\# string :test #'char= :start start)
		     (length string)))
	 (conns '())
	 (i start)
	 )
    (loop
      (let* ((name-start nil)
	     (name-end nil)
	     (link-char nil)
	     (terminal-p nil)
	     (cost-expression nil))
	(loop
	  (let* ((c (char string i)))
	    (case c
	      ((#\! #\@ #\: #\%)
	       (assert (null link-char) () "3Line ~D: multiple link-chars, ~A and ~A*" line-number link-char c)
	       (setq link-char c)
	       (when (and name-start (null name-end))
		 (setq name-end i))
	       )

	      (#\<
	       (setq terminal-p t))

	      (#\>
	       (when (or (null terminal-p) (eq terminal-p :closed))
		 (error "3Line ~D: mismatched <> delimiters.*" line-number))
	       (setq terminal-p :closed)
	       (unless name-end (setq name-end i)))
	      
	      (#\(
	       (multiple-value-bind (ignore list-end)
				    (let* ((*read-suppress* t))
				      (read-from-string string t nil :start i :preserve-whitespace t))
		 (unless name-end (setq name-end i))
		 (setq cost-expression (connection-value-tokenize string (1+ i) (1- list-end))
		       i (1- list-end))))
	      
	      (#\) (error "3Line ~D: mismatched parens at position ~D of ~S.*" line-number i string))
	      
	      ((#\Space #\Tab #\,)
	       (unless name-end (setq name-end i))
	       (incf i)
	       (return))
	      (t
	       (unless name-start (setq name-start i)))))
	  (when (>= (incf i) length) (return)))
	(when name-start
	  (assert (member terminal-p '(nil :closed)) () "3Line ~D: a < delimiter was not closed.*" line-number)
	  (check-type string string)
	  (let* ((conn (make-connection :name (subseq string name-start (or name-end i))
					:char link-char
					:symbolic-cost cost-expression
					:terminal-p (not (null terminal-p))
					)))
	    (push conn conns)))
	)
      (when (>= i length) (return)))
    (nreverse conns)))




(defun 4connection-tokenize* (string &optional (start 0) end)
  (unless end (setq end (length string)))
  (macrolet ((make-token (string start end)
	       `(intern (subseq ,string ,start ,end) "3UUMAP*")))
    (let* ((tokens '())
	   (last-nonwhite-pos nil))
      (do* ((pos start (1+ pos)))
	   ((>= pos end)
	    (when last-nonwhite-pos
	      (push (make-token string last-nonwhite-pos end) tokens)))
	(let* ((c (char string pos)))
	  (cond ((char= c #\()
		 (multiple-value-bind (ignore list-end)
				      (let* ((*read-suppress* t))
					(read-from-string string t nil :start pos :preserve-whitespace t))
		   (let* ((list (connection-value-tokenize string (1+ pos) (1- list-end)))
			  (name (and last-nonwhite-pos (make-token string last-nonwhite-pos pos)))
			  (cost (if list (eval list) 0)))
		     (unless (integerp cost) (setq cost (float cost 1.0s0)))
		     (check-type name symbol)
		     (push (make-connection :name name :symbolic-cost list :cost cost)
			   tokens))
		   (setq pos (1- list-end))
		   (setq last-nonwhite-pos nil)))
		
		((and (graphic-char-p c) (char/= c #\Space) (char/= c #\,))
		 (unless last-nonwhite-pos (setq last-nonwhite-pos pos)))
		
		(last-nonwhite-pos                            ;1 we've encountered whitespace after nonwhite.*
		 (push (make-token string last-nonwhite-pos pos) tokens)
		 (setq last-nonwhite-pos nil)
		 (setq start pos)))))
      (nreverse tokens))))


(defun 4connection-value-tokenize *(string &optional (start 0) end)
  "2Given a string of the form ``DAILY+FAST'' returns a sexpr of the form (+ DAILY FAST).*"
  (let* ((arith (char-pos '(#\+ #\- #\/ #\*) string start end t))
	 (ch (and arith (char string arith)))
	 (sym (and ch (intern (string ch)))))
    (if sym
	(let* ((left  (nstring-upcase (subseq string start arith)))
	       (right (nstring-upcase (subseq string (1+ arith) end))))
	  (setq right (if (every #'digit-char-p right) (parse-integer right) (intern right "3UUMAP*")))
	  (setq left  (if (every #'digit-char-p left)
			  (parse-integer left)
			  (if (every #'alpha-char-p left)
			      (intern left "3UUMAP*")
			      (connection-value-tokenize left 0))))
	  (list sym left right))
	(let* ((sub (subseq string start end)))
	  (if (every #'digit-char-p sub)
	      (parse-integer sub)
	      (values (intern (string-upcase sub) "3UUMAP*")))))))


;(setq foo (directory "3eti:/usr3/uumap/u.*.**"))
;(setq foo (directory "3pt:/usr/netnews/uumap/u.*.**"))

(defun 4parse-site-directory *()
  (dolist (path foo)
    (parse-site-file path))
  (parse-site-file "3eti:/usr3/uumap/d.AProject*")
  (parse-site-file "3eti:/usr3/uumap/d.Country*")
  (parse-site-file "3eti:/usr3/uumap/d.Glue*")
  (parse-site-file "3eti:/usr3/uumap/d.Top*")
  )

;(defun 4parse-site-directory *(&optional (directory "3pt:/usr/netnews/uumap/u.*.**"))
;  (dolist (path (directory directory))
;    (parse-site-file path)))



;1;; Some real stuff.*


(defun 4find-path *(from-machine to-machine &optional (npaths 1) debug-p)
  "2Returns, as multiple values, the N best paths from one machine to the other.
  The values are lists; the CAR of each list is a number, where smaller numbers mean better connections.
  The CDRs of the lists are the path between the two machines (inclusive).*"
  (declare (special npaths))
  (let* ((tried (list from-machine))
	 (valid-paths '())
	 )
    (declare (special tried valid-paths))
    (find-path-1 from-machine to-machine (list from-machine) 0 10 debug-p)
    (values-list (nreverse valid-paths))
    ))

(defun 4find-path-1 *(from-machine to-machine path-so-far cost-so-far max-depth debug-p)
  (declare (special tried valid-paths npaths)
	   (symbol from-machine to-machine)
	   (fixnum cost-so-far max-depth)
	   (list path-so-far)
	   (optimize speed))
  (let* ((conns (get from-machine 'connections)))
    (dolist (conn conns)
      (unless (symbolp conn)   ;1 ## hack*
      (let* ((conn-host (connection-name conn))
	     (conn-cost (round (connection-cost conn)))
	     (alias (get conn-host 'ALIAS-OF))
	     (max-cost (or (car (car (the list valid-paths))) most-positive-fixnum))
	     )
	(declare (symbol conn-host)
		 (integer conn-cost)
		 (fixnum max-cost)
		 (optimize speed))
	(flet ((bug-out ()
		 "2Done, finit, kaput, we're outa here.*"
		 (return (values 
			   (nreverse (cons conn-host (copy-list path-so-far)))   ;1 copy the stack-list*
			   cost-so-far
			   ))))
	  (when (eq conn-host to-machine) (bug-out))
	  (when alias (setq conn-host alias))
	  (when (eq conn-host to-machine) (bug-out)))
	
	(cond ((connection-terminal-p conn)
	       (and debug-p (format t "3~&BUG-OUT - ~A is terminal, don't descend it.*" conn-host))
	       nil)
	      
	      ((or (>= conn-cost DEAD)
		   (>= (incf cost-so-far conn-cost) max-cost))
	       (and debug-p (format t "3~&BUG OUT1 -* too expensive at ~A ~A*" cost-so-far conn-host))
	       nil)
	      
	      ((= 1 max-depth)
	       (and debug-p (format t "3~&BUG OUT - too deep.*"))
	       nil)
	      
	      ((member conn-host tried :test #'eq)
	       (and debug-p (format t "3~&BUG OUT - tried ~a already.*" conn-host))
	       nil)
	      
	      (t
	       (with-stack-list* (tried conn-host tried)	;1 This prevents looping.*
		 (declare (special tried valid-paths))
		 (with-stack-list* (try-path conn-host path-so-far)
		   (multiple-value-bind (sublist cost)
					(find-path-1 conn-host to-machine try-path cost-so-far (1- max-depth) debug-p)
		     (declare (fixnum cost))
		     (when sublist
		       (setq cost-so-far cost)
		       (cond ((= (length valid-paths) npaths)
			      (cond ((< cost-so-far max-cost)
				     (and debug-p (format t "3~&replace ~s with ~s*" max-cost cost-so-far))
				     (setf (car valid-paths) (cons cost-so-far sublist))
				     (setf valid-paths (sort (the list valid-paths) #'> :key #'car))
				     )
				    (t
				     (and debug-p (format t "3~&dont replace ~s with ~s*" max-cost cost-so-far))
				     )))
			     (t
			      (and debug-p (format t "3~&PUSH ~a*" cost-so-far))
			      (push (cons cost-so-far sublist) valid-paths)
			      (setf valid-paths (sort (the list valid-paths) #'> :key #'car))
			      ))))))))))))
  )
