;;; -*- Mode:Common-Lisp; Package:CHAOS; Base:10 -*-

;;; ChangeLog:
;;;
;;; 15 May 88  Jamie Zawinski  Created.
;;; 30 Jun 88  Jamie Zawinski  Made talkserver notify on errors.
;;; 13 Jul 88  Jamie Zawinski  Added some more output.
;;; 22 Aug 89  Jamie Zawinski  Made the talkserver process be launched from the chaos:server-alist instead of 
;;;			       always being around.  Made length be transmitted in 4 bytes instead of 2.
;;;

(unintern 'user::talk-on "USER")
(unintern 'talk-on)
(export 'ticl::talk-on "TICL")

(export '(*talkserver-verbose* talk-on))


(defvar *talkserver-contact-name* "TALKSERVER" "This is the ID name that requests to the talkserver use.")
(defvar *talkserver-verbose* nil
  "If this is T, then the talkserver contact will print diagnostics on *STANDARD-OUTPUT* and will not trap errors.")

(defun talkserver-process ()
  "Wait for a connection, read a sound array, and play it."
  (let* ((conn (ignore-errors (chaos:listen *talkserver-contact-name*))))
    (when conn
      (chaos:accept conn)
      (let* ((chaos-stream (chaos:make-stream conn :characters nil))
	     array)
	(condition-call-if (not *talkserver-verbose*)
			   (condition-instance)
	    (unwind-protect
		(progn
		  (when *talkserver-verbose*
		    (let* ((foreign-host (si:get-host-from-address (chaos:foreign-address conn) :chaos)))
		      (format t "~&Talkserver request from ~A - " foreign-host)))
		  (setq array (talkserver-read-array chaos-stream))
		  (setf (get 'TALKSERVER-SOUND 'TV:SOUND-ARRAY) array)
		  (tv:play 'TALKSERVER-SOUND)
		  (when *talkserver-verbose* (format t " done.~%"))
		  )
	      ;; Protected.
	      (send chaos-stream :finish)
	      (chaos:remove-conn conn)
	      (when array
		(setf (get 'TALKSERVER-SOUND 'TV:SOUND-ARRAY) nil)
		(deallocate-resource 'TV:SOUND-ARRAY array)))
	  (t (tv:notify nil "Process ~A got an error: ~A" (send tv:current-process :name) condition-instance))
	  )))))


(defun talkserver-read-array (stream)
  "Returns an array from the TV:SOUND-ARRAY resource.
  STREAM should be a connection to a talkserver client.
  The caller is responsible for deallocating this array."
  ;; First read the length of the sample as 4 bytes ((MSB,LSB),(MSB,LSB)).  Then construct an array and read the data.
  (let* ((length 0))
    (setq length (dpb (read-byte stream) (byte 8 24) length))
    (setq length (dpb (read-byte stream) (byte 8 16) length))
    (setq length (dpb (read-byte stream) (byte 8  8) length))
    (setq length (dpb (read-byte stream) (byte 8  0) length))
    (let* ((array (allocate-resource 'TV:SOUND-ARRAY length)))
      (declare (type (array (integer 0 255) 1) array))
      (when *talkserver-verbose* (format t " length=~D bytes; " length))
      (setf (fill-pointer array) length)
      (dotimes (i length)
	(declare (fixnum i)
		 (optimize speed))
	(setf (aref array i) (read-byte stream)))
      (when *talkserver-verbose* (format t " done reading - "))
      array)))


(defun talk-on (host &optional (sample nil sample-supplied-p))
  "Contact the talkserver on HOST.  If SAMPLE is supplied, it should be an array made with TV:RECORD.
  If it is not supplied, the user will be prompted for a number of seconds to record, and a new sample will be made."
  (let* ((deallocate nil))
    (when (and sample-supplied-p (symbolp sample))
      (setq sample (get sample 'TV:SOUND-ARRAY)))
    (unless sample-supplied-p
      (let* ((sec nil))
	(do* ()
	     ((numberp sec))
	  (format t "~&How many seconds? ")
	  (setq sec (read)))
	(format t "~&Type any character to begin recording.")
	(read-char)
	(let* ((sym (gensym)))
	  (tv:record sym sec)
	  (setq sample (get sym 'TV:SOUND-ARRAY))
	  (setq deallocate t))))
    (when (> (fill-pointer sample) #.(1- (expt 2 32)))
      (format t "~&Warning: this sample's length is greater than can be represented in 32 bits.  It will be truncated."))
    (format t "~&Contacting server...")
    (with-open-stream (stream (chaos:open-stream host *talkserver-contact-name* :characters nil))
      (format t "~&Transmitting...")
      ;;
      ;; First write out the length of the sample as four bytes.
      ;;
      (let* ((length (fill-pointer sample))
	     (b1 (ldb (byte 8 24) length))
	     (b2 (ldb (byte 8 16) length))
	     (b3 (ldb (byte 8  8) length))
	     (b4 (ldb (byte 8  0) length)))
	(write-byte b1 stream)
	(write-byte b2 stream)
	(write-byte b3 stream)
	(write-byte b4 stream))
      ;;
      ;; Then write out the sample data.
      ;;
      (dotimes (i (fill-pointer sample))
	(declare (fixnum i)
		 (optimize speed))
	(write-byte (aref (the (array (integer 0 255) 1) sample)
			  (the fixnum i))
		    stream)))
    (when deallocate (deallocate-resource 'TV:SOUND-ARRAY sample)))
  (format t "~&Done.~%")
  (values))


(add-initialization *talkserver-contact-name*
		    '(process-run-function "TalkServer" 'talkserver-process)
		    ()
		    'chaos:server-alist)
