;;; -*- Mode:Common-Lisp; Package:USER; Base:10; Fonts:(MEDFNT MEDFNTI MEDFNTBI MEDFNT MEDFNTB) -*-

(provide "3reservations*")

1;;; simple reservation system for machines.
;;; functions to be used at `top level':
;;;   RESERVE*		1- reserve time on a machine
;;;   CANCEL*		1- cancel time already reserved on a machine
;;;   SHOW-RESERVATIONS*	1- show who has what time reserved
;;; on login, the function CHECK-RESERVED-AT-LOGIN is called to determine
;;; if the machine has been reserved by another user, and to warn accordingly.*

(defconstant 4reservations-filename* "3lm:site;reservations.data#>*")

(defun 4show-reservations-internal* (machine start times)
   "2Show reserved time on MACHINE after time START.
   If TIMES is nil, read the list from the reservation file on MACHINE, otherwise use TIMES.*"
   (let* ((reservation-file (if (null times)
				(make-pathname :host machine
					       :defaults reservations-filename)))
	  (times (or times
		     (and (probe-file reservation-file)
			  (with-open-file (res reservation-file :direction :input)
			    (read res)))))
	  (reservations (delete-if
			   #'(lambda (item)
			       (< (third item) start))
			   (sort times #'lessp :key #'second))))
     (if reservations
	 (format t "3~2&The following times are reserved on ~:(~a~):~2%~*
		3       User~16tStart~36tEnd~56tPurpose~%~*
		3       ----~16t-----~36t---~56t-------~%~*
		3       ~{~{~@[~a~16t~\\time\\~36t~\\time\\~56t~@[~a~]~%~]~}~}*"
		 (if (equalp machine "3lm*") sys:local-host machine)
		 reservations)
	 (format t "3~2&No reservations are outstanding.~2%*"))))



(defun 4show-reservations* (&optional
			  &key
			  (machine "3lm*")
			  (start (get-universal-time) startp))
   "2Show reserved time on MACHINE after time START.*"
   (show-reservations-internal machine 
			       (if (and startp (typep start 'string))
				   (time::parse-universal-time start)
				   start)
			       nil))


(defun 4reserve* (&key
		(machine sys:local-host)
		(start (get-universal-time) startp)
		(duration 1.0)
		(user (string-capitalize user-id))
		(purpose nil))
   "2Reserve DURATION hours of time on MACHINE starting at START (universal time) for USER
    to do PURPOSE.*"
   (if startp
       (if (typep start 'string)
	   (setq start (time:parse-universal-time start)))
       (if (typep *terminal-io* 'w:window)
	   (w:choose-variable-values
	      `((,(locf start) "3Start time*" :date)
		(,(locf duration) "3Duration*" :number)
		(,(locf user) "3Your name*" :string)
		(,(locf machine) "3Machine*" :string)
		(,(locf purpose) "3Purpose*" :string))
	      :label "3Please enter the time, etc., for which you wish to reserve the machine*")
	   (let (answer)
	     (macrolet ((get-answer (prompt def var parser)
			  `(progn
			      (format t "3~&~a [default: ~a]? *" ,prompt ,def)
			      (setf answer (read-line))
			      (when (not (equal answer ""))
				(setf ,var (,parser answer))))))
	       (get-answer "3Start time*" "3now*" start time::parse-universal-time)
	       (get-answer "3Duration*" (format nil "3~a hour~:p*" duration) duration parse-number)
	       (get-answer "3Your name*" user user string)
	       (get-answer "3Machine*" machine machine string)
	       (get-answer "3Purpose*" (or purpose "") purpose string)
	       ))))
   (setf user (string user)) 1;; make sure it's a string*
   (if (< duration 0)
       (format t "3~&The duration must be non-negative.~%*")
       (let* ((reservation-file (make-pathname :host machine
					       :defaults reservations-filename))
	      (existsp (probe-file reservation-file))
	      (now (get-universal-time))
	      (end (+ start (round (* duration 3600))))
	      times)
	 (when existsp
	   (setf reservation-file existsp))
	 (with-open-file (res reservation-file :direction :io :if-exists :overwrite)
	   (when existsp
	     (setf times (read res)))
	   (if (check-reserved start end user t machine times)
	       (block no-reserve
		 (format t "3~2&Sorry, someone else already has some of that time reserved.~2%*")
		 (show-reservations-internal machine now times))
	       (block do-reserve
		 (push (list user start end purpose) times)
		 (setf times (sort times #'< :key #'second))
		 (file-position res :start)
		 (write times :stream res :escape t)
		 (write-char #\space res)
		 (format t "3~&~:(~a~) reserved by ~a~@[ for ~a~] from ~\\time\\ until ~\\time\\.~%*"
			 (if (equalp machine "3lm*") sys:local-host machine)
			 user purpose start end))))
	 (check-reserved-at-login nil)
	 ))
   (values))


(defun 4cancel* (&key
	       (machine "3lm*")
	       (start (get-universal-time) startp)
	       (duration 1.0)
	       (user (string-capitalize user-id)))
   "2Cancel time reserved by USER on MACHINE in the period of length DURATION (in
    hours) after START (universal time).*"
   (if startp
       (if (typep start 'string)
	   (setq start (time::parse-universal-time start)))
       (if (typep *terminal-io* 'w:window)
	   (w:choose-variable-values
	      `((,(locf start) "3Start time*" :date)
		(,(locf duration) "3Duration*" :number)
		(,(locf user) "3Your name*" :string)
		(,(locf machine) "3Machine*" :string))
	      :label "3Please enter the time reserved on the machine to be cancelled*")
	   (let (answer)
	     (macrolet ((get-answer (prompt def var parser)
			  `(progn
			      (format t "3~&~a [default: ~a]? *" ,prompt ,def)
			      (setf answer (read-line))
			      (when (not (equal answer ""))
				(setf ,var (,parser answer))))))
	       (get-answer "3Start time*" "3now*" start time::parse-universal-time)
	       (get-answer "3Duration*" (format nil "3~a hour~:p*" duration) duration parse-number)
	       (get-answer "3Your name*" user-id user string)
	       (get-answer "3Machine*" sys:local-host machine string)
	       ))))
   (setf user (string user)) 1;; make sure it's a string*
   (let* ((reservation-file (make-pathname :host machine
					   :defaults reservations-filename))
	  (existsp (probe-file reservation-file))
	  (end (+ start (round (* duration 3600))))
	  times
	  candidates)
     (when existsp
       (setf reservation-file existsp))
     (with-open-file (res reservation-file :direction :io :if-exists :overwrite)
       (when existsp
	 (setf times (read res)))
       (setf candidates (check-reserved start end user nil machine times))
       (if candidates
	   (block do-cancel
	     (format t "3~&The following reservations are to be cancelled on ~:(~a~):~2%~*
			3  User~16tStart~36tEnd~56tPurpose~%~*
			3  ----~16t-----~36t---~56t-------~%~*
			3  ~{~{~@[~a~16t~\\time\\~36t~\\time\\~56t~@[~a~]~%~]~}~}*"
		     (if (equalp machine "3lm*") sys:local-host machine)
		     candidates)
	     (when (yes-or-no-p "3Cancel them?*")
	       (file-position res :start)
	       (write (set-difference times candidates) :stream res)
	       (write-char #\space res)))
	   (format t "3~&Sorry, I couldn't find any such time.~%*")))
     (check-reserved-at-login nil)))


(defun 4check-reserved* (&optional
		       (start (get-universal-time))
		       (end start)
		       (user user-id)
		       (test-equal t)
		       (machine "3lm*")
		       (times nil times-p)
		       check-start-only)
   (let* ((reservation-file (if (null times)
				(make-pathname :host machine
					       :defaults reservations-filename))))
     (when (and (not times-p)
		(probe-file reservation-file))
       (setf times (with-open-file (res reservation-file :direction :input)
		     (read res))))
     (setf user (string user)) 1;; make sure it's a string*
     (remove-if-not
	#'(lambda (item)
	    (and (xor test-equal
		      (equalp (string user) (first item)))
		 (if check-start-only
		     (<= start (second item) (or end (second item)))
		     (or (<= (second item) start (third item))
			 (and end
			      (<= (second item) end (third item)))
			 (<= start (second item) (or end (second item)))
			 (<= start (third item) (or end (third item)))))))
	times)))


(defun 4next-user-reminder* (user time)
  (let ((time-to-warn (- time (get-universal-time) (* 15 60)))) 1; 15 minutes warning*
    (when (plusp time-to-warn)
      (sleep time-to-warn))
    (w:notify nil "3~:(~a~) will want the machine in 15 minutes.~%~*
		3   Please be ready to logout then.*" user)
    1;; don't forget to check again, once we've warned about one user*
    1;; -- want to queue up another warning, but not commit suicide*
    (check-reserved-at-login nil nil)
    ))


(defun 4delete-old-reminder-processes* ()
   (dolist (process sys:all-processes)
     (if (equal (send process :name) "3Next User Reminder*")
	 (send process :kill))))


1;;; Check and possibly warn*
(defun 4check-reserved-at-login* (&optional (warn-on-soon t) (kill-other-warnings t))
  (let* ((now (get-universal-time))
	 (reservations (check-reserved now nil)) 1;; check for any future time*
	 (next-day     (check-reserved (+ now 901) nil user-id t "3lm*" reservations t)) 1;; then after 15 min*
	 (soon         (check-reserved now (+ now 900) user-id t "3lm*" reservations))) 1;; then next 1/2 hour*
    (when kill-other-warnings
      (delete-old-reminder-processes))
    (cond
       ((and soon warn-on-soon)
	(format t "3~2&~:(~a~) is reserved~*
		3    ~{~{~12tby ~a from ~\\time\\ until ~\\time\\ for ~a~}~%~}~*
		3   You may use it for now, but please be ready to give it up if necessary.~2%*"
		sys:local-host
		soon))
       (next-day
	(sys:process-run-function
	  "3Next User Reminder*" 'next-user-reminder (caar next-day) (cadar next-day))))
    ))


(add-initialization "3Check for reserved times.*" '(check-reserved-at-login) :login)
(add-initialization "3Remove check for reserved times.*" '(delete-old-reminder-processes) :logout)