;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:SYSTEM -*-

;;; File PIPE-STREAM.LISP
;;; A TI Explorer implementation of Unix-like pipes.
;;;
;;;  past...    Jamie Zawinski   Created.
;;;  13 feb 90  Jamie Zawinski   Made the major-mode of a read-time uncompressed buffer be defaulted better.
;;;				 Started working on compressing-output streams.

;;; Usage:
;;; ======
;;; Calling MAKE-PIPE-STREAM with a function (of one argument, an output stream) will return a
;;; PIPE-STREAM.  Reading the first character from that stream invokes the function.  As soon as the
;;; function writes its first character, it's execution is suspended until the next character is read
;;; from the pipe.  When that happens, the function's execution state is restored, and it is allowed to
;;; run until it writes the next character.  This continues until the function returns, after which the
;;; pipe-stream will be seen to be at EOF; or until the pipe-stream is closed, at which point the
;;; function will be terminated (it may catch this through the usual unwind-protect mechanisms).
;;;
;;; This file also defines WITH-OPEN-DECOMPRESSING-STREAM which is just like WITH-OPEN-FILE :DIRECTION :INPUT
;;; except that if the file name ends in .Z, you will get a pipe running COMPRESS:DECOMPRESS-FILE on the 
;;; other end instead of a file stream.  This lets you deal with compressed files as if they were uncompressed
;;; already (like the Unix incantation 'zcat file | grep foo').
;;;
;;;
;;; Unbuffered-Pipe-Stream Implementation:
;;; ======================================
;;; When the stream is made, a stopped stack-group running the given function is also made.  There are
;;; two streams involved: the one given to the caller of MAKE-PIPE-STREAM, and one given to the function.
;;; Each of these is one end of the pipe.  When the caller reads a character from its side of the pipe,
;;; the character-generating stack-group is swapped in and allowed to run.  As soon as the other stack-group
;;; writes a character, the original stack-group is swapped back in (via stack-group-return) and the written
;;; character is returned.  In this way, the number of characters read and the number of characters
;;; written is always exactly the same.
;;;
;;; Buffered-Pipe-Stream Implementation:
;;; ====================================
;;; Just like the unbuffered stream, except that when the caller of MAKE-PIPE-STREAM reads a character,
;;; the other end of the pipe is swapped in and allowed to run until it fills up its output buffer.  If an
;;; output buffer is ready, the reading side will take characters off of that; when it runs out, then the
;;; other side is allowed to run again.  The size of the buffer is user-settable.
;;;
;;; If the filter/stream-source you are using is computationally inexpensive, then buffered streams are a
;;; good idea; there is a substantial overhead in stack-group swapping, and you probably don't want to do
;;; one twice for every character you read.
;;;
;;; However, if your source is computationally expensive and has a 1-character quantum (like, say, computing
;;; prime numbers) then an unbuffered stream might be faster; but there will probably not be much difference
;;; between using an unbuffered stream and using a buffered-stream with a buffer-size of 1.
;;;
;;; Automatic Uncompressing:
;;; ========================
;;; This file defines a kind of stream which pipes through Paul Fuqua's port of the Unix "compress" program.
;;; This is patched in to Zmacs so that visiting a file which ends in .Z will automagically decompress it
;;; into a buffer.  (This works when typing 'e' in Dired as well.)
;;;
;;; To Do:
;;; ======
;;; There should be a flavor of pipe-stream which runs the right half in its own process, with the left
;;; half blocking until a character becomes available.  If both left and right are computationally
;;; expensive, and produce their output/expect their input in different sized packets, this would win.
;;;
;;; A filtering-stream (taking a function of two args, in and out) would be nice.
;;;


(export '(unbuffered-pipe-stream make-unbuffered-pipe-stream
	  buffered-pipe-stream make-buffered-pipe-stream
	  ))
(export '(with-open-decompressing-stream make-decompressing-stream decompressing-open))


(defflavor basic-pipe-stream
  ((source nil)
   (source-sg nil)
   (state nil)
   (internal-stream nil)
   )
  (sys:input-stream)
  :gettable-instance-variables
  (:initable-instance-variables source)
  (:documentation :combination "The ``left half'' of a pipe stream.")
  )

(defflavor basic-internal-pipe-stream
  ((pipe-stream nil))
  (sys:output-stream)
  :gettable-instance-variables
  :initable-instance-variables
  (:documentation :combination "The ``right half'' of a pipe stream.  When an instance of PIPE-STREAM is made,
 one of these is created, and handed to the output-producing function to write on.")
  )

(defmethod (basic-pipe-stream :print-self) (stream &rest ignore)
  (sys:printing-random-object (self stream :typep)
    (if (eq state :closed)
	(princ "(closed)")
	(prin1 (or (and (functionp source) (function-name source))
		   source)))))


(defwhopper (basic-pipe-stream :line-in) (leader)
  "If the stream is closed, don't try to read."
  (if (eq state :closed)
      (values nil t)
      (continue-whopper leader)))

(defmethod (basic-pipe-stream :internal-close) (&optional ignore)
  (setq state :closed))


;;; Unbuffered pipes.


(defflavor unbuffered-pipe-stream
  ((unreadable-char nil))
  (sys:unbuffered-line-input-stream basic-pipe-stream)
  :gettable-instance-variables
  )

(defflavor internal-unbuffered-pipe-stream
  ((pipe-stream nil))
  (sys:output-stream)
  :gettable-instance-variables
  :initable-instance-variables
  )

(defun make-unbuffered-pipe-stream (function)
  "Creates and returns a stream which reads input from FUNCTION, which runs on an as-needed basis.
  FUNCTION should be a function of one argument, a stream.  Output written on this stream will be read
  by reading from the PIPE-STREAM.  The catch is, FUNCTION runs concurrently; it need not generate
  all of its output before you can start reading it."
  (make-instance 'unbuffered-pipe-stream :source function))

(defmethod (unbuffered-pipe-stream :after :init) (ignore)
  (assert (functionp source) (source) "No source supplied!")
  (setq source-sg (make-stack-group "Pipe SG"))
  (stack-group-preset source-sg 'pipe-sg-internal self source *terminal-io*)
  (setq internal-stream (make-instance 'internal-unbuffered-pipe-stream :pipe-stream self))
  )

(defun pipe-sg-internal (pipe-stream pipe-source-function &optional (*terminal-io* *terminal-io*))
  "The top-level function that runs in a pipe's stack group."
  (catch 'PIPE-CLOSE
    (let* ((right-half (send pipe-stream :internal-stream)))
      (unless (eq :closed (send pipe-stream :state))		; Closed before we were ever run...
	(unwind-protect (funcall pipe-source-function right-half)
	  (send right-half :close)
	  )))))

(defmethod (unbuffered-pipe-stream :untyi) (c)
  (setf unreadable-char c))


(defmethod (unbuffered-pipe-stream :tyi) (&optional (eof-error-p t))
  (if (eq state :closed)
      (and eof-error-p
	   (ferror 'SYS:END-OF-FILE-1 (if (stringp eof-error-p) eof-error-p "End of file on ~S.") self))
      (if unreadable-char
	  (prog1 unreadable-char (setq unreadable-char nil))
	  (funcall source-sg nil))))


(defmethod (internal-unbuffered-pipe-stream :tyo) (c)
  (when (eq :closed (send pipe-stream :state))
    (error "Internal error: right side of a pipe still runing on a closed stream."))
  ;; This swaps out the current stack-group; this won't return until the left-side of the pipe is again read from.
  ;; As we swap out, we pass back the character that was just written.
  (stack-group-return c)
  ;; If we were resumed after being closed, then that means that the left half of the pipe has been closed and
  ;; the code on the right half (that's us) should not execute any further, except to do cleanup.  So throw.
  (when (eq :closed (send pipe-stream :state))
    (throw 'PIPE-CLOSE nil))
  c)

(defmethod (internal-unbuffered-pipe-stream :close) (&optional error-p)
  (send pipe-stream :internal-close error-p)
  (send pipe-stream :close error-p))


(defmethod (unbuffered-pipe-stream :close) (&optional error-p)
  (unless (eq :closed state)
    (send self :internal-close error-p)
    (funcall source-sg nil)) ; wake it one last time for cleanup.
  (setq source nil
	source-sg nil))


;;; Buffered pipes.

(defflavor buffered-pipe-stream
  ((buffer-size 1024))
  (sys:buffered-line-input-stream basic-pipe-stream)
  :gettable-instance-variables
  :initable-instance-variables
  )

(defflavor internal-buffered-pipe-stream
  ()
  (sys:buffered-output-stream sys:line-output-stream-mixin basic-internal-pipe-stream)
  :gettable-instance-variables
  :initable-instance-variables
  )


(defmethod (internal-buffered-pipe-stream :after :close) (&optional error-p)
  (send pipe-stream :internal-close error-p)
  (send pipe-stream :close error-p))


(defmethod (buffered-pipe-stream :after :close) (&optional error-p)
  (unless (eq :closed state)
    (send self :internal-close error-p)
    (funcall source-sg nil)) ; wake it one last time for cleanup.
  (setq source nil
	source-sg nil))


(defun make-buffered-pipe-stream (function &optional (buffer-size 1024))
  "Creates and returns a stream which reads input from FUNCTION, which runs on an as-needed basis.
  FUNCTION should be a function of one argument, a stream.  Output written on this stream will be read
  by reading from the PIPE-STREAM.  The catch is, FUNCTION runs concurrently; it need not generate
  all of its output before you can start reading it.  The output will be generated BUFFER-SIZE characters
  at a time; it might make sense to tune BUFFER-SIZE to be appropriate to FUNCTION, if FUNCTION generates
  its output in a recognisably-sized quantum."
  (make-instance 'buffered-pipe-stream :source function :buffer-size buffer-size))


(defmethod (buffered-pipe-stream :after :init) (ignore)
  (assert (functionp source) (source) "No source supplied!")
  (check-type buffer-size (and fixnum (satisfies plusp)) "a positive fixnum")
  (setq source-sg (make-stack-group "Buffered Pipe SG"))
  (stack-group-preset source-sg 'pipe-sg-internal self source *terminal-io*)
  (setq internal-stream (make-instance 'internal-buffered-pipe-stream :pipe-stream self))
  )

(defmethod (buffered-pipe-stream :next-input-buffer) (&optional no-hang-p)
  (cond ((eq :closed state)
	 (if no-hang-p
	     nil
	     (ferror 'SYS:END-OF-FILE-1 "End of file on ~S." self))
	 )
	(t
	 ;; let the source-sg run until it fills another input buffer.
	 (let* ((cons (funcall source-sg nil))  ; can't do stack-group-return on multiple values.
		(array (car cons))
		(index (cdr cons)))
	   (values array 0 index)))))


;;; I had been using sys:get-readstring to get buffers, but the readstrings are only 128 elements long.
;;; So, now I use the ip:tcp-stream-ascii-buffer resource, which is of art-8b's, and which are >8k long.
;;; Defining my own resource for this seems a bit silly.

(defmethod (internal-buffered-pipe-stream :new-output-buffer) ()
  (let* ((size (send pipe-stream :buffer-size))
;	 (array (sys:get-readstring))
	 (array (allocate-resource 'ip:tcp-stream-ascii-buffer)))
    (setf (fill-pointer array)
	  (min size (array-length array))) ; in case they specified a buffer-size larger than the readstring.
    (values array 0 (min size (array-length array)))))

(defmethod (buffered-pipe-stream :discard-input-buffer) (array)
  ;; Since the array we hold as an input-buffer is the same as the array our conterpart holds as
  ;; an output-buffer, only one of us should deallocate it.
  (declare (ignore array)))

(defmethod (internal-buffered-pipe-stream :discard-output-buffer) (array)
; (sys:return-readstring array)
  (deallocate-resource 'ip:tcp-stream-ascii-buffer array))


(defmethod (internal-buffered-pipe-stream :send-output-buffer) (array end-index)
  (when (eq :closed (send pipe-stream :state))
    (error "Internal error: right side of a pipe still runing on a closed stream."))
  ;; This swaps out the current stack-group; this won't return until the left-side of the pipe is again read from.
  ;; As we swap out, we pass back the input buffer we have just filled.
  (stack-group-return (cons array end-index))
  ;; If we were resumed after being closed, then that means that the left half of the pipe has been closed and
  ;; the code on the right half (that's us) should not execute any further, except to do cleanup.  So throw.
  (when (eq :closed (send pipe-stream :state))
    (throw 'PIPE-CLOSE nil))
  )


(defwhopper (buffered-pipe-stream :line-in) (leader)
  "What a crock.  Trap wrong-type-argument conditions around :LINE-IN, since :LINE-IN expects that
  :SETUP-NEXT-INPUT-BUFFER will never cause the stream to become closed."
  (condition-call (c)
      (continue-whopper leader)
    ((condition-typep c 'eh:wrong-type-argument)
     (values (cond ((fixnump leader) (make-array 0 :element-type 'string-char :fill-pointer 0 :leader-length leader))
		   (t ""))
	     t))))


;;; A decompressing stream; uses Paul Fuqua's port of the Unix "compress" program.
;;; This is patched in to Zmacs so that visiting a file which ends in .Z will automagically decompress it
;;; into a buffer.  It might be nice if it was automatically compressed again on output.

(defun pathname-without-Z (pathname)
  "Given a pathname of type \"Z\", or of a type ending in \".Z\", returns a new pathname without the \".Z\" suffix.
  This reparses the pathname, so that (:NAME \"test\" :TYPE \"txt.Z\") will turn into (:NAME \"test\" :TYPE \"txt\")
  instead of (:NAME \"test.txt\" :TYPE :UNSPECIFIC)."
  (let* ((name (namestring pathname)))
    (if (string= ".Z" name :start2 (- (length name) 2))
	(pathname (subseq name 0 (- (length name) 2)))
	pathname)))

(defun pathname-with-Z (pathname)
  "Given a pathname, return a new pathname which has a suffix ending in .Z; that is,
  (:NAME \"test\" :TYPE \"txt\") will turn into (:NAME \"test\" :TYPE \"txt.Z\")"
  (let* ((otype (send pathname :raw-type)))
    (make-pathname :host (pathname-host pathname) :device (pathname-device pathname)
		   :directory (pathname-directory pathname) :name (pathname-name pathname)
		   :version (pathname-version pathname)
		   :raw-type (if (stringp otype)
				 (string-append otype ".Z")
				 "Z"))))


(defflavor compression-stream
	   (pathname
	    truename
	    file-stream)
	   (buffered-pipe-stream)
  :gettable-instance-variables
  :initable-instance-variables)

(defflavor decompressing-stream ()
	   (compression-stream)
  (:default-init-plist :source 'decompressing-stream-internal)
  :gettable-instance-variables
  :initable-instance-variables)

(defflavor compressing-stream ()
	   (compression-stream)
  (:default-init-plist :source 'compressing-stream-internal)
  :gettable-instance-variables
  :initable-instance-variables)


(defmethod (compression-stream :info) ()
  (send file-stream :info))

;(defmethod (decompressing-stream :info) ()
;  "Cons up a new info spec - instead of (<file.txt.Z> . <creation-date>) we return (<file.txt> . <current-time>).
;  This will make Zmacs complain if they try to write it back (though the warning message should be better)."
;  (let* ((source-info (send file-stream :info)))
;    (cons (pathname-without-Z (car source-info))
;	  (get-universal-time))))


(defun decompressing-stream-internal (pipe-right-side)
  (let* ((stream (send (send pipe-right-side :pipe-stream) :file-stream)))
    (unwind-protect (compress:internal-decompress-stream stream pipe-right-side nil)
      (close stream))))

(defun compressing-stream-internal (pipe-right-side)
  (let* ((stream (send (send pipe-right-side :pipe-stream) :file-stream)))
    (unwind-protect (compress:do-compression stream pipe-right-side nil)
      (close stream))))

;;; ## problem here is, pipes only work one way.  When you make a pipe, it is something you can
;;; ## read from, but not something you can write to.  I would like the semantics of make-decompressing-stream
;;; ## to be: input returns an uncompressing input stream, and output returns a compressing output stream.
;;; ## Instead of copying the buffer to the (compressing) file, we could copy the (compressing) buffer to
;;; ## a normal file.  But pipes really should work both ways...

(defun make-decompressing-stream (input-file &rest args &key (direction :input) &allow-other-keys)
  (when (or (getf args :characters)
	    (getf args :byte-size)
	    (getf args :element-type))
    ;; Characters is always NIL, and byte-size is always 8 (for now).
    ;; This will cause problems if someone really does want to open the stream for byte-size 16.
    (setq args (copy-list args))
    (remf args :characters)
    (remf args :byte-size)
    (remf args :element-type)
    )
  (let* ((file-stream (apply 'open input-file :characters nil :byte-size 8 args))
	 (pipe-stream (if (eq direction :output)
			  (make-instance 'COMPRESSING-STREAM
					 :pathname (send file-stream :pathname)
					 :truename (send file-stream :truename)
					 :file-stream file-stream)
			  (make-instance 'DECOMPRESSING-STREAM
					 :pathname (pathname input-file)
					 :truename (send file-stream :truename)
					 :file-stream file-stream))))
    (if (eq direction :output)
	(send pipe-stream :internal-stream)	; ## can't hand back the opposite side of a pipe like this!
	pipe-stream)))


(defmacro with-open-decompressing-stream ((stream input-file &rest args &key (characters t) byte-size element-type)
					  &body body)
  "Just like WITH-OPEN-FILE, except that if direction is :input and the file is compressed (ends in .Z, or has
  an extension that ends in .Z) then what you get in STREAM is a pipe; reading from it uncompresses on the fly.
  Likewise for output - the file written will be in compressed format (if it ends in .Z)."
  (declare (ignore characters byte-size element-type))
  (let* ((file (gensym))
	 (raw-type (gensym))
	 (compressed-file-p (gensym)))
    `(let* ((,file (pathname ,input-file))
	    (,raw-type (send ,file :raw-type))
	    (,compressed-file-p (and (stringp ,raw-type)
				     (or (string= "Z" ,raw-type)
					 (eql (search ".Z" ,raw-type :test #'char= :from-end t)
					      (- (length ,raw-type) 2))
					 ))))
       (with-open-stream (,stream
			  (if ,compressed-file-p
			      (make-decompressing-stream ,file ,@args)
			      (open ,file :direction :input ,@args)))
	 ,@body))))


(defun decompressing-open (input-file &rest args)
  (let* ((file (pathname input-file))
	 (raw-type (send file :raw-type))
	 (compressed-file-p (and (stringp raw-type)
				 (or (string= "Z" raw-type)
				     (eql (search ".Z" raw-type :test #'char= :from-end t)
					  (- (length raw-type) 2))
				     ))))
    (if compressed-file-p
	(apply #'make-decompressing-stream input-file args)
	(apply #'open input-file args))))


(compiler-let ((sys:compile-encapsulations-flag t))
  (sys:advise-within ZWEI:REVERT-FILE-BUFFER OPEN :around decompress nil
    ;;
    ;; Instead of the real open, call decompressing-open.
    ;;
    (apply 'decompressing-open sys:arglist)))


(compiler-let ((sys:compile-encapsulations-flag t))
  (sys:advise ZWEI:REVERT-FILE-BUFFER :around decompress nil
    ;;
    ;; Store a property on the buffer saying that it was decompressed.
    ;; Also, if the file does not have a -*- line, default the major mode properly -
    ;; "file.text.Z" should default to "text" mode.  Looks at ".text" instead of ".Z".
    ;;
    (let* ((buffer (first sys:arglist))
	   (input-file (second sys:arglist))
	   (uncompressed-pathname (and input-file (pathname-without-Z input-file)))
	   (file-mode (and input-file
			   (cdr (assoc (send uncompressed-pathname :canonical-type)
				       fs::*file-type-mode-alist* :test #'string-equal))))
	   (raw-type (and input-file (send input-file :raw-type)))
	   (compressed-file-p (and buffer input-file
				   (stringp raw-type)
				   (or (string= "Z" raw-type)
				       (eql (search ".Z" raw-type :test #'char= :from-end t)
					    (- (length raw-type) 2))))))
      (when compressed-file-p
	(setf (get buffer 'DECOMPRESSED) input-file))
      (let-if file-mode
	      ((fs::*file-type-mode-alist* (cons (cons :Z file-mode) fs::*file-type-mode-alist*)))
	:DO-IT))))


(compiler-let ((sys:compile-encapsulations-flag t))
  (sys:advise ZWEI:WRITE-FILE-INTERNAL :around decompress nil
    ;;
    ;; Instead of the real open, call decompressing-open.
    ;;
    (let* ((buffer (or (second sys:arglist) zwei:*interval*))
	   (target-file (pathname (car sys:arglist)))
	   (decompressed-p (get buffer 'DECOMPRESSED))
	   (compress-on-write
	     (and decompressed-p
		  ;(progn (format t "~&~s ~s ~s" buffer target-file decompressed-p) t)
		  (let* ((fquery-options `(:select t :type :readline :choices ,format:yes-or-no-p-choices)))
		    (if (and (equal (pathname-host decompressed-p)      (pathname-host target-file))
			     (equal (pathname-directory decompressed-p) (pathname-directory target-file))
			     (equal (pathname-name decompressed-p)      (pathname-name target-file))
			     (equal (pathname-type decompressed-p)      (pathname-type target-file)))
			(fquery fquery-options
				"The file ~A is in compressed format.~%Compress this file on output? " target-file)
			(fquery fquery-options
				"This buffer was read from the file ~A, which is in compressed format.~%~
			         Compress the file ~A on output? " decompressed-p target-file))))))
      (declare (special compress-on-write))
      (let ((real-output-file (if compress-on-write
				  target-file
				  (pathname-without-Z target-file))))
	(declare (special real-output-file))
;	(format t "~&~s ~s ~s ~s" target-file decompressed-p compress-on-write real-output-file)
	:DO-IT))))


(compiler-let ((sys:compile-encapsulations-flag t))
  (sys:advise-within ZWEI:WRITE-FILE-INTERNAL OPEN :around decompress nil
    ;;
    ;; Instead of the real open, maybe call decompressing-open.
    ;;
    (locally (declare (special real-output-file compress-on-write))
;      (format t "~&~s ~s" (and (boundp 'real-output-file) real-output-file) compress-on-write)
      (if (and (boundp 'real-output-file) real-output-file)
	  (if compress-on-write
	      (apply 'decompressing-open real-output-file (cdr sys:arglist))
	      :DO-IT)
	  :DO-IT))))




;;; Some silly example hacks - a fibonacci stream, and a stream of english numbers.

#|

(defun lie-to-me ()
  "Creates an infinite stream of fibonacci numbers.  Pop them by sending the stream the :TYI message."
  (make-unbuffered-pipe-stream #'(lambda (out)
				   (let ((p2 0)
					 (p1 1))
				     (loop
				       (send out :tyo p1)
				       (psetq p2 p1
					      p1 (+ p2 p1)))))))



(defun los-numeros (&optional how-many (buffer-size 10))
  "Creates an infinite stream of numbers.  Better than shortwave radio."
  (make-buffered-pipe-stream #'(lambda (out)
				 ;(format tv:selected-window "~&numbers starting...")
				 ;(unwind-protect
				     (let ((i 0))
				       (loop
					 ;(format tv:selected-window "~&--- writing ~s" (1+ i))
					 (format out "~R~%" (incf i))
					 (when (and how-many (>= i how-many)) (return))
					 ))
				   ;(format tv:selected-window "~&numbers shutting down.")
				   ;)
				 )
			     buffer-size))


 |#
