;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:TV; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B TR12BI ILSYMBOLS10 TR12BI); Vsp:0 -*-

;1;; File "3PHONES*"*
;1;; Written by Eric Gardner and Jamie Zawinski.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   distant past*	1Eric Gardner*	1 Created.*
;1;;    25 Apr 89*	1Jamie Zawinski*	1 Added changelog.  Removed the5 :PRINT-FUNCTION*s, since they didn't work anyway.*
;1;;    *				1 Phone numbers like 5621-0818* were printing as 5621-818* because 5~A* was being used instead of 5~4,'0D.**
;1;;    27 Apr 89*	1Jamie Zawinski *	1 Added a 5plist* slot to 5person* structure for the window interface to cache information.*
;1;;*				1 Made the reader not put NIL in slots declared to be of type STRING.*
;1;;    *			1 *	1 Made searches be case-insensitive.*
;1;;    11 May 89*	1Jamie Zawinski *	1 Defined *WRITE-PHONE-FILE1.*
;1;;      9 Jun 89*	1Jamie Zawinski *	1 Made *PHONE-LIST<1 be case-insensitive.*
;1;;* 1   23 Aug 89*	1Jamie Zawinski*	1 Cleaned up handling of db file.*


(export '(*my-area-code* 4*long-distance-dial-one-p** *default-phone-db-file*
	  phone phone-search))

(defparameter *my-area-code* 415 "2When dialing numbers, the area code will not be dialed if it is this.*")
(defparameter 4*long-distance-dial-one-p* *t "2Whether it is necessary to dial 1 before the area code when making a long distance call.*")
(defparameter *default-phone-db-file* nil "2The default file from which to load a database.*")

(defvar *phone-list* '() "2A 1list of PERSON structures*, the currently loaded phone database1.**")


(defstruct 4phone*
  (where	""  :type string)
  (area-code	0   :type integer)
  (prefix	0   :type integer)
  (extension	0   :type integer)
  (ext		nil :type (or null integer)))


(defun 4print-phone* (phone-rec stream &optional ignore)
  (if (phone-ext phone-rec)
      (format stream "3~&    ~10D: (~3D) ~3D-~4,'0D  ext: ~D~%*"
	      (phone-where phone-rec)
	      (phone-area-code phone-rec)
	      (phone-prefix phone-rec)
	      (phone-extension phone-rec)
	      (phone-ext phone-rec))
      (format stream "3~&    ~10D: (~3D) ~3D-~4,'0D~%*"
	      (phone-where phone-rec)
	      (phone-area-code phone-rec)
	      (phone-prefix phone-rec)
	      (phone-extension phone-rec))))

(defstruct 4address*
  (where   ""  :type string)
  (street1 ""  :type string)
  (street2 ""  :type string)
  (city    ""  :type string)
  (state   ""  :type string)
  (zip     nil :type (or null integer)))

(defun 4print-address* (address-rec stream &optional ignore)
  (if (string= (address-street2 address-rec) "")
      (format stream "3~&    ~A:~10,10T~A~%~10,10T~A, ~A  ~D*"
	      (address-where address-rec)
	      (address-street1 address-rec)
	      (address-city address-rec)
	      (address-state address-rec)
	      (address-zip address-rec))
      (format stream "3~&    ~A:~10,10T~A~%~10,10T~A~%~10,10T~A, ~A  ~D*"
	      (address-where address-rec)
	      (address-street1 address-rec)
	      (address-street2 address-rec)
	      (address-city address-rec)
	      (address-state address-rec)
	      (address-zip address-rec))))

(defstruct 4name*
  (first "" :type string)
  (last  "" :type string)
  (title "" :type string))


(defun 4pretty-print-person-name* (name &key (title nil) (print-first t) (print-last t))
  (setq title (and title (name-title name)))
  (setq print-first (and print-first (name-first name)))
  (setq print-last (and print-last (name-last name)))
  (cond ((null name)
	 "")
	(t
	 (format nil "3~:[~*~;~A ~]~:[~*~;~A~]~:[~; ~]~:[~*~;~A~]*"
		 title (name-title name)
		 print-first (name-first name)
		 print-last
		 print-last (name-last name)))))

(defstruct 4person*
  (company   ""  :type string)
  (name      nil :type (or null name))
  (phones    '() :type list)
  (addresses '() :type list)
  (net       ""  :type string)
  (note      ""  :type string)
  (plist     nil :type list)   ;1 the window interface puts properties here.*
  )

(defun 4print-person* (person stream &optional ignore)
  (cond ((and (person-name person) (string/= (person-company person) ""))
	 (format stream "3~&~A : ~A~%*" (pretty-print-person-name (person-name person)) (person-company person)))
	((person-name person)
	 (format stream "3~&~A~%*" (pretty-print-person-name (person-name person))))
	((string/= (person-company person) "")
	 (format stream "3~&~A~%*" (person-company person))))
  (when (person-phones person)
    (format stream "3~&  PHONES:~%*")
    (dolist (phone-rec (person-phones person))
      (princ phone-rec stream)))
  (when (plusp (length (person-addresses person)))
    (format stream "3~&  Addresses:~%*")
    (dolist (address-rec (person-addresses person))
      (princ address-rec stream)))
  (when (string/= (person-net person) "")
    (format stream "3~&  NET  : ~A~%*" (person-net person)))
  (when (string/= (person-note person) "")
    (format stream "3~&  NOTES: ~A~%*" (person-note person)))
  (terpri)
  nil)




;1;;  Functions for reading the phone database.*
;1;;*

(defun 4phone-list<* (person1 person2)
  (let* ((string1 (if (person-name person1)
		      (string-append (name-last (person-name person1)) (name-first (person-name person1)))
		      (person-company person1)))
	 (string2 (if (person-name person2)
		      (string-append (name-last (person-name person2)) (name-first (person-name person2)))
		      (person-company person2))))
  (string-lessp string1 string2)))

(defun 4read-person* (infile)
  "2Reads a structure of type PERSON from INFILE.*"
  (let* ((person (make-person))
	 (rec (read infile)))
    (do* ()
	 ((not rec))
      (case (car rec)
	(:company
	 (setf (person-company person) (or (cadr rec) ""))
	 (pop rec)
	 (pop rec))
	(:name
	 (let* ((name (cadr rec))
		(first (or (first name) ""))
		(last (or (second name) ""))
		(title (or (third name) ""))
		(name-rec (make-name :first first :last last :title title)))
	   (setf (person-name person) name-rec))
	 (pop rec)
	 (pop rec))
	(:phone
	 (let* ((number (cadr rec))
		(where (or (first number) "3phone:*"))
		(area-code (or (second number) *my-area-code*))
		(prefix (third number))
		(extension (fourth number))
		(ext (fifth number))
		(phone-rec (make-phone :where where
				       :area-code area-code
				       :prefix prefix
				       :extension extension
				       :ext ext)))
	   (push phone-rec (person-phones person)))
	 (pop rec)
	 (pop rec))
	(:address
	 (let* ((address (cadr rec))
		(where (first address))
		(street1 (or (second address) ""))
		(street2 (or (third address) ""))
		(city (or (fourth address) ""))
		(state (or (fifth address) ""))
		(zip (sixth address))
		(address-rec (make-address :where where
					   :street1 street1
					   :street2 street2
					   :city city
					   :state state
					   :zip zip)))
	   (push address-rec (person-addresses person)))
	 (pop rec)
	 (pop rec))
	(:net
	 (setf (person-net person) (cadr rec))
	 (pop rec)
	 (pop rec))
	(:note
	 (setf (person-note person) (cadr rec))
	 (pop rec)
	 (pop rec))
	(t
	 (format t "3~&Unknown Field ~A with value ~A.~%*" (pop rec) (pop rec))
	 )))
    (setf (person-phones person) (nreverse (person-phones person)))		;1 Preserve order from file.*
    (setf (person-addresses person) (nreverse (person-addresses person)))
    person))

(defun 4read-phone-file* (&optional (file-name *default-phone-db-file*) ok-if-not-found)
  "2Read a file of structures of type PERSON from infile.*"
  (unless file-name
    (setq file-name (make-pathname :defaults (user-homedir-pathname) :name "3PHONES*" :type "3TEXT*" :version :NEWEST)))
  (with-open-file (infile file-name :direction :input :if-does-not-exist (if ok-if-not-found nil :error))
    (when infile
      (setq *phone-list* '())
      (setq *default-phone-db-file* file-name) ;1 Set default now, we know it exists.*
      (do* ()
	   ((not (listen infile))
	    (setq *phone-list* (sort *phone-list* 'phone-list<)))
	(push (read-person infile) *phone-list*))
      t)))


(defun 4write*-phone-file (file-name)
  "2Write a file of structures of type PERSON to file-name.*"
  (setq *phone-list* (sort *phone-list* 'phone-list<))
  (with-open-file (stream file-name :direction :output)
    (format stream "3;;; -*- Mode:Common-Lisp -*-~2%*")
    (dolist (person *phone-list*)
      (princ "3(*" stream)
      (when (person-name person)
	(format stream "3:name (~S ~S*" (name-first (person-name person)) (name-last (person-name person)))
	(unless (string= "" (name-title (person-name person)))
	  (format stream "3 ~S*" (name-title (person-name person))))
	(princ "3)*" stream))
      (unless (string= "" (person-company person))
	(format stream "3 :company ~S*" (person-company person)))
      (dolist (phone (person-phones person))
	(format stream "3 :phone (~S ~3,'0D ~3,'0D ~4,'0D*"
		(phone-where phone) (phone-area-code phone) (phone-prefix phone) (phone-extension phone))
	(when (phone-ext phone) (princ "3 *" stream) (prin1 (phone-ext phone) stream))
	(princ "3)*" stream))
      (dolist (address (person-addresses person))
	(format stream "3 :address (~S ~S ~S ~S ~S ~5,'0D)*"
		(address-where address) (address-street1 address) (address-street2 address)
		(address-city address) (address-state address) (or (address-zip address) 0)))
      (when (and (person-net person) (string/= "" (person-net person)))
	(format stream "3 :net ~S*" (person-net person)))
      (when (and (person-note person) (string/= "" (person-note person)))
	(format stream "3 :note ~S*" (person-note person)))
      (format stream "3)~%*"))
    (truename stream)))



;1;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;*
;1;;*
;1;;  Functions for searching the phone database.*
;1;;*

(defmacro 4phone-string=* (str1 str2)
  `(or (not ,str1)
       (string-equal (string-upcase ,str1) (string-upcase ,str2))))

(defmacro 4phone-substring* (str1 str2)
  `(or (not ,str1)
       (zlc:string-search (string-upcase ,str1) (string-upcase ,str2) 0 nil 0 nil nil)))

(defun 4phone-number-search* (person search-val type)
  (if search-val
      (dolist (phone-rec (person-phones person) nil)
	(case type
	  (:area-code (when (= search-val (phone-area-code phone-rec))
			(return t)))
	  (:prefix (when (= search-val (phone-prefix phone-rec))
		     (return t)))
	  (:extension (when (= search-val (phone-extension phone-rec))
			(return t)))))
      t))

(defun 4phone-search* (&key (company nil) (name nil) (area-code nil) (phone-prefix nil) (phone-extension nil)
		          (net-address nil) (notes nil))
  (let* (person-list)
    (dolist (person *phone-list*)
      (when (and ;(or (string= company "") (phone-string= company (person-company person)))
	         (phone-substring (or company "") (or (person-company person) ""))
		 (phone-substring (or name "") (if (person-name person)
						   (string-append (name-last (person-name person))
								  #\Space
								  (name-first (person-name person)))
						   ""))
		 (phone-substring (or net-address "") (or (person-net person) ""))
		 (phone-substring (or notes "") (or (person-note person) ""))
		 (phone-number-search person area-code :area-code)
		 (phone-number-search person phone-prefix :prefix)
		 (phone-number-search person phone-extension :extension)
		 )
      (push person person-list)))
    (setq person-list (nreverse person-list))
    person-list))


(defun 4pretty-print-net-address *(person)
  (let* ((name-obj (tv:person-name person))
	 (name (and name-obj (tv:pretty-print-person-name name-obj)))
	 (net (tv:person-net person)))
    (values
      (when (and name net (string/= "" net))
	(cond ((or (position #\< net)		;1 already formatted.*
		   (< (or (position #\. net :start (or (position #\! net :from-end t) 0)) 9999)
		      (min (or (position #\@ net) 99999)	;1 in *fn.ln@host1 or *fn.ln@h1@h21 or *h3!h2!fn.ln@h11 form.*
			   (or (position #\% net) 99999))))
	       (format nil "3~A*" net))
	      ((position #\, net)		;1 multiple addresses*
	       (format nil "3~A <~A>, ~A*" name
		       (subseq net 0 (position #\, net)) (subseq net (1+ (position #\, net)))))
	      (t (format nil "3~A <~A>*" name net))))
      name)))


(defun 4map-net-addresses *(function)
  "2Call function with three arguments; a person-object, a pretty-printed name, and a pretty-printed network address.*"
  (dolist (person tv:*phone-list*)
    (multiple-value-bind (a n) (4pretty-print-net-address* person)
      (and a n (funcall function person n a)))))


(defun 4format-phone* (phone-rec)
  (if (phone-ext phone-rec)
      (format nil "3    ~10A: (~3D) ~3D-~4,'0D  ext: ~D*"
	      (phone-where phone-rec)
	      (phone-area-code phone-rec)
	      (phone-prefix phone-rec)
	      (phone-extension phone-rec)
	      (phone-ext phone-rec))
      (format nil "3    ~10A: (~3D) ~3D-~4,'0D*"
	      (phone-where phone-rec)
	      (phone-area-code phone-rec)
	      (phone-prefix phone-rec)
	      (phone-extension phone-rec))))


(defun 4format-person* (person)
  (string-append
    (cond ((and (person-name person) (string/= (person-company person) ""))
	   (format nil "3~A - ~A~%*" (pretty-print-person-name (person-name person)) (person-company person)))
	  ((person-name person)
	   (format nil "3~A~%*" (pretty-print-person-name (person-name person))))
	  ((not (string= (person-company person) ""))
	   (format nil "3~A~%*" (person-company person))))
    (if (person-phones person)
	(let* ((format-list ""))
	  (dolist (phone-rec (person-phones person))
	    (setq format-list (string-append format-list (format-phone phone-rec) #\Newline)))
	  format-list)
	"")
    (if (string/= (person-net person) "")
	(format nil "3    Net       : ~A~%*" (person-net person)) "")
    (apply #'string-append
	   (or (mapcar #'(lambda (address)
			   (string-append
			     (format nil "3    ~10A:*" (address-where address))
			     (if (string/= (address-street1 address) "")
				 (format nil "3 ~A~%*" (address-street1 address)) "")
			     (if (string/= (address-street2 address) "")
				 (format nil "3~16t~A~%*" (address-street2 address)) "")
			     (format nil "3~16t~A, ~A  ~5,'0D~%*" (or (address-city address) "")
				     (or (address-state address) "") (or (address-zip address) 0))))
		       (person-addresses person))
	       '("")))
    (if (string/= (person-note person) "")
	(format nil "3~%    ~A~&*" (person-note person))
	"")
    ))


(defun 4phone* (string)
  "2The command-line interface.*"
  (let* ((list (delete-duplicates (append (phone-search :name string) (phone-search :company string)))))
    (dolist (person list)
      (format t "3~2&~A*" (format-person person)))
    (values)))



;1;;  Functions for dialing a phone number.*
;1;;*

(defun dial-digit (digit)
  "2Produce the telephone tone corresponding to the given digit (0-9, #, or *).*"
  (setq digit (or (digit-char-p digit) digit))
  (multiple-value-bind (tone1 tone2)
      (ecase digit
	(1   (values 697 1209))		;1 A phone keypad is arranged in a 3x4 grid.*
	(2   (values 697 1336))		;1 Each button produces a two-note chord, with one note from each axis:*
	(3   (values 697 1477))		;1 *
	(4   (values 770 1209))		;	       81209*   81336*    81477*
	(5   (values 770 1336))		;		:	:	:
	(6   (values 770 1477))		;	8697* --	51* -----	52* -----	53*
	(7   (values 852 1209))		;		:	:	:
	(8   (values 852 1336))		;	8770* --	54* -----	55* -----	56*
	(9   (values 852 1477))		;		:	:	:
	(#\* (values 941 1209))		;	8852* --	57* -----	58* -----	59*
	(0   (values 941 1336))		;		:	:	:
	(#\# (values 941 1477))		;	8941* --	3** -----	50* -----	5#*
	)
    (tv:with-sound-enabled
      (tv:do-sound (tv:volume 0 :on))
      (tv:do-sound (tv:volume 1 :on))
      (tv:do-sound (tv:tone 0 tone1))
      (tv:do-sound (tv:tone 1 tone2))
      (sleep .05))))

(defun dial-number (number &optional digits)
  "2Produce the telephone tones corresponding to the given number (an integer or a string).*"
  (if (numberp number)
      (let* ((n (max (or digits 0) (1+ (floor (log number 10))))))
	(dotimes (i n)
	  (let* ((digit (- (truncate number (expt 10 (- n i 1)))
			   (* 10 (truncate number (expt 10 (- n i)))))))
	    (princ digit)
	    (dial-digit digit))
	  (sleep .1)))
      (dotimes (i (length number))
	(princ (char number i))
	(dial-digit (char number i))
	(sleep .1))))

(defun dial (number)
  "2Produce the telephone tones corresponding to the given PHONE structure.*"
  (let* ((area-code (phone-area-code number))
	 (prefix (phone-prefix number))
	 (extension (phone-extension number)))
    (unless (= *my-area-code* area-code)
      (when 4*long-distance-dial-one-p**
	(dial-number 1)
	(princ #\-))
      (princ #\()
      (dial-number area-code 3)
      (princ #\)))
    (dial-number prefix 3)
    (princ #\-)
    (dial-number extension 4)
    (terpri)))
