;;; -*- Mode:Common-Lisp; Package:USER; Base:10 -*-

;;; Code which emulates a Heath-19 terminal talking to a modem over the serial port of the local machine.
;;; This code is pretty useless if you have your modem hooked to a unix box - you can use VT100 to get to
;;; the unix world, and then use 'tip' to talk to the modem.  (When I wrote this, we didn't have anything but TIs.)
;;;
;;; But this might be good example code if for some reason you need to interpret H19 control sequences ... (why?)


;;; ChangeLog:
;;;
;;; 20 Mar 87  Jamie Zawinski  Added changelog, rewrote lots.  Gave it its own process, made it almost full screen.
;;; 23 Mar 87  Jamie Zawinski  Fixed Direct Cursor Addressing, and more generally, stripped of the high bits.
;;; 26 Mar 87  Jamie Zawinski  Defined H19-START-RECORDING and H19-STOP-RECORDING.
;;;  4 Apr 87  Jamie Zawinski  Added the *ADDITIONAL-H19-KEYBINDINGS*, the HELP message, and make ABORT send Control-C.
;;;  6 Apr 87  Jamie Zawinski  Defined COMMON-TO-ASCII-SERIAL-STREAM flavor.
;;;                            Made :H19-ON-SELF print its errors instead of erroring in the background stream.
;;;  7 Apr 87  Jamie Zawinski  Wrote CALL-CMU-EFE and CALL-CMU-FE, and the code that made them easy.
;;;                            Got COMMON-TO-ASCII-SERIAL-STREAM to treat CRLF as CR while accepting CR and LF separately.
;;;

;;; USER::H19 (&OPTIONAL RESTART)
;;;   Brings up a Modem Window, 120 characters by 55 lines, emulating a Heath-19 Terminal.
;;;   It talks over the serial port at 1200 baud.
;;;   If RESTART is T and The Modem Window already exists, then the Modem Process is restarted.
;;;    Do this if you get a Lisp error.
;;;   If RESTART is :ALL, then the window is reconstructed.  Do this is the window instance is damaged.
;;;
;;; USER::H19-START-RECORDING (FILENAME)
;;;   Transcribe all Modem I/O to the file specified.
;;; USER::H19-STOP-RECORDING ()
;;;   Close the file and record no more.
;;;
;;; USER::CALL-CMU-EFE
;;;   Dials up to the CMU ethernet front end, calling a different number until it connects, sets the BREAK option so that
;;;    The BREAK key will bring you back to the front end, and starts H19.
;;;
;;; USER::CAL-CMU-FE
;;;   Just like CALL-CMU-EFE, but does the old front end instead.
;;;

(defvar *modem-stream* nil  "A stream to the Serial port.")
(defvar *modem-window* nil  "The window H19 runs in.")
(defvar *h19-record-stream* nil "If non-NIL, a stream to which all IO is copied.")

(defparameter *additional-h19-keybindings* nil
  "An Alist of nonstandard keybindings.  The KEY should be a character, and the value should be a character, 
integer ASCII code, string, or list of characters or ASCII codes.")

(defvar *escape-mode-p* nil)
(defvar *insert-mode-p* nil)
(defvar *reverse-video-mode-p* nil)
(defvar *saved-pos-x* 0)
(defvar *saved-pos-y* 0)

(defflavor modem-window ()
	   (tv:process-mixin
	    tv:auto-scrolling-mixin   ; does scrolling the RIGHT way.
	    tv:truncating-window)    ; truncates lines at the right margin.
  (:default-init-plist
    :process '(start-h19-process :regular-pdl-size 7000
				 :special-pdl-size 7000)))

(defmethod (modem-window :after :init) (&rest ignore)
  (setq tv:char-aluf tv:alu-seta))  ; Otherwise it doesn't do INSERT-CHAR right.

(defflavor common-to-ascii-serial-stream
	   ((character-translation-p t)     ; Whether to translate the characters.
	    (last-char #\Null))             ; The last character read.
	   (si::serial-stream)
  :gettable-instance-variables
  :settable-instance-variables
  (:documentation :combination "RS 232 Serial Stream, with very good CommonLisp <==> ASCII character translation."))


(defwrapper (common-to-ascii-serial-stream :tyi) ((&optional no-hang-p) . body)
  `(when (or (neq no-hang-p :nohang) (listen self))
     (let* ((char ,@body))
       (setq char (if character-translation-p
		      (char-int (ascii-to-common char))
		      (char-int char)))
       (prog1
	 (if (or (and character-translation-p (char= char #\Linefeed) (char= last-char #\Return))
		 (and (char= (ascii-to-common char) #\Linefeed) (char= (ascii-to-common last-char) #\Return)))
	     (if (eq no-hang-p :nohang)
		 (when (listen self) (send self :tyi :nohang))
		 (send self :tyi))
	     char)
	 (setq last-char char)))))

(defmethod (common-to-ascii-serial-stream :tyi-no-hang) (&rest ignore)
  (send self :tyi :nohang))

(defmethod (common-to-ascii-serial-stream :tyo) (char &optional (handle-meta-p t))
  (when (characterp char) (setq char (int-char char)))
  (setq char (if character-translation-p
		 (common-to-ascii char handle-meta-p)
		 char))
  (when char (funcall #'(:method si::serial-stream-mixin :tyo) :tyo (char-int char))))


(defmacro with-ascii-translation ((stream) &body body)
  (let* ((st (gensym))
	 (old-val (gensym)))
    `(let* ((,st ,stream)
	    (,old-val (send ,st :character-translation-p)))
       (unwind-protect (progn (send ,st :set-character-translation-p t)
			      ,@body)
	 (send ,st :set-character-translation-p ,old-val)))))

(defmacro without-ascii-translation ((stream) &body body)
  (let* ((st (gensym))
	 (old-val (gensym)))
    `(let* ((,st ,stream)
	    (,old-val (send ,st :character-translation-p)))
       (unwind-protect (progn (send ,st :set-character-translation-p nil)
			      ,@body)
	 (send ,st :set-character-translation-p ,old-val)))))

(defun start-h19-process (window)
  "Called by the Scheduler.  Runs H19 in WINDOW."
  (initialize-modem)
  (send window :h19-on-self))

(defun h19 (&optional reset)
"Brings up a Modem Window, 120 characters by 55 lines, emulating a Heath-19 Terminal.
It talks over the serial port at 1200 baud.
If RESTART is T and The Modem Window already exists, then the Modem Process is restarted.
 Do this if you get a Lisp error.
If RESTART is :ALL, then the window is reconstructed.  Do this is the window instance is damaged.

When running the Heath-19 Emulator, you have some nice keyboard-assistance:
  END            ==> Exit the program.
  CLEAR-SCREEN   ==> Control-L
  RUBOUT         ==> Backspace (ASCII 8)
  CONTROL-RUBOUT ==> Rubout (ASCII 127)
  BACK-ARROW     ==> Control-B
  FORWARD-ARROW  ==> Control-F
  DOWN-ARROW     ==> Control-N
  UP-ARROW       ==> Control-P
  BREAK          ==> Break (ASCII 31, Control-_)
  ABORT          ==> Control-C

Certain keystrokes translate into two characters, for example:
  META-A          ==> ESC a
  META-SHIFT-A    ==> ESC A
  META-CONTROL-A  ==> ESC CONTROL-A
  CONTROL-SHIFT-A ==> ESC a

HELP tells you all of this stuff again.

The variable *ADDITIONAL-H19-KEYBINDINGS* lets you define other bindings.  It associates a character or integer
with another character or integer, or with a string or list of characters or integers.
"
  (setup-modem-window (eq reset :all))
  (when reset
    (send (send *modem-window* :process) :reset)
    (when *modem-stream* (send *modem-stream* :clear-input))
    (send *modem-window* :clear-input))
  (send *modem-window* :select))

(defun h19-help ()
  "Print keystroke help on the H19 window."
  ;; Save what's on the screen in the saved-bits array.
  (send *modem-window* :bitblt-from-sheet tv:alu-seta (send *modem-window* :width) (send *modem-window* :height) 0 0
	               (send *modem-window* :bit-array) 0 0)
  ;; save the current cursor position.
  (let* ((x (- (send *modem-window* :cursor-x) (send *modem-window* :top-margin-size)))
	 (y (- (send *modem-window* :cursor-y) (send *modem-window* :left-margin-size))))
    (send *modem-window* :clear-screen)
    (format *modem-window* "Explorer Heath 19 Keybindings:~%~%   ~
                          END             ==> Exit the program.~%   ~
                          CLEAR-SCREEN    ==> Control-L~%   ~
                          RUBOUT          ==> Backspace (ASCII 8)~%   ~
                          CONTROL-RUBOUT  ==> Rubout (ASCII 127)~%   ~
                          BACK-ARROW      ==> Control-B~%   ~
                          FORWARD-ARROW   ==> Control-F~%   ~
                          DOWN-ARROW      ==> Control-N~%   ~
                          UP-ARROW        ==> Control-P~%   ~
                          BREAK           ==> Break (ASCII 31, Control-_)~%   ~
                          ABORT           ==> Control-C~%~%~
                          Certain keystrokes translate into two characters, for example:~%   ~
                          META-A          ==> ESC a~%   ~
                          META-SHIFT-A    ==> ESC A~%   ~
                          META-CONTROL-A  ==> ESC CONTROL-A~%   ~
                          CONTROL-SHIFT-A ==> ESC a~%~%~
              The variable *ADDITIONAL-H19-KEYBINDINGS* lets you define other bindings.  It associates a character or integer
              with another character or integer, or with a string or list of characters or integers.~
              ~:[~;~%~%Current user-bindings:~%~%~
              ~:{     ~:C~19t==>~{ ~:C~}~%~}~]~%~%~
              Type any character to remove this message."
	    *additional-h19-keybindings*
	    (mapcar #'(lambda (x)
			(list (car x)
			      (if (typep (cdr x) 'sequence)
				  (mapcar #'ascii-to-common (coerce (cdr x) 'list))
				  (list (ascii-to-common (cdr x))))))
		    *additional-h19-keybindings*))
    (send *modem-window* :tyi)
    (send *modem-window* :set-cursorpos x y))
  (send *modem-window* :bitblt tv:alu-seta (send *modem-window* :width) (send *modem-window* :height)
	               (send *modem-window* :bit-array) 0 0 0 0)
  nil)

(defun setup-modem-window (&optional recreate-p)
  "Initialize the Modem Window.  If RECREATE-P is T, then the old instance is discarded."
  (when (or recreate-p (null *modem-window*))
    (when *modem-window* (send (send *modem-window* :process) :kill))
    (let ((winx 10) (winy 10))
      (setq *modem-window* (make-instance 'modem-window
					  :scroll-pixel-increment (tv:font-char-height fonts:cptfont)
					  :edges (list winx winy (+ 20 winx) (+ 20 winy))
					  :borders 2
					  :save-bits t
					  :more-p nil
					  :truncate-line-out-flag 1
					  :backspace-not-overprinting-flag 0
					  :name "Modem Window"
					  :font-map (make-array 26 :initial-element fonts::cptfont)
					  :label fonts::tr10i))))
  (send *modem-window* :set-size-in-characters 120 55))

(defun initialize-modem (&optional recreate-p)
  "Initialize the Modem Serial Stream.  If RECREATE-P is T, then the old instance is closed and discarded."
  (when (or recreate-p (null *modem-stream*))
    (when *modem-stream* (close *modem-stream*))
    (setq *modem-stream* (si:make-serial-stream :flavor 'common-to-ascii-serial-stream
						:ascii-characters nil
						:xon-xoff-protocol nil
						:force-output t
						:number-of-stop-bits 1
						:baud 1200)))
  (send *modem-stream* :init nil)
  (send *modem-stream* :reset)
  t)

(defun strip-high-bit (num)
  "Treating NUM as 8 bits, returns the bottom 7."
  (values (rem num 128)))

(defun char-shifted-p (char)
  "Whether the SHIFT key was used in the production of CHAR.
But, even though you type + with SHIFT-=, = is not CHAR-SHIFTED-P.
This is primarily useful for seeing the difference between CONTROL-A and CONTROL-SHIFT-A."
  (let* ((stripped-char (make-char char))  ; strip the bits.
	 (result (and (both-case-p stripped-char)
		      (char/= stripped-char (char-upcase stripped-char)))))
    (if (and (char= char stripped-char) (both-case-p char))
	(not result)
	result)))

(defun common-to-ascii (char &optional handle-meta-p)
  "Converts a CommonLisp character into an ASCII code (integer).  If HANDLE-META-P is true, then (when a Meta-character is
passed in) ESC will be sent to *MODEM-STREAM* before the ASCII version of CHAR is returned.
The value returned may be NIL if there is no ASCII equivalent."
  (unless (characterp char) (setq char (int-char char)))
  (when handle-meta-p
    (cond ((and (char-bit char :meta) (not (char-bit char :control)))  ; META-A        ==>  ESC-a
	   (send *modem-stream* :tyo 27)                               ; META-SHIFT-A  ==>  ESC-A
	   (if (char-shifted-p char)    
	       (setq char (char-upcase (make-char char)))
	       (setq char (char-downcase (make-char char)))))
	  ((and (char-bit char :meta) (char-bit char :control))        ; META-CONTROL-A  ==>  ESC-CONTROL-A
	   (send *modem-stream* :tyo 27)
	   (setq char (make-char char 1)))
	  ((and (char-bit char :control) (char-shifted-p char))        ; CONTROL-SHIFT-A ==>  ESC-a
	   (send *modem-stream* :tyo 27)
	   (setq char (char-downcase (make-char char))))
	  (t  nil)))
  (case char             ;; special cases.
	(#\Rubout       8)    ; backspace
	(#\C-Rubout     127)  ; rubout
	(#\Escape       27)
	(#\C-\[         27)   ; escape
	(#\C-\]         28)
	(#\Return       13)
	(#\LineFeed     10)
	(#\Tab          9)
	(#\C-@          0)
	(#\C-Space      0)
	(#\C-_          31)
	(#\            2)    ; Control-B
	(#\            6)    ; Control-F
	(#\            14)   ; Control-N
	(#\            16)   ; Control-P
	(#\Clear-Screen 12)   ; Control-L
	(#\Abort        3)    ; Control-C
	(#\Break        31)   ; Control-_
	(#\Help         (h19-help) nil)
	(t (let* ((personalized-binding (cdr (assoc char *additional-h19-keybindings*
						    :test #'(lambda (x y) (= (char-int x) (char-int y)))))))
	     ;; Make (A B) work the same as (A . B) without messing up (A B C)
	     (when (and personalized-binding (listp personalized-binding) (= (length personalized-binding) 1))
	       (setq personalized-binding (car personalized-binding)))
	     (cond (personalized-binding
		    (cond ((or (characterp personalized-binding) (integerp personalized-binding))
			   (send *modem-stream* :tyo (int-char personalized-binding)))
			  ((typep personalized-binding 'sequence)
			   (dotimes (x (length personalized-binding))
			     (send *modem-stream* :tyo (int-char (elt personalized-binding x)))))
			  (t (beep)))
		    nil) ; Return NIL since we already wrote characters onto the stream.
		   ((member char '(#\Resume #\Break #\Network #\Status #\Undo #\Clear-Input #\F1 #\F2 #\F3 #\F4)
			    :test #'equalp)
		    (beep) nil)  ; Illegal characters unless they are rebound in *ADDITIONAL-H19-KEYBINDINGS*.
		   (t  (if (zerop (char-bits char))
			   ;; if no bits are set, then pass it through,
			   (char-code char)
			   ;; otherwise strip the bits.
			   (let ((stripped-char (make-char char)))
			     (if (alpha-char-p stripped-char)
				 (if (char-bit char :control)
				     (- (char-code (char-upcase stripped-char)) 64) ;META-A => A, META-CONTROL-A => CONTROL-A
				     (char-code (char-upcase stripped-char)))
				 stripped-char)))))))))

(defun ascii-to-common (char)
  "Converts an ASCII character code to a CommonLisp character."
  (unless (integerp char) (setq char (char-int char)))
  (setq char (strip-high-bit char))
  (case char
	(10  #\Linefeed)
	(13  #\return)
	((8 127) #\backspace)
	(32 #\space)
	(27 #\escape)
	(t (if (< char 32)
	       (make-char (+ 64 char) 1)
	       (make-char char)))))


;; Internal functions of HANDLE-H19-ESCAPE-CODES.

(defun cursor-up () (send *modem-window* :increment-cursorpos 0 -1 :character))
(defun cursor-down () (send *modem-window* :increment-cursorpos 0 1 :character))
(defun cursor-forward () (send *modem-window* :forward-char))
(defun cursor-back () (send *modem-window* :backward-char))
(defun cursor-home () (send *modem-window* :home-cursor))
(defun clear-screen () (send *modem-window* :clear-screen))
(defun clear-to-end-of-screen () (send *modem-window* :clear-eof))
(defun clear-to-eol () (send *modem-window* :clear-eol))

(defun insert-line ()
  (send *modem-window* :insert-line 1 :character)
  (multiple-value-bind (x y) (send *modem-window* :read-cursorpos :character)
    (declare (ignore x))
    (send *modem-window* :set-cursorpos 0 y)))

(defun delete-line ()
  (send *modem-window* :delete-line 1 :character)
  (multiple-value-bind (x y) (send *modem-window* :read-cursorpos :character)
    (declare (ignore x))
    (send *modem-window* :set-cursorpos 0 y)))			                            

(defun delete-character ()  (send *modem-window* :delete-char 1 :character))
(defun clear-to-beginning-of-screen () 
  (multiple-value-bind (curs-x curs-y)
      (send *modem-window* :read-cursorpos :character)
    (send *modem-window* :clear-between-cursorposes 0 0 curs-x curs-y)))

(defun save-cursor-position ()
  (multiple-value-setq (*saved-pos-x* *saved-pos-y*) (send *modem-window* :read-cursorpos :character)))

(defun cursor-to-saved-pos ()
  (send *modem-window* :set-cursorpos *saved-pos-x* *saved-pos-y* :characters))

(defun erase-line ()
  (multiple-value-bind (curs-x curs-y)
      (send *modem-window* :read-cursorpos :character)
    (declare (ignore curs-x))
    (send *modem-window* :clear-between-cursorposes 0 curs-y (send *modem-window* :size-in-characters) curs-y)))

(defun erase-to-bol ()
  (multiple-value-bind (curs-x curs-y)
      (send *modem-window* :read-cursorpos :character)
    (send *modem-window* :clear-between-cursorposes 0 curs-y curs-x curs-y)))

(defun back-tab ()
  (multiple-value-bind (x y)
      (send *modem-window* :read-cursorpos :character)
    (send *modem-window* :set-cursorpos (* 8 (floor x 8)) y :character)))

(defun goto-direct ()
  (let* ((newy (- (strip-high-bit (char-code (read-char *modem-stream* :tyi))) 32))
	 (newx (- (strip-high-bit (char-code (read-char *modem-stream* :tyi))) 32)))
;		  (format t "direct cursor addressing -- x: ~d   y: ~d~%" newx newy)
    (send *modem-window* :set-cursorpos newx newy :character)))

(defun line-wrap-mode ()  (send *modem-window* :set-truncate-line-out-flag 0))
(defun line-trunk-mode ()  (send *modem-window* :set-truncate-line-out-flag 1))

(defun handle-h19-escape-codes (char)
  "Processes CHAR as appropriate.  CHAR came from the Serial port, from a computer that thinks we are a Heath-19."
  (declare (inline cursor-up cursor-down cursor-forward cursor-back cursor-home clear-screen clear-to-end-of-screen
		   erase-eol insert-line delete-line delete-char clear-to-beginning-of-screen save-cursor-position
		   cursor-to-saved-pos erase-line erase-to-bol back-tab goto-direct line-wrap-mode line-trunk-mode))
  (unless (characterp char) (setq char (int-char char)))
  (setq char (int-char (strip-high-bit (char-int char))))
  (without-ascii-translation (*modem-stream*)
    (cond (*escape-mode-p*
	   (setq *escape-mode-p* nil)
	   (case char
		 (#\A (cursor-up))
		 (#\B (cursor-down))
		 (#\C (cursor-forward))
		 (#\D (cursor-back))
		 (#\H (cursor-home))
		 (#\E (clear-screen))
		 (#\J (clear-to-end-of-screen))
		 (#\K (clear-to-eol))
		 (#\L (insert-line))
		 (#\M (delete-line))
		 (#\N (delete-character))
		 (#\@ (setq *insert-mode-p* t))
		 (#\O (setq *insert-mode-p* nil))
		 (#\b (clear-to-beginning-of-screen))
		 (#\j (save-cursor-position))
		 (#\k (cursor-to-saved-pos))
		 (#\I (erase-line))
		 (#\o (erase-to-bol))
		 (#\- (back-tab))
		 (#\Y (goto-direct))
		 (#\p (setq *reverse-video-mode-p* t))
		 (#\q (setq *reverse-video-mode-p* nil))
		 (#\v (line-wrap-mode))
		 (#\w (line-trunk-mode))
		 (t
		  (send *modem-window* :beep)
		  ; (format t "~&WARNING: Character ~S (~D) is an unknown Escape Code~%" char (char-code char))
		  )))
	  (t (case (char-int char)
		   (0 nil)			; ignore nulls (padding)
		   ((10 13)  (multiple-value-bind (x y) (send *modem-window* :read-cursorpos :character)
			  (declare (ignore x))
			  (send *modem-window* :set-cursorpos 0 (1+ y) :character))
			(when *h19-record-stream* (terpri *h19-record-stream*)))
;		   (13  (beep :doorbell))
		   ((8 127) (send *modem-window* :backward-char)
			    (when *h19-record-stream* (princ #\Backspace *h19-record-stream*)))
		   (27 (setq *escape-mode-p* t))
		   (12 (send *modem-window* :clear-screen)
		       (when *h19-record-stream* (princ #\Page *h19-record-stream*)))
		   (7  (send *modem-window* :beep :vt100-beep))
		   
		   (t   ;; all other characters
		    ;; strip the high bit; ASCII is <= 127.
		    (setq char (if (> (int-char char) 127)
				   (code-char (- (int-char char) 128))
				   (code-char (int-char char))))
		    (when *insert-mode-p* (send *modem-window* :insert-char))
		    (send *modem-window* :tyo char)
		    (when *h19-record-stream* (princ (int-char char) *h19-record-stream*))))))))

(defmethod (modem-window :after :tyo) (&rest ignore)
  "Do magic for *REVERSE-VIDEO-MODE-P*."
  (when *reverse-video-mode-p*
    (tv:prepare-sheet (self)
      (tv::%draw-rectangle-clipped tv:char-width (- tv:line-height 2) (- tv:cursor-x tv:char-width) tv:cursor-y
				   tv:alu-xor self))))

(defmethod (modem-window :h19-on-self) ()
  "The internal loop of the Heath-19 emulator."
  (send self :clear-screen)
  (setq *escape-mode-p* nil)
  (setq *insert-mode-p* nil)
  (setq *reverse-video-mode-p* nil)
  (let* ((char nil) (back nil) (tv:kbd-intercepted-characters nil))
    (loop
      (condition-call (condition)
	  (block INTERCEPT-CHARACTER
	    (process-wait "Keyboard or Serial" #'(lambda () (or (listen *modem-stream*) (listen *modem-window*))))
	    (process-wait "Wait until Exposed" #'(lambda () (send *modem-window* :exposed-p)))
	    (without-ascii-translation (*modem-stream*)
	      (setq back (send *modem-stream* :tyi-no-hang))
	      (when back (handle-h19-escape-codes (strip-high-bit back))))
	    (setq char (send self :tyi-no-hang))
	    (when (eql char (char-code #\End))
	      (send self :deactivate)
	      (return-from INTERCEPT-CHARACTER))
	    (when char (princ (int-char char) *modem-stream*)))
	(t (format t "H19 got an error: ~a" condition))))))

(defun h19-start-recording (filename)
  "Says that H19 should record all transaction in the file FILENAME."
  (h19-stop-recording)
  (setq *h19-record-stream* (open filename :direction :output)))

(defun h19-stop-recording ()
  (when *h19-record-stream*
    (close *h19-record-stream*)
    (setq *h19-record-stream* nil)))

(compile-flavor-methods MODEM-WINDOW)


(defun dumb-tty (&optional reset-p)
  (format t "~&DUMB TTY to Serial Port at 1200 Baud.  Type  to exit.~%")
  (initialize-modem reset-p)
  (let* ((char nil) (back nil) (tv:kbd-intercepted-characters nil))
    (loop
      (process-wait "Keyboard or Serial" #'(lambda (termio) (or (listen *modem-stream*) (listen termio)))
		                         *terminal-io*)
      (setq back (send *modem-stream* :tyi-no-hang))
      (when back (princ (int-char back) *terminal-io*))
      (setq char (send *terminal-io* :tyi-no-hang))
      (when (eql char (char-code #\End))
	(return-from DUMB-TTY))
      (when char (princ (int-char char) *modem-stream*)))))

(defun modem-command-reset ()
  "Software-resets the modem - Go to LOCAL state, hang up the phone, remote echo mode, default registers."
  (sleep 1.1) (princ "+++" *modem-stream*) (sleep 1.1)
  (format *modem-stream* "ATZ~%")
  (sleep 2) (send *modem-stream* :clear-input)
  (modem-command "ATH")
  t)

(defun read-line-echoing (stream-to-read stream-to-echo)
  "Does a READ-LINE on STREM-TO-READ, and returns it after printing it on STREAM-TO-ECHO."
  (let* ((l (read-line stream-to-read)))
    (when stream-to-echo
      (fresh-line stream-to-echo)
      (princ l stream-to-echo)
      (terpri stream-to-echo))
    l))

(defun modem-command (string &optional (echo-stream *terminal-io*))
  "Sends the STRING to the modem, waits for a response, and returns a string of the modem's response.
ECHO-STREAM, if non-NIL, says whether and where to echo all transactions with the modem."
  (send *modem-stream* :clear-input)
  (princ string *modem-stream*)
  (terpri *modem-stream*)
  ;; Discard the modem's echoback.
  (do* ((resp (read-line-echoing *modem-stream* echo-stream)
	      (read-line-echoing *modem-stream* echo-stream)))
       ((not (string= resp ""))))
  ;; Return the response.
  (do* ((resp (read-line-echoing *modem-stream* echo-stream)
	      (read-line-echoing *modem-stream* echo-stream)))
       ((not (string= resp ""))
	resp)))

(defun wait-for (string &optional (echo-stream *terminal-io*))
  "Read lines from the modem until one contains the string STRING.  Return that line."
  (do* ((resp (read-line-echoing *modem-stream* echo-stream)
	      (read-line-echoing *modem-stream* echo-stream)))
       ((search string resp)
	resp)))

(defvar *cmu-efe-numbers* '(6218363 6218368 6218389 6218486 6218499 6218516 6218519 6218542
			    6218548 6218576 6218626 6218669 6218676 6218720 6218770 6218812))

(defvar *cmu-fe-numbers* '(6213812 6215448 6215497 6217266 6217371 6213561 6213564 6213575
			   6213585 6213592 6213633 6213637 6213659 6213671 6213677 6213698))


(defun call-cmu-efe (&optional (numbers *cmu-efe-numbers*) reset-p)
  (format t "~&; Resetting modem...")
  (modem-command-reset)
  (block DIAL-LOOP
    (loop
      (dolist (num numbers)
	(send *modem-stream* :clear-input)
	(let* ((response (modem-command (string-append "ATDT" (princ-to-string num)))))
	  (when (search "CONNECT" response)
	    ;; Get to the front end.  Every second type a return until we get a prompt, or 30 seconds.
	    (with-timeout (1800 (format t "~&Timeout connecting.~%"))
	      (sleep 0.5)
	      (terpri *modem-stream*)  (terpri *modem-stream*)
	      (wait-for "ommand")
	      (return-from DIAL-LOOP)))))))
  (sleep 0.5)
  (princ "option break" *modem-stream*) (terpri *modem-stream*)
  (wait-for "opt")
  (princ "option no halt" *modem-stream*) (terpri *modem-stream*)
  (wait-for "opt")
  (send *modem-stream* :clear-input)
  (format t "~&Connected.  Starting H19...")
  (h19 reset-p)
  (send *modem-window* :clear-screen))

(defun call-cmu-fe (&optional (numbers *cmu-fe-numbers*) reset-p)
  (format t "~&; Resetting modem...")
  (modem-command-reset)
  (block DIAL-LOOP
    (loop
      (dolist (num numbers)
	(send *modem-stream* :clear-input)
	(let* ((response (modem-command (string-append "ATDT" (princ-to-string num)))))
	  (when (search "CONNECT" response)
	    ;; Get to the front end.  Every second type a return until we get a prompt.
	    (sleep 0.5)
	    (terpri *modem-stream*)  (terpri *modem-stream*)
	    (wait-for "ost")
	    (return-from DIAL-LOOP))))))
  (send *modem-stream* :clear-input)
  (format t "~&Connected.  Starting H19...")
  (h19 reset-p)
  (send *modem-window* :clear-screen))