;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:ZWEI; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER ADOBE-COURIER14B HL12B HL12I HL12BI) -*-

;1;; File "3MAIL-VOTE-PATCH*".*
;1;; Making the TI mail reader understand automatic mail voting.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   30 Mar 90*	1Jamie Zawinski*	1Created.*
;1;;*   110 Apr 90*	1Jamie Zawinski *	1Changed vote-reply format to conform to Andrew's.*
;1;;*

;1;; The 5Vote* feature.*
;1;; *
;1;; It is possible to send a vote accompanying a message. * 1Compose the*
;1;; message to explain the issue you want people to submit a vote on*
;1;; and* 1then say 5Meta-X Insert Vote*. * 1You will be prompted in the echo*
;1;; area for* 1the vote topic, and for the vote's choices, one at a time.*
;1;; You may enter* 1as many vote choices as you wish. * 1When you have*
;1;; entered all the choices,* 1hit return (an empty choice). * 1You will then be*
;1;; asked whether you want to allow write-in* 1votes.*
;1;;*
;1;; When the recipient reads the message for the first time, they will be*
;1;; presented with a menu containing the vote question and the vote*
;1;; choices,* 1along with a 5No Vote* choice. * 1When they choose one, their*
;1;; vote is sent to* 1you through the mail.*
;1;;*
;1;;* 1After selecting an answer from the pop-up menu, the user is given a*
;1;; chance* 1to change their vote. * 1After making a final choice, they are*
;1;; placed in* 1a send-mail buffer with the appropriate information filled in:*
;1;; the vote* 1answer, the vote-creator's destination address, etc. * 1They*
;1;; may then edit* 1the message, add more information, or simply hit 5End* to*
;1;; send it off.*  
;1;;*
;1;; The vote information lives in three message-header fields: 5Vote-Request*,*
;1;; 5Vote-To*, and 5Vote-Choices*.  5Vote-Request* contains two items: a string*
;1;; uniquely identifying this vote, and the vote question itself.  5Vote-To**
;1;; contains the address or addresses to which votes should be delivered.*
;1;; And 5Vote-Choices* contains the legitimate answers, separated by commas.*
;1;; A write-in vote is indicated by a choice consisting only of an asterisk.*
;1;;*
;1;; For example, here is a vote as it appeared on an Andrew bboard:*
;1;;*
;1;; 5Vote-Request:**	1 2ligonier.andrew.cmu.edu.1493.3, What Movies do you want to see?**
;1;; 5Vote-To:**	1 2David Allen Markley <dm3e+Vote2@andrew.cmu.edu>**
;1;; 5Vote-Choices:**	1 2The Princess Bride, Any Ralph Bakshi Movie, Baron Munchausen, Without A Clue, Spaceballs, ***
;1;;*
;1;; Vote replies are of the form `` 5My vote on '2<vote-id>*' is '2<vote-choice>*'.* '' where*
;1;; 2<vote-id>* is the first field of the 5Vote-Request* header, and 2<vote-choice>* is*
;1;; the answer the user picked from the menu (or a write-in).  This reply *
;1;; appears both in the 5Subject* line and in the body of the message.*
;1;;*
;1;; The vote mechanism implemented in this file is fully compatible with the*
;1;; vote mechanism used by the Andrew Message System at CMU.*
;1;;*


(defvar 4*voting-enabled* *t "2Set this to NIL to turn off the mail-reader's Vote feature.*")

(pushnew :VOTE-TO mail:*address-header-types*)	;1 Tell the header parser that this contains addresses.*
(pushnew :VOTED *mail-attribute-list*)		;1 Tell various things that *VOTED1 is a valid attribute.*


(defun 4parse-message-vote *(message)
  "2Given a message (like *msg*) parse out the headers relevant to automatic voting.
  If there are vote headers, return their (munged) values.*"
  (declare (values vote-id-string vote-subject-string vote-reply-to-address vote-choices))
  (assure-message-parsed message t)
  (let* ((request-id nil)
	 (request (get-message-header message :vote-request :interval))
	 (vote-to (get-message-header message :vote-to :header))
	 (choices (get-message-header message :vote-choices :interval)))
    (unless (or request vote-to choices)
      (return-from PARSE-MESSAGE-VOTE nil))
    (unless request (barf "3No Vote-Request field!*"))
    (unless vote-to (barf "3No Vote-To field!*"))
    (unless choices (barf "3No Vote-Choices field!*"))
    (setq request (string-interval (zwei:search (interval-first-bp request) "3:*" nil t 1) (interval-last-bp request))
	  choices (string-interval (zwei:search (interval-first-bp choices) "3:*" nil t 1) (interval-last-bp choices))
	  vote-to (send vote-to :address-list))
    (flet ((white-p (char)
	     (declare (character char))
	     (or (char-equal char #\Space) (char-equal char #\Tab) (char-equal char #\Newline))))
      (let* ((comma (position #\, request :test #'char-equal)))
	(when comma
	  (psetq request (subseq request (position-if-not #'white-p request :start (1+ comma)))
		 request-id (subseq request 0 comma))))
      (let* ((result '()))
	(do* ((last-pos 0 pos)
	      (pos 0 (and last-pos (position #\, choices :test #'char-equal :start (1+ last-pos)))))
	     ((null last-pos))
	  (unless (eql pos last-pos)
	    (let* ((start (if (zerop last-pos) last-pos (1+ last-pos))))
	      (push (subseq choices (position-if-not #'white-p choices :start start) (or pos (length choices)))
		    result))))
	(values request-id request vote-to (nreverse result))))))


(defun 4prompt-for-message-vote *(message &optional (force-p t))
  "2Given a message (like *msg*) parse out the headers relevant to automatic voting.
  If there are vote headers, prompt the user for their vote, and return it.
  The reply may be NIL, meaning either there was no vote, or the user chose not to respond.*"
  (declare (values vote-response vote-request-id vote-to))
  (multiple-value-bind (request-id request vote-to choices)
		       (parse-message-vote message)
    (when request
      (let* ((wild-p (member "3**" choices :test #'string-equal))
	     (label-string (format nil "3~:[~;~:*~A~%~]Vote:~%~A*" request-id request)))
	(when wild-p (setq choices (delete "3**" choices :test #'string-equal)))
	(flet ((vote ()
		(w:menu-choose
		  (append '(("" :no-select t))
			  (mapcar #'(lambda (choice) `(,choice :value ,choice :font fonts:hl12))
				  choices)
			  (when (or wild-p force-p)
			    '(("" :no-select t)))
			  (when wild-p
			    '(("3Write-In Vote*" :value :WRITEIN :font fonts:hl12b)))
			  (when force-p
			    '(("3No Vote*" :value :NOVOTE :font fonts:hl12b)))
			  )
		 :columns 1 :item-alignment :center
		 :label `(:string ,label-string :font fonts:hl12b :centered))))
	  (add-message-attribute :voted message)
	  (let* ((answer (vote)))
	    (when force-p
	      (do* () (answer) (setq answer (vote))))
	    (setq answer
	     (case answer
	       ((:NOVOTE NIL) nil)
	       (:WRITEIN
		(condition-call (c)
		    (progn
		     ;1 I would use this, but it's 5sooo* ugly...*
		     ;1 *(w:pop-up-prompt-and-read :string '(:mouse) :default "3Write-In Vote:*")
		     
		     (w:using-resource (w w:prompting-window)
		       (let* ((*query-io* w))
			 (send w :set-current-font FONTS:HL12B t)
			 (send w :set-size-in-characters 50 4)
			 (send w :expose-near '(:mouse))
			 (w:window-call (w :deactivate)
			   (send w :clear-screen)
			   (send w :set-current-font FONTS:HL12B t)
			   (format w "3Write-In Vote:~2%*")
			   (send w :set-current-font FONTS:CPTFONT t)
			   (prompt-and-read :string "")))))
		  ((condition-typep c 'SYS:ABORT)	;1 If they hit abort, pop up the menu again.*
		   (prompt-for-message-vote message force-p))))
	       
	       (t answer)))
	    (values 
	      (when answer
		(if (tv:mouse-confirm (format nil "3Your vote is ``~A''*" answer)
				      #.(format nil "3Click mouse or press ~C if this is correct;~%~*
						3   Move off window, or press N to re-enter your vote.*"
						#\End))
		    answer
		    (prompt-for-message-vote message force-p)))
	      (or request-id request)
	      vote-to)))))))

(defun 4vote-on-message *(message &optional force-p)
  "2Given a message (like *msg*) parse out the headers relevant to automatic voting.
  If there are vote headers, prompt the user for their vote, and, if they elect to vote,
  put them in a mail buffer filled in with their response.*"
  (multiple-value-bind (answer vote-id vote-to) (prompt-for-message-vote message force-p)
    (when answer
      (let* ((to (format nil "3~{~A~^, ~}*" (mapcar #'(lambda (x) (send x :string-for-message)) vote-to)))
	     (reply1 (format nil "3~A (my vote on '~A')*" answer vote-id))
	     (reply2 (format nil "3My vote on '~A' is '~A'.*" vote-id answer)))
	(zwei:mail-thing to (string-append reply2 #\Newline) reply1))))
  DIS-NONE)


(defun 4maybe-vote-on-message *(msg window)
  "2Pop up a menu letting the user cast a vote, if this message has a vote,
 and if the user has not already voted on it.  This function redisplays the
 window displaying the message before popping up the menu so that the
 user has a chance to read the message before voting.*"
  (when (and msg
	     *voting-enabled*
	     ;1 *(message-attribute-p :unseen msg)
	     (not (message-attribute-p :voted msg)))
    (when window
      (must-redisplay window DIS-ALL)
      (redisplay window))
    (dolist (w *window-list*) (redisplay w))
;    (delete-message-attribute :unseen msg)
    (vote-on-message msg t)))


(compiler-let ((sys:compile-encapsulations-flag t))
  (sys:advise 4must-redisplay-mail-buffer* :after 4mail-vote* nil
    ;1;  2When we select a message that has a vote in it, and the user has not **
    ;1;  2already seen this message, let them vote.**
    (let* ((buf (car sys:arglist))
	   (msg (and buf (current-message buf nil)))
	   (win (or (fourth sys:arglist) (and buf (mail-buffer-window buf)))))
      (maybe-vote-on-message msg win)))
  )


(defcom 4com-vote-on-message *"2Reply to the vote contained in the current message.
3 * Use this only if you didn't respond to the vote when you first read it, or 
3 *if **VOTING-ENABLED*2 is NIL.*"
	()
  (in-mail-context (:require-message t :require-buffer t)
    (delete-message-attribute :unseen *msg*)
    (add-message-attribute :voted *msg*)
    (vote-on-message *msg* t))
  DIS-NONE)

(set-comtab *read-mail-comtab* () '(("2Vote on Message*" . com-vote-on-message)))


;1;; Sending votes.*

(defcom 4com-insert-vote *"2Insert a Vote into this send-mail buffer.
  Recipients of this mail will be able to automatically reply to this vote 
  if their mail software is as sophisticated as yours is.*"
	()
  (let* ((subject (prompt-and-read :string "3Vote Topic: *"))
	 (vote-id (format nil "3~A.~D*" si:local-host (time:time)))
	 (choices '())
	 (i 0))
    (loop
      (let* ((string (prompt-and-read :string "3Vote Choice #~D: *" (incf i))))
	(when (zerop (length string)) (return))
	(cond ((string-equal string "3**")
	       (beep)
	       (typein-line "3\"*\" is an invalid choice.*")
	       (decf i))
	      (t
	       (nsubstitute #\- #\, string :test #'char-equal)
	       (push string choices)))))
    (when (y-or-n-p "3Do you want to allow write-in votes? *")
      (push "3**" choices))
    (unless choices (barf "3This vote has no choices!*"))
    (setq choices (nreverse choices))
    (let* ((bp (zwei:search (interval-first-bp *interval*) #.(string-append #\Newline #\Newline) nil nil)))
      (unless bp (barf "3Couldn't find end of headers!*"))
      (move-bp bp (beg-line bp -1 t))
      (insert-header-field bp :VOTE-REQUEST (format nil "3~A, ~A*" vote-id subject))
      (insert-header-field bp :VOTE-TO      (send (mail:default-from-address) :string-for-message))
      (insert-header-field bp :VOTE-CHOICES (format nil "3~{~A~^, ~}*" choices))
      (flush-bp bp))
    DIS-TEXT))

;1;; This magic is to make 5Insert Vote* be available only within Send-Mail mode without*
;1;; re-evaluating the whole 5defminor* of of 5com-mail-mode*.  Lousy interface...*
;1;;*
(pushnew '(set-comtab *mode-comtab* () '(("2Insert Vote*" . com-insert-vote)))
	 (get 'mail-mode 'mode)
	 :test #'equalp)
