;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:ZWEI; VSP:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B) -*-

;1;; File "3MAIL-CLASSES*"*
;1;; Some read-mail commands that relieve you of the burden of remembering mail file names.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;     1 Sep 89*	1Jamie Zawinski*	1 Created.*
;1;;*

;1;;*
;1;;   5Meta-X Read Mail Class**	1Just like 5Read Mail*, but prompts you (with completion) for the name of a mail class instead of *
;1;;*				1the name of a file.*
;1;;*
;1;;   5Meta-X Classify Message**	1Bound to 5Meta-C* in read-mail mode.  Just like the 5Copy Message* command (which is bound to 5C*) *
;1;;*				1except that it prompts you (with completion) for the name of a mail class instead of the name of a *
;1;;*				1mail file.  If you type the name of a mail class which does not exist, you have the option of creating it.*
;1;;*
;1;;   5*mail-classes-alist***		1An association list which defines the names of the mail classes.  Mail class names are strings, *
;1;;*				1which are compared with 5string-equal* (case does not matter).  You should set this in your init file.*
;1;;*				1The mail class 5Default* always means the file in 5*user-default-mail-file**, so it should not appear *
;1;;*				1on this list.*
;1;;*
;1;;  5*delete-reclassified-messages***	1If T, then copying a message to a different mail class will automatically mark it for deletion from *
;1;;*				1the old class.*
;1;;*


(defvar 4*mail-classes-alist* *'() "2An association list, associating mail class names with mail files.*")

(defun 4get-mail-class-file *(class-name)
  (if (string-equal class-name "3Default*")
      (default-mail-file)
      (cdr (assoc class-name *mail-classes-alist* :test #'equalp))))


(defun 4change-mail-class-file *(class-name new-file)
  "2Change what file is associated with a mail class.  If the mail class doesn't exist, it is created.
  If NEW-FILE is NIL, then the mail class is deleted.*"
  (let* ((acons (assoc class-name *mail-classes-alist* :test #'string-equal)))
    (cond ((null new-file)
	   (setq *mail-classes-alist* (delete acons *mail-classes-alist* :test #'eq)))
	  (acons
	   (setf (car acons) (string (car acons)))
	   (setf (cdr acons) new-file))
	  (t
	   (push (cons (string class-name) new-file) *mail-classes-alist*)))
    (when (string-equal class-name "3Default*")
      (setq *user-default-mail-file* new-file)))
  new-file)


(defvar 4*last-mail-class* *"3Default*" "2The name of the last mail class read.*")

(defun 4prompt-for-mail-class *(&optional (prompt "3Mail Class:*"))
  (let* ((default 4*last-mail-class**)
	 (result (completing-read-from-mini-buffer (if default
						       (format nil "2~A (default ~A)*" prompt default)
						       prompt)
						   (cons (cons "3Default*" (default-mail-file))
							 *mail-classes-alist*)
						   'MAYBE nil "2Type the name of a mail class.*")))
    (cond ((consp result)
	   (setq 4*last-mail-class** (car result))
	   (cdr result))
	  ((equal result "")
	   (get-mail-class-file default))
	  (t
	   (let* ((path (or (cdar *mail-classes-alist*) (default-other-mail-file)))
		  (default-file (and path (make-pathname :name result :type (pathname-type path) :defaults path)))
		  (file (read-defaulted-pathname (if default
						     (format nil "2Mail file to associate with mail class ~A (default ~A)*"
							     result default-file)
						     (format nil "2Mail file to associate with mail class ~A*" result))
						 default (if default (pathname-type default-file) :babyl) nil
						 :write nil nil)))
	     (setq 4*last-mail-class** result)
	     (change-mail-class-file result file)
	     (format *query-io* "2~&Mail class ~A created.*" result)
	     file)))))


(defcom 4com-read-mail-class *"2Read a named mail class.*" ()
  (get-mail-file (prompt-for-mail-class "3Read mail class: *"))
  DIS-NONE)


;1;;*
;1;; Contents of this function almost completely lifted from 5copy-message-to-mail-file*.*
;1;;*
(defun 4copy-message-to-mail-class *(message-list prompt)
  (let* ((*mini-buffer-dont-record* t)
	 (pathname (prompt-for-mail-class (format nil prompt (length message-list))))
	 (abort-flag :abort)
	 (mail-buffer (if (filter-summary-p *interval*)
			  (mail-file-buffer-of *interval*)
			  *mail-buffer*))
	 whereto format)
    (setq *default-other-mail-file* pathname)
    (unwind-protect
	(progn
	  (multiple-value-setq (whereto format)
	    (copy-message-init pathname))
	  (format *query-io* "3~&Copied *")
	  (loop for msg in message-list
		as firstp = t then nil
		doing
		(if (streamp whereto)
		    (write-message msg format whereto)
		  (let ((copy (copy-message-object msg)))
		    (delete-message-attribute :apply copy)	;1 In case of apply-command usage.*
		    (add-message-to-buffer copy whereto)))
		(add-message-attribute :filed msg)
		(when *delete-message-after-copy*
		  (add-message-attribute :deleted msg))
		(format *query-io* "3~:[, ~;~]~D*"
			firstp
			(1+ (message-index msg mail-buffer))))
	  (setq abort-flag nil)
	  (when (mail-reader-buffer-p whereto)
	    (when (mail-summary-of whereto)
	      (update-summary (mail-summary-of whereto)))
	    (format *query-io* "3~&Be sure to save the other buffer to reflect changes.*")))
      (when (streamp whereto)
	(close whereto :abort abort-flag)))
    (format *query-io* "3 to ~A ~A~@[ in ~A format~].*"
	    (if (streamp whereto) "3file*" "3buffer*")
	    (if (streamp whereto) pathname whereto)
	    format)))


(defvar 4*delete-reclassified-messages* *t
  "2If T, then copying a message to a different mail class will automatically mark it for deletion from the old class.*")


(defcom 4com-classify-message *"2Copy this message into another mail class, and maybe delete it.*" ()
  (in-mail-context (:require-message t :require-buffer t)
    (copy-message-to-mail-class
      (loop repeat (abs *numeric-arg*)
	    as *msg* = *msg* then (select-next-message-search :undeleted *mail-buffer* (signum *numeric-arg*))
	    until (null *msg*)
	    collecting *msg*)
      "2Copy message~P to mail class:*")
    (when 4*delete-reclassified-messages** (com-delete-message)))
  DIS-TEXT)

(define-mail-apply-command 4classify*-message
			   (:name "3Classify Message*"
			    :function copy-message-to-mail-class
			    :message-arg :message-list
			    :args ("2Copy message~P to mail class:*"))
			   "2Copy all marked messages to a mail class.*")


(set-comtab *read-mail-comtab*
	    '(#\Meta-C com-classify-message)
	    ())

(set-comtab *zmacs-comtab* () '(("3Read Mail Class*" . com-read-mail-class)))
