;; -*- Mode:Common-Lisp; Package:FS; Base:10 -*-

;;;
;;; Copyright (c) 1987  David R. Forster, all rights reserved.
;;;


;; ********************************************************
;; ****       WARNING!!! Heavy-duty tricky magic       ****
;; ********************************************************
;; ****        Common-LISP functions and macros        ****
;; ****                being redefined!!               ****
;; ********************************************************
;; ****                   Load, Open,                  ****
;; ****  Defun, Defmacro, Defstruct, Defsetf, Deftype  ****
;; ****       Defconstant, Defvar, Defparameter        ****
;; ****    Define-Setf-Method, Define-Modify-Macro     ****
;; ********************************************************
;;
;; This code implements logical pathnames for the VAX, and arranges for the
;; OPEN function to use these, as well as trying to parse pathnames for other
;; hosts.  In addition, the LOAD function assigns the given namestring to the
;; dynamic global variable FDEFINE-FILE-PATHNAME, for use by Defun, etc. in
;; recording the file in which functions, etc. were defined.
;; RECORD-SOURCE-FILE-NAME can be used in user-defined macros to update this
;; information, and REPORT-SOURCE-FILE-NAME can be used to access it.
;; In order to make this functionality available, it was necessary to `wrap'
;; code around Defun, etc. (see list above) and to redefine these functions
;; in terms of the new macros.  This file includes code to prevent the
;; recursive redefinition of the macros (i.e. reloading this file should not
;; cause problems, the redefinitions of Defun, etc. will remain).  To restore
;; Defun, etc. to the original definitions, simply execute the function
;; FS::RESTORE-COMMON-LISP-DEFINITIONS

;; Logical Pathnames:
;; A logical pathname has the syntax:
;;	logical-host:dir1.dir2...;filename.filetype#version
;; It allows you to access different directories independent of the specific
;; host on which they are located (note that they must all be located on the
;; same host, however), and somewhat independent of the actual directories
;; involved.  This can best be illustrated by an example.  Imagine the logical
;; host Foo with the following definition:
;;   (fs:add-logical-pathname-host
;;     "Foo"  "vax1"
;;     '(("Bar"       "[brillig]")
;;       ("Baz"       "[slithy]")
;;       ("Bar.Baz"   "[slithy.brillig]")
;;       (nil         "[other-stuff]")))
;; and the physical host VAX1, with the following directory structure:
;;   vax1:sys$disk:[brillig]
;;   vax1:sys$disk:[brillig.slithy]
;;   vax1:sys$disk:[brillig.baz]
;;   vax1:sys$disk:[slithy]
;;   vax1:sys$disk:[slithy.brillig]
;;   vax1:sys$disk:[other-stuff]
;; then the following translations would be correct:
;;   Foo:Bar;x.y                vax1:sys$disk:[brillig]x.y
;;   Foo:Bar.slithy;x.y         vax1:sys$disk:[brillig.slithy]x.y
;;   Foo:Baz.brillig;x.y        vax1:sys$disk:[slithy.brillig]x.y      **
;;   Foo:Bar.Baz;x.y            vax1:sys$disk:[slithy.brillig]x.y      **
;;   Foo:Baz.Bar;x.y            vax1:sys$disk:[other-stuff.baz.bar]x.y **
;;   Foo:Brillig.Baz;x.y        vax1:sys$disk:[brillig.baz]x.y         **
;; Note especially the translations marked `**': subdirectories of Brillig may
;; be accessed as subdirectories of the logical directory `Bar', excepting the
;; directory `Baz', which has been specifically given a different translation.
;; Top-level physical directories not specifically mentioned in the translation
;; are accessed as subdirectories of the directory given in the NIL translation.
;; In general, this is a pretty straight-forward re-implementation of the
;; logical pathnames found on Lisp Machines.

(in-package "FS")

(export '(add-physical-host		; add type translation for physical host
	  add-logical-pathname-host	; add pathname translation for logical
					; host
	  parse-pathname		; get pathname for logical or physical
					; namestring
	  fdefine-file-pathname		; name of file being loaded via FS::LOAD
	  inhibit-fdefine-warnings	; whether or not RECORD-SOURCE-FILE-NAME
					; should complain about redefinitions
	  record-source-file-name	; record name of file being loaded on
					; :source-file-name property of symbol
	  report-source-file-name	; report names of files associated with
					; :source-file-name property of symbol
	  ))

;; *******************************
;; **** change for your site: ****
;; *******************************
(defconstant local-host-table
  (cond (#+vms t nil "chaos$root:[host.tables]nethosts.txt")
	(t "nethosts.txt")))

;;; These characters are defined by referencing strings because #\; and friends
;;; mess up Zmacs. 

(defconstant logical-directory-separator (schar "." 0))
(defconstant logical-host-delimiter      (schar ":" 0))
(defconstant logical-directory-delimiter (schar ";" 0))

(defconstant lisp-type (lisp-implementation-type))

(defconstant explode-directory-default
  (cond ((equal lisp-type "VAX LISP") nil)
	((equal lisp-type "CMU Common Lisp") 'vector)
	(#+lispm t
	 #+kcl   t
	 #+hp    t
		 nil 'list)))

(defconstant explode-directory-type (or explode-directory-default 'list))


(defun directory-separator-string (host-type)
  "Returns string for denoting a directory separator in pathname on host of
   type HOST-TYPE"
  (case host-type
    (:symbolics ">")
    (otherwise "."))) ;; includes: :vms, :logical, :explorer


(defvar logical-host-table (make-hash-table)
  "Alist of Logical hosts and translation tables for those hosts.")

(defvar physical-host-table (make-hash-table)
  "Table of Physical hosts and system types for those hosts.
   Valid (implemented) types include - :VMS, :EXPLORER, :SYMBOLICS")

(defconstant fs-package (find-package 'fs))

(defmacro mksym (symbol-or-string)
   `(typecase ,symbol-or-string
      (string (intern (string-upcase ,symbol-or-string) fs-package))
      (otherwise ,symbol-or-string)))

(defun add-physical-host (physical-host system-type)
  "Add PHYSICAL-HOST to a table of SYSTEM-TYPEs"
  (declare (special physical-host-table))
  (setf (gethash (mksym physical-host) physical-host-table)
	system-type))


(defun add-logical-pathname-host (logical-host physical-host translations)
  "Add LOGICAL-HOST to table of host translations.  Real host is PHYSICAL-HOST;
   Real pathnames are composed from TRANSLATIONS, which is an list of logical
   pathname string and physical pathname string pairs."
  (declare (special logical-host-table))
  (let* ((phys-host (mksym physical-host))
	 (host-type (physical-host-type phys-host)))
    (if host-type
	(setf (gethash (mksym logical-host) logical-host-table)
	      (cons phys-host
		    (mapcar #'(lambda (xlation)
				(setf (cadr xlation)
				      (multiple-value-list
					(parse-physical-pathname (cadr xlation) host-type)))
				xlation)
			    translations)))
	(format t "~&No physical host defined for ~s~%" physical-host))))

(defun parse-physical-pathname (pathname host-type &key (explode-directories explode-directory-default))
  (case host-type
    (:vms       (parse-vms-pathname
		  pathname
		  :explode-directories explode-directories))
    (:unix      (parse-unix-pathname
		  pathname
		  :explode-directories explode-directories))
    (:symbolics (parse-symbolics-pathname
		  pathname
		  :explode-directories explode-directories))
    (:explorer  (parse-explorer-pathname
		  pathname
		  :explode-directories explode-directories))
    (nil	nil)
    (otherwise  (format t "~&Parse-~a-pathname not yet implemented.~%" host-type)
	       nil)))


(defun logical-host-translation (logical-host)
  "Find translations for LOGICAL-HOST"
  (declare (special logical-host-table))
  (gethash (mksym logical-host) logical-host-table))

(defun physical-host-type (physical-host)
  "Find type of PHYSICAL-HOST"
  (declare (special physical-host-table))
  (gethash (mksym physical-host) physical-host-table))

(defmacro parse-with-string-delimiter (delim string start)
  `(let ((p%pos (search ,delim ,string :start2 ,start)))
     (if (null p%pos)
	 nil
	 (prog1
	   (subseq ,string ,start p%pos)
	   (setq ,start (+ p%pos (length ,delim)))))
     ))


(defun parse-with-string-delimiter* (pattern source)
  (let ((elt-list nil)
	(p 0))
    (do ((elt (parse-with-string-delimiter pattern source p)
	      (parse-with-string-delimiter pattern source p)))
      ((null elt))
      (push elt elt-list))
    (push (subseq source p) elt-list)
    (nreverse elt-list)))

(defun gen-with-string-delimiter* (delim elt-list)
  (do ((source (car elt-list) (concatenate 'string source delim (car elt-list)))
       (elt-list (cdr elt-list) (cdr elt-list)))
      ((null elt-list)
       source)
      ))



;; All PARSE-machine-PATHNAME calls return six values:
;;  1. host
;;  2. device
;;  3. directory list
;;  4. name
;;  5. type
;;  6. version


(defun parse-vms-pathname (pathname &key (explode-directories explode-directory-default))
  ;; pattern:  host::device:[dir1.dir2...]name.type;version
  ;; or:       host::device:<dir1.dir2...>name.type.version
  (let ((host nil)
	(device nil)
	(directories nil)
	(name nil)
	(type nil)
	(version nil)
	(pos 0))
    (setq host (parse-with-string-delimiter "::" pathname pos))
    (setq device (parse-with-string-delimiter ":" pathname pos))
    (when (< pos (length pathname))
      (case (char pathname pos)
	(#\[	(incf pos)
		(setq directories (parse-with-string-delimiter
				    "]" pathname pos)))
	(#\<	(incf pos)
		(setq directories (parse-with-string-delimiter
				    ">" pathname pos))))
      (when (and explode-directories directories)
	(setq directories
	      (coerce (parse-with-string-delimiter* "." directories)
		      explode-directories))))
    (setq name (parse-with-string-delimiter "." pathname pos))
    (setq type (parse-with-string-delimiter ";" pathname pos))
    (if (null type)
	(setq type (parse-with-string-delimiter "." pathname pos)))
    (if type
	(setq version (subseq pathname pos))
	(if name
	    (setq type (subseq pathname pos))
	    (setq name (subseq pathname pos))))
    (values host device directories name type version)))

(defun parse-explorer-pathname (pathname &key (explode-directories explode-directory-default))
  (parse-logical-pathname pathname :explode-directories explode-directories))

(defun parse-symbolics-pathname (pathname &key (explode-directories explode-directory-default))
  ;; pattern:  host:>dir1>dir2>**>name.type.version
  ;; or:       host|fep0:>dir1>dir2>**>name.type.version  -- not implemented
  (let ((host nil)
	(device nil)
	(directories nil)
	(name nil)
	(type nil)
	(version nil)
	(pos 0)
	dir-pos)
    (setq host (parse-with-string-delimiter ":>" pathname pos))
    (setq dir-pos (position #\> pathname :from-end t :start pos))
    (when dir-pos
      (setq directories (subseq pathname pos dir-pos))
      (setq pos (1+ dir-pos))
      (when (and explode-directories directories)
	(setq directories
	      (coerce (parse-with-string-delimiter* ">" directories)
		      explode-directories))))
    (setq name (parse-with-string-delimiter "." pathname pos))
    (setq type (parse-with-string-delimiter "." pathname pos))
    (if type
	(setq version (subseq pathname pos))
	(if name
	    (setq type (subseq pathname pos))
	    (setq name (subseq pathname pos))))
    (values host device directories name type version)))


(defun parse-unix-pathname (pathname &key (explode-directories explode-directory-default))
  ;; pattern:  /dir1/dir2/name.type   (:absolute device)
  ;; or:       dir1/dir2/name.type    ("Default" device)
  ;; our extension to it (for now):   host:...
  (let ((host nil)
	(device "Default")
	(directories nil)
	(name nil)
	(type nil)
	(version nil)
	(pos 0)
	dir-pos)
    (setq host (parse-with-string-delimiter ":" pathname pos))
    (setq dir-pos (position #\/ pathname :from-end t :start pos))
    (when dir-pos
      (setq directories (subseq pathname pos dir-pos))
      (setq pos (1+ dir-pos))
      (when (and explode-directories directories)
	(setq directories (parse-with-string-delimiter* "/" directories))
	(when (equal (car directories) "")
	  (pop directories)
	  #+(and sun kcl unix) (push :root directories)
	  #+:cmu (setq device :absolute))
	(setq directories (coerce directories explode-directories))
	))
    (setq name (parse-with-string-delimiter "." pathname pos))
    (if name
	(setq type (subseq pathname pos))
	(setq name (subseq pathname pos)))
    (values host device directories name type version)))



(defun parse-logical-pathname (pathname &key (explode-directories explode-directory-default))
  ;; pattern:  host:dir1.dir2...;name.type#version
  (let ((host nil)
	(device nil) ;; not handled
	(directories nil)
	(name nil)
	(type nil)
	(version nil)
	(pos 0))
    (setq host (parse-with-string-delimiter ":" pathname pos))
    (setq directories (parse-with-string-delimiter ";" pathname pos))
    (when (and explode-directories directories)
      (setq directories
	    (coerce (parse-with-string-delimiter* "." directories)
		    explode-directories)))
    (setq name (parse-with-string-delimiter "." pathname pos))
    (setq type (parse-with-string-delimiter "#" pathname pos))
    (if type
	(setq version (subseq pathname pos))
	(if name
	    (setq type (subseq pathname pos))
	    (setq name (subseq pathname pos))))
    (values host device directories name type version)))




(defun parse-pathname (pathname)
  "Try to parse PATHNAME first as a logical pathname, then as physical."
  (let ((logical (translate-logical-pathname pathname)))
    (if logical
	logical
	(let ((host-type (physical-host-type
			   (subseq pathname
				   0
				   (position logical-host-delimiter pathname)))))
	  (if host-type
	      (multiple-value-bind
		(host device directories name type version)
		  (parse-physical-pathname pathname host-type)
		(make-pathname
		  :host (if host (string host) host)
		  :device device
		  :directory directories
		  :name name
		  :type type
		  :version version))
	      (parse-namestring pathname))	;; unknown host -- wing it
	  ))))


(defun translate-logical-pathname (pathname &aux ltbl)
  "Translate PATHNAME from a logical pathname to a physical pathname, according
   to the specifications made via ADD-LOGICAL-PATHNAME and stored in HOST-TABLE."
  (etypecase pathname
    (string nil)
    (pathname (return-from translate-logical-pathname pathname)))
  (multiple-value-bind
    (host device directories name type version)
      (parse-logical-pathname pathname :explode-directories nil)
      (declare (ignore device))
    (if (or (null host)
	    (null (setq ltbl (logical-host-translation host))))
	nil	;; couldn't find both - not translatable
	(let* ((longest-match-length 0)
	       (longest-match nil)
	       (default	      nil)
	       (physical-host (car ltbl))
	       (tbl	      (cdr ltbl))
	       (host-type     (physical-host-type physical-host))
	       l)
	  ;; find the longest pattern in the host table which matches
	  (dolist (pair tbl)
	    (if (null (car pair))
		(setq default pair)
		(block check-for-match
		  (setq l (string-not-equal directories (car pair)))
		  ;; if nil, match is complete, otherwise use index
		  (if (null l)
		      (block exact-match
			(setq longest-match pair)
			(setq longest-match-length nil)
			(return))
		      (when (and (> l longest-match-length)    ;; longest yet?
				 (char-equal (char directories l) ;; full dir?
					     logical-directory-separator))
			(setq longest-match pair)
			(setq longest-match-length (1+ l))))
		  )))
	  ;; compose directory from match found, if any
	  (unless longest-match
	    (setq longest-match default))
	  (multiple-value-bind
	    (phost pdevice pdirectories pname ptype pversion)
	      (values-list (cadr longest-match))
	      (declare (ignore phost pname ptype pversion))
	    (setq directories
		  (if explode-directory-default
		      (if longest-match-length
			  (concatenate explode-directory-default
				       pdirectories
				       (coerce (parse-with-string-delimiter*
						  (directory-separator-string :logical)
						  (subseq directories longest-match-length))
					       explode-directory-default))
			  pdirectories)
		      (if (or (null directories)
			      (null longest-match-length))
			  pdirectories
			  (let* ((src-ch (char (directory-separator-string :logical) 0))
				 (target (directory-separator-string host-type))
				 (tgt-ch (char target 0))
				 (newdir (substitute tgt-ch src-ch
						     (subseq directories longest-match-length))))
			    (if pdirectories
				(concatenate 'string pdirectories target newdir)
				newdir)))))
	    (make-pathname
	      :host (if physical-host (string physical-host) physical-host)
	      :device pdevice
	      :directory directories
	      :name name
	      :type type
	      :version version)
	    )))))


;;; This character is defined by referencing a string because #\( messes up Zmacs

(defconstant left-parenthesis (schar "(" 0))

(defun compose-physical-hostab (&optional (local-namestring local-host-table))
  (when local-namestring
    (with-open-file (hostab local-namestring :direction :input)
      (let ((eof-token (gensym))
	    line)
	(do ((host? (read hostab nil eof-token) (read hostab nil eof-token)))
	    ((eq host? eof-token))
	  ;; we have a token -- it should be either NET or HOST (otherwise error)
	  (setq line (read-line hostab nil))
	  ;;(format t "~&DBG: ~a ~s~%" host? line)
	  (cond ((null line)  ;; unexpected EOF - graceful recovery - return
		 (return))
		((equal (symbol-name host?) "HOST")
		 ;; e.g. HOST HILLARY, CHAOS 570,SERVER,LISP,EXPLORER,[EDMUND]
		 ;; -> ((HILLARY)(CHAOS 570)(SERVER)(LISP)(EXPLORER)((EDMUND)))
		 (setq line (delete #\space line))
		 (setq line (delete #\tab line))
		 (let* ((pos 0)
			(name (parse-with-string-delimiter "," line pos))
			(nicknames nil)
			(system  nil)
			(machine nil))
		   ;; throw away host #'s
		   (if (char-equal left-parenthesis (char line pos))
		       (parse-with-string-delimiter ")," line pos)
		       (parse-with-string-delimiter "," line pos))
		   (parse-with-string-delimiter "," line pos) ;; ... and status
		   ;; record system and machine types
		   (setq system  (parse-with-string-delimiter "," line pos))
		   (setq machine (parse-with-string-delimiter "," line pos))
		   (if (null machine)		; any nicknames?
		       (setq machine (subseq line pos))	; no, read rest of line
		       (when (> (length line) pos)
			 (incf pos)
			 (setq nicknames (parse-with-string-delimiter*
					   ","
					   (parse-with-string-delimiter "]" line pos)))))
		   (unless (equal "" system)
		     (if (equal "LISP" system)
			 (setq system machine))
		     (setq system (intern system 'keyword))
		     (case system
		       (:mach (setq system :unix))
		       (:lisp (setq system :symbolics))
		       (:appaloosa (setq system :explorer)))
		     (add-physical-host name system)
		     (dolist (name nicknames)
		       (add-physical-host name system)))
		   ))
		)))
      )))

(defvar canonical-types nil
  "Alist of canonical types and translations to try, depending on the system")

(defmacro define-canonical-type (canonical-type default &rest translations)
  `(pushnew ',(list canonical-type default (copy-list translations))
	    canonical-types
	    :key #'car))


(defmacro translate-canonical-type (canonical-type host)
  `(let ((host	      (if (keywordp ,host) ,host (physical-host-type ,host)))
	 (translation (cdr (assoc ,canonical-type canonical-types)))
	 types)
     (setf types (assoc host (cadr translation)))
     (if types
	 (cdr types)
	 (list (car translation)))))

;; *************************************************
;; ****   WARNING!!! Heavy-duty tricky magic    ****
;; *************************************************
;; **** Common-LISP functions being redefined!! ****
;; *************************************************

(defmacro redefine-macro (name internal-name alternate-name &body body)
  `(progn
     (if (not (macro-function ',internal-name))
	 (setf (macro-function ',internal-name)
	       (macro-function ',name)))
     (defmacro ,alternate-name . ,body)
     (setf (macro-function ',name)
	   (macro-function ',alternate-name))
     ',name))

(defmacro redefine-function (name internal-name alternate-name &body body)
  `(progn
     (if (not (fboundp ',internal-name))
	 (setf (symbol-function ',internal-name)
	       (symbol-function ',name)))
     (defun ,alternate-name . ,body)
     (setf (symbol-function ',name)
	   (symbol-function ',alternate-name))
     ',name))


(defvar fdefine-file-pathname nil
  "Namestring of file currently being loaded, or NIL, if at top level")

(defvar inhibit-fdefine-warnings nil
  "Whether or not to allow functions, etc. to be redefined without querying the user.")

(redefine-function
  lisp::load lisp-load load-with-pathname-patch
  (pathname &rest args &key if-does-not-exist print &allow-other-keys)
  "Modified LOAD function which makes the name of the file being loaded 
   available in the variable FDEFINE-FILE-PATHNAME.  For a full suite of 
   functions, we would also have to shadow DEFUN, DEFMACRO, DEFVAR, DEFCONSTANT,
   DEFSTRUCT, DEFINE-MODIFY-MACRO, DEFINE-SETF-METHOD, DEFPARAMETER, DEFSETF, 
   DEFTYPE, and maybe others, so that they record the source file name thus 
   made available as a property of the symbol, after the style of lispm's, 
   something like:
        (pushnew `(<defun-etc> ,fdefine-file-pathname)
		 (get :source-file-name '<symbol>)
		 :test #'equal)
   or:   (record-source-file-name '<symbol> '<defun-etc>)"
  ;;
  ;; Let+Declare special defines the pathname:
  (let ((fdefine-file-pathname pathname))
    (declare (special fdefine-file-pathname))
    (let* ((pathname     (parse-pathname pathname))
	   (pathnm-type  (pathname-type pathname))
	   (pathnm-host  (pathname-host pathname)))
      (cond ((null pathnm-type)
	     (setq pathnm-type
		   (append (translate-canonical-type :lisp pathnm-host)
			   (translate-canonical-type :fasl pathnm-host))))
	    ((keywordp pathnm-type)
	     (setq pathnm-type
		   (translate-canonical-type pathnm-type pathnm-host))))
      (when (listp pathnm-type)
	(let ((date-of-newest-file  -1)
	      (type-of-newest-file  nil))
	  (dolist (f-t pathnm-type)
	    (setq pathname (make-pathname :defaults pathname :type f-t))
	    (when (probe-file pathname)
	      (let ((fwd (abs (file-write-date pathname))))
		(when (> fwd date-of-newest-file)
		  (setq type-of-newest-file f-t)
		  (setq date-of-newest-file fwd)))))
	  (if (null type-of-newest-file)
	      (if if-does-not-exist
		  (error "File ~s was merged with types ~s - none exist."
			 pathname pathnm-type)
		  (return-from load-with-pathname-patch nil))
	      (setf pathname (make-pathname :defaults pathname :type  type-of-newest-file)))))
      (apply #'lisp-load pathname args))))


(defun ynp-p (&optional proceed-string format-string &rest args)
  (do ((answer #\x)
       (first-time t nil))
      ((find answer "ynpYNP")
       (or (char-equal #\y answer)
	   (and (char-equal #\p answer) :proceed)))
    (fresh-line *query-io*)
    (format *query-io* "~? (Y, N, or P)" format-string args)
    (if first-time
	(peek-char t *query-io*))
    (setq answer (read-char *query-io*))
    (if (char-equal #\? answer)
	(format *query-io* "~&Type: Y: Yes, N: No, P: Proceed (~a)" proceed-string))))


(defun string-compare (s1 s2)
  (let ((cmp (string-not-equal s1 s2)))
    (cond ((null cmp)                               :equal)
	  ((>= cmp (length s1))                     :less)
	  ((>= cmp (length s2))                     :greater)
	  ((char-lessp (char s1 cmp) (char s2 cmp)) :less)
	  (t                                        :greater))))

(defun matching-pair-in-list? (item-1 item-2 list)
  (case (string-compare item-1 item-2)
    (:equal   t)
    (:less    (member (list item-1 item-2) list :test #'equal))
    (:greater (member (list item-2 item-1) list :test #'equal))))

(defmacro make-matching-pair-in-list (item-1 item-2 list)
  `(case (string-compare ,item-1 ,item-2)
     (:equal   t)
     (:less    (push (list ,item-1 ,item-2) ,list))
     (:greater (push (list ,item-2 ,item-1) ,list))))

(defvar ignore-source-file-list nil
  "List of pairs of files which should be considered equivalent for
   (dis-)allowing redefinitions")

(defun record-source-file-name (function-name
				&optional (type 'defun)
				(no-query-p inhibit-fdefine-warnings))
  "Record the place in which FUNCTION-NAME of TYPE (e.g. defun, defvar, ...) was
   defined, as given in FS:FDEFINE-FILE-PATHNAME (set automatically in LOAD).
   If NO-QUERY-P is T, do not warn about redefinitions;
   If NO-QUERY-P is NIL, ask the user whether or not it should be redefined.
   NO-QUERY-P defaults to the value of FS:INHIBIT-FDEFINE-WARNINGS"
  (declare (special fdefine-file-pathname
		    inhibit-fdefine-warnings
		    ignore-source-file-list))
  (let* ((previous-definition (assoc type (get function-name :source-file-name)))
	 (location	      (cadr previous-definition))
	 (redefinition?	      t)
	 (file-pathname	      (if (pathnamep fdefine-file-pathname)
				  (namestring fdefine-file-pathname)
				  fdefine-file-pathname))
	 msg)
    (when (and (not (eq no-query-p t))
	       previous-definition
	       (null (matching-pair-in-list?
		       file-pathname location
		       ignore-source-file-list)))
      (setq msg (format nil  "Redefining ~a ~s (previously defined ~a ~s)."
			   type function-name
			   (if location "in" "at")
			   (or location "Top Level")))
      (if (eq no-query-p :warn)
	  (format t "~&~a~%" msg)
	  (setq redefinition? (ynp-p
				"don't ask again for the same pair of files"
				"~a  Continue?"
				msg))))
    (when redefinition?
      (pushnew (list type file-pathname)
	       (get function-name :source-file-name)
	       :test #'equal)
      (when (eq redefinition? :proceed)
	(make-matching-pair-in-list
	  file-pathname location
	  ignore-source-file-list)))
    redefinition?))

(defun report-source-file-name (function-name &optional type)
  (let ((def-list (get function-name :source-file-name)))
    (if (and type (consp def-list))
	(cadr (assoc type def-list))
	def-list)))


(redefine-function lisp::open lisp-open open-with-pathname-patch (pathname &rest args &key &allow-other-keys)
  "Modified OPEN function which allows the use of logical pathnames also"
  (apply #'lisp-open (parse-pathname pathname) args))



;; Warning: in VAX lisp, this hack is only accessible via the loader, i.e.
;; the source file name is not recorded when the code is compiled!
(redefine-macro lisp::defun lisp-defun local-defun (name args &body body)
  `(progn
     (when (fboundp ',name)
       (push (symbol-function ',name) (get ',name :old-function-definition))
       (push (documentation ',name 'function) (get ',name :old-function-documentation)))
     (lisp-defun ,name ,args . ,body)
     (if (record-source-file-name ',name 'defun)
	 (if (get ',name :old-function-definition)
	     (format t "~&Old definition of function ~a ~
			saved under its :Old-Function-Definition property.~%"
		       ',name))
	 (progn
	   (setf (symbol-function ',name) (pop (get ',name :old-function-definition)))
	   (setf (documentation ',name 'function) (pop (get ',name :old-function-documentation)))))
     ',name))



(redefine-macro lisp::defvar lisp-defvar local-defvar
		(name &optional (initial-value :unbound initial-value-p)
				(documentation :unbound documentation-p))
  `(progn
     (when (boundp ',name)
       (push (symbol-value ',name) (get ',name :old-value-definition))
       (push (documentation ',name 'variable) (get ',name :old-variable-documentation)))
     ,(cond (documentation-p  `(lisp-defvar ,name ,initial-value ,documentation))
	    (initial-value-p  `(lisp-defvar ,name ,initial-value))
	    (t		      `(lisp-defvar ,name)))
     (if (record-source-file-name ',name 'defvar)
	 (if (get ',name :old-value-definition)
	     (format t "~&Old definition of variable ~a ~
			saved under its :Old-Value-Definition property.~%"
		       ',name))
	 (progn
	   (setf (symbol-value ',name) (pop (get ',name :old-value-definition)))
	   (setf (documentation ',name 'value) (pop (get ',name :old-value-documentation)))))
     ',name))



;; constants aren't really allowed to be rebound, but at least VAXlisp merely
;; warns about redefinition.
(redefine-macro lisp::defconstant lisp-defconstant local-defconstant
		(name initial-value &optional (documentation :unbound documentation-p))
  `(progn
     (when (boundp ',name)
       (push (symbol-value ',name) (get ',name :old-value-definition))
       (push (documentation ',name 'variable) (get ',name :old-variable-documentation)))
     ,(if documentation-p
	  `(lisp-defconstant ,name ,initial-value ,documentation)
	  `(lisp-defconstant ,name ,initial-value))
     (if (record-source-file-name ',name 'defconstant)
	 (if (get ',name :old-value-definition)
	     (format t "~&Old definition of constant ~a ~
			saved under its :Old-Value-Definition property.~%"
		       ',name))
	 (progn
	   (setf (symbol-value ',name) (pop (get ',name :old-value-definition)))
	   (setf (documentation ',name 'value) (pop (get ',name :old-value-documentation)))))
     ',name))



(redefine-macro lisp::defparameter lisp-defparameter local-defparameter
		(name initial-value &optional (documentation :unbound documentation-p))
  `(progn
     (when (boundp ',name)
       (push (symbol-value ',name) (get ',name :old-value-definition))
       (push (documentation ',name 'variable) (get ',name :old-variable-documentation)))
     ,(if documentation-p
	  `(lisp-defparameter ,name ,initial-value ,documentation)
	  `(lisp-defparameter ,name ,initial-value))
     (if (record-source-file-name ',name 'defparameter)
	 (if (get ',name :old-value-definition)
	     (format t "~&Old definition of parameter ~a ~
			saved under its :Old-Value-Definition property.~%"
		       ',name))
	 (progn
	   (setf (symbol-value ',name) (pop (get ',name :old-value-definition)))
	   (setf (documentation ',name 'value) (pop (get ',name :old-value-documentation)))))
     ',name))


;; still to be added:
;; (redefine-macro defmacro old-defmacro new-defmacro (name &body body))
;; (redefine-macro defstruct old-defstruct new-defstruct (name &body body))
;; (redefine-macro defsetf old-defsetf new-defsetf (name &body body))
;; (redefine-macro deftype old-deftype new-deftype (name &body body))
;; (redefine-macro define-modify-macro old-define-modify-macro new-define-modify-macro (name &body body)
;; (redefine-macro define-modify-macro old-define-modify-macro new-define-modify-macro (name &body body)
;; (redefine-macro define-setf-method old-define-setf-method new-define-setf-method (name &body body))

(defmacro restore-common-lisp-function (common-name private-name)
  `(if (fboundp ',private-name)
       (setf (symbol-function ',common-name) (symbol-function ',private-name))))

(defmacro restore-common-lisp-macro (common-name private-name)
  `(if (macro-function ',private-name)
       (setf (macro-function ',common-name) (macro-function ',private-name))))

(defun restore-common-lisp-definitions ()
   (restore-common-lisp-function lisp::load lisp-load)
   (restore-common-lisp-function lisp::open lisp-open)
   (restore-common-lisp-macro    lisp::defun lisp-defun)
   (restore-common-lisp-macro    lisp::defvar lisp-defvar)
   (restore-common-lisp-macro    lisp::defconstant lisp-defconstant)
   (restore-common-lisp-macro    lisp::defparameter lisp-defparameter))


(eval-when (load eval)
  (add-physical-host nil (or #+vms       :vms
			     #+explorer  :explorer
			     #+symbolics :symbolics
			     #+unix	 :unix
			     #+hp	 :unix
			     (if (equal lisp-type "CMU Common Lisp")
				 :unix)))
  (compose-physical-hostab)
  (define-canonical-type :lisp "LISP"
    (:VMS "LSP" "LISP")
    ;;(:VMS4 "LSP" "LISP")
    ((:VMS :MSDOS) "LSP")
    (:UNIX-UCB "LISP")
    (:UNIX #+(and sun kcl unix) "lsp"
	   "l" "lisp")
    ((:TENEX :TOPS-20) "LISP" "LSP"))
  (define-canonical-type :fasl "XFASL"
    (:unix #+hp "b"
	   #+(and sun kcl unix) "o"
	   "fasl")
    (:vms "FAS")
    (:explorer "XLD")
    (:symbolics "BIN")))
