;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:FILE-SYSTEM; VSP:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B); Patch-File:T -*-

;1;; File "*LOAD-PATCH1"*
;1;; Redefine *LOAD-11 to sometimes query the user when a binary is newer than source, or when binary doesn't exist.*
;1;; If *LOAD1 is called on a pathname or string with no *TYPE1 component, then the file loaded is as specified by*
;1;; the variables* *LOAD-WHEN-SOURCE-NEWER*1 and* *LOAD-WHEN-SOURCE-ONLY*.
;1;; If the pathname's component was specified, then the behavior is as before.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   distant past*	1Jamie Zawinski*	1Created.*
;1;;      5 Jun 89*	1Jamie Zawinski*	1In *LOAD-11 I had changed the condition 5multiple-file-not-found* to be 5file-not-found*, which was, like,*
;1;;*				1 totally wrong.  It caused strange abortive behavior on nested loads, I'm such a bozo.*
;1;;*

(export '(*load-when-source-newer* *load-when-source-only*))


(defparameter *load-when-source-newer* :query
  "What to do when LOAD is called on a pathname without an extension specified, and the source is newer.
  One of :SOURCE, :BINARY, :COMPILE, or QUERY.")

(defparameter *load-when-source-only* :load
  "What to do when LOAD is called on a pathname without an extension specified, and there is no binary.
  One of :LOAD, :COMPILE, or QUERY.")


(defun file-write-date-or-nil (filename-or-stream)
  "Just like FILE-WRITE-DATE, but returns NIL if the file doesn't exist.
  This is faster than calling PROBE-FILE first."
  (if (or (stringp filename-or-stream)
	  (symbolp filename-or-stream)
	  (typep filename-or-stream 'pathname))
      (condition-case (stream)
	  (open filename-or-stream :direction nil)
	(open-deleted-file nil)
	(file-not-found nil)
	(directory-not-found nil)
	(:no-error (prog1
		     (send stream :creation-date)
		     (close stream))))
      (send filename-or-stream :creation-date)))


(defun normalize-extensionless-file (pathname)
  "If FILE has no TYPE component, then default it according to *LOAD-WHEN-SOURCE-NEWER* and *LOAD-WHEN-SOURCE-ONLY*.
  This may involve querying the user.
  Returns a new pathname."
  (let* ((type-specified (pathname-type pathname)))
    (unless type-specified
      (let* ((binary-type (or (send (send pathname :generic-pathname) :get :default-binary-file-type)
			      (local-binary-file-type)))
	     (src (fs:merge-pathname-defaults pathname load-pathname-defaults :lisp))
	     (bin (fs:merge-pathname-defaults pathname load-pathname-defaults binary-type))
	     (src-date (file-write-date-or-nil src))
	     (bin-date (file-write-date-or-nil bin))
	     )
	(cond ;; Source and Binary exist, and Source is newer.
	      ;;
	      ((and src-date bin-date (> src-date bin-date))
	       (check-type *load-when-source-newer* (member :source :binary :compile :query nil))
	       (let* ((action
			(case *load-when-source-newer*
			  ((:source :binary :compile) *load-when-source-newer*)
			  (t (fquery '(:fresh-line t
				       :beep t
				       :choices (((:source "Load source.") #\S)
						 ((:binary "Load binary.") #\B)
						 ((:compile "Compile, then load binary.") #\C)))
				     "Source file ~A is newer than it's corresponding binary;~%~
                                      load source, binary, or compiled binary? " src)))))
		 (case action
		   (:source  (setq pathname src))
		   (:binary  (setq pathname bin))
		   (:compile (setq pathname (compile-file src))))))
	      ;;
	      ;; Source and Binary exist, and Binary is newer.
	      ((and src-date bin-date)
	       (setq pathname bin))
	      ;;
	      ;; Source Only.
	      ((and src-date (not bin-date))
	       (check-type *load-when-source-only* (member :load :compile :query nil))
	       (let* ((action
			(case *load-when-source-only*
			  ((:load :compile) *load-when-source-only*)
			  (t (if (yes-or-no-p "Source file ~S has no corresponding binary;~%compile it first?"
					      src)
				 :compile
				 :load)))))
		 (case action
		   (:source  (setq pathname src))
		   (:compile (setq pathname (compile-file src))))))
	      ;;
	      ;; Binary Only.
	      (bin-date (setq pathname bin))
	      ;;
	      ;; Neither exists - don't change PATHNAME, return the same extensionless one passed in.
	      (t nil)))))
  pathname)


;;;
;;; Originally defined in files "SYS: IO; OPEN.LISP#25" and "SYS: PATCH.IO; IO-3-28.LISP#1"
;;;
(DEFUN 4LOAD-1* (FILE &OPTIONAL PKG NONEXISTENT-OK-FLAG DONT-SET-DEFAULT-P NO-MSG-P)
  ;1; Merge everything, defaulting type component to NIL.*
  (IF (STREAMP FILE)
      (PROGN
	;1; Set the defaults from the pathname we finally opened*
	(OR DONT-SET-DEFAULT-P (SET-DEFAULT-PATHNAME (SEND FILE :PATHNAME) LOAD-PATHNAME-DEFAULTS))
	(CATCH-ERROR-RESTART (ERROR "Give up on loading ~A." (SEND FILE :PATHNAME))
	  ;1; If the file was a character file, read it, else try to fasload it.*
	  (FUNCALL (IF (SEND FILE :CHARACTERS)
		       #'SI::READFILE-INTERNAL
		       #'SI::FASLOAD-INTERNAL)
		   FILE PKG NO-MSG-P)
	  (OR (SEND FILE :SEND-IF-HANDLES :TRUENAME) T)))
      (LET ((PATHNAME (PARSE-PATHNAME FILE)))
	(CATCH-ERROR-RESTART (ERROR "Give up on loading ~A." FILE)
	  (CONDITION-CASE-IF (and NONEXISTENT-OK-FLAG (neq NONEXISTENT-OK-FLAG  :error)) ()   ;107-30-87 dab*
	      ;;
	      ;; start of Jamie's changes
	      ;;
	      (file-retry-new-pathname-if (not nonexistent-ok-flag) (pathname fs:file-error)
	       (setq pathname (normalize-extensionless-file pathname))
	       (with-open-file (stream pathname :direction :input :characters :default)
		 ;;
		 ;; end of Jamie's changes
		 ;;
		;1; Set the defaults from the pathname we finally opened*
		(OR DONT-SET-DEFAULT-P
		    (SET-DEFAULT-PATHNAME (SEND STREAM :PATHNAME) LOAD-PATHNAME-DEFAULTS))
		;1; If the file was a character file, read it, else try to fasload it.*
		(FUNCALL
		  (IF (SEND STREAM :CHARACTERS)
		      #'SI::READFILE-INTERNAL
		      #'SI::FASLOAD-INTERNAL)
		  STREAM PKG NO-MSG-P)
		(SEND STREAM :TRUENAME))
	       )
	    (MULTIPLE-FILE-NOT-FOUND NIL)
	    (file-not-found nil)		; Use this since we now call OPEN instead of OPEN-FILE-SEARCH.
	    (DIRECTORY-NOT-FOUND NIL)
	    )))))
