;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:NAME -*-

;;; File "GENERATE-ETC-HOSTS"
;;; Code to look at a TI namespace and generate an /etc/hosts or /usr/lib/aliases file from it.
;;; Also code to print lists of hosts sorted by addresses, which the NSE doesn't let you do!
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;; 16 Dec 88  Jamie Zawinski    Created.
;;;


(defun generate-/etc/hosts ()
  "Write on *STANDARD-OUTPUT* a line for each IP host in the current namespace.
  These lines are suitable for inclusion in a Unix /etc/hosts file."
  (let* ((hosts-alist (name:list-objects-from-attributes :class :HOST :brief t))
	 (non-aliased-hosts '()))
    (dolist (list hosts-alist)
      (let* ((name (car list)))
	(unless (string-equal name "SYS")
	  (let* ((object (name:lookup-object name :HOST :read-only t)))
	    (unless (name:get-attribute-value object :*alias-of*)
	      (push object non-aliased-hosts))))))

    ;; Sort by ascending IP addresses.
    (setq non-aliased-hosts (sort non-aliased-hosts #'<
				  :key #'(lambda (x)
					   (or (second (assoc :IP (name:get-attribute-value x :addresses)))
					       0))))
    
    (dolist (object non-aliased-hosts)
      (let* ((ip-address (second (assoc :IP (name:get-attribute-value object :addresses)))))
	(when ip-address
	  (princ (ip:ip-address-display ip-address))
	  (princ #\Tab)
	  (princ (nsubstitute #\_ #\Space (string-downcase (name:object-name object))))
	  (let* ((aliases (remove "lm" (name:get-attribute-value object :aliases) :test #'string-equal)))
	    (dolist (alias aliases)
	      (princ #\Space)
	      (princ (nsubstitute #\_ #\Space (string-downcase alias))))
	    (terpri)))))
    (values)))


(defun show-hosts (&optional sort)
  "Print a list of all of the hosts known in the current namespace.
  SORT is one of NIL, :ALPHA, :CHAOS, or :IP."
  (let* ((hosts-alist (name:list-objects-from-attributes :class :HOST :brief t))
	 (non-aliased-hosts '()))
    (dolist (list hosts-alist)
      (let* ((name (car list)))
	(unless (string-equal name "SYS")
	  (let* ((object (name:lookup-object name :HOST :read-only t)))
	    (unless (name:get-attribute-value object :*alias-of*)
	      (push object non-aliased-hosts))))))
    (ecase sort
      (NIL  nil)
      (:ALPHA (setq non-aliased-hosts (sort non-aliased-hosts #'string-lessp :key #'name:object-name)))
      ((:IP :CHAOS) (setq non-aliased-hosts (sort non-aliased-hosts #'<
						  :key #'(lambda (x)
							   (or (second (assoc sort
									      (name:get-attribute-value x :addresses)))
							       0))))))
    (dolist (object non-aliased-hosts)
      (let* ((addresses (name:get-attribute-value object :addresses))
	     (ip (second (assoc :IP addresses)))
	     (chaos (second (assoc :CHAOS addresses)))
	     (aliases (remove "lm" (name:get-attribute-value object :aliases) :test #'string-equal)))
	(format t "~&~A~20t ~A~28t ~A~41t ~A~56t"
		(name:object-name object) (or chaos "-") (or ip "-") (if ip (ip:ip-address-display ip) "-"))
	(dolist (alias aliases)
	  (princ #\Space)
	  (princ alias))
	(terpri))))
  (values))


(defun generate-/usr/lib/aliases ()
  "Write on *STANDARD-OUTPUT* a line for each mailing list in the current namespace.
  These lines are suitable for inclusion in a Unix /usr/lib/aliases file."
  (let* ((mailing-list-alist (name:list-objects-from-attributes :class :MAILING-LIST :brief t)))
    (dolist (list mailing-list-alist)
      (let* ((name (car list))
	     (object (name:lookup-object name :MAILING-LIST :read-only t :merge-aliases nil))
	     (contents (name:get-attribute-value object :ADDRESS-LIST))
	     (alias-of (name:get-attribute-value object :*ALIAS-OF*)))
	(fresh-line)
	(princ (string-downcase name))
	(princ #\:)
	(if alias-of
	    (princ (string-downcase alias-of))
	    (let* ((first-p t))
	      (dolist (obj contents)
		(if first-p
		    (setq first-p nil)
		    (princ #\,))
		(princ (string-downcase obj))))))
      (terpri)))
  (values))
